/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "m4/adv_r/adv_inv.h"
#include "m4/riddle/inventory.h"
#include "m4/riddle/vars.h"
#include "m4/m4.h"

namespace M4 {
namespace Riddle {

struct InvObject {
	const char *_name;
	const char *_german;
	const char *_spanish;
	const char *_french;
	const char *_italian;
	int16 _scene, _cel, _cursor;
};

static const InvObject INVENTORY_ITEMS[] = {
	{ "SOLDIER'S HELMET",       "ARMEEHELM",					"CASCO DE SOLDADO",				"CASQUE DE SOLDAT",				"ELMETTO DEL SOLDATO",			203,  14,  10 },
	{ "BUCKET",					"EIMER",						"CUBO",							"SEAU",							"CESTO",						203,  15,  11 },
	{ "RIPLEY PHOTO",			"RIPLEYS FOTO",					"FOTO DE RIPLEY",				"PHOTO DE RIPLEY",				"FOTO DI RIPLEY",				998,  16,  12 },
	{ "LEAD PIPE",				"BLEIROHR",						"TUBO DE PLOMO",				"TUYAU DE PLOMB",				"TUBO",							207,  17,  13 },
	{ "SEVEN SPOKES",			"SIEBEN SPEICHEN",				"SIETE RADIOS",					"SEPT BARREAUX",				"SETTE PIOLI",					207,  18,  14 },
	{ "MALLET",					"SCHLEGEL",						"MACILLO",						"MAILLET",						"MAZZA",						204,  55,  21 },
	{ "GONG",					"GONG",							"GONG",							"GONG",							"GONG",							204,  56,  22 },
	{ "WHEELED TOY",			"SPIELZEUG AUF R\x01""DERN",	"JUGUETE CON RUEDAS",			"JOUET A ROUES",				"GIOCATTOLO",					504,  57,  53 },
	{ "CHARCOAL",				"HOLZKOHLE",					"CARB\xd3""N",					"CHARBON",						"CARBONE",						205,  58,  54 },
	{ "POST MARKED ENVELOPE",	"ABGESTEMPELTER UMSCHLAG",		"SOBRE",						"ENVELOPPE TIMBREE",			"BUSTA AFFRANCATA",				304,  25,  34 },
	{ "TURTLE",					"SCHILDKR\x03TE",				"TORTUGA",						"TORTUE",						"TARTARUGA",					999,  59,  55 },
	{ "TURTLE TREATS",			"SCHILDKR\x03TENBONBONS",		"GOLOSINA DE TORTUGA",			"FRIANDISES POUR TORTUE",		"CARAMELLA D'ACERO",			305,  60,  56 },
	{ "PLANK",					"PLANKE",						"TABL\xd3""N",					"PLANCHE",						"TAVOLA",						408,  26,  23 },
	{ "PASS",					"VISUM",						"PASE",							"PASSE",						"PERMESSO",						204,  61,  57 },
	{ "KEY",					"SCHL\x04SSEL",					"LLAVE",						"CLE",							"CHIAVE",						409,  27,  35 },
	{ "ACE OF SPADES CARD",		"PIK-AS-SPIELKARTE",			"AS DE PICAS",					"AS DE PIQUE",					"CARTA ASSO DI PICCHE",			406,  28,  24 },
	{ "STEP LADDER",			"TRITTLEITER",					"ESCALERA",						"ECHELLE",						"SCALA",						403,  62,  58 },
	{ "BILLIARD BALL",			"BILLARDKUGEL",					"BOLA DE BILLAR",				"BOULE DE BILLARD",				"PALLA DA BILIARDO",			406,  29,  25 },
	{ "CIGAR",					"ZIGARRE",						"PURO",							"CIGARE",						"SIGARO",						456,  30,  26 },
	{ "GLASS JAR",				"GLASGEF\x01SS",				"BOTELLA DE CRISTAL",			"POT EN VERRE",					"VASO DI VETRO",				407,  31,  27 },
	{ "ROMANOV EMERALD",		"ROMANOW-SMARAGD",				"ESMERALDA ROMANOV",			"EMERAUDE DES ROMANOV",			"SMERALDO ROMANOV",				407,  32,  28 },
	{ "CORK",					"KORKEN",						"CORCHO",						"BOUCHON",						"TAPPO",						407,  33,  29 },
	{ "LEVER KEY",				"HEBELSCHL\x04SSEL",			"PALANCA",						"LEVIER",						"BARRA",						407,  34,  30 },
	{ "EDGER",					"KANTENSCHNEIDER",				"\xda""TIL DE JARDINER\xcd""A",	"COUPE-BORDURE",				"VANGA",						408,  88,  84 },
	{ "SURGICAL TUBE",			"GUMMISCHLAUCH",				"TUBO QUIR\xda""RGICO",			"TUBE CHIRURGICAL",				"TUBO CHIRURGICO",				407,  45,  43 },
	{ "PUMP ROD",				"TAUCHKOLBEN",					"\xc9""MBOLO",					"TIGE DE PISTON",				"MANICO DELLA POMPA",			407,  35,  31 },
	{ "FAUCET PIPE",			"LEITUNG DES WASSERHAHNS",		"TUBER\xcd""A DEL GRIFO",		"COL DE CYGNE",					"RUBINETTO",					407,  46,  44 },
	{ "FAUCET HANDLE",			"GRIFF DES WASSERHAHNS",		"LLAVE DEL GRIFO",				"POIGNEES DE ROBINET",			"LEVA",							407,  36,  32 },
	{ "RUBBER PLUG",			"GUMMIPFROPFEN",				"CLAVIJA DE GOMA",				"BOUCHON EN CAOUTCHOUC",		"TAPPO DI GOMMA",				407,  37,  33 },
	{ "GARDEN HOSE",			"GARTENSCHLAUCH",				"MANGUERA",						"TUYAU D'ARROSAGE",				"TUBO PER ANNAFFIARE",			407,  38,  36 },
	{ "MENENDEZ'S LETTER",		"MENENDEZ' BRIEF",				"CARTA DE MEN\xc9""NDEZ",		"LETTRE DE MENENDEZ",			"LETTERA DI MENENDEZ",			407,  39,  37 },
	{ "KEYS",					"SCHL\x04SSELBUND",				"LLAVES",						"CLES",							"CHIAVI",						406,  40,  38 },
	{ "BROKEN WHEEL",			"ZERBROCHENES RAD",				"RUEDA ROTA",					"ROUE BRISEE",					"RUOTA ROTTA",					207,  63,  59 },
	{ "PUMP GRIPS",				"PUMPENHEBEL",					"AGARRADERA DE LA BOMBA",		"POIGNEES DE POMPE",			"IMPUGNATURA DELLA POMPA",		407,  41,  39 },
	{ "CLOCK FACING",			"ZIFFERBLATT",					"ESFERA DE RELOJ",				"FA\x0c""ADE PENDULE",			"QUADRANTE",					501,  47,  45 },
	{ "SHOVEL",					"SCHAUFEL",						"PALA",							"PELLE",						"PALA",							504,  64,  60 },
	{ "WOODEN LADDER",			"HOLZLEITER",					"ESCALERA DE MADERA",			"ECHELLE EN BOIS",				"SCALA",						504,  65,  61 },
	{ "ROPE",					"SEIL",							"CUERDA",						"CORDE",						"CORDA",						504,  66,  62 },
	{ "GREEN VINE",				"GR\x04NE RANKE",				"LIANA VERDE",					"LIERRE VERT",					"RAMPICANTE VERDE",				504,  67,  63 },
	{ "CRYSTAL SKULL",			"KRISTALLSCH\x01""DEL",			"CALAVERA DE CRISTAL",			"CRANE DE CRISTAL",				"TESCHIO DI CRISTALLO",			510,  42,  40 },
	{ "BROWN VINE",				"BRAUNE RANKE",					"LIANA MARR\xd3""N",			"LIERRE MARRON",				"RAMPICANTE SCURO",				504,  68,  64 },
	{ "PULL CORD",				"REISSLEINE",					"CORD\xd3""N DE ARRANQUE",		"CORDE DE DEMARRAGE",			"CINGHIA",						604,  69,  65 },
	{ "SPARK PLUG TOOL",		"Z\x04NDKERZENSCHL\x04SSEL",	"HERRAMIENTA PARA BUJ\xcd""AS",	"CLE A BOUGIE",					"CHIAVE SPECIALE",				604,  70,  66 },
	{ "RONGORONGO TABLET",		"RONGORONGO-TAFEL",				"TABLILLA RONGORONGO",			"TABLETTE RONGORONGO",			"TAVOLA RONGORONGO",			607,  71,  67 },
	{ "SLEEVE",					"\x01RMEL",						"MANGA",						"MANCHE",						"MANICA",						603,  72,  68 },
	{ "LIGHTER",				"FEUERZEUG",					"ENCENDEDOR",					"BRIQUET",						"ACCENDINO",					608,  73,  69 },
	{ "STICK AND SHELL MAP",	"STOCK- U. MUSCHEL-KARTE",		"MAPA DE PALO Y CONCHAS",		"CARTE",						"MAPPA CONCHIGLIA-BASTONCINI",	608,  74,  70 },
	{ "DRIFTWOOD STUMP",		"STUMPF AUS TREIBHOLZ",			"TOC\xd3""N",					"SOUCHE PETRIFIEE",				"CEPPO",						600,  75,  71 },
	{ "OBSIDIAN DISK",			"OBSIDIANSCHEIBE",				"DISCO DE OBSIDIANA",			"DISQUE OBSIDIENNE",			"DISCO DI OSSIDIANA",			605,  76,  72 },
	{ "SOLDIER'S SHIELD",		"ARMEESCHILD",					"ESCUDO DE SOLDADO",			"BOUCLIER DE SOLDAT",			"SCUDO DEL SOLDATO",			805,  77,  73 },
	{ "TWELVETREES' NOTE",		"TWELVETREES' NACHRICHT",		"NOTA DE TWELVETREES",			"NOTE DE TWELVETREES",			"MESSAGGIO DI TWELVETREES",		603,  89,  85 },
	{ "TWO SOLDIERS' SHIELDS",	"ZWEI ARMEESCHILDE",			"DOS ESCUDOS DE SOLDADO",		"DEUX BOUCLIERS DE SOLDAT",		"SCUDI DEI DUE SOLDATI",		805,  78,  74 },
	{ "WOODEN POST",			"HOLZPFOSTEN",					"TROZO DE VIGA",				"POTEAU EN BOIS",				"INSEGNA",						805,  79,  75 },
	{ "CRANK",					"KURBEL",						"MANIVELA",						"MANIVELLE",					"MANOVELLA",					806,  80,  76 },
	{ "WOODEN BEAM",			"HOLZBALKEN",					"VIGA DE MADERA",				"POUTRE EN BOIS",				"TRAVE",						844,  81,  77 },
	{ "EMPEROR'S SEAL",			"KAISERLICHES SIEGEL",			"SELLO DEL EMPERADOR",			"SCEAU DE L'EMPEREUR",			"SIGILLO DELL'IMPERATORE",		810,  82,  78 },
	{ "POLE",					"STANGE",						"POSTE",						"POTEAU",						"PALO",							603,  90,  86 },
	{ "REBUS AMULET",			"REBUS-AMULETT",				"AMULETO REBUS",				"AMULETTE-REBUS",				"AMULETO REBUS",				207,  19,  15 },
	{ "DRIFTWOOD PUFFIN",		"TREIBHOLZ-PAPAGEIENTAUCHER",	"FRAILECILLO DE MADERA",		"MACAREUX PETRIFIE",			"PUFFINO DI LEGNO",				608,  20,  16 },
	{ "SHRUNKEN HEAD",			"SCHRUMPFKOPF",					"CABEZA REDUCIDA",				"TETE REDUITE",					"TESTA RIMPICCIOLITA",			504,  43,  41 },
	{ "METAL RIM",				"METALLFELGE",					"CUBIERTA DE METAL",			"JANTE METALLIQUE",				"CERCHIO IN METALLO",			207,  23,  19 },
	{ "CHINESE YUAN",			"CHINESISCHE Y\x04""AN",		"YUANES CHINOS",				"YUAN CHINOIS",					"YUAN CINESI",					999,  22,  18 },
	{ "JOURNAL",				"TAGEBUCH",						"DIARIO",						"JOURNAL",						"GIORNALE",						998,  24,  20 },
	{ "SILVER BUTTERFLY",		"SILBERNER SCHMETTERLING",		"MARIPOSA DE PLATA",			"PAPILLON D'ARGENT",			"FARFALLA D'ARGENTO",			204,  44,  42 },
	{ "POSTAGE STAMP",			"BRIEFMARKE",					"SELLO DE CORREOS",				"TIMBRE-POSTE",					"FRANCOBOLLO",					999,  83,  79 },
	{ "GERMAN BANKNOTE",		"DEUTSCHE BANKNOTE",			"BILLETE ALEM\xc1""N",			"BILLET ALLEMAND",				"BANCONOTA TEDESCA",			405,  84,  80 },
	{ "WHALE BONE HORN",		"HORN AUS WALFISCHKNOCHEN",		"CUERNO DE BARBA DE BALLENA",	"CORNE EN OS DE BALEINE",		"CORNO DI OSSO",				608,  85,  81 },
	{ "CHISEL",					"MEISSEL",						"CINCEL",						"CISEAU",						"CESELLO",						709,  86,  82 },
	{ "INCENSE BURNER",			"WEIHRAUCHPFANNE",				"INCENSARIO",					"ENCENSOIR",					"INCENSIERE",					709,  87,  83 },
	{ "POMERANIAN MARKS",		"POMMERSCHE MARK",				"MARCOS POMERANOS",				"MARKS POMERANIENS",			"MARCHI POMERANIANI",			401,  22,  18 },
	{ "PERUVIAN INTI",			"PERUANISCHE INTI",				"INTIS PERUANOS",				"INTI PERUVIENS",				"INTI PERUVIANI",				501,  22,  18 },
	{ "SIKKIMESE RUPEE",		"RUPIEN AUS SIKKIM",			"RUPIAS SIKKIMESAS",			"ROUPIES DE SIKKIM",			"RUPIE SIKKIMESI",				701,  22,  18 },
	{ "SERENITY WHEEL",			"M\x04HLE DER GELASSENHEIT",	"MOLINO DE LA SERENIDAD",		"ROUE DE LA SERENITE",			"RUOTA DELLA SERENITA'",		703,  48,  46 },
	{ "PEACE WHEEL",			"M\x04HLE DES FRIEDENS",		"MOLINO DE LA PAZ",				"ROUE DE LA PAIX",				"RUOTA DELLA PACE",				704,  49,  47 },
	{ "INSIGHT WHEEL",			"M\x04HLE DES VERST\x01NDNISSES", "MOLINO DE LA PERSPICACIA",	"ROUE DE LA PERSPICACITE",		"RUOTA DELL'INTROSPEZIONE",		705,  50,  48 },
	{ "TRUTH WHEEL",			"M\x04HLE DER WAHRHEIT",		"MOLINO DE LA VERDAD",			"ROUE DE LA VERITE",			"RUOTA DELLA VERITA'",			705,  51,  49 },
	{ "WISDOM WHEEL",			"M\x04HLE DER WEISHEIT",		"MOLINO DE LA SABIDUR\xcd""A",	"ROUE DE LA SAGESSE",			"RUOTA DELLA SAGGEZZA",			706,  52,  50 },
	{ "CIGAR WITHOUT BAND",		"ZIGARRE OHNE BANDEROLE",		"PURO SIN VITOLA",				"CIGARE SANS BAGUE",			"SIGARO SENZA FASCETTA",		406,  53,  51 },
	{ "CIGAR BAND",				"ZIGARRENBANDEROLE",			"VITOLA",						"BAGUE DE CIGARE",				"FASCETTA DEL SIGARO",			406,  54,  52 },
	{ "FARMER'S SHOVEL",		"SPATEN",						"PALA DEL GRANJERO",			"PELLE  DU FERMIER",			"PALA DEL CONTADINO",			801,  91,  87 },
	{ "MATCH",					"STREICHHOLZ",					"CERILLA",						"ALLUMETTE",					"FIAMMIFERO",					801,  92,  88 },
	{ "LIT LIGHTER",			"ANGEZ\x04NDETES FEUERZEUG",	"ENCENDEDOR ENCENDIDO",			"BRIQUET ALLUME",				"ACCENDINO ACCESO",				999,  96,  92 },
	{ "VON SELTSAM'S NOTE",		"VON SELTSAMS NACHRICHT",		"NOTA DE VON SELTSAM",			"NOTE DE VON SELTSAM",			"MESSAGGIO DI VON SELTSAM",		999,  97,  93 },
	{ "PRAYER WHEEL BROCHURE",	"GEBETSM\x04HLEN-BROSCH\x04RE",	"FOLLETO MOLINO DE ORACI\xd3""N", "PUB SUR ROUE DE PRIERE",		"OPUSCOLO",						999,  98,  94 },
	{ "ENVELOPE",				"BRIEFUMSCHLAG",				"SOBRE",						"ENVELOPPE",					"BUSTA",						406,  99,  95 },
	{ "VON SELTSAM'S LETTER",	"VON SELTSAMS BRIEF",			"CARTA DE VON SELTSAM",			"LETTRE DE VON SELTSAM",		"LETTERA DI VON SELTSAM",		999, 100,  96 },
	{ "HORN/PULL CORD",			"HORN/REISSLEINE",				"CUERNO/CORD\xd3""N DE ARRANQUE", "CORNE/CORDE DE DEMAR.",		"CORNO/CINGHIA",				999, 101,  97 },
	{ "FAUCET PIPE/HOSE",		"LEITUNG/SCHLAUCH",				"TUBER\xcd""A/MANGUERA",		"COL DE CYGNE/TUYAU",			"RUBINETTO/TUBO",				999, 102,  98 },
	{ "FAUCET PIPE/HOSE/TUBE",	"HAHN/SCHLAUCH/GUMMIROHR",		"TUBER\xcd""A/MANGUERA/TUBO.",	"COL DE CYGNE/TUYAU/TUBE",		"RUBINETTO/TUBO/TUBO CHIR.",	999, 103,  99 },
	{ "JAR/CORK",				"GLAS/KORKEN",					"BOTELLA/CORCHO",				"POT/BOUCHON",					"VASO/TAPPO",					999, 104, 100 },
	{ "JAR/RUBBER PLUG",		"GLAS/GUMMIPFROPFEN",			"BOTELLA/CLAVIJA",				"POT/BOUCHON EN CAOUTCHOUC",	"VASO/TAPPO DI GOMMA",			999, 105, 101 },
	{ "JAR/GRIPS",				"GLAS/PUMPENHEBEL",				"BOTELLA/AGARRADERA",			"POT/POIGNEES",					"VASO/IMPUGNATURA",				999, 106, 102 },
	{ "JAR/PLUG/GRIPS",			"GLAS/PFROPFEN/PUMPENHEBEL",	"BOTELLA/CLAVIJA/AGARRADERA.",	"POT/BOUCHON/POIGNEES",			"VASO/TAPPO GOMMA /IMPUGNATURA", 999, 107, 103 },
	{ "JAR/PLUG/CORK/GRIPS",	"GLAS/PFROPFEN/KORK/HEBEL",		"BOT./CLAVIJA/CORCHO/AGARR.",	"POT/BOUCHONS/POIGNEES",		"VASO/TAPPO GOMMA/TAPPO/IMPUGNATURA", 999, 108, 104 },
	{ "JAR/CORK/PLUG",			"GLAS/KORKEN/PFROPFEN",			"BOTELLA/CORCHO/CLAVIJA",		"POT/BOUCHONS",					"VASO/TAPPO/TAPPO GOMMA",		999, 109, 105 },
	{ "JAR/CORK/GRIPS",			"GLAS/KORKEN/PUMPENHEBEL",		"BOTELLA/CORCHO/AGARRADERA",	"POT/BOUCHON/POIGNEES",			"VASO/TAPPO/IMPUGNATURA",		999, 110, 106 },
	{ "VINES",					"RANKEN",						"LIANAS",						"LIERRES",						"RAMPICANTI",					999, 111, 107 },
	{ "TUBE/HOSE",				"GUMMISCHLAUCH/SCHLAUCH",		"TUBO/MANGUERA",				"TUBE/TUYAU",					"TUBO CHIR./TUBO",				407, 112, 108 },
	{ "US DOLLARS",				"US-DOLLAR",					"D\xd3""LARES AMERICANOS",		"DOLLARS US",					"DOLLARI",						998,  21,  17 },
	{ "JAR/FAUCET PIPE",		"GLAS/HAHNLEITUNG",				"BOTELLA/TUBER\xcd""A",			"POT/COL DE CYGNE",				"VASO/RUBINETTO",				407, 113, 109 },
	{ "EMERALD/CORK",			"SMARAGD/KORKEN",				"ESMERALDA/CORCHO",				"EMERAUDE/BOUCHON",				"SMERALDO/TAPPO",				407, 114, 110 },
	{ "TWELVETREES' MAP",		"TWELVETREES' LANDKARTE",		"MAPA DE TWELVETREES",			"CARTE DE TWELVETREES",			"MAPPA DI TWELVETREES",			603,  95,  91 },
	{ "PERIODIC TABLE",			"PERIODENSYSTEM DER ELEMENTE",	"TABLA PERI\xd3""DICA",			"LISTE PERIODIQUE",				"TAVOLA PERIODICA",				407,  93,  89 },
	{ "LEVER KEY/PUMP ROD",		"HEBELSCHL\x04SSEL/PUMPENKOPF",	"PALANCA/\xc9""MBOLO",			"LEVIER/TIGE DE PISTON",		"BARRA/MANICO DELLA POMPA",		407, 115, 111 },
	{ "JAR/LEVER KEY",			"GLAS/HEBELSCHL\x04SSEL",		"BOTELLA/PALANCA",				"POT/LEVIER",					"VASO/BARRA",					407, 116, 112 },
	{ "PUMP ROD/GRIPS",			"PUMPENKOPF/PUMPENHEBEL",		"\xc9""MBOLO/AGARRADERA",		"TIGE DE PISTON/POIGNEES",		"MANICO/IMPUGNATURA",			407, 117, 113 },
	{ "JAR/PERIODIC TABLE",		"GLAS/PERIODENSYSTEM",			"BOTELLA/TABLA PERI\xd3""DICA",	"POT/LISTE PERIODIQUE",			"VASO/TAVOLA PERIODICA",		407, 118, 114 },
	{ "ROPE/GREEN VINE",		"SEIL/GR\x04NE RANKE",			"CUERDA/LIANA VERDE",			"CORDE/LIERRE VERT",			"CORDA/RAMPICANTE VERDE",		999, 119, 115 },
	{ "ROPE/VINES",				"SEIL/RANKEN",					"CUERDA/LIANAS",				"CORDE/LIERRE",					"CORDA/RAMPICANTI",				999, 120, 116 },
	{ "HORN/PULL CORD/WATER",	"HORN/REISSLEINE/WASSER",		"CUERNO/CORD\xd3""N/AGUA",		"CORNE/CORDE DEMAR./EAU",		"CORNO/CINGHIA/ACQUA",			999, 101,  97 },
	{ "LADDER/ROPE",			"LEITER/SEIL",					"ESCALERA/CUERDA",				"ECHELLE/CORDE",				"SCALA/CORDA",					999, 121, 117 },
	{ "LADDER/GREEN VINE",		"LEITER/GR\x04NE RANKE",		"ESCALERA/LIANA VERDE",			"ECHELLE/LIERRE VERT",			"SCALA/RAMPICANTE VERDE",		999, 122, 118 },
	{ "LADDER/VINES",			"LEITER/RANKE",					"ESCALERA/LIANAS",				"ECHELLE/LIERRE",				"SCALA/RAMPICANTI",				999, 123, 119 },
	{ "WITHERED GREEN VINE",	"VERWELKTE GR\x04NE RANKE",		"LIANA VERDE AJADA",			"LIERRE DESSECHE (VERT)",		"RAMPICANTE VERDE AVVIZZITO",	999,  67,  63 },
	{ "WITHERED VINES",			"VERWELKTE RANKEN",				"LIANAS AJADAS",				"LIERRE DESSECHE",				"RAMPICANTI AVVIZZITI",			999, 111, 107 },
	{ "VINE DUST",				"RANKENRESTE",					"POLVO DE LIANA",				"POUSSIERE DE LIERRE ",			"POLVERE DI RAMPICANTE",		999, 124, 120 },
	{ "JAR/PIPE/HOSE",			"GLAS/LEITUNG/SCHLAUCH",		"BOTELLA/TUBER\xcd""A/MANGUERA","POT/TUYAU D'ARROSAGE",			"VASO/RUBINETTO/TUBO",			999, 125, 121 },
	{ "LADDER/BROWN VINE",		"LEITER/BRAUNE RANKE",			"ESCALERA/LIANA MARR\xd3""N",	"ECHELLE/LIERRE MARRON",		"SCALA/RAMPICANTE SCURO",		999, 126, 122 },
	{ "ROPE/BROWN VINE",		"SEIL/BRAUNE RANKE",			"CUERDA/LIANA MARR\xd3""N",		"CORDE/LIERRE MARRON",			"CORDA/RAMPICANTE SCURO",		999, 127, 123 },
	{ "MESSAGE LOG",			"LOGBUCH F\x04R TELEGRAMME",	"LISTA DE MENSAJES",			"CARNET DE MESSAGES",			"REGISTRO MESSAGGI",			999,  94,  90 },
	{ "WITHERED BROWN VINE",	"VERWELKTE BRAUNE RANKE",		"LIANA MARR\xd3""N AJADA",		"LIERRE MARRON DESSECHE",		"RAMPICANTE SCURO AVVIZZITO",	999,  68,  64 },
	{ "TOMB MESSAGE",			"NACHRICHT AUS GRABMAL",		"MENSAJE EN TUMBA",				"MESSAGE DU TOMBEAU",			"MESSAGGIO DELLA TOMBA",		409, 128, 124 },
	{ "FAUCET PIPE/TUBE",		"HAHNLEITUNG/GUMMISCHLAUCH",	"TUBER\xcd""A/TUBO",			"COL DE CYGNE/TUBE",			"RUBINETTO/TUBO CHIR.",			999, 137, 125 },
	{ nullptr,                  nullptr,						nullptr,						nullptr,						nullptr,						  0,   0,   0 }
};

void Inventory::init() {
	Common::Language lang = g_engine->getLanguage();

	for (const InvObject *item = INVENTORY_ITEMS; item->_name; ++item) {
		const char *foreignName = item->_name;
		if (lang == Common::DE_DEU)
			foreignName = item->_german;
		else if (lang == Common::ES_ESP)
			foreignName = item->_spanish;
		else if (lang == Common::FR_FRA)
			foreignName = item->_french;
		else if (lang == Common::IT_ITA)
			foreignName = item->_italian;

		inv_register_thing(item->_name, foreignName, item->_scene, item->_cel, item->_cursor);

		_items.push_back(InventoryItem(item->_name, item->_scene));
	}
}

void Inventory::add(const Common::String &name, const Common::String &verb, int32 sprite, int32 cursor) {
	_GI(inventory)->add(name, verb, sprite, cursor);
	_GI(inventory)->_must_redraw_all = true;
}

void Inventory::set_scroll(int32 scroll) {
	_GI(inventory)->set_scroll(scroll);
}

void Inventory::remove(const Common::String &name) {
	_GI(inventory)->remove(name);
	_GI(inventory)->_must_redraw_all = true;
}

// Added for room 205
const char * Inventory::get_name(int32 id) {
	if (id < 0 || id >= INVENTORY_COUNT)
		return "";

	return INVENTORY_ITEMS[id]._name;
}

void Inventory::reset() {
	for (const InvObject *item = INVENTORY_ITEMS; item->_name; ++item)
		inv_move_object(item->_name, item->_scene);
}
} // namespace Riddle
} // namespace M4
