%feature("docstring") OT::MartinezSensitivityAlgorithm
R"RAW(Sensitivity analysis using Martinez method.

Available constructors:
    MartinezSensitivityAlgorithm(*inputDesign, outputDesign, N*)

    MartinezSensitivityAlgorithm(*distribution, N, model, computeSecondOrder*)

    MartinezSensitivityAlgorithm(*experiment, model, computeSecondOrder*)

Parameters
----------
inputDesign : :class:`~openturns.Sample`
    Design for the evaluation of sensitivity indices, obtained with the
    :class:`~openturns.SobolIndicesExperiment`:code:`.`:meth:`~openturns.SobolIndicesExperiment.generate`
    method
outputDesign : :class:`~openturns.Sample`
    Design for the evaluation of sensitivity indices, obtained as the evaluation
    of a Function (model) on the previous inputDesign
distribution : :class:`~openturns.Distribution`
    Input probabilistic model.
    Should have independent copula
experiment : :class:`~openturns.WeightedExperiment`
    Experiment for the generation of two independent samples.
N : int
    Size of samples to generate
model : :class:`~openturns.Function`
    Model to evaluate input samples.
computeSecondOrder : bool
    If True, design that will be generated contains elements for the evaluation
    of second order indices.

Notes
-----
This class analyzes the influence of each component of a random vector
:math:`\vect{X} = \left( X_1, \ldots, X_{n_X} \right)` on a random vector
:math:`\vect{Y} = \left( Y_1, \ldots, Y_{n_Y} \right)` by computing Sobol' indices (see also [sobol1993]_).
The [martinez2011]_ method is used to estimate both first
and total order indices.
Notations are defined in the documentation page of the :class:`~openturns.SobolIndicesAlgorithm` class.
The estimators of the first and total order Sobol' indices used by this class are respectively:

.. math::

   \hat{S}_i & = \tilde{\rho}_n\left(\tilde{\vect{g}}(\mat{B}), \tilde{\vect{g}}(\mat{E}^i)\right) \\
   \widehat{S}_i^T & = 1 - \tilde{\rho}_n\left(\tilde{\vect{g}}(\mat{A}), \tilde{\vect{g}}(\mat{E}^i)\right) \\

where :math:`\tilde{\rho}_n` is the modified sample correlation (under the assumption that the
sample has a zero mean) defined, for any pair of vectors
:math:`(\vect{z}, \vect{z}') \in \Rset^N \times \Rset^N`, by:

.. math::

   \tilde{\rho}_n \left(\vect{z}, \vect{z}'\right) = 
   \frac{\sum_{k=1}^N z_k z_k'}
   {\sqrt{\sum_{k=1}^N z_k^2} \sqrt{\sum_{k=1}^N \left( z_k' \right)^2}}

where :math:`\tilde{\vect{g}}` is the centered model based on the sample.

The class constructor ``MartinezSensitivityAlgorithm(inputDesign, outputDesign, N)``
requires a specific structure for the ``outputDesign``, and therefore for the  ``inputDesign``.
The latter should be generated using :class:`~openturns.SobolIndicesExperiment`
(see example below). Otherwise, results will be **worthless**.



See also
--------
SobolIndicesAlgorithm

Examples
--------

Estimate first and total order Sobol' indices:

>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> formula = ['sin(pi_*X1)+7*sin(pi_*X2)^2+0.1*(pi_*X3)^4*sin(pi_*X1)']
>>> model = ot.SymbolicFunction(['X1', 'X2', 'X3'], formula)
>>> distribution = ot.JointDistribution([ot.Uniform(-1.0, 1.0)] * 3)
>>> # Define designs to pre-compute
>>> size = 10000
>>> inputDesign = ot.SobolIndicesExperiment(distribution, size).generate()
>>> outputDesign = model(inputDesign)
>>> # sensitivity analysis algorithm
>>> sensitivityAnalysis = ot.MartinezSensitivityAlgorithm(inputDesign, outputDesign, size)
>>> print(sensitivityAnalysis.getFirstOrderIndices())
[0.308902,0.459187,0.00683867]
>>> print(sensitivityAnalysis.getTotalOrderIndices())
[0.567786,0.430754,0.244293])RAW"

