#include "ggml.h"
#include "ggml-cpu.h"
#include "ggml-alloc.h"
#include "ggml-backend.h"

#ifdef GGML_USE_CUDA
#include "ggml-cuda.h"
#endif

#ifdef GGML_USE_METAL
#include "ggml-metal.h"
#endif

#include <cstdio>
#include <cstring>
#include <fstream>
#include <map>
#include <string>
#include <vector>
#include <cmath>
#include <iostream>

static void ggml_log_callback_default(ggml_log_level level, const char * text, void * user_data) {
    (void) level;
    (void) user_data;
    fputs(text, stderr);
    fflush(stderr);
}

struct test_model {
    struct ggml_tensor * a_0;
    struct ggml_tensor * b_0;

    struct ggml_tensor * a_1;
    struct ggml_tensor * b_1;

    struct ggml_tensor * a_2;
    struct ggml_tensor * b_2;

    struct ggml_tensor * a_3;
    struct ggml_tensor * b_3;

    struct ggml_tensor * a_4;
    struct ggml_tensor * b_4;


    ggml_backend_t backend = NULL;
    ggml_backend_buffer_t buffer;
    struct ggml_context * ctx;
};

void load_model(test_model & model, bool use_gpu = false) {


    float adata_0[] = {1,2,3};
    float bdata_0[] = {1,2};

    float adata_1[] = {1,2,3,3,2,1};
    float bdata_1[] = {2,3,1,1,3,2};

    float adata_2[] =  {3,2,1,1,2,3,1,2,3,3,2,1};
    float bdata_2[] =  {2,3,1,1,3,2};

    float data[16*32*32];
    for (int i = 0; i < 16*32*32; ++i) {
        data[i] = (float)(i%1024);
    }




    size_t buffer_size = 0;
    {
        buffer_size += 3* ggml_type_size(GGML_TYPE_F32); // tensor a_0
        buffer_size += 2* ggml_type_size(GGML_TYPE_F32); // tensor b_0

        buffer_size += 6* ggml_type_size(GGML_TYPE_F32); // tensor a_1
        buffer_size += 6* ggml_type_size(GGML_TYPE_F32); // tensor b_1

        buffer_size += 12* ggml_type_size(GGML_TYPE_F32); // tensor a_2
        buffer_size += 6* ggml_type_size(GGML_TYPE_F32); // tensor b_2

        buffer_size += 2 * 3 * 2 * ggml_type_size(GGML_TYPE_F32); // tensor a_3
        buffer_size += 3 * 2* ggml_type_size(GGML_TYPE_F32); // tensor b_3

        buffer_size += 16 * 32 * 32 * ggml_type_size(GGML_TYPE_F32); // tensor a_4
        buffer_size += 197 * 32* ggml_type_size(GGML_TYPE_F32); // tensor b_4

        buffer_size += 1024;
    }

    printf("%s: ggml tensor size    = %d bytes\n", __func__, (int) sizeof(ggml_tensor));
    printf("%s: backend buffer size = %0.2f MB\n", __func__, (buffer_size/ 1024.f/ 1024.f));

    int num_tensors = 10;
    struct ggml_init_params params {
            /*.mem_size   =*/ ggml_tensor_overhead() * num_tensors,
            /*.mem_buffer =*/ NULL,
            /*.no_alloc   =*/ true,
    };

    ggml_log_set(ggml_log_callback_default, nullptr);

    // initialize the backend
#ifdef GGML_USE_CUDA
    if (use_gpu) {
        fprintf(stderr, "%s: using CUDA backend\n", __func__);
        model.backend = ggml_backend_cuda_init(0);
        if (!model.backend) {
            fprintf(stderr, "%s: ggml_backend_cuda_init() failed\n", __func__);
        }
    }
#endif

#ifdef GGML_USE_METAL
    if (use_gpu) {
        fprintf(stderr, "%s: using Metal backend\n", __func__);
        model.backend = ggml_backend_metal_init();
        if (!model.backend) {
            fprintf(stderr, "%s: ggml_backend_metal_init() failed\n", __func__);
        }
    }
#endif

    if(!model.backend) {
        // fallback to CPU backend
        model.backend = ggml_backend_cpu_init();
    }

    model.buffer = ggml_backend_alloc_buffer(model.backend, buffer_size);

    // create context
    model.ctx = ggml_init(params);

    // create tensors
    model.a_0 = ggml_new_tensor_1d(model.ctx, GGML_TYPE_F32, 3);
    model.b_0 = ggml_new_tensor_1d(model.ctx, GGML_TYPE_F32, 2);


    model.a_1 = ggml_new_tensor_3d(model.ctx, GGML_TYPE_F32, 3,1,2);
    model.b_1 = ggml_new_tensor_2d(model.ctx, GGML_TYPE_F32, 3,2);

    model.a_2 = ggml_new_tensor_3d(model.ctx, GGML_TYPE_F32, 3,2,2);
    model.b_2 = ggml_new_tensor_2d(model.ctx, GGML_TYPE_F32, 3,2);

    model.a_3 = ggml_new_tensor_3d(model.ctx, GGML_TYPE_F32, 2,3,2);
    model.b_3 = ggml_new_tensor_2d(model.ctx, GGML_TYPE_F32, 3,2);

    model.a_4 = ggml_new_tensor_3d(model.ctx, GGML_TYPE_F32, 16,32,32);
    model.b_4 = ggml_new_tensor_2d(model.ctx, GGML_TYPE_F32, 197,32);


    // create a allocator
    ggml_tallocr alloc = ggml_tallocr_new(model.buffer);

    // alloc memory
    ggml_tallocr_alloc(&alloc, model.a_0);

    // load data to buffer
    if(ggml_backend_is_cpu(model.backend)) {
        memcpy(model.a_0->data, adata_0, ggml_nbytes(model.a_0));
    } else {
        ggml_backend_tensor_set(model.a_0, adata_0, 0, ggml_nbytes(model.a_0));
    }


    // alloc memory
    ggml_tallocr_alloc(&alloc, model.a_1);

    // load data to buffer
    if(ggml_backend_is_cpu(model.backend)) {
        memcpy(model.a_1->data, adata_1, ggml_nbytes(model.a_1));
    } else {
        ggml_backend_tensor_set(model.a_1, adata_1, 0, ggml_nbytes(model.a_1));
    }

     // alloc memory
    ggml_tallocr_alloc(&alloc, model.a_2);

    // load data to buffer
    if(ggml_backend_is_cpu(model.backend)) {
        memcpy(model.a_2->data, adata_2, ggml_nbytes(model.a_2));
    } else {
        ggml_backend_tensor_set(model.a_2, adata_2, 0, ggml_nbytes(model.a_2));
    }

      // alloc memory
    ggml_tallocr_alloc(&alloc, model.a_3);

    // load data to buffer
    if(ggml_backend_is_cpu(model.backend)) {
        memcpy(model.a_3->data, data, ggml_nbytes(model.a_3));
    } else {
        ggml_backend_tensor_set(model.a_3, data, 0, ggml_nbytes(model.a_3));
    }


      // alloc memory
    ggml_tallocr_alloc(&alloc, model.a_4);

    // load data to buffer
    if(ggml_backend_is_cpu(model.backend)) {
        memcpy(model.a_4->data, data, ggml_nbytes(model.a_4));
    } else {
        ggml_backend_tensor_set(model.a_4, data, 0, ggml_nbytes(model.a_4));
    }



    // alloc memory
    ggml_tallocr_alloc(&alloc, model.b_0);

    if(ggml_backend_is_cpu(model.backend)
#ifdef GGML_USE_METAL
                || ggml_backend_is_metal(model.backend)
#endif
    ) {
        memcpy(model.b_0->data, bdata_0, ggml_nbytes(model.b_0));
    } else {
        ggml_backend_tensor_set(model.b_0, bdata_0, 0, ggml_nbytes(model.b_0));
    }

    // alloc memory
    ggml_tallocr_alloc(&alloc, model.b_1);

    if(ggml_backend_is_cpu(model.backend)
#ifdef GGML_USE_METAL
                || ggml_backend_is_metal(model.backend)
#endif
    ) {
        memcpy(model.b_1->data, bdata_1, ggml_nbytes(model.b_1));
    } else {
        ggml_backend_tensor_set(model.b_1, bdata_1, 0, ggml_nbytes(model.b_1));
    }

     // alloc memory
    ggml_tallocr_alloc(&alloc, model.b_2);

    if(ggml_backend_is_cpu(model.backend)
#ifdef GGML_USE_METAL
                || ggml_backend_is_metal(model.backend)
#endif
    ) {
        memcpy(model.b_2->data, bdata_2, ggml_nbytes(model.b_2));
    } else {
        ggml_backend_tensor_set(model.b_2, bdata_2, 0, ggml_nbytes(model.b_2));
    }

    // alloc memory
    ggml_tallocr_alloc(&alloc, model.b_3);

    if(ggml_backend_is_cpu(model.backend)
#ifdef GGML_USE_METAL
                || ggml_backend_is_metal(model.backend)
#endif
    ) {
        memcpy(model.b_3->data, data, ggml_nbytes(model.b_3));
    } else {
        ggml_backend_tensor_set(model.b_3, data, 0, ggml_nbytes(model.b_3));
    }


   // alloc memory
    ggml_tallocr_alloc(&alloc, model.b_4);

    if(ggml_backend_is_cpu(model.backend)
#ifdef GGML_USE_METAL
                || ggml_backend_is_metal(model.backend)
#endif
    ) {
        memcpy(model.b_4->data, data, ggml_nbytes(model.b_4));
    } else {
        ggml_backend_tensor_set(model.b_4, data, 0, ggml_nbytes(model.b_4));
    }


}

struct ggml_cgraph * build_graph(const test_model& model) {
    static size_t buf_size = ggml_tensor_overhead()*GGML_DEFAULT_GRAPH_SIZE + ggml_graph_overhead();
    static std::vector<uint8_t> buf(buf_size);

    struct ggml_init_params params0 = {
        /*.mem_size   =*/ buf_size,
        /*.mem_buffer =*/ buf.data(),
        /*.no_alloc   =*/ true, // the tensors will be allocated later by ggml_gallocr_alloc_graph()
    };

    // create a temporally context to build the graph
    struct ggml_context * ctx0 = ggml_init(params0);

    struct ggml_cgraph  * gf = ggml_new_graph(ctx0);

    int s0 = 1;
    int p0 = 0;
    int d0 = 1;

    struct ggml_tensor* conv1d_transpose_res_0 = ggml_conv_transpose_1d(ctx0, model.a_0, model.b_0, s0, p0, d0);
    ggml_set_name(conv1d_transpose_res_0, "conv1d_transpose_res_0");
    ggml_build_forward_expand(gf, conv1d_transpose_res_0);


    s0 = 1;
    p0 = 0;
    d0 = 1;

    struct ggml_tensor* conv1d_transpose_res_1 = ggml_conv_transpose_1d(ctx0, model.a_1, model.b_1, s0, p0, d0);
    ggml_set_name(conv1d_transpose_res_1, "conv1d_transpose_res_1");
    ggml_build_forward_expand(gf, conv1d_transpose_res_1);

    s0 = 1;
    p0 = 0;
    d0 = 1;

    struct ggml_tensor* conv1d_transpose_res_2 = ggml_conv_transpose_1d(ctx0, model.a_2, model.b_2, s0, p0, d0);
    ggml_set_name(conv1d_transpose_res_2, "conv1d_transpose_res_2");
    ggml_build_forward_expand(gf, conv1d_transpose_res_2);

    s0 = 2;
    p0 = 0;
    d0 = 1;

    struct ggml_tensor* conv1d_transpose_res_3 = ggml_conv_transpose_1d(ctx0, model.a_2, model.b_2, s0, p0, d0);
    ggml_set_name(conv1d_transpose_res_3, "conv1d_transpose_res_3");
    ggml_build_forward_expand(gf, conv1d_transpose_res_3);

    s0 = 1;
    p0 = 0;
    d0 = 1;

    struct ggml_tensor* conv1d_transpose_res_4 = ggml_conv_transpose_1d(ctx0, model.a_3, model.b_3, s0, p0, d0);
    ggml_set_name(conv1d_transpose_res_4, "conv1d_transpose_res_4");
    ggml_build_forward_expand(gf, conv1d_transpose_res_4);

    s0 = 2;
    p0 = 0;
    d0 = 1;

    struct ggml_tensor* conv1d_transpose_res_5 = ggml_conv_transpose_1d(ctx0, model.a_3, model.b_3, s0, p0, d0);
    ggml_set_name(conv1d_transpose_res_5, "conv1d_transpose_res_5");
    ggml_build_forward_expand(gf, conv1d_transpose_res_5);

    s0 = 3;
    p0 = 0;
    d0 = 1;

    struct ggml_tensor* conv1d_transpose_res_6 = ggml_conv_transpose_1d(ctx0, model.a_3, model.b_3, s0, p0, d0);
    ggml_set_name(conv1d_transpose_res_6, "conv1d_transpose_res_6");
    ggml_build_forward_expand(gf, conv1d_transpose_res_6);


    s0 = 8;
    p0 = 0;
    d0 = 1;

    struct ggml_tensor* conv1d_transpose_res_7 = ggml_conv_transpose_1d(ctx0, model.a_4, model.b_4, s0, p0, d0);
    ggml_set_name(conv1d_transpose_res_7, "conv1d_transpose_res_7");
    ggml_build_forward_expand(gf, conv1d_transpose_res_7);



    // delete the temporally context used to build the graph
    ggml_free(ctx0);
    return gf;
}

struct ggml_cgraph* compute_graph(const test_model & model, ggml_gallocr_t allocr) {
    struct ggml_cgraph * gf = build_graph(model);

    // allocate tensors
    ggml_gallocr_alloc_graph(allocr, gf);
    int n_threads = 1;

    if (ggml_backend_is_cpu(model.backend)) {
        ggml_backend_cpu_set_n_threads(model.backend, n_threads);
    }

    ggml_backend_graph_compute(model.backend, gf);

    //ggml_graph_print(gf);

    return gf;
}

int main(void)
{
    ggml_time_init();

    test_model model;
    load_model(model, true);

    ggml_gallocr_t allocr = NULL;

    {
        allocr = ggml_gallocr_new(ggml_backend_get_default_buffer_type(model.backend));

        //create the worst case graph for memory usage estimation
        struct ggml_cgraph * gf = build_graph(model);

        // compute the required memory
        ggml_gallocr_reserve(allocr, gf);
        size_t mem_size = ggml_gallocr_get_buffer_size(allocr, 0);
        fprintf(stderr, "%s: compute buffer size: %.2f MB\n", __func__, mem_size/1024.0f/1024.0f);
    }

    struct ggml_cgraph * gf_res = compute_graph(model, allocr);

    struct ggml_tensor * conv1d_transpose_res_0 = NULL;

    for(int i = 0; i < ggml_graph_n_nodes(gf_res); i++) {
       if(strcmp(ggml_get_name(ggml_graph_node(gf_res, i)), "conv1d_transpose_res_0") == 0) {
            conv1d_transpose_res_0 = ggml_graph_node(gf_res, i);
        }
    }

    std::vector<float> conv1d_transpose_data_0(ggml_nelements(conv1d_transpose_res_0));

    ggml_backend_tensor_get(conv1d_transpose_res_0, conv1d_transpose_data_0.data(), 0, ggml_nbytes(conv1d_transpose_res_0));

    const int n_conv_transpose_1d_test_0 = 4;

    float expected_conv1d_0[n_conv_transpose_1d_test_0] = {
       1.00f,4.00f,7.00f,6.00f
    };


    struct ggml_tensor * conv1d_transpose_res_1 = NULL;

    for(int i = 0; i < ggml_graph_n_nodes(gf_res); i++) {
       if(strcmp(ggml_get_name(ggml_graph_node(gf_res, i)), "conv1d_transpose_res_1") == 0) {
            conv1d_transpose_res_1 = ggml_graph_node(gf_res, i);
        }
    }

    std::vector<float> conv1d_transpose_data_1(ggml_nelements(conv1d_transpose_res_1));

    ggml_backend_tensor_get(conv1d_transpose_res_1, conv1d_transpose_data_1.data(), 0, ggml_nbytes(conv1d_transpose_res_1));





    const int n_conv_transpose_1d_test_1 = 5;

    float expected_conv1d_1[n_conv_transpose_1d_test_1] =
       {5.0f, 18.0f, 26.0f, 18.0f,  5.0f};


    struct ggml_tensor * conv1d_transpose_res_2 = NULL;

    for(int i = 0; i < ggml_graph_n_nodes(gf_res); i++) {
       if(strcmp(ggml_get_name(ggml_graph_node(gf_res, i)), "conv1d_transpose_res_2") == 0) {
            conv1d_transpose_res_2 = ggml_graph_node(gf_res, i);
        }
    }

    std::vector<float> conv1d_transpose_data_2(ggml_nelements(conv1d_transpose_res_2));

    ggml_backend_tensor_get(conv1d_transpose_res_2, conv1d_transpose_data_2.data(), 0, ggml_nbytes(conv1d_transpose_res_2));


    const int n_conv_transpose_1d_test_2 = 10;

    float expected_conv1d_2[n_conv_transpose_1d_test_2] =
       {7.0f, 18.0f, 22.0f, 18.0f,  7.0f,
         5.0f, 18.0f, 26.0f, 18.0f,  5.0f};

    struct ggml_tensor * conv1d_transpose_res_3 = NULL;

    for(int i = 0; i < ggml_graph_n_nodes(gf_res); i++) {
       if(strcmp(ggml_get_name(ggml_graph_node(gf_res, i)), "conv1d_transpose_res_3") == 0) {
            conv1d_transpose_res_3 = ggml_graph_node(gf_res, i);
        }
    }

    std::vector<float> conv1d_transpose_data_3(ggml_nelements(conv1d_transpose_res_3));

    ggml_backend_tensor_get(conv1d_transpose_res_3, conv1d_transpose_data_3.data(), 0, ggml_nbytes(conv1d_transpose_res_3));


    const int n_conv_transpose_1d_test_3 = 14;

    float expected_conv1d_3[n_conv_transpose_1d_test_3] =
       {7.0f,  6.0f, 17.0f, 12.0f, 17.0f,  6.0f,  7.0f
         ,5.0f,  6.0f, 19.0f, 12.0f, 19.0f,  6.0f,  5.0f};


    struct ggml_tensor * conv1d_transpose_res_4 = NULL;

    for(int i = 0; i < ggml_graph_n_nodes(gf_res); i++) {
       if(strcmp(ggml_get_name(ggml_graph_node(gf_res, i)), "conv1d_transpose_res_4") == 0) {
            conv1d_transpose_res_4 = ggml_graph_node(gf_res, i);
        }
    }

    std::vector<float> conv1d_transpose_data_4(ggml_nelements(conv1d_transpose_res_4));

    ggml_backend_tensor_get(conv1d_transpose_res_4, conv1d_transpose_data_4.data(), 0, ggml_nbytes(conv1d_transpose_res_4));


    const int n_conv_transpose_1d_test_4 = 12;

    float expected_conv1d_4[3*4] = {
        18.0, 45.0, 59.0, 37.0,
        24.0, 61.0, 83.0, 51.0,
        30.0, 77.0, 107.0, 65.0
    };

    struct ggml_tensor * conv1d_transpose_res_5 = NULL;

    for(int i = 0; i < ggml_graph_n_nodes(gf_res); i++) {
       if(strcmp(ggml_get_name(ggml_graph_node(gf_res, i)), "conv1d_transpose_res_5") == 0) {
            conv1d_transpose_res_5 = ggml_graph_node(gf_res, i);
        }
    }

    std::vector<float> conv1d_transpose_data_5(ggml_nelements(conv1d_transpose_res_5));

    ggml_backend_tensor_get(conv1d_transpose_res_5, conv1d_transpose_data_5.data(), 0, ggml_nbytes(conv1d_transpose_res_5));


    const int n_conv_transpose_1d_test_5 = 18;

    float expected_conv1d_5[3*6] = {
        18.0, 21.0, 24.0, 29.0, 30.0, 37.0,
        24.0, 27.0, 34.0, 39.0, 44.0, 51.0,
        30.0, 33.0, 44.0, 49.0, 58.0, 65.0
        };

    struct ggml_tensor * conv1d_transpose_res_6 = NULL;

    for(int i = 0; i < ggml_graph_n_nodes(gf_res); i++) {
       if(strcmp(ggml_get_name(ggml_graph_node(gf_res, i)), "conv1d_transpose_res_6") == 0) {
            conv1d_transpose_res_6 = ggml_graph_node(gf_res, i);
        }
    }

    std::vector<float> conv1d_transpose_data_6(ggml_nelements(conv1d_transpose_res_6));

    ggml_backend_tensor_get(conv1d_transpose_res_6, conv1d_transpose_data_6.data(), 0, ggml_nbytes(conv1d_transpose_res_6));


    const int n_conv_transpose_1d_test_6 = 24;

    float expected_conv1d_6[3*8] = {
        18.0, 21.0, 0.0, 24.0, 29.0, 0.0, 30.0, 37.0,
        24.0, 27.0, 0.0, 34.0, 39.0, 0.0, 44.0, 51.0,
        30.0, 33.0, 0.0, 44.0, 49.0, 0.0, 58.0, 65.0};



    struct ggml_tensor * conv1d_transpose_res_7 = NULL;

    for(int i = 0; i < ggml_graph_n_nodes(gf_res); i++) {
       if(strcmp(ggml_get_name(ggml_graph_node(gf_res, i)), "conv1d_transpose_res_7") == 0) {
            conv1d_transpose_res_7 = ggml_graph_node(gf_res, i);
        }
    }

    std::vector<float> conv1d_transpose_data_7(ggml_nelements(conv1d_transpose_res_7));

    ggml_backend_tensor_get(conv1d_transpose_res_7, conv1d_transpose_data_7.data(), 0, ggml_nbytes(conv1d_transpose_res_7));


    const int n_conv_transpose_1d_test_7 = 32*1584;

    float expected_conv1d_7[32*1584] = {4325376.0,4341168.0,4356960.0,4372752.0,4388544.0,4404336.0,4420128.0,4435920.0,8785280.0,8816896.0,8848512.0,8880128.0,8911744.0,8943360.0,8974976.0,9006592.0,8801920.0,8833600.0,8865280.0,8896960.0,8928640.0,8960320.0,8992000.0,9023680.0,8818560.0,8850304.0,8882048.0,8913792.0,8945536.0,8977280.0,9009024.0,9040768.0,8835200.0,8867008.0,8898816.0,8930624.0,8962432.0,8994240.0,9026048.0,9057856.0,8851840.0,8883712.0,8915584.0,8947456.0,8979328.0,9011200.0,9043072.0,9074944.0,8868480.0,8900416.0,8932352.0,8964288.0,8996224.0,9028160.0,9060096.0,9092032.0,8885120.0,8917120.0,8949120.0,8981120.0,9013120.0,9045120.0,9077120.0,9109120.0,8901760.0,8933824.0,8965888.0,8997952.0,9030016.0,9062080.0,9094144.0,9126208.0,8918400.0,8950528.0,8982656.0,9014784.0,9046912.0,9079040.0,9111168.0,9143296.0,8935040.0,8967232.0,8999424.0,9031616.0,9063808.0,9096000.0,9128192.0,9160384.0,8951680.0,8983936.0,9016192.0,9048448.0,9080704.0,9112960.0,9145216.0,9177472.0,8968320.0,9000640.0,9032960.0,9065280.0,9097600.0,9129920.0,9162240.0,9194560.0,8984960.0,9017344.0,9049728.0,9082112.0,9114496.0,9146880.0,9179264.0,9211648.0,9001600.0,9034048.0,9066496.0,9098944.0,9131392.0,9163840.0,9196288.0,9228736.0,9018240.0,9050752.0,9083264.0,9115776.0,9148288.0,9180800.0,9213312.0,9245824.0,9034880.0,9067456.0,9100032.0,9132608.0,9165184.0,9197760.0,9230336.0,9262912.0,9051520.0,9084160.0,9116800.0,9149440.0,9182080.0,9214720.0,9247360.0,9280000.0,9068160.0,9100864.0,9133568.0,9166272.0,9198976.0,9231680.0,9264384.0,9297088.0,9084800.0,9117568.0,9150336.0,9183104.0,9215872.0,9248640.0,9281408.0,9314176.0,9101440.0,9134272.0,9167104.0,9199936.0,9232768.0,9265600.0,9298432.0,9331264.0,9118080.0,9150976.0,9183872.0,9216768.0,9249664.0,9282560.0,9315456.0,9348352.0,9134720.0,9167680.0,9200640.0,9233600.0,9266560.0,9299520.0,9332480.0,9365440.0,9151360.0,9184384.0,9217408.0,9250432.0,9283456.0,9316480.0,9349504.0,9382528.0,9168000.0,9201088.0,9234176.0,9267264.0,9300352.0,9333440.0,9366528.0,9399616.0,9184640.0,9217792.0,9250944.0,9284096.0,9317248.0,9350400.0,9383552.0,9416704.0,9201280.0,9234496.0,9267712.0,9300928.0,9334144.0,9367360.0,9400576.0,9433792.0,9217920.0,9251200.0,9284480.0,9317760.0,9351040.0,9384320.0,9417600.0,9450880.0,9234560.0,9267904.0,9301248.0,9334592.0,9367936.0,9401280.0,9434624.0,9467968.0,9251200.0,9284608.0,9318016.0,9351424.0,9384832.0,9418240.0,9451648.0,9485056.0,9267840.0,9301312.0,9334784.0,9368256.0,9401728.0,9435200.0,9468672.0,9502144.0,9284480.0,9318016.0,9351552.0,9385088.0,9418624.0,9452160.0,9485696.0,9519232.0,9301120.0,9334720.0,9368320.0,9401920.0,9435520.0,9469120.0,9502720.0,9536320.0,9317760.0,9351424.0,9385088.0,9418752.0,9452416.0,9486080.0,9519744.0,9553408.0,9334400.0,9368128.0,9401856.0,9435584.0,9469312.0,9503040.0,9536768.0,9570496.0,9351040.0,9384832.0,9418624.0,9452416.0,9486208.0,9520000.0,9553792.0,9587584.0,9367680.0,9401536.0,9435392.0,9469248.0,9503104.0,9536960.0,9570816.0,9604672.0,8860032.0,8892928.0,8925824.0,8958720.0,8991616.0,9024512.0,9057408.0,9090304.0,8344192.0,8376128.0,8408064.0,8440000.0,8471936.0,8503872.0,8535808.0,8567744.0,7836544.0,7867520.0,7898496.0,7929472.0,7960448.0,7991424.0,8022400.0,8053376.0,7320704.0,7350720.0,7380736.0,7410752.0,7440768.0,7470784.0,7500800.0,7530816.0,7337344.0,7367424.0,7397504.0,7427584.0,7457664.0,7487744.0,7517824.0,7547904.0,7353984.0,7384128.0,7414272.0,7444416.0,7474560.0,7504704.0,7534848.0,7564992.0,7370624.0,7400832.0,7431040.0,7461248.0,7491456.0,7521664.0,7551872.0,7582080.0,7387264.0,7417536.0,7447808.0,7478080.0,7508352.0,7538624.0,7568896.0,7599168.0,7403904.0,7434240.0,7464576.0,7494912.0,7525248.0,7555584.0,7585920.0,7616256.0,7420544.0,7450944.0,7481344.0,7511744.0,7542144.0,7572544.0,7602944.0,7633344.0,7437184.0,7467648.0,7498112.0,7528576.0,7559040.0,7589504.0,7619968.0,7650432.0,7453824.0,7484352.0,7514880.0,7545408.0,7575936.0,7606464.0,7636992.0,7667520.0,7470464.0,7501056.0,7531648.0,7562240.0,7592832.0,7623424.0,7654016.0,7684608.0,7487104.0,7517760.0,7548416.0,7579072.0,7609728.0,7640384.0,7671040.0,7701696.0,7503744.0,7534464.0,7565184.0,7595904.0,7626624.0,7657344.0,7688064.0,7718784.0,7520384.0,7551168.0,7581952.0,7612736.0,7643520.0,7674304.0,7705088.0,7735872.0,7537024.0,7567872.0,7598720.0,7629568.0,7660416.0,7691264.0,7722112.0,7752960.0,7553664.0,7584576.0,7615488.0,7646400.0,7677312.0,7708224.0,7739136.0,7770048.0,7570304.0,7601280.0,7632256.0,7663232.0,7694208.0,7725184.0,7756160.0,7787136.0,7586944.0,7617984.0,7649024.0,7680064.0,7711104.0,7742144.0,7773184.0,7804224.0,7603584.0,7634688.0,7665792.0,7696896.0,7728000.0,7759104.0,7790208.0,7821312.0,7620224.0,7651392.0,7682560.0,7713728.0,7744896.0,7776064.0,7807232.0,7838400.0,7636864.0,7668096.0,7699328.0,7730560.0,7761792.0,7793024.0,7824256.0,7855488.0,7653504.0,7684800.0,7716096.0,7747392.0,7778688.0,7809984.0,7841280.0,7872576.0,7670144.0,7701504.0,7732864.0,7764224.0,7795584.0,7826944.0,7858304.0,7889664.0,7686784.0,7718208.0,7749632.0,7781056.0,7812480.0,7843904.0,7875328.0,7906752.0,7703424.0,7734912.0,7766400.0,7797888.0,7829376.0,7860864.0,7892352.0,7923840.0,7720064.0,7751616.0,7783168.0,7814720.0,7846272.0,7877824.0,7909376.0,7940928.0,7736704.0,7768320.0,7799936.0,7831552.0,7863168.0,7894784.0,7926400.0,7958016.0,7753344.0,7785024.0,7816704.0,7848384.0,7880064.0,7911744.0,7943424.0,7975104.0,7769984.0,7801728.0,7833472.0,7865216.0,7896960.0,7928704.0,7960448.0,7992192.0,7786624.0,7818432.0,7850240.0,7882048.0,7913856.0,7945664.0,7977472.0,8009280.0,7803264.0,7835136.0,7867008.0,7898880.0,7930752.0,7962624.0,7994496.0,8026368.0,7819904.0,7851840.0,7883776.0,7915712.0,7947648.0,7979584.0,8011520.0,8043456.0,7836544.0,7868544.0,7900544.0,7932544.0,7964544.0,7996544.0,8028544.0,8060544.0,7853184.0,7885248.0,7917312.0,7949376.0,7981440.0,8013504.0,8045568.0,8077632.0,7869824.0,7901952.0,7934080.0,7966208.0,7998336.0,8030464.0,8062592.0,8094720.0,7886464.0,7918656.0,7950848.0,7983040.0,8015232.0,8047424.0,8079616.0,8111808.0,7903104.0,7935360.0,7967616.0,7999872.0,8032128.0,8064384.0,8096640.0,8128896.0,7919744.0,7952064.0,7984384.0,8016704.0,8049024.0,8081344.0,8113664.0,8145984.0,7936384.0,7968768.0,8001152.0,8033536.0,8065920.0,8098304.0,8130688.0,8163072.0,7953024.0,7984448.0,8015872.0,8047296.0,8078720.0,8110144.0,8141568.0,8172992.0,7961472.0,7991936.0,8022400.0,8052864.0,8083328.0,8113792.0,8144256.0,8174720.0,7978112.0,8008640.0,8039168.0,8069696.0,8100224.0,8130752.0,8161280.0,8191808.0,7994752.0,8025344.0,8055936.0,8086528.0,8117120.0,8147712.0,8178304.0,8208896.0,8011392.0,8042048.0,8072704.0,8103360.0,8134016.0,8164672.0,8195328.0,8225984.0,8028032.0,8058752.0,8089472.0,8120192.0,8150912.0,8181632.0,8212352.0,8243072.0,8044672.0,8075456.0,8106240.0,8137024.0,8167808.0,8198592.0,8229376.0,8260160.0,8061312.0,8092160.0,8123008.0,8153856.0,8184704.0,8215552.0,8246400.0,8277248.0,8077952.0,8108864.0,8139776.0,8170688.0,8201600.0,8232512.0,8263424.0,8294336.0,8094592.0,8125568.0,8156544.0,8187520.0,8218496.0,8249472.0,8280448.0,8311424.0,8111232.0,8142272.0,8173312.0,8204352.0,8235392.0,8266432.0,8297472.0,8328512.0,8127872.0,8158976.0,8190080.0,8221184.0,8252288.0,8283392.0,8314496.0,8345600.0,8144512.0,8175680.0,8206848.0,8238016.0,8269184.0,8300352.0,8331520.0,8362688.0,8161152.0,8192384.0,8223616.0,8254848.0,8286080.0,8317312.0,8348544.0,8379776.0,8177792.0,8209088.0,8240384.0,8271680.0,8302976.0,8334272.0,8365568.0,8396864.0,8194432.0,8225792.0,8257152.0,8288512.0,8319872.0,8351232.0,8382592.0,8413952.0,8211072.0,8242496.0,8273920.0,8305344.0,8336768.0,8368192.0,8399616.0,8431040.0,8227712.0,8259200.0,8290688.0,8322176.0,8353664.0,8385152.0,8416640.0,8448128.0,8244352.0,8275904.0,8307456.0,8339008.0,8370560.0,8402112.0,8433664.0,8465216.0,8260992.0,8292608.0,8324224.0,8355840.0,8387456.0,8419072.0,8450688.0,8482304.0,8277632.0,8309312.0,8340992.0,8372672.0,8404352.0,8436032.0,8467712.0,8499392.0,8294272.0,8326016.0,8357760.0,8389504.0,8421248.0,8452992.0,8484736.0,8516480.0,8310912.0,8342720.0,8374528.0,8406336.0,8438144.0,8469952.0,8501760.0,8533568.0,8327552.0,8359424.0,8391296.0,8423168.0,8455040.0,8486912.0,8518784.0,8550656.0,8344192.0,8376128.0,8408064.0,8440000.0,8471936.0,8503872.0,8535808.0,8567744.0,8360832.0,8392832.0,8424832.0,8456832.0,8488832.0,8520832.0,8552832.0,8584832.0,8377472.0,8409536.0,8441600.0,8473664.0,8505728.0,8537792.0,8569856.0,8601920.0,8394112.0,8426240.0,8458368.0,8490496.0,8522624.0,8554752.0,8586880.0,8619008.0,8410752.0,8442944.0,8475136.0,8507328.0,8539520.0,8571712.0,8603904.0,8636096.0,8427392.0,8459648.0,8491904.0,8524160.0,8556416.0,8588672.0,8620928.0,8653184.0,8444032.0,8476352.0,8508672.0,8540992.0,8573312.0,8605632.0,8637952.0,8670272.0,8460672.0,8493056.0,8525440.0,8557824.0,8590208.0,8622592.0,8654976.0,8687360.0,8477312.0,8509760.0,8542208.0,8574656.0,8607104.0,8639552.0,8672000.0,8704448.0,8493952.0,8526464.0,8558976.0,8591488.0,8624000.0,8656512.0,8689024.0,8721536.0,8510592.0,8543168.0,8575744.0,8608320.0,8640896.0,8673472.0,8706048.0,8738624.0,8527232.0,8559872.0,8592512.0,8625152.0,8657792.0,8690432.0,8723072.0,8755712.0,8543872.0,8576576.0,8609280.0,8641984.0,8674688.0,8707392.0,8740096.0,8772800.0,8560512.0,8593280.0,8626048.0,8658816.0,8691584.0,8724352.0,8757120.0,8789888.0,8577152.0,8609984.0,8642816.0,8675648.0,8708480.0,8741312.0,8774144.0,8806976.0,8069504.0,8101376.0,8133248.0,8165120.0,8196992.0,8228864.0,8260736.0,8292608.0,7553664.0,7584576.0,7615488.0,7646400.0,7677312.0,7708224.0,7739136.0,7770048.0,7570304.0,7601280.0,7632256.0,7663232.0,7694208.0,7725184.0,7756160.0,7787136.0,7586944.0,7617984.0,7649024.0,7680064.0,7711104.0,7742144.0,7773184.0,7804224.0,7603584.0,7634688.0,7665792.0,7696896.0,7728000.0,7759104.0,7790208.0,7821312.0,7620224.0,7651392.0,7682560.0,7713728.0,7744896.0,7776064.0,7807232.0,7838400.0,7636864.0,7668096.0,7699328.0,7730560.0,7761792.0,7793024.0,7824256.0,7855488.0,7653504.0,7684800.0,7716096.0,7747392.0,7778688.0,7809984.0,7841280.0,7872576.0,7670144.0,7701504.0,7732864.0,7764224.0,7795584.0,7826944.0,7858304.0,7889664.0,7686784.0,7718208.0,7749632.0,7781056.0,7812480.0,7843904.0,7875328.0,7906752.0,7703424.0,7734912.0,7766400.0,7797888.0,7829376.0,7860864.0,7892352.0,7923840.0,7720064.0,7751616.0,7783168.0,7814720.0,7846272.0,7877824.0,7909376.0,7940928.0,7736704.0,7768320.0,7799936.0,7831552.0,7863168.0,7894784.0,7926400.0,7958016.0,7753344.0,7785024.0,7816704.0,7848384.0,7880064.0,7911744.0,7943424.0,7975104.0,7769984.0,7801728.0,7833472.0,7865216.0,7896960.0,7928704.0,7960448.0,7992192.0,7786624.0,7818432.0,7850240.0,7882048.0,7913856.0,7945664.0,7977472.0,8009280.0,7803264.0,7835136.0,7867008.0,7898880.0,7930752.0,7962624.0,7994496.0,8026368.0,7819904.0,7851840.0,7883776.0,7915712.0,7947648.0,7979584.0,8011520.0,8043456.0,7836544.0,7868544.0,7900544.0,7932544.0,7964544.0,7996544.0,8028544.0,8060544.0,7853184.0,7885248.0,7917312.0,7949376.0,7981440.0,8013504.0,8045568.0,8077632.0,7869824.0,7901952.0,7934080.0,7966208.0,7998336.0,8030464.0,8062592.0,8094720.0,7886464.0,7918656.0,7950848.0,7983040.0,8015232.0,8047424.0,8079616.0,8111808.0,7903104.0,7935360.0,7967616.0,7999872.0,8032128.0,8064384.0,8096640.0,8128896.0,7919744.0,7952064.0,7984384.0,8016704.0,8049024.0,8081344.0,8113664.0,8145984.0,7936384.0,7968768.0,8001152.0,8033536.0,8065920.0,8098304.0,8130688.0,8163072.0,7953024.0,7985472.0,8017920.0,8050368.0,8082816.0,8115264.0,8147712.0,8180160.0,7969664.0,8002176.0,8034688.0,8067200.0,8099712.0,8132224.0,8164736.0,8197248.0,7986304.0,8018880.0,8051456.0,8084032.0,8116608.0,8149184.0,8181760.0,8214336.0,8002944.0,8035584.0,8068224.0,8100864.0,8133504.0,8166144.0,8198784.0,8231424.0,8019584.0,8052288.0,8084992.0,8117696.0,8150400.0,8183104.0,8215808.0,8248512.0,8036224.0,8068992.0,8101760.0,8134528.0,8167296.0,8200064.0,8232832.0,8265600.0,8052864.0,8085696.0,8118528.0,8151360.0,8184192.0,8217024.0,8249856.0,8282688.0,8069504.0,8102400.0,8135296.0,8168192.0,8201088.0,8233984.0,8266880.0,8299776.0,8086144.0,8119104.0,8152064.0,8185024.0,8217984.0,8250944.0,8283904.0,8316864.0,8102784.0,8135808.0,8168832.0,8201856.0,8234880.0,8267904.0,8300928.0,8333952.0,8119424.0,8152512.0,8185600.0,8218688.0,8251776.0,8284864.0,8317952.0,8351040.0,8136064.0,8169216.0,8202368.0,8235520.0,8268672.0,8301824.0,8334976.0,8368128.0,8152704.0,8185920.0,8219136.0,8252352.0,8285568.0,8318784.0,8352000.0,8385216.0,8169344.0,8202624.0,8235904.0,8269184.0,8302464.0,8335744.0,8369024.0,8402304.0,8185984.0,8218304.0,8250624.0,8282944.0,8315264.0,8347584.0,8379904.0,8412224.0,8194432.0,8225792.0,8257152.0,8288512.0,8319872.0,8351232.0,8382592.0,8413952.0,8211072.0,8242496.0,8273920.0,8305344.0,8336768.0,8368192.0,8399616.0,8431040.0,8227712.0,8259200.0,8290688.0,8322176.0,8353664.0,8385152.0,8416640.0,8448128.0,8244352.0,8275904.0,8307456.0,8339008.0,8370560.0,8402112.0,8433664.0,8465216.0,8260992.0,8292608.0,8324224.0,8355840.0,8387456.0,8419072.0,8450688.0,8482304.0,8277632.0,8309312.0,8340992.0,8372672.0,8404352.0,8436032.0,8467712.0,8499392.0,8294272.0,8326016.0,8357760.0,8389504.0,8421248.0,8452992.0,8484736.0,8516480.0,8310912.0,8342720.0,8374528.0,8406336.0,8438144.0,8469952.0,8501760.0,8533568.0,8327552.0,8359424.0,8391296.0,8423168.0,8455040.0,8486912.0,8518784.0,8550656.0,8344192.0,8376128.0,8408064.0,8440000.0,8471936.0,8503872.0,8535808.0,8567744.0,8360832.0,8392832.0,8424832.0,8456832.0,8488832.0,8520832.0,8552832.0,8584832.0,8377472.0,8409536.0,8441600.0,8473664.0,8505728.0,8537792.0,8569856.0,8601920.0,8394112.0,8426240.0,8458368.0,8490496.0,8522624.0,8554752.0,8586880.0,8619008.0,8410752.0,8442944.0,8475136.0,8507328.0,8539520.0,8571712.0,8603904.0,8636096.0,8427392.0,8459648.0,8491904.0,8524160.0,8556416.0,8588672.0,8620928.0,8653184.0,8444032.0,8476352.0,8508672.0,8540992.0,8573312.0,8605632.0,8637952.0,8670272.0,8460672.0,8493056.0,8525440.0,8557824.0,8590208.0,8622592.0,8654976.0,8687360.0,8477312.0,8509760.0,8542208.0,8574656.0,8607104.0,8639552.0,8672000.0,8704448.0,8493952.0,8526464.0,8558976.0,8591488.0,8624000.0,8656512.0,8689024.0,8721536.0,8510592.0,8543168.0,8575744.0,8608320.0,8640896.0,8673472.0,8706048.0,8738624.0,8527232.0,8559872.0,8592512.0,8625152.0,8657792.0,8690432.0,8723072.0,8755712.0,8543872.0,8576576.0,8609280.0,8641984.0,8674688.0,8707392.0,8740096.0,8772800.0,8560512.0,8593280.0,8626048.0,8658816.0,8691584.0,8724352.0,8757120.0,8789888.0,8577152.0,8609984.0,8642816.0,8675648.0,8708480.0,8741312.0,8774144.0,8806976.0,8593792.0,8626688.0,8659584.0,8692480.0,8725376.0,8758272.0,8791168.0,8824064.0,8610432.0,8643392.0,8676352.0,8709312.0,8742272.0,8775232.0,8808192.0,8841152.0,8627072.0,8660096.0,8693120.0,8726144.0,8759168.0,8792192.0,8825216.0,8858240.0,8643712.0,8676800.0,8709888.0,8742976.0,8776064.0,8809152.0,8842240.0,8875328.0,8660352.0,8693504.0,8726656.0,8759808.0,8792960.0,8826112.0,8859264.0,8892416.0,8676992.0,8710208.0,8743424.0,8776640.0,8809856.0,8843072.0,8876288.0,8909504.0,8693632.0,8726912.0,8760192.0,8793472.0,8826752.0,8860032.0,8893312.0,8926592.0,8710272.0,8743616.0,8776960.0,8810304.0,8843648.0,8876992.0,8910336.0,8943680.0,8726912.0,8760320.0,8793728.0,8827136.0,8860544.0,8893952.0,8927360.0,8960768.0,8743552.0,8777024.0,8810496.0,8843968.0,8877440.0,8910912.0,8944384.0,8977856.0,8760192.0,8793728.0,8827264.0,8860800.0,8894336.0,8927872.0,8961408.0,8994944.0,8776832.0,8810432.0,8844032.0,8877632.0,8911232.0,8944832.0,8978432.0,9012032.0,8793472.0,8827136.0,8860800.0,8894464.0,8928128.0,8961792.0,8995456.0,9029120.0,8810112.0,8843840.0,8877568.0,8911296.0,8945024.0,8978752.0,9012480.0,9046208.0,8302464.0,8335232.0,8368000.0,8400768.0,8433536.0,8466304.0,8499072.0,8531840.0,7786624.0,7818432.0,7850240.0,7882048.0,7913856.0,7945664.0,7977472.0,8009280.0,3961216.0,3977136.0,3993056.0,4008976.0,4024896.0,4040816.0,4056736.0,4072656.0,4578048.0,4593840.0,4609632.0,4625424.0,4641216.0,4657008.0,4672800.0,4688592.0,9291136.0,9322752.0,9354368.0,9385984.0,9417600.0,9449216.0,9480832.0,9512448.0,9308800.0,9340480.0,9372160.0,9403840.0,9435520.0,9467200.0,9498880.0,9530560.0,9326464.0,9358208.0,9389952.0,9421696.0,9453440.0,9485184.0,9516928.0,9548672.0,9344128.0,9375936.0,9407744.0,9439552.0,9471360.0,9503168.0,9534976.0,9566784.0,9361792.0,9393664.0,9425536.0,9457408.0,9489280.0,9521152.0,9553024.0,9584896.0,9379456.0,9411392.0,9443328.0,9475264.0,9507200.0,9539136.0,9571072.0,9603008.0,9397120.0,9429120.0,9461120.0,9493120.0,9525120.0,9557120.0,9589120.0,9621120.0,9414784.0,9446848.0,9478912.0,9510976.0,9543040.0,9575104.0,9607168.0,9639232.0,9432448.0,9464576.0,9496704.0,9528832.0,9560960.0,9593088.0,9625216.0,9657344.0,9450112.0,9482304.0,9514496.0,9546688.0,9578880.0,9611072.0,9643264.0,9675456.0,9467776.0,9500032.0,9532288.0,9564544.0,9596800.0,9629056.0,9661312.0,9693568.0,9485440.0,9517760.0,9550080.0,9582400.0,9614720.0,9647040.0,9679360.0,9711680.0,9503104.0,9535488.0,9567872.0,9600256.0,9632640.0,9665024.0,9697408.0,9729792.0,9520768.0,9553216.0,9585664.0,9618112.0,9650560.0,9683008.0,9715456.0,9747904.0,9538432.0,9570944.0,9603456.0,9635968.0,9668480.0,9700992.0,9733504.0,9766016.0,9556096.0,9588672.0,9621248.0,9653824.0,9686400.0,9718976.0,9751552.0,9784128.0,9573760.0,9606400.0,9639040.0,9671680.0,9704320.0,9736960.0,9769600.0,9802240.0,9591424.0,9624128.0,9656832.0,9689536.0,9722240.0,9754944.0,9787648.0,9820352.0,9609088.0,9641856.0,9674624.0,9707392.0,9740160.0,9772928.0,9805696.0,9838464.0,9626752.0,9659584.0,9692416.0,9725248.0,9758080.0,9790912.0,9823744.0,9856576.0,9644416.0,9677312.0,9710208.0,9743104.0,9776000.0,9808896.0,9841792.0,9874688.0,9662080.0,9695040.0,9728000.0,9760960.0,9793920.0,9826880.0,9859840.0,9892800.0,9679744.0,9712768.0,9745792.0,9778816.0,9811840.0,9844864.0,9877888.0,9910912.0,9697408.0,9730496.0,9763584.0,9796672.0,9829760.0,9862848.0,9895936.0,9929024.0,9715072.0,9748224.0,9781376.0,9814528.0,9847680.0,9880832.0,9913984.0,9947136.0,9732736.0,9765952.0,9799168.0,9832384.0,9865600.0,9898816.0,9932032.0,9965248.0,9750400.0,9783680.0,9816960.0,9850240.0,9883520.0,9916800.0,9950080.0,9983360.0,9768064.0,9801408.0,9834752.0,9868096.0,9901440.0,9934784.0,9968128.0,10001472.0,9785728.0,9819136.0,9852544.0,9885952.0,9919360.0,9952768.0,9986176.0,10019584.0,9803392.0,9836864.0,9870336.0,9903808.0,9937280.0,9970752.0,10004224.0,10037696.0,9821056.0,9854592.0,9888128.0,9921664.0,9955200.0,9988736.0,10022272.0,10055808.0,9838720.0,9872320.0,9905920.0,9939520.0,9973120.0,10006720.0,10040320.0,10073920.0,9856384.0,9890048.0,9923712.0,9957376.0,9991040.0,10024704.0,10058368.0,10092032.0,9874048.0,9907776.0,9941504.0,9975232.0,10008960.0,10042688.0,10076416.0,10110144.0,9891712.0,9925504.0,9959296.0,9993088.0,10026880.0,10060672.0,10094464.0,10128256.0,9909376.0,9943232.0,9977088.0,10010944.0,10044800.0,10078656.0,10112512.0,10146368.0,9386368.0,9419264.0,9452160.0,9485056.0,9517952.0,9550848.0,9583744.0,9616640.0,8855168.0,8887104.0,8919040.0,8950976.0,8982912.0,9014848.0,9046784.0,9078720.0,8332160.0,8363136.0,8394112.0,8425088.0,8456064.0,8487040.0,8518016.0,8548992.0,7800960.0,7830976.0,7860992.0,7891008.0,7921024.0,7951040.0,7981056.0,8011072.0,7818624.0,7848704.0,7878784.0,7908864.0,7938944.0,7969024.0,7999104.0,8029184.0,7836288.0,7866432.0,7896576.0,7926720.0,7956864.0,7987008.0,8017152.0,8047296.0,7853952.0,7884160.0,7914368.0,7944576.0,7974784.0,8004992.0,8035200.0,8065408.0,7871616.0,7901888.0,7932160.0,7962432.0,7992704.0,8022976.0,8053248.0,8083520.0,7889280.0,7919616.0,7949952.0,7980288.0,8010624.0,8040960.0,8071296.0,8101632.0,7906944.0,7937344.0,7967744.0,7998144.0,8028544.0,8058944.0,8089344.0,8119744.0,7924608.0,7955072.0,7985536.0,8016000.0,8046464.0,8076928.0,8107392.0,8137856.0,7942272.0,7972800.0,8003328.0,8033856.0,8064384.0,8094912.0,8125440.0,8155968.0,7959936.0,7990528.0,8021120.0,8051712.0,8082304.0,8112896.0,8143488.0,8174080.0,7977600.0,8008256.0,8038912.0,8069568.0,8100224.0,8130880.0,8161536.0,8192192.0,7995264.0,8025984.0,8056704.0,8087424.0,8118144.0,8148864.0,8179584.0,8210304.0,8012928.0,8043712.0,8074496.0,8105280.0,8136064.0,8166848.0,8197632.0,8228416.0,8030592.0,8061440.0,8092288.0,8123136.0,8153984.0,8184832.0,8215680.0,8246528.0,8048256.0,8079168.0,8110080.0,8140992.0,8171904.0,8202816.0,8233728.0,8264640.0,8065920.0,8096896.0,8127872.0,8158848.0,8189824.0,8220800.0,8251776.0,8282752.0,8083584.0,8114624.0,8145664.0,8176704.0,8207744.0,8238784.0,8269824.0,8300864.0,8101248.0,8132352.0,8163456.0,8194560.0,8225664.0,8256768.0,8287872.0,8318976.0,8118912.0,8150080.0,8181248.0,8212416.0,8243584.0,8274752.0,8305920.0,8337088.0,8136576.0,8167808.0,8199040.0,8230272.0,8261504.0,8292736.0,8323968.0,8355200.0,8154240.0,8185536.0,8216832.0,8248128.0,8279424.0,8310720.0,8342016.0,8373312.0,8171904.0,8203264.0,8234624.0,8265984.0,8297344.0,8328704.0,8360064.0,8391424.0,8189568.0,8220992.0,8252416.0,8283840.0,8315264.0,8346688.0,8378112.0,8409536.0,8207232.0,8238720.0,8270208.0,8301696.0,8333184.0,8364672.0,8396160.0,8427648.0,8224896.0,8256448.0,8288000.0,8319552.0,8351104.0,8382656.0,8414208.0,8445760.0,8242560.0,8274176.0,8305792.0,8337408.0,8369024.0,8400640.0,8432256.0,8463872.0,8260224.0,8291904.0,8323584.0,8355264.0,8386944.0,8418624.0,8450304.0,8481984.0,8277888.0,8309632.0,8341376.0,8373120.0,8404864.0,8436608.0,8468352.0,8500096.0,8295552.0,8327360.0,8359168.0,8390976.0,8422784.0,8454592.0,8486400.0,8518208.0,8313216.0,8345088.0,8376960.0,8408832.0,8440704.0,8472576.0,8504448.0,8536320.0,8330880.0,8362816.0,8394752.0,8426688.0,8458624.0,8490560.0,8522496.0,8554432.0,8348544.0,8380544.0,8412544.0,8444544.0,8476544.0,8508544.0,8540544.0,8572544.0,8366208.0,8398272.0,8430336.0,8462400.0,8494464.0,8526528.0,8558592.0,8590656.0,8383872.0,8416000.0,8448128.0,8480256.0,8512384.0,8544512.0,8576640.0,8608768.0,8401536.0,8433728.0,8465920.0,8498112.0,8530304.0,8562496.0,8594688.0,8626880.0,8419200.0,8451456.0,8483712.0,8515968.0,8548224.0,8580480.0,8612736.0,8644992.0,8436864.0,8469184.0,8501504.0,8533824.0,8566144.0,8598464.0,8630784.0,8663104.0,8454528.0,8486912.0,8519296.0,8551680.0,8584064.0,8616448.0,8648832.0,8681216.0,8455808.0,8487232.0,8518656.0,8550080.0,8581504.0,8612928.0,8644352.0,8675776.0,8448896.0,8479360.0,8509824.0,8540288.0,8570752.0,8601216.0,8631680.0,8662144.0,8466560.0,8497088.0,8527616.0,8558144.0,8588672.0,8619200.0,8649728.0,8680256.0,8484224.0,8514816.0,8545408.0,8576000.0,8606592.0,8637184.0,8667776.0,8698368.0,8501888.0,8532544.0,8563200.0,8593856.0,8624512.0,8655168.0,8685824.0,8716480.0,8519552.0,8550272.0,8580992.0,8611712.0,8642432.0,8673152.0,8703872.0,8734592.0,8537216.0,8568000.0,8598784.0,8629568.0,8660352.0,8691136.0,8721920.0,8752704.0,8554880.0,8585728.0,8616576.0,8647424.0,8678272.0,8709120.0,8739968.0,8770816.0,8572544.0,8603456.0,8634368.0,8665280.0,8696192.0,8727104.0,8758016.0,8788928.0,8590208.0,8621184.0,8652160.0,8683136.0,8714112.0,8745088.0,8776064.0,8807040.0,8607872.0,8638912.0,8669952.0,8700992.0,8732032.0,8763072.0,8794112.0,8825152.0,8625536.0,8656640.0,8687744.0,8718848.0,8749952.0,8781056.0,8812160.0,8843264.0,8643200.0,8674368.0,8705536.0,8736704.0,8767872.0,8799040.0,8830208.0,8861376.0,8660864.0,8692096.0,8723328.0,8754560.0,8785792.0,8817024.0,8848256.0,8879488.0,8678528.0,8709824.0,8741120.0,8772416.0,8803712.0,8835008.0,8866304.0,8897600.0,8696192.0,8727552.0,8758912.0,8790272.0,8821632.0,8852992.0,8884352.0,8915712.0,8713856.0,8745280.0,8776704.0,8808128.0,8839552.0,8870976.0,8902400.0,8933824.0,8731520.0,8763008.0,8794496.0,8825984.0,8857472.0,8888960.0,8920448.0,8951936.0,8749184.0,8780736.0,8812288.0,8843840.0,8875392.0,8906944.0,8938496.0,8970048.0,8766848.0,8798464.0,8830080.0,8861696.0,8893312.0,8924928.0,8956544.0,8988160.0,8784512.0,8816192.0,8847872.0,8879552.0,8911232.0,8942912.0,8974592.0,9006272.0,8802176.0,8833920.0,8865664.0,8897408.0,8929152.0,8960896.0,8992640.0,9024384.0,8819840.0,8851648.0,8883456.0,8915264.0,8947072.0,8978880.0,9010688.0,9042496.0,8837504.0,8869376.0,8901248.0,8933120.0,8964992.0,8996864.0,9028736.0,9060608.0,8855168.0,8887104.0,8919040.0,8950976.0,8982912.0,9014848.0,9046784.0,9078720.0,8872832.0,8904832.0,8936832.0,8968832.0,9000832.0,9032832.0,9064832.0,9096832.0,8890496.0,8922560.0,8954624.0,8986688.0,9018752.0,9050816.0,9082880.0,9114944.0,8908160.0,8940288.0,8972416.0,9004544.0,9036672.0,9068800.0,9100928.0,9133056.0,8925824.0,8958016.0,8990208.0,9022400.0,9054592.0,9086784.0,9118976.0,9151168.0,8943488.0,8975744.0,9008000.0,9040256.0,9072512.0,9104768.0,9137024.0,9169280.0,8961152.0,8993472.0,9025792.0,9058112.0,9090432.0,9122752.0,9155072.0,9187392.0,8978816.0,9011200.0,9043584.0,9075968.0,9108352.0,9140736.0,9173120.0,9205504.0,8996480.0,9028928.0,9061376.0,9093824.0,9126272.0,9158720.0,9191168.0,9223616.0,9014144.0,9046656.0,9079168.0,9111680.0,9144192.0,9176704.0,9209216.0,9241728.0,9031808.0,9064384.0,9096960.0,9129536.0,9162112.0,9194688.0,9227264.0,9259840.0,9049472.0,9082112.0,9114752.0,9147392.0,9180032.0,9212672.0,9245312.0,9277952.0,9067136.0,9099840.0,9132544.0,9165248.0,9197952.0,9230656.0,9263360.0,9296064.0,9084800.0,9117568.0,9150336.0,9183104.0,9215872.0,9248640.0,9281408.0,9314176.0,9102464.0,9135296.0,9168128.0,9200960.0,9233792.0,9266624.0,9299456.0,9332288.0,8579456.0,8611328.0,8643200.0,8675072.0,8706944.0,8738816.0,8770688.0,8802560.0,8048256.0,8079168.0,8110080.0,8140992.0,8171904.0,8202816.0,8233728.0,8264640.0,8065920.0,8096896.0,8127872.0,8158848.0,8189824.0,8220800.0,8251776.0,8282752.0,8083584.0,8114624.0,8145664.0,8176704.0,8207744.0,8238784.0,8269824.0,8300864.0,8101248.0,8132352.0,8163456.0,8194560.0,8225664.0,8256768.0,8287872.0,8318976.0,8118912.0,8150080.0,8181248.0,8212416.0,8243584.0,8274752.0,8305920.0,8337088.0,8136576.0,8167808.0,8199040.0,8230272.0,8261504.0,8292736.0,8323968.0,8355200.0,8154240.0,8185536.0,8216832.0,8248128.0,8279424.0,8310720.0,8342016.0,8373312.0,8171904.0,8203264.0,8234624.0,8265984.0,8297344.0,8328704.0,8360064.0,8391424.0,8189568.0,8220992.0,8252416.0,8283840.0,8315264.0,8346688.0,8378112.0,8409536.0,8207232.0,8238720.0,8270208.0,8301696.0,8333184.0,8364672.0,8396160.0,8427648.0,8224896.0,8256448.0,8288000.0,8319552.0,8351104.0,8382656.0,8414208.0,8445760.0,8242560.0,8274176.0,8305792.0,8337408.0,8369024.0,8400640.0,8432256.0,8463872.0,8260224.0,8291904.0,8323584.0,8355264.0,8386944.0,8418624.0,8450304.0,8481984.0,8277888.0,8309632.0,8341376.0,8373120.0,8404864.0,8436608.0,8468352.0,8500096.0,8295552.0,8327360.0,8359168.0,8390976.0,8422784.0,8454592.0,8486400.0,8518208.0,8313216.0,8345088.0,8376960.0,8408832.0,8440704.0,8472576.0,8504448.0,8536320.0,8330880.0,8362816.0,8394752.0,8426688.0,8458624.0,8490560.0,8522496.0,8554432.0,8348544.0,8380544.0,8412544.0,8444544.0,8476544.0,8508544.0,8540544.0,8572544.0,8366208.0,8398272.0,8430336.0,8462400.0,8494464.0,8526528.0,8558592.0,8590656.0,8383872.0,8416000.0,8448128.0,8480256.0,8512384.0,8544512.0,8576640.0,8608768.0,8401536.0,8433728.0,8465920.0,8498112.0,8530304.0,8562496.0,8594688.0,8626880.0,8419200.0,8451456.0,8483712.0,8515968.0,8548224.0,8580480.0,8612736.0,8644992.0,8436864.0,8469184.0,8501504.0,8533824.0,8566144.0,8598464.0,8630784.0,8663104.0,8454528.0,8486912.0,8519296.0,8551680.0,8584064.0,8616448.0,8648832.0,8681216.0,8472192.0,8504640.0,8537088.0,8569536.0,8601984.0,8634432.0,8666880.0,8699328.0,8489856.0,8522368.0,8554880.0,8587392.0,8619904.0,8652416.0,8684928.0,8717440.0,8507520.0,8540096.0,8572672.0,8605248.0,8637824.0,8670400.0,8702976.0,8735552.0,8525184.0,8557824.0,8590464.0,8623104.0,8655744.0,8688384.0,8721024.0,8753664.0,8542848.0,8575552.0,8608256.0,8640960.0,8673664.0,8706368.0,8739072.0,8771776.0,8560512.0,8593280.0,8626048.0,8658816.0,8691584.0,8724352.0,8757120.0,8789888.0,8578176.0,8611008.0,8643840.0,8676672.0,8709504.0,8742336.0,8775168.0,8808000.0,8595840.0,8628736.0,8661632.0,8694528.0,8727424.0,8760320.0,8793216.0,8826112.0,8613504.0,8646464.0,8679424.0,8712384.0,8745344.0,8778304.0,8811264.0,8844224.0,8631168.0,8664192.0,8697216.0,8730240.0,8763264.0,8796288.0,8829312.0,8862336.0,8648832.0,8681920.0,8715008.0,8748096.0,8781184.0,8814272.0,8847360.0,8880448.0,8666496.0,8699648.0,8732800.0,8765952.0,8799104.0,8832256.0,8865408.0,8898560.0,8684160.0,8717376.0,8750592.0,8783808.0,8817024.0,8850240.0,8883456.0,8916672.0,8701824.0,8735104.0,8768384.0,8801664.0,8834944.0,8868224.0,8901504.0,8934784.0,8703104.0,8735424.0,8767744.0,8800064.0,8832384.0,8864704.0,8897024.0,8929344.0,8696192.0,8727552.0,8758912.0,8790272.0,8821632.0,8852992.0,8884352.0,8915712.0,8713856.0,8745280.0,8776704.0,8808128.0,8839552.0,8870976.0,8902400.0,8933824.0,8731520.0,8763008.0,8794496.0,8825984.0,8857472.0,8888960.0,8920448.0,8951936.0,8749184.0,8780736.0,8812288.0,8843840.0,8875392.0,8906944.0,8938496.0,8970048.0,8766848.0,8798464.0,8830080.0,8861696.0,8893312.0,8924928.0,8956544.0,8988160.0,8784512.0,8816192.0,8847872.0,8879552.0,8911232.0,8942912.0,8974592.0,9006272.0,8802176.0,8833920.0,8865664.0,8897408.0,8929152.0,8960896.0,8992640.0,9024384.0,8819840.0,8851648.0,8883456.0,8915264.0,8947072.0,8978880.0,9010688.0,9042496.0,8837504.0,8869376.0,8901248.0,8933120.0,8964992.0,8996864.0,9028736.0,9060608.0,8855168.0,8887104.0,8919040.0,8950976.0,8982912.0,9014848.0,9046784.0,9078720.0,8872832.0,8904832.0,8936832.0,8968832.0,9000832.0,9032832.0,9064832.0,9096832.0,8890496.0,8922560.0,8954624.0,8986688.0,9018752.0,9050816.0,9082880.0,9114944.0,8908160.0,8940288.0,8972416.0,9004544.0,9036672.0,9068800.0,9100928.0,9133056.0,8925824.0,8958016.0,8990208.0,9022400.0,9054592.0,9086784.0,9118976.0,9151168.0,8943488.0,8975744.0,9008000.0,9040256.0,9072512.0,9104768.0,9137024.0,9169280.0,8961152.0,8993472.0,9025792.0,9058112.0,9090432.0,9122752.0,9155072.0,9187392.0,8978816.0,9011200.0,9043584.0,9075968.0,9108352.0,9140736.0,9173120.0,9205504.0,8996480.0,9028928.0,9061376.0,9093824.0,9126272.0,9158720.0,9191168.0,9223616.0,9014144.0,9046656.0,9079168.0,9111680.0,9144192.0,9176704.0,9209216.0,9241728.0,9031808.0,9064384.0,9096960.0,9129536.0,9162112.0,9194688.0,9227264.0,9259840.0,9049472.0,9082112.0,9114752.0,9147392.0,9180032.0,9212672.0,9245312.0,9277952.0,9067136.0,9099840.0,9132544.0,9165248.0,9197952.0,9230656.0,9263360.0,9296064.0,9084800.0,9117568.0,9150336.0,9183104.0,9215872.0,9248640.0,9281408.0,9314176.0,9102464.0,9135296.0,9168128.0,9200960.0,9233792.0,9266624.0,9299456.0,9332288.0,9120128.0,9153024.0,9185920.0,9218816.0,9251712.0,9284608.0,9317504.0,9350400.0,9137792.0,9170752.0,9203712.0,9236672.0,9269632.0,9302592.0,9335552.0,9368512.0,9155456.0,9188480.0,9221504.0,9254528.0,9287552.0,9320576.0,9353600.0,9386624.0,9173120.0,9206208.0,9239296.0,9272384.0,9305472.0,9338560.0,9371648.0,9404736.0,9190784.0,9223936.0,9257088.0,9290240.0,9323392.0,9356544.0,9389696.0,9422848.0,9208448.0,9241664.0,9274880.0,9308096.0,9341312.0,9374528.0,9407744.0,9440960.0,9226112.0,9259392.0,9292672.0,9325952.0,9359232.0,9392512.0,9425792.0,9459072.0,9243776.0,9277120.0,9310464.0,9343808.0,9377152.0,9410496.0,9443840.0,9477184.0,9261440.0,9294848.0,9328256.0,9361664.0,9395072.0,9428480.0,9461888.0,9495296.0,9279104.0,9312576.0,9346048.0,9379520.0,9412992.0,9446464.0,9479936.0,9513408.0,9296768.0,9330304.0,9363840.0,9397376.0,9430912.0,9464448.0,9497984.0,9531520.0,9314432.0,9348032.0,9381632.0,9415232.0,9448832.0,9482432.0,9516032.0,9549632.0,9332096.0,9365760.0,9399424.0,9433088.0,9466752.0,9500416.0,9534080.0,9567744.0,9349760.0,9383488.0,9417216.0,9450944.0,9484672.0,9518400.0,9552128.0,9585856.0,8826752.0,8859520.0,8892288.0,8925056.0,8957824.0,8990592.0,9023360.0,9056128.0,8295552.0,8327360.0,8359168.0,8390976.0,8422784.0,8454592.0,8486400.0,8518208.0,4215936.0,4231856.0,4247776.0,4263696.0,4279616.0,4295536.0,4311456.0,4327376.0,4830720.0,4846512.0,4862304.0,4878096.0,4893888.0,4909680.0,4925472.0,4941264.0,9796992.0,9828608.0,9860224.0,9891840.0,9923456.0,9955072.0,9986688.0,10018304.0,9815680.0,9847360.0,9879040.0,9910720.0,9942400.0,9974080.0,10005760.0,10037440.0,9834368.0,9866112.0,9897856.0,9929600.0,9961344.0,9993088.0,10024832.0,10056576.0,9853056.0,9884864.0,9916672.0,9948480.0,9980288.0,10012096.0,10043904.0,10075712.0,9871744.0,9903616.0,9935488.0,9967360.0,9999232.0,10031104.0,10062976.0,10094848.0,9890432.0,9922368.0,9954304.0,9986240.0,10018176.0,10050112.0,10082048.0,10113984.0,9909120.0,9941120.0,9973120.0,10005120.0,10037120.0,10069120.0,10101120.0,10133120.0,9927808.0,9959872.0,9991936.0,10024000.0,10056064.0,10088128.0,10120192.0,10152256.0,9946496.0,9978624.0,10010752.0,10042880.0,10075008.0,10107136.0,10139264.0,10171392.0,9965184.0,9997376.0,10029568.0,10061760.0,10093952.0,10126144.0,10158336.0,10190528.0,9983872.0,10016128.0,10048384.0,10080640.0,10112896.0,10145152.0,10177408.0,10209664.0,10002560.0,10034880.0,10067200.0,10099520.0,10131840.0,10164160.0,10196480.0,10228800.0,10021248.0,10053632.0,10086016.0,10118400.0,10150784.0,10183168.0,10215552.0,10247936.0,10039936.0,10072384.0,10104832.0,10137280.0,10169728.0,10202176.0,10234624.0,10267072.0,10058624.0,10091136.0,10123648.0,10156160.0,10188672.0,10221184.0,10253696.0,10286208.0,10077312.0,10109888.0,10142464.0,10175040.0,10207616.0,10240192.0,10272768.0,10305344.0,10096000.0,10128640.0,10161280.0,10193920.0,10226560.0,10259200.0,10291840.0,10324480.0,10114688.0,10147392.0,10180096.0,10212800.0,10245504.0,10278208.0,10310912.0,10343616.0,10133376.0,10166144.0,10198912.0,10231680.0,10264448.0,10297216.0,10329984.0,10362752.0,10152064.0,10184896.0,10217728.0,10250560.0,10283392.0,10316224.0,10349056.0,10381888.0,10170752.0,10203648.0,10236544.0,10269440.0,10302336.0,10335232.0,10368128.0,10401024.0,10189440.0,10222400.0,10255360.0,10288320.0,10321280.0,10354240.0,10387200.0,10420160.0,10208128.0,10241152.0,10274176.0,10307200.0,10340224.0,10373248.0,10406272.0,10439296.0,10226816.0,10259904.0,10292992.0,10326080.0,10359168.0,10392256.0,10425344.0,10458432.0,10245504.0,10278656.0,10311808.0,10344960.0,10378112.0,10411264.0,10444416.0,10477568.0,10264192.0,10297408.0,10330624.0,10363840.0,10397056.0,10430272.0,10463488.0,10496704.0,10282880.0,10316160.0,10349440.0,10382720.0,10416000.0,10449280.0,10482560.0,10515840.0,10301568.0,10334912.0,10368256.0,10401600.0,10434944.0,10468288.0,10501632.0,10534976.0,10320256.0,10353664.0,10387072.0,10420480.0,10453888.0,10487296.0,10520704.0,10554112.0,10338944.0,10372416.0,10405888.0,10439360.0,10472832.0,10506304.0,10539776.0,10573248.0,10357632.0,10391168.0,10424704.0,10458240.0,10491776.0,10525312.0,10558848.0,10592384.0,10376320.0,10409920.0,10443520.0,10477120.0,10510720.0,10544320.0,10577920.0,10611520.0,10395008.0,10428672.0,10462336.0,10496000.0,10529664.0,10563328.0,10596992.0,10630656.0,10413696.0,10447424.0,10481152.0,10514880.0,10548608.0,10582336.0,10616064.0,10649792.0,10432384.0,10466176.0,10499968.0,10533760.0,10567552.0,10601344.0,10635136.0,10668928.0,10451072.0,10484928.0,10518784.0,10552640.0,10586496.0,10620352.0,10654208.0,10688064.0,9912704.0,9945600.0,9978496.0,10011392.0,10044288.0,10077184.0,10110080.0,10142976.0,9366144.0,9398080.0,9430016.0,9461952.0,9493888.0,9525824.0,9557760.0,9589696.0,8827776.0,8858752.0,8889728.0,8920704.0,8951680.0,8982656.0,9013632.0,9044608.0,8281216.0,8311232.0,8341248.0,8371264.0,8401280.0,8431296.0,8461312.0,8491328.0,8299904.0,8329984.0,8360064.0,8390144.0,8420224.0,8450304.0,8480384.0,8510464.0,8318592.0,8348736.0,8378880.0,8409024.0,8439168.0,8469312.0,8499456.0,8529600.0,8337280.0,8367488.0,8397696.0,8427904.0,8458112.0,8488320.0,8518528.0,8548736.0,8355968.0,8386240.0,8416512.0,8446784.0,8477056.0,8507328.0,8537600.0,8567872.0,8374656.0,8404992.0,8435328.0,8465664.0,8496000.0,8526336.0,8556672.0,8587008.0,8393344.0,8423744.0,8454144.0,8484544.0,8514944.0,8545344.0,8575744.0,8606144.0,8412032.0,8442496.0,8472960.0,8503424.0,8533888.0,8564352.0,8594816.0,8625280.0,8430720.0,8461248.0,8491776.0,8522304.0,8552832.0,8583360.0,8613888.0,8644416.0,8449408.0,8480000.0,8510592.0,8541184.0,8571776.0,8602368.0,8632960.0,8663552.0,8468096.0,8498752.0,8529408.0,8560064.0,8590720.0,8621376.0,8652032.0,8682688.0,8486784.0,8517504.0,8548224.0,8578944.0,8609664.0,8640384.0,8671104.0,8701824.0,8505472.0,8536256.0,8567040.0,8597824.0,8628608.0,8659392.0,8690176.0,8720960.0,8524160.0,8555008.0,8585856.0,8616704.0,8647552.0,8678400.0,8709248.0,8740096.0,8542848.0,8573760.0,8604672.0,8635584.0,8666496.0,8697408.0,8728320.0,8759232.0,8561536.0,8592512.0,8623488.0,8654464.0,8685440.0,8716416.0,8747392.0,8778368.0,8580224.0,8611264.0,8642304.0,8673344.0,8704384.0,8735424.0,8766464.0,8797504.0,8598912.0,8630016.0,8661120.0,8692224.0,8723328.0,8754432.0,8785536.0,8816640.0,8617600.0,8648768.0,8679936.0,8711104.0,8742272.0,8773440.0,8804608.0,8835776.0,8636288.0,8667520.0,8698752.0,8729984.0,8761216.0,8792448.0,8823680.0,8854912.0,8654976.0,8686272.0,8717568.0,8748864.0,8780160.0,8811456.0,8842752.0,8874048.0,8673664.0,8705024.0,8736384.0,8767744.0,8799104.0,8830464.0,8861824.0,8893184.0,8692352.0,8723776.0,8755200.0,8786624.0,8818048.0,8849472.0,8880896.0,8912320.0,8711040.0,8742528.0,8774016.0,8805504.0,8836992.0,8868480.0,8899968.0,8931456.0,8729728.0,8761280.0,8792832.0,8824384.0,8855936.0,8887488.0,8919040.0,8950592.0,8748416.0,8780032.0,8811648.0,8843264.0,8874880.0,8906496.0,8938112.0,8969728.0,8767104.0,8798784.0,8830464.0,8862144.0,8893824.0,8925504.0,8957184.0,8988864.0,8785792.0,8817536.0,8849280.0,8881024.0,8912768.0,8944512.0,8976256.0,9008000.0,8804480.0,8836288.0,8868096.0,8899904.0,8931712.0,8963520.0,8995328.0,9027136.0,8823168.0,8855040.0,8886912.0,8918784.0,8950656.0,8982528.0,9014400.0,9046272.0,8841856.0,8873792.0,8905728.0,8937664.0,8969600.0,9001536.0,9033472.0,9065408.0,8860544.0,8892544.0,8924544.0,8956544.0,8988544.0,9020544.0,9052544.0,9084544.0,8879232.0,8911296.0,8943360.0,8975424.0,9007488.0,9039552.0,9071616.0,9103680.0,8897920.0,8930048.0,8962176.0,8994304.0,9026432.0,9058560.0,9090688.0,9122816.0,8916608.0,8948800.0,8980992.0,9013184.0,9045376.0,9077568.0,9109760.0,9141952.0,8935296.0,8967552.0,8999808.0,9032064.0,9064320.0,9096576.0,9128832.0,9161088.0,8953984.0,8986304.0,9018624.0,9050944.0,9083264.0,9115584.0,9147904.0,9180224.0,8972672.0,9005056.0,9037440.0,9069824.0,9102208.0,9134592.0,9166976.0,9199360.0,8958592.0,8990016.0,9021440.0,9052864.0,9084288.0,9115712.0,9147136.0,9178560.0,8936320.0,8966784.0,8997248.0,9027712.0,9058176.0,9088640.0,9119104.0,9149568.0,8955008.0,8985536.0,9016064.0,9046592.0,9077120.0,9107648.0,9138176.0,9168704.0,8973696.0,9004288.0,9034880.0,9065472.0,9096064.0,9126656.0,9157248.0,9187840.0,8992384.0,9023040.0,9053696.0,9084352.0,9115008.0,9145664.0,9176320.0,9206976.0,9011072.0,9041792.0,9072512.0,9103232.0,9133952.0,9164672.0,9195392.0,9226112.0,9029760.0,9060544.0,9091328.0,9122112.0,9152896.0,9183680.0,9214464.0,9245248.0,9048448.0,9079296.0,9110144.0,9140992.0,9171840.0,9202688.0,9233536.0,9264384.0,9067136.0,9098048.0,9128960.0,9159872.0,9190784.0,9221696.0,9252608.0,9283520.0,9085824.0,9116800.0,9147776.0,9178752.0,9209728.0,9240704.0,9271680.0,9302656.0,9104512.0,9135552.0,9166592.0,9197632.0,9228672.0,9259712.0,9290752.0,9321792.0,9123200.0,9154304.0,9185408.0,9216512.0,9247616.0,9278720.0,9309824.0,9340928.0,9141888.0,9173056.0,9204224.0,9235392.0,9266560.0,9297728.0,9328896.0,9360064.0,9160576.0,9191808.0,9223040.0,9254272.0,9285504.0,9316736.0,9347968.0,9379200.0,9179264.0,9210560.0,9241856.0,9273152.0,9304448.0,9335744.0,9367040.0,9398336.0,9197952.0,9229312.0,9260672.0,9292032.0,9323392.0,9354752.0,9386112.0,9417472.0,9216640.0,9248064.0,9279488.0,9310912.0,9342336.0,9373760.0,9405184.0,9436608.0,9235328.0,9266816.0,9298304.0,9329792.0,9361280.0,9392768.0,9424256.0,9455744.0,9254016.0,9285568.0,9317120.0,9348672.0,9380224.0,9411776.0,9443328.0,9474880.0,9272704.0,9304320.0,9335936.0,9367552.0,9399168.0,9430784.0,9462400.0,9494016.0,9291392.0,9323072.0,9354752.0,9386432.0,9418112.0,9449792.0,9481472.0,9513152.0,9310080.0,9341824.0,9373568.0,9405312.0,9437056.0,9468800.0,9500544.0,9532288.0,9328768.0,9360576.0,9392384.0,9424192.0,9456000.0,9487808.0,9519616.0,9551424.0,9347456.0,9379328.0,9411200.0,9443072.0,9474944.0,9506816.0,9538688.0,9570560.0,9366144.0,9398080.0,9430016.0,9461952.0,9493888.0,9525824.0,9557760.0,9589696.0,9384832.0,9416832.0,9448832.0,9480832.0,9512832.0,9544832.0,9576832.0,9608832.0,9403520.0,9435584.0,9467648.0,9499712.0,9531776.0,9563840.0,9595904.0,9627968.0,9422208.0,9454336.0,9486464.0,9518592.0,9550720.0,9582848.0,9614976.0,9647104.0,9440896.0,9473088.0,9505280.0,9537472.0,9569664.0,9601856.0,9634048.0,9666240.0,9459584.0,9491840.0,9524096.0,9556352.0,9588608.0,9620864.0,9653120.0,9685376.0,9478272.0,9510592.0,9542912.0,9575232.0,9607552.0,9639872.0,9672192.0,9704512.0,9496960.0,9529344.0,9561728.0,9594112.0,9626496.0,9658880.0,9691264.0,9723648.0,9515648.0,9548096.0,9580544.0,9612992.0,9645440.0,9677888.0,9710336.0,9742784.0,9534336.0,9566848.0,9599360.0,9631872.0,9664384.0,9696896.0,9729408.0,9761920.0,9553024.0,9585600.0,9618176.0,9650752.0,9683328.0,9715904.0,9748480.0,9781056.0,9571712.0,9604352.0,9636992.0,9669632.0,9702272.0,9734912.0,9767552.0,9800192.0,9590400.0,9623104.0,9655808.0,9688512.0,9721216.0,9753920.0,9786624.0,9819328.0,9609088.0,9641856.0,9674624.0,9707392.0,9740160.0,9772928.0,9805696.0,9838464.0,9627776.0,9660608.0,9693440.0,9726272.0,9759104.0,9791936.0,9824768.0,9857600.0,9089408.0,9121280.0,9153152.0,9185024.0,9216896.0,9248768.0,9280640.0,9312512.0,8542848.0,8573760.0,8604672.0,8635584.0,8666496.0,8697408.0,8728320.0,8759232.0,8561536.0,8592512.0,8623488.0,8654464.0,8685440.0,8716416.0,8747392.0,8778368.0,8580224.0,8611264.0,8642304.0,8673344.0,8704384.0,8735424.0,8766464.0,8797504.0,8598912.0,8630016.0,8661120.0,8692224.0,8723328.0,8754432.0,8785536.0,8816640.0,8617600.0,8648768.0,8679936.0,8711104.0,8742272.0,8773440.0,8804608.0,8835776.0,8636288.0,8667520.0,8698752.0,8729984.0,8761216.0,8792448.0,8823680.0,8854912.0,8654976.0,8686272.0,8717568.0,8748864.0,8780160.0,8811456.0,8842752.0,8874048.0,8673664.0,8705024.0,8736384.0,8767744.0,8799104.0,8830464.0,8861824.0,8893184.0,8692352.0,8723776.0,8755200.0,8786624.0,8818048.0,8849472.0,8880896.0,8912320.0,8711040.0,8742528.0,8774016.0,8805504.0,8836992.0,8868480.0,8899968.0,8931456.0,8729728.0,8761280.0,8792832.0,8824384.0,8855936.0,8887488.0,8919040.0,8950592.0,8748416.0,8780032.0,8811648.0,8843264.0,8874880.0,8906496.0,8938112.0,8969728.0,8767104.0,8798784.0,8830464.0,8862144.0,8893824.0,8925504.0,8957184.0,8988864.0,8785792.0,8817536.0,8849280.0,8881024.0,8912768.0,8944512.0,8976256.0,9008000.0,8804480.0,8836288.0,8868096.0,8899904.0,8931712.0,8963520.0,8995328.0,9027136.0,8823168.0,8855040.0,8886912.0,8918784.0,8950656.0,8982528.0,9014400.0,9046272.0,8841856.0,8873792.0,8905728.0,8937664.0,8969600.0,9001536.0,9033472.0,9065408.0,8860544.0,8892544.0,8924544.0,8956544.0,8988544.0,9020544.0,9052544.0,9084544.0,8879232.0,8911296.0,8943360.0,8975424.0,9007488.0,9039552.0,9071616.0,9103680.0,8897920.0,8930048.0,8962176.0,8994304.0,9026432.0,9058560.0,9090688.0,9122816.0,8916608.0,8948800.0,8980992.0,9013184.0,9045376.0,9077568.0,9109760.0,9141952.0,8935296.0,8967552.0,8999808.0,9032064.0,9064320.0,9096576.0,9128832.0,9161088.0,8953984.0,8986304.0,9018624.0,9050944.0,9083264.0,9115584.0,9147904.0,9180224.0,8972672.0,9005056.0,9037440.0,9069824.0,9102208.0,9134592.0,9166976.0,9199360.0,8991360.0,9023808.0,9056256.0,9088704.0,9121152.0,9153600.0,9186048.0,9218496.0,9010048.0,9042560.0,9075072.0,9107584.0,9140096.0,9172608.0,9205120.0,9237632.0,9028736.0,9061312.0,9093888.0,9126464.0,9159040.0,9191616.0,9224192.0,9256768.0,9047424.0,9080064.0,9112704.0,9145344.0,9177984.0,9210624.0,9243264.0,9275904.0,9066112.0,9098816.0,9131520.0,9164224.0,9196928.0,9229632.0,9262336.0,9295040.0,9084800.0,9117568.0,9150336.0,9183104.0,9215872.0,9248640.0,9281408.0,9314176.0,9103488.0,9136320.0,9169152.0,9201984.0,9234816.0,9267648.0,9300480.0,9333312.0,9122176.0,9155072.0,9187968.0,9220864.0,9253760.0,9286656.0,9319552.0,9352448.0,9140864.0,9173824.0,9206784.0,9239744.0,9272704.0,9305664.0,9338624.0,9371584.0,9159552.0,9192576.0,9225600.0,9258624.0,9291648.0,9324672.0,9357696.0,9390720.0,9178240.0,9211328.0,9244416.0,9277504.0,9310592.0,9343680.0,9376768.0,9409856.0,9196928.0,9230080.0,9263232.0,9296384.0,9329536.0,9362688.0,9395840.0,9428992.0,9215616.0,9248832.0,9282048.0,9315264.0,9348480.0,9381696.0,9414912.0,9448128.0,9234304.0,9267584.0,9300864.0,9334144.0,9367424.0,9400704.0,9433984.0,9467264.0,9220224.0,9252544.0,9284864.0,9317184.0,9349504.0,9381824.0,9414144.0,9446464.0,9197952.0,9229312.0,9260672.0,9292032.0,9323392.0,9354752.0,9386112.0,9417472.0,9216640.0,9248064.0,9279488.0,9310912.0,9342336.0,9373760.0,9405184.0,9436608.0,9235328.0,9266816.0,9298304.0,9329792.0,9361280.0,9392768.0,9424256.0,9455744.0,9254016.0,9285568.0,9317120.0,9348672.0,9380224.0,9411776.0,9443328.0,9474880.0,9272704.0,9304320.0,9335936.0,9367552.0,9399168.0,9430784.0,9462400.0,9494016.0,9291392.0,9323072.0,9354752.0,9386432.0,9418112.0,9449792.0,9481472.0,9513152.0,9310080.0,9341824.0,9373568.0,9405312.0,9437056.0,9468800.0,9500544.0,9532288.0,9328768.0,9360576.0,9392384.0,9424192.0,9456000.0,9487808.0,9519616.0,9551424.0,9347456.0,9379328.0,9411200.0,9443072.0,9474944.0,9506816.0,9538688.0,9570560.0,9366144.0,9398080.0,9430016.0,9461952.0,9493888.0,9525824.0,9557760.0,9589696.0,9384832.0,9416832.0,9448832.0,9480832.0,9512832.0,9544832.0,9576832.0,9608832.0,9403520.0,9435584.0,9467648.0,9499712.0,9531776.0,9563840.0,9595904.0,9627968.0,9422208.0,9454336.0,9486464.0,9518592.0,9550720.0,9582848.0,9614976.0,9647104.0,9440896.0,9473088.0,9505280.0,9537472.0,9569664.0,9601856.0,9634048.0,9666240.0,9459584.0,9491840.0,9524096.0,9556352.0,9588608.0,9620864.0,9653120.0,9685376.0,9478272.0,9510592.0,9542912.0,9575232.0,9607552.0,9639872.0,9672192.0,9704512.0,9496960.0,9529344.0,9561728.0,9594112.0,9626496.0,9658880.0,9691264.0,9723648.0,9515648.0,9548096.0,9580544.0,9612992.0,9645440.0,9677888.0,9710336.0,9742784.0,9534336.0,9566848.0,9599360.0,9631872.0,9664384.0,9696896.0,9729408.0,9761920.0,9553024.0,9585600.0,9618176.0,9650752.0,9683328.0,9715904.0,9748480.0,9781056.0,9571712.0,9604352.0,9636992.0,9669632.0,9702272.0,9734912.0,9767552.0,9800192.0,9590400.0,9623104.0,9655808.0,9688512.0,9721216.0,9753920.0,9786624.0,9819328.0,9609088.0,9641856.0,9674624.0,9707392.0,9740160.0,9772928.0,9805696.0,9838464.0,9627776.0,9660608.0,9693440.0,9726272.0,9759104.0,9791936.0,9824768.0,9857600.0,9646464.0,9679360.0,9712256.0,9745152.0,9778048.0,9810944.0,9843840.0,9876736.0,9665152.0,9698112.0,9731072.0,9764032.0,9796992.0,9829952.0,9862912.0,9895872.0,9683840.0,9716864.0,9749888.0,9782912.0,9815936.0,9848960.0,9881984.0,9915008.0,9702528.0,9735616.0,9768704.0,9801792.0,9834880.0,9867968.0,9901056.0,9934144.0,9721216.0,9754368.0,9787520.0,9820672.0,9853824.0,9886976.0,9920128.0,9953280.0,9739904.0,9773120.0,9806336.0,9839552.0,9872768.0,9905984.0,9939200.0,9972416.0,9758592.0,9791872.0,9825152.0,9858432.0,9891712.0,9924992.0,9958272.0,9991552.0,9777280.0,9810624.0,9843968.0,9877312.0,9910656.0,9944000.0,9977344.0,10010688.0,9795968.0,9829376.0,9862784.0,9896192.0,9929600.0,9963008.0,9996416.0,10029824.0,9814656.0,9848128.0,9881600.0,9915072.0,9948544.0,9982016.0,10015488.0,10048960.0,9833344.0,9866880.0,9900416.0,9933952.0,9967488.0,10001024.0,10034560.0,10068096.0,9852032.0,9885632.0,9919232.0,9952832.0,9986432.0,10020032.0,10053632.0,10087232.0,9870720.0,9904384.0,9938048.0,9971712.0,10005376.0,10039040.0,10072704.0,10106368.0,9889408.0,9923136.0,9956864.0,9990592.0,10024320.0,10058048.0,10091776.0,10125504.0,9351040.0,9383808.0,9416576.0,9449344.0,9482112.0,9514880.0,9547648.0,9580416.0,8804480.0,8836288.0,8868096.0,8899904.0,8931712.0,8963520.0,8995328.0,9027136.0,4470656.0,4486576.0,4502496.0,4518416.0,4534336.0,4550256.0,4566176.0,4582096.0,5083392.0,5099184.0,5114976.0,5130768.0,5146560.0,5162352.0,5178144.0,5193936.0,10302848.0,10334464.0,10366080.0,10397696.0,10429312.0,10460928.0,10492544.0,10524160.0,10322560.0,10354240.0,10385920.0,10417600.0,10449280.0,10480960.0,10512640.0,10544320.0,10342272.0,10374016.0,10405760.0,10437504.0,10469248.0,10500992.0,10532736.0,10564480.0,10361984.0,10393792.0,10425600.0,10457408.0,10489216.0,10521024.0,10552832.0,10584640.0,10381696.0,10413568.0,10445440.0,10477312.0,10509184.0,10541056.0,10572928.0,10604800.0,10401408.0,10433344.0,10465280.0,10497216.0,10529152.0,10561088.0,10593024.0,10624960.0,10421120.0,10453120.0,10485120.0,10517120.0,10549120.0,10581120.0,10613120.0,10645120.0,10440832.0,10472896.0,10504960.0,10537024.0,10569088.0,10601152.0,10633216.0,10665280.0,10460544.0,10492672.0,10524800.0,10556928.0,10589056.0,10621184.0,10653312.0,10685440.0,10480256.0,10512448.0,10544640.0,10576832.0,10609024.0,10641216.0,10673408.0,10705600.0,10499968.0,10532224.0,10564480.0,10596736.0,10628992.0,10661248.0,10693504.0,10725760.0,10519680.0,10552000.0,10584320.0,10616640.0,10648960.0,10681280.0,10713600.0,10745920.0,10539392.0,10571776.0,10604160.0,10636544.0,10668928.0,10701312.0,10733696.0,10766080.0,10559104.0,10591552.0,10624000.0,10656448.0,10688896.0,10721344.0,10753792.0,10786240.0,10578816.0,10611328.0,10643840.0,10676352.0,10708864.0,10741376.0,10773888.0,10806400.0,10598528.0,10631104.0,10663680.0,10696256.0,10728832.0,10761408.0,10793984.0,10826560.0,10618240.0,10650880.0,10683520.0,10716160.0,10748800.0,10781440.0,10814080.0,10846720.0,10637952.0,10670656.0,10703360.0,10736064.0,10768768.0,10801472.0,10834176.0,10866880.0,10657664.0,10690432.0,10723200.0,10755968.0,10788736.0,10821504.0,10854272.0,10887040.0,10677376.0,10710208.0,10743040.0,10775872.0,10808704.0,10841536.0,10874368.0,10907200.0,10697088.0,10729984.0,10762880.0,10795776.0,10828672.0,10861568.0,10894464.0,10927360.0,10716800.0,10749760.0,10782720.0,10815680.0,10848640.0,10881600.0,10914560.0,10947520.0,10736512.0,10769536.0,10802560.0,10835584.0,10868608.0,10901632.0,10934656.0,10967680.0,10756224.0,10789312.0,10822400.0,10855488.0,10888576.0,10921664.0,10954752.0,10987840.0,10775936.0,10809088.0,10842240.0,10875392.0,10908544.0,10941696.0,10974848.0,11008000.0,10795648.0,10828864.0,10862080.0,10895296.0,10928512.0,10961728.0,10994944.0,11028160.0,10815360.0,10848640.0,10881920.0,10915200.0,10948480.0,10981760.0,11015040.0,11048320.0,10835072.0,10868416.0,10901760.0,10935104.0,10968448.0,11001792.0,11035136.0,11068480.0,10854784.0,10888192.0,10921600.0,10955008.0,10988416.0,11021824.0,11055232.0,11088640.0,10874496.0,10907968.0,10941440.0,10974912.0,11008384.0,11041856.0,11075328.0,11108800.0,10894208.0,10927744.0,10961280.0,10994816.0,11028352.0,11061888.0,11095424.0,11128960.0,10913920.0,10947520.0,10981120.0,11014720.0,11048320.0,11081920.0,11115520.0,11149120.0,10933632.0,10967296.0,11000960.0,11034624.0,11068288.0,11101952.0,11135616.0,11169280.0,10953344.0,10987072.0,11020800.0,11054528.0,11088256.0,11121984.0,11155712.0,11189440.0,10973056.0,11006848.0,11040640.0,11074432.0,11108224.0,11142016.0,11175808.0,11209600.0,10992768.0,11026624.0,11060480.0,11094336.0,11128192.0,11162048.0,11195904.0,11229760.0,10439040.0,10471936.0,10504832.0,10537728.0,10570624.0,10603520.0,10636416.0,10669312.0,9877120.0,9909056.0,9940992.0,9972928.0,10004864.0,10036800.0,10068736.0,10100672.0,9323392.0,9354368.0,9385344.0,9416320.0,9447296.0,9478272.0,9509248.0,9540224.0,8761472.0,8791488.0,8821504.0,8851520.0,8881536.0,8911552.0,8941568.0,8971584.0,8781184.0,8811264.0,8841344.0,8871424.0,8901504.0,8931584.0,8961664.0,8991744.0,8800896.0,8831040.0,8861184.0,8891328.0,8921472.0,8951616.0,8981760.0,9011904.0,8820608.0,8850816.0,8881024.0,8911232.0,8941440.0,8971648.0,9001856.0,9032064.0,8840320.0,8870592.0,8900864.0,8931136.0,8961408.0,8991680.0,9021952.0,9052224.0,8860032.0,8890368.0,8920704.0,8951040.0,8981376.0,9011712.0,9042048.0,9072384.0,8879744.0,8910144.0,8940544.0,8970944.0,9001344.0,9031744.0,9062144.0,9092544.0,8899456.0,8929920.0,8960384.0,8990848.0,9021312.0,9051776.0,9082240.0,9112704.0,8919168.0,8949696.0,8980224.0,9010752.0,9041280.0,9071808.0,9102336.0,9132864.0,8938880.0,8969472.0,9000064.0,9030656.0,9061248.0,9091840.0,9122432.0,9153024.0,8958592.0,8989248.0,9019904.0,9050560.0,9081216.0,9111872.0,9142528.0,9173184.0,8978304.0,9009024.0,9039744.0,9070464.0,9101184.0,9131904.0,9162624.0,9193344.0,8998016.0,9028800.0,9059584.0,9090368.0,9121152.0,9151936.0,9182720.0,9213504.0,9017728.0,9048576.0,9079424.0,9110272.0,9141120.0,9171968.0,9202816.0,9233664.0,9037440.0,9068352.0,9099264.0,9130176.0,9161088.0,9192000.0,9222912.0,9253824.0,9057152.0,9088128.0,9119104.0,9150080.0,9181056.0,9212032.0,9243008.0,9273984.0,9076864.0,9107904.0,9138944.0,9169984.0,9201024.0,9232064.0,9263104.0,9294144.0,9096576.0,9127680.0,9158784.0,9189888.0,9220992.0,9252096.0,9283200.0,9314304.0,9116288.0,9147456.0,9178624.0,9209792.0,9240960.0,9272128.0,9303296.0,9334464.0,9136000.0,9167232.0,9198464.0,9229696.0,9260928.0,9292160.0,9323392.0,9354624.0,9155712.0,9187008.0,9218304.0,9249600.0,9280896.0,9312192.0,9343488.0,9374784.0,9175424.0,9206784.0,9238144.0,9269504.0,9300864.0,9332224.0,9363584.0,9394944.0,9195136.0,9226560.0,9257984.0,9289408.0,9320832.0,9352256.0,9383680.0,9415104.0,9214848.0,9246336.0,9277824.0,9309312.0,9340800.0,9372288.0,9403776.0,9435264.0,9234560.0,9266112.0,9297664.0,9329216.0,9360768.0,9392320.0,9423872.0,9455424.0,9254272.0,9285888.0,9317504.0,9349120.0,9380736.0,9412352.0,9443968.0,9475584.0,9273984.0,9305664.0,9337344.0,9369024.0,9400704.0,9432384.0,9464064.0,9495744.0,9293696.0,9325440.0,9357184.0,9388928.0,9420672.0,9452416.0,9484160.0,9515904.0,9313408.0,9345216.0,9377024.0,9408832.0,9440640.0,9472448.0,9504256.0,9536064.0,9333120.0,9364992.0,9396864.0,9428736.0,9460608.0,9492480.0,9524352.0,9556224.0,9352832.0,9384768.0,9416704.0,9448640.0,9480576.0,9512512.0,9544448.0,9576384.0,9372544.0,9404544.0,9436544.0,9468544.0,9500544.0,9532544.0,9564544.0,9596544.0,9392256.0,9424320.0,9456384.0,9488448.0,9520512.0,9552576.0,9584640.0,9616704.0,9411968.0,9444096.0,9476224.0,9508352.0,9540480.0,9572608.0,9604736.0,9636864.0,9431680.0,9463872.0,9496064.0,9528256.0,9560448.0,9592640.0,9624832.0,9657024.0,9451392.0,9483648.0,9515904.0,9548160.0,9580416.0,9612672.0,9644928.0,9677184.0,9471104.0,9503424.0,9535744.0,9568064.0,9600384.0,9632704.0,9665024.0,9697344.0,9490816.0,9523200.0,9555584.0,9587968.0,9620352.0,9652736.0,9685120.0,9717504.0,9461376.0,9492800.0,9524224.0,9555648.0,9587072.0,9618496.0,9649920.0,9681344.0,9423744.0,9454208.0,9484672.0,9515136.0,9545600.0,9576064.0,9606528.0,9636992.0,9443456.0,9473984.0,9504512.0,9535040.0,9565568.0,9596096.0,9626624.0,9657152.0,9463168.0,9493760.0,9524352.0,9554944.0,9585536.0,9616128.0,9646720.0,9677312.0,9482880.0,9513536.0,9544192.0,9574848.0,9605504.0,9636160.0,9666816.0,9697472.0,9502592.0,9533312.0,9564032.0,9594752.0,9625472.0,9656192.0,9686912.0,9717632.0,9522304.0,9553088.0,9583872.0,9614656.0,9645440.0,9676224.0,9707008.0,9737792.0,9542016.0,9572864.0,9603712.0,9634560.0,9665408.0,9696256.0,9727104.0,9757952.0,9561728.0,9592640.0,9623552.0,9654464.0,9685376.0,9716288.0,9747200.0,9778112.0,9581440.0,9612416.0,9643392.0,9674368.0,9705344.0,9736320.0,9767296.0,9798272.0,9601152.0,9632192.0,9663232.0,9694272.0,9725312.0,9756352.0,9787392.0,9818432.0,9620864.0,9651968.0,9683072.0,9714176.0,9745280.0,9776384.0,9807488.0,9838592.0,9640576.0,9671744.0,9702912.0,9734080.0,9765248.0,9796416.0,9827584.0,9858752.0,9660288.0,9691520.0,9722752.0,9753984.0,9785216.0,9816448.0,9847680.0,9878912.0,9680000.0,9711296.0,9742592.0,9773888.0,9805184.0,9836480.0,9867776.0,9899072.0,9699712.0,9731072.0,9762432.0,9793792.0,9825152.0,9856512.0,9887872.0,9919232.0,9719424.0,9750848.0,9782272.0,9813696.0,9845120.0,9876544.0,9907968.0,9939392.0,9739136.0,9770624.0,9802112.0,9833600.0,9865088.0,9896576.0,9928064.0,9959552.0,9758848.0,9790400.0,9821952.0,9853504.0,9885056.0,9916608.0,9948160.0,9979712.0,9778560.0,9810176.0,9841792.0,9873408.0,9905024.0,9936640.0,9968256.0,9999872.0,9798272.0,9829952.0,9861632.0,9893312.0,9924992.0,9956672.0,9988352.0,10020032.0,9817984.0,9849728.0,9881472.0,9913216.0,9944960.0,9976704.0,10008448.0,10040192.0,9837696.0,9869504.0,9901312.0,9933120.0,9964928.0,9996736.0,10028544.0,10060352.0,9857408.0,9889280.0,9921152.0,9953024.0,9984896.0,10016768.0,10048640.0,10080512.0,9877120.0,9909056.0,9940992.0,9972928.0,10004864.0,10036800.0,10068736.0,10100672.0,9896832.0,9928832.0,9960832.0,9992832.0,10024832.0,10056832.0,10088832.0,10120832.0,9916544.0,9948608.0,9980672.0,10012736.0,10044800.0,10076864.0,10108928.0,10140992.0,9936256.0,9968384.0,10000512.0,10032640.0,10064768.0,10096896.0,10129024.0,10161152.0,9955968.0,9988160.0,10020352.0,10052544.0,10084736.0,10116928.0,10149120.0,10181312.0,9975680.0,10007936.0,10040192.0,10072448.0,10104704.0,10136960.0,10169216.0,10201472.0,9995392.0,10027712.0,10060032.0,10092352.0,10124672.0,10156992.0,10189312.0,10221632.0,10015104.0,10047488.0,10079872.0,10112256.0,10144640.0,10177024.0,10209408.0,10241792.0,10034816.0,10067264.0,10099712.0,10132160.0,10164608.0,10197056.0,10229504.0,10261952.0,10054528.0,10087040.0,10119552.0,10152064.0,10184576.0,10217088.0,10249600.0,10282112.0,10074240.0,10106816.0,10139392.0,10171968.0,10204544.0,10237120.0,10269696.0,10302272.0,10093952.0,10126592.0,10159232.0,10191872.0,10224512.0,10257152.0,10289792.0,10322432.0,10113664.0,10146368.0,10179072.0,10211776.0,10244480.0,10277184.0,10309888.0,10342592.0,10133376.0,10166144.0,10198912.0,10231680.0,10264448.0,10297216.0,10329984.0,10362752.0,10153088.0,10185920.0,10218752.0,10251584.0,10284416.0,10317248.0,10350080.0,10382912.0,9599360.0,9631232.0,9663104.0,9694976.0,9726848.0,9758720.0,9790592.0,9822464.0,9037440.0,9068352.0,9099264.0,9130176.0,9161088.0,9192000.0,9222912.0,9253824.0,9057152.0,9088128.0,9119104.0,9150080.0,9181056.0,9212032.0,9243008.0,9273984.0,9076864.0,9107904.0,9138944.0,9169984.0,9201024.0,9232064.0,9263104.0,9294144.0,9096576.0,9127680.0,9158784.0,9189888.0,9220992.0,9252096.0,9283200.0,9314304.0,9116288.0,9147456.0,9178624.0,9209792.0,9240960.0,9272128.0,9303296.0,9334464.0,9136000.0,9167232.0,9198464.0,9229696.0,9260928.0,9292160.0,9323392.0,9354624.0,9155712.0,9187008.0,9218304.0,9249600.0,9280896.0,9312192.0,9343488.0,9374784.0,9175424.0,9206784.0,9238144.0,9269504.0,9300864.0,9332224.0,9363584.0,9394944.0,9195136.0,9226560.0,9257984.0,9289408.0,9320832.0,9352256.0,9383680.0,9415104.0,9214848.0,9246336.0,9277824.0,9309312.0,9340800.0,9372288.0,9403776.0,9435264.0,9234560.0,9266112.0,9297664.0,9329216.0,9360768.0,9392320.0,9423872.0,9455424.0,9254272.0,9285888.0,9317504.0,9349120.0,9380736.0,9412352.0,9443968.0,9475584.0,9273984.0,9305664.0,9337344.0,9369024.0,9400704.0,9432384.0,9464064.0,9495744.0,9293696.0,9325440.0,9357184.0,9388928.0,9420672.0,9452416.0,9484160.0,9515904.0,9313408.0,9345216.0,9377024.0,9408832.0,9440640.0,9472448.0,9504256.0,9536064.0,9333120.0,9364992.0,9396864.0,9428736.0,9460608.0,9492480.0,9524352.0,9556224.0,9352832.0,9384768.0,9416704.0,9448640.0,9480576.0,9512512.0,9544448.0,9576384.0,9372544.0,9404544.0,9436544.0,9468544.0,9500544.0,9532544.0,9564544.0,9596544.0,9392256.0,9424320.0,9456384.0,9488448.0,9520512.0,9552576.0,9584640.0,9616704.0,9411968.0,9444096.0,9476224.0,9508352.0,9540480.0,9572608.0,9604736.0,9636864.0,9431680.0,9463872.0,9496064.0,9528256.0,9560448.0,9592640.0,9624832.0,9657024.0,9451392.0,9483648.0,9515904.0,9548160.0,9580416.0,9612672.0,9644928.0,9677184.0,9471104.0,9503424.0,9535744.0,9568064.0,9600384.0,9632704.0,9665024.0,9697344.0,9490816.0,9523200.0,9555584.0,9587968.0,9620352.0,9652736.0,9685120.0,9717504.0,9510528.0,9542976.0,9575424.0,9607872.0,9640320.0,9672768.0,9705216.0,9737664.0,9530240.0,9562752.0,9595264.0,9627776.0,9660288.0,9692800.0,9725312.0,9757824.0,9549952.0,9582528.0,9615104.0,9647680.0,9680256.0,9712832.0,9745408.0,9777984.0,9569664.0,9602304.0,9634944.0,9667584.0,9700224.0,9732864.0,9765504.0,9798144.0,9589376.0,9622080.0,9654784.0,9687488.0,9720192.0,9752896.0,9785600.0,9818304.0,9609088.0,9641856.0,9674624.0,9707392.0,9740160.0,9772928.0,9805696.0,9838464.0,9628800.0,9661632.0,9694464.0,9727296.0,9760128.0,9792960.0,9825792.0,9858624.0,9648512.0,9681408.0,9714304.0,9747200.0,9780096.0,9812992.0,9845888.0,9878784.0,9668224.0,9701184.0,9734144.0,9767104.0,9800064.0,9833024.0,9865984.0,9898944.0,9687936.0,9720960.0,9753984.0,9787008.0,9820032.0,9853056.0,9886080.0,9919104.0,9707648.0,9740736.0,9773824.0,9806912.0,9840000.0,9873088.0,9906176.0,9939264.0,9727360.0,9760512.0,9793664.0,9826816.0,9859968.0,9893120.0,9926272.0,9959424.0,9747072.0,9780288.0,9813504.0,9846720.0,9879936.0,9913152.0,9946368.0,9979584.0,9766784.0,9800064.0,9833344.0,9866624.0,9899904.0,9933184.0,9966464.0,9999744.0,9737344.0,9769664.0,9801984.0,9834304.0,9866624.0,9898944.0,9931264.0,9963584.0,9699712.0,9731072.0,9762432.0,9793792.0,9825152.0,9856512.0,9887872.0,9919232.0,9719424.0,9750848.0,9782272.0,9813696.0,9845120.0,9876544.0,9907968.0,9939392.0,9739136.0,9770624.0,9802112.0,9833600.0,9865088.0,9896576.0,9928064.0,9959552.0,9758848.0,9790400.0,9821952.0,9853504.0,9885056.0,9916608.0,9948160.0,9979712.0,9778560.0,9810176.0,9841792.0,9873408.0,9905024.0,9936640.0,9968256.0,9999872.0,9798272.0,9829952.0,9861632.0,9893312.0,9924992.0,9956672.0,9988352.0,10020032.0,9817984.0,9849728.0,9881472.0,9913216.0,9944960.0,9976704.0,10008448.0,10040192.0,9837696.0,9869504.0,9901312.0,9933120.0,9964928.0,9996736.0,10028544.0,10060352.0,9857408.0,9889280.0,9921152.0,9953024.0,9984896.0,10016768.0,10048640.0,10080512.0,9877120.0,9909056.0,9940992.0,9972928.0,10004864.0,10036800.0,10068736.0,10100672.0,9896832.0,9928832.0,9960832.0,9992832.0,10024832.0,10056832.0,10088832.0,10120832.0,9916544.0,9948608.0,9980672.0,10012736.0,10044800.0,10076864.0,10108928.0,10140992.0,9936256.0,9968384.0,10000512.0,10032640.0,10064768.0,10096896.0,10129024.0,10161152.0,9955968.0,9988160.0,10020352.0,10052544.0,10084736.0,10116928.0,10149120.0,10181312.0,9975680.0,10007936.0,10040192.0,10072448.0,10104704.0,10136960.0,10169216.0,10201472.0,9995392.0,10027712.0,10060032.0,10092352.0,10124672.0,10156992.0,10189312.0,10221632.0,10015104.0,10047488.0,10079872.0,10112256.0,10144640.0,10177024.0,10209408.0,10241792.0,10034816.0,10067264.0,10099712.0,10132160.0,10164608.0,10197056.0,10229504.0,10261952.0,10054528.0,10087040.0,10119552.0,10152064.0,10184576.0,10217088.0,10249600.0,10282112.0,10074240.0,10106816.0,10139392.0,10171968.0,10204544.0,10237120.0,10269696.0,10302272.0,10093952.0,10126592.0,10159232.0,10191872.0,10224512.0,10257152.0,10289792.0,10322432.0,10113664.0,10146368.0,10179072.0,10211776.0,10244480.0,10277184.0,10309888.0,10342592.0,10133376.0,10166144.0,10198912.0,10231680.0,10264448.0,10297216.0,10329984.0,10362752.0,10153088.0,10185920.0,10218752.0,10251584.0,10284416.0,10317248.0,10350080.0,10382912.0,10172800.0,10205696.0,10238592.0,10271488.0,10304384.0,10337280.0,10370176.0,10403072.0,10192512.0,10225472.0,10258432.0,10291392.0,10324352.0,10357312.0,10390272.0,10423232.0,10212224.0,10245248.0,10278272.0,10311296.0,10344320.0,10377344.0,10410368.0,10443392.0,10231936.0,10265024.0,10298112.0,10331200.0,10364288.0,10397376.0,10430464.0,10463552.0,10251648.0,10284800.0,10317952.0,10351104.0,10384256.0,10417408.0,10450560.0,10483712.0,10271360.0,10304576.0,10337792.0,10371008.0,10404224.0,10437440.0,10470656.0,10503872.0,10291072.0,10324352.0,10357632.0,10390912.0,10424192.0,10457472.0,10490752.0,10524032.0,10310784.0,10344128.0,10377472.0,10410816.0,10444160.0,10477504.0,10510848.0,10544192.0,10330496.0,10363904.0,10397312.0,10430720.0,10464128.0,10497536.0,10530944.0,10564352.0,10350208.0,10383680.0,10417152.0,10450624.0,10484096.0,10517568.0,10551040.0,10584512.0,10369920.0,10403456.0,10436992.0,10470528.0,10504064.0,10537600.0,10571136.0,10604672.0,10389632.0,10423232.0,10456832.0,10490432.0,10524032.0,10557632.0,10591232.0,10624832.0,10409344.0,10443008.0,10476672.0,10510336.0,10544000.0,10577664.0,10611328.0,10644992.0,10429056.0,10462784.0,10496512.0,10530240.0,10563968.0,10597696.0,10631424.0,10665152.0,9875328.0,9908096.0,9940864.0,9973632.0,10006400.0,10039168.0,10071936.0,10104704.0,9313408.0,9345216.0,9377024.0,9408832.0,9440640.0,9472448.0,9504256.0,9536064.0,4725376.0,4741296.0,4757216.0,4773136.0,4789056.0,4804976.0,4820896.0,4836816.0,5336064.0,5351856.0,5367648.0,5383440.0,5399232.0,5415024.0,5430816.0,5446608.0,10808704.0,10840320.0,10871936.0,10903552.0,10935168.0,10966784.0,10998400.0,11030016.0,10829440.0,10861120.0,10892800.0,10924480.0,10956160.0,10987840.0,11019520.0,11051200.0,10850176.0,10881920.0,10913664.0,10945408.0,10977152.0,11008896.0,11040640.0,11072384.0,10870912.0,10902720.0,10934528.0,10966336.0,10998144.0,11029952.0,11061760.0,11093568.0,10891648.0,10923520.0,10955392.0,10987264.0,11019136.0,11051008.0,11082880.0,11114752.0,10912384.0,10944320.0,10976256.0,11008192.0,11040128.0,11072064.0,11104000.0,11135936.0,10933120.0,10965120.0,10997120.0,11029120.0,11061120.0,11093120.0,11125120.0,11157120.0,10953856.0,10985920.0,11017984.0,11050048.0,11082112.0,11114176.0,11146240.0,11178304.0,10974592.0,11006720.0,11038848.0,11070976.0,11103104.0,11135232.0,11167360.0,11199488.0,10995328.0,11027520.0,11059712.0,11091904.0,11124096.0,11156288.0,11188480.0,11220672.0,11016064.0,11048320.0,11080576.0,11112832.0,11145088.0,11177344.0,11209600.0,11241856.0,11036800.0,11069120.0,11101440.0,11133760.0,11166080.0,11198400.0,11230720.0,11263040.0,11057536.0,11089920.0,11122304.0,11154688.0,11187072.0,11219456.0,11251840.0,11284224.0,11078272.0,11110720.0,11143168.0,11175616.0,11208064.0,11240512.0,11272960.0,11305408.0,11099008.0,11131520.0,11164032.0,11196544.0,11229056.0,11261568.0,11294080.0,11326592.0,11119744.0,11152320.0,11184896.0,11217472.0,11250048.0,11282624.0,11315200.0,11347776.0,11140480.0,11173120.0,11205760.0,11238400.0,11271040.0,11303680.0,11336320.0,11368960.0,11161216.0,11193920.0,11226624.0,11259328.0,11292032.0,11324736.0,11357440.0,11390144.0,11181952.0,11214720.0,11247488.0,11280256.0,11313024.0,11345792.0,11378560.0,11411328.0,11202688.0,11235520.0,11268352.0,11301184.0,11334016.0,11366848.0,11399680.0,11432512.0,11223424.0,11256320.0,11289216.0,11322112.0,11355008.0,11387904.0,11420800.0,11453696.0,11244160.0,11277120.0,11310080.0,11343040.0,11376000.0,11408960.0,11441920.0,11474880.0,11264896.0,11297920.0,11330944.0,11363968.0,11396992.0,11430016.0,11463040.0,11496064.0,11285632.0,11318720.0,11351808.0,11384896.0,11417984.0,11451072.0,11484160.0,11517248.0,11306368.0,11339520.0,11372672.0,11405824.0,11438976.0,11472128.0,11505280.0,11538432.0,11327104.0,11360320.0,11393536.0,11426752.0,11459968.0,11493184.0,11526400.0,11559616.0,11347840.0,11381120.0,11414400.0,11447680.0,11480960.0,11514240.0,11547520.0,11580800.0,11368576.0,11401920.0,11435264.0,11468608.0,11501952.0,11535296.0,11568640.0,11601984.0,11389312.0,11422720.0,11456128.0,11489536.0,11522944.0,11556352.0,11589760.0,11623168.0,11410048.0,11443520.0,11476992.0,11510464.0,11543936.0,11577408.0,11610880.0,11644352.0,11430784.0,11464320.0,11497856.0,11531392.0,11564928.0,11598464.0,11632000.0,11665536.0,11451520.0,11485120.0,11518720.0,11552320.0,11585920.0,11619520.0,11653120.0,11686720.0,11472256.0,11505920.0,11539584.0,11573248.0,11606912.0,11640576.0,11674240.0,11707904.0,11492992.0,11526720.0,11560448.0,11594176.0,11627904.0,11661632.0,11695360.0,11729088.0,11513728.0,11547520.0,11581312.0,11615104.0,11648896.0,11682688.0,11716480.0,11750272.0,11534464.0,11568320.0,11602176.0,11636032.0,11669888.0,11703744.0,11737600.0,11771456.0,10965376.0,10998272.0,11031168.0,11064064.0,11096960.0,11129856.0,11162752.0,11195648.0,10388096.0,10420032.0,10451968.0,10483904.0,10515840.0,10547776.0,10579712.0,10611648.0,9819008.0,9849984.0,9880960.0,9911936.0,9942912.0,9973888.0,10004864.0,10035840.0,9241728.0,9271744.0,9301760.0,9331776.0,9361792.0,9391808.0,9421824.0,9451840.0,9262464.0,9292544.0,9322624.0,9352704.0,9382784.0,9412864.0,9442944.0,9473024.0,9283200.0,9313344.0,9343488.0,9373632.0,9403776.0,9433920.0,9464064.0,9494208.0,9303936.0,9334144.0,9364352.0,9394560.0,9424768.0,9454976.0,9485184.0,9515392.0,9324672.0,9354944.0,9385216.0,9415488.0,9445760.0,9476032.0,9506304.0,9536576.0,9345408.0,9375744.0,9406080.0,9436416.0,9466752.0,9497088.0,9527424.0,9557760.0,9366144.0,9396544.0,9426944.0,9457344.0,9487744.0,9518144.0,9548544.0,9578944.0,9386880.0,9417344.0,9447808.0,9478272.0,9508736.0,9539200.0,9569664.0,9600128.0,9407616.0,9438144.0,9468672.0,9499200.0,9529728.0,9560256.0,9590784.0,9621312.0,9428352.0,9458944.0,9489536.0,9520128.0,9550720.0,9581312.0,9611904.0,9642496.0,9449088.0,9479744.0,9510400.0,9541056.0,9571712.0,9602368.0,9633024.0,9663680.0,9469824.0,9500544.0,9531264.0,9561984.0,9592704.0,9623424.0,9654144.0,9684864.0,9490560.0,9521344.0,9552128.0,9582912.0,9613696.0,9644480.0,9675264.0,9706048.0,9511296.0,9542144.0,9572992.0,9603840.0,9634688.0,9665536.0,9696384.0,9727232.0,9532032.0,9562944.0,9593856.0,9624768.0,9655680.0,9686592.0,9717504.0,9748416.0,9552768.0,9583744.0,9614720.0,9645696.0,9676672.0,9707648.0,9738624.0,9769600.0,9573504.0,9604544.0,9635584.0,9666624.0,9697664.0,9728704.0,9759744.0,9790784.0,9594240.0,9625344.0,9656448.0,9687552.0,9718656.0,9749760.0,9780864.0,9811968.0,9614976.0,9646144.0,9677312.0,9708480.0,9739648.0,9770816.0,9801984.0,9833152.0,9635712.0,9666944.0,9698176.0,9729408.0,9760640.0,9791872.0,9823104.0,9854336.0,9656448.0,9687744.0,9719040.0,9750336.0,9781632.0,9812928.0,9844224.0,9875520.0,9677184.0,9708544.0,9739904.0,9771264.0,9802624.0,9833984.0,9865344.0,9896704.0,9697920.0,9729344.0,9760768.0,9792192.0,9823616.0,9855040.0,9886464.0,9917888.0,9718656.0,9750144.0,9781632.0,9813120.0,9844608.0,9876096.0,9907584.0,9939072.0,9739392.0,9770944.0,9802496.0,9834048.0,9865600.0,9897152.0,9928704.0,9960256.0,9760128.0,9791744.0,9823360.0,9854976.0,9886592.0,9918208.0,9949824.0,9981440.0,9780864.0,9812544.0,9844224.0,9875904.0,9907584.0,9939264.0,9970944.0,10002624.0,9801600.0,9833344.0,9865088.0,9896832.0,9928576.0,9960320.0,9992064.0,10023808.0,9822336.0,9854144.0,9885952.0,9917760.0,9949568.0,9981376.0,10013184.0,10044992.0,9843072.0,9874944.0,9906816.0,9938688.0,9970560.0,10002432.0,10034304.0,10066176.0,9863808.0,9895744.0,9927680.0,9959616.0,9991552.0,10023488.0,10055424.0,10087360.0,9884544.0,9916544.0,9948544.0,9980544.0,10012544.0,10044544.0,10076544.0,10108544.0,9905280.0,9937344.0,9969408.0,10001472.0,10033536.0,10065600.0,10097664.0,10129728.0,9926016.0,9958144.0,9990272.0,10022400.0,10054528.0,10086656.0,10118784.0,10150912.0,9946752.0,9978944.0,10011136.0,10043328.0,10075520.0,10107712.0,10139904.0,10172096.0,9967488.0,9999744.0,10032000.0,10064256.0,10096512.0,10128768.0,10161024.0,10193280.0,9988224.0,10020544.0,10052864.0,10085184.0,10117504.0,10149824.0,10182144.0,10214464.0,10008960.0,10041344.0,10073728.0,10106112.0,10138496.0,10170880.0,10203264.0,10235648.0,9964160.0,9995584.0,10027008.0,10058432.0,10089856.0,10121280.0,10152704.0,10184128.0,9911168.0,9941632.0,9972096.0,10002560.0,10033024.0,10063488.0,10093952.0,10124416.0,9931904.0,9962432.0,9992960.0,10023488.0,10054016.0,10084544.0,10115072.0,10145600.0,9952640.0,9983232.0,10013824.0,10044416.0,10075008.0,10105600.0,10136192.0,10166784.0,9973376.0,10004032.0,10034688.0,10065344.0,10096000.0,10126656.0,10157312.0,10187968.0,9994112.0,10024832.0,10055552.0,10086272.0,10116992.0,10147712.0,10178432.0,10209152.0,10014848.0,10045632.0,10076416.0,10107200.0,10137984.0,10168768.0,10199552.0,10230336.0,10035584.0,10066432.0,10097280.0,10128128.0,10158976.0,10189824.0,10220672.0,10251520.0,10056320.0,10087232.0,10118144.0,10149056.0,10179968.0,10210880.0,10241792.0,10272704.0,10077056.0,10108032.0,10139008.0,10169984.0,10200960.0,10231936.0,10262912.0,10293888.0,10097792.0,10128832.0,10159872.0,10190912.0,10221952.0,10252992.0,10284032.0,10315072.0,10118528.0,10149632.0,10180736.0,10211840.0,10242944.0,10274048.0,10305152.0,10336256.0,10139264.0,10170432.0,10201600.0,10232768.0,10263936.0,10295104.0,10326272.0,10357440.0,10160000.0,10191232.0,10222464.0,10253696.0,10284928.0,10316160.0,10347392.0,10378624.0,10180736.0,10212032.0,10243328.0,10274624.0,10305920.0,10337216.0,10368512.0,10399808.0,10201472.0,10232832.0,10264192.0,10295552.0,10326912.0,10358272.0,10389632.0,10420992.0,10222208.0,10253632.0,10285056.0,10316480.0,10347904.0,10379328.0,10410752.0,10442176.0,10242944.0,10274432.0,10305920.0,10337408.0,10368896.0,10400384.0,10431872.0,10463360.0,10263680.0,10295232.0,10326784.0,10358336.0,10389888.0,10421440.0,10452992.0,10484544.0,10284416.0,10316032.0,10347648.0,10379264.0,10410880.0,10442496.0,10474112.0,10505728.0,10305152.0,10336832.0,10368512.0,10400192.0,10431872.0,10463552.0,10495232.0,10526912.0,10325888.0,10357632.0,10389376.0,10421120.0,10452864.0,10484608.0,10516352.0,10548096.0,10346624.0,10378432.0,10410240.0,10442048.0,10473856.0,10505664.0,10537472.0,10569280.0,10367360.0,10399232.0,10431104.0,10462976.0,10494848.0,10526720.0,10558592.0,10590464.0,10388096.0,10420032.0,10451968.0,10483904.0,10515840.0,10547776.0,10579712.0,10611648.0,10408832.0,10440832.0,10472832.0,10504832.0,10536832.0,10568832.0,10600832.0,10632832.0,10429568.0,10461632.0,10493696.0,10525760.0,10557824.0,10589888.0,10621952.0,10654016.0,10450304.0,10482432.0,10514560.0,10546688.0,10578816.0,10610944.0,10643072.0,10675200.0,10471040.0,10503232.0,10535424.0,10567616.0,10599808.0,10632000.0,10664192.0,10696384.0,10491776.0,10524032.0,10556288.0,10588544.0,10620800.0,10653056.0,10685312.0,10717568.0,10512512.0,10544832.0,10577152.0,10609472.0,10641792.0,10674112.0,10706432.0,10738752.0,10533248.0,10565632.0,10598016.0,10630400.0,10662784.0,10695168.0,10727552.0,10759936.0,10553984.0,10586432.0,10618880.0,10651328.0,10683776.0,10716224.0,10748672.0,10781120.0,10574720.0,10607232.0,10639744.0,10672256.0,10704768.0,10737280.0,10769792.0,10802304.0,10595456.0,10628032.0,10660608.0,10693184.0,10725760.0,10758336.0,10790912.0,10823488.0,10616192.0,10648832.0,10681472.0,10714112.0,10746752.0,10779392.0,10812032.0,10844672.0,10636928.0,10669632.0,10702336.0,10735040.0,10767744.0,10800448.0,10833152.0,10865856.0,10657664.0,10690432.0,10723200.0,10755968.0,10788736.0,10821504.0,10854272.0,10887040.0,10678400.0,10711232.0,10744064.0,10776896.0,10809728.0,10842560.0,10875392.0,10908224.0,10109312.0,10141184.0,10173056.0,10204928.0,10236800.0,10268672.0,10300544.0,10332416.0,9532032.0,9562944.0,9593856.0,9624768.0,9655680.0,9686592.0,9717504.0,9748416.0,9552768.0,9583744.0,9614720.0,9645696.0,9676672.0,9707648.0,9738624.0,9769600.0,9573504.0,9604544.0,9635584.0,9666624.0,9697664.0,9728704.0,9759744.0,9790784.0,9594240.0,9625344.0,9656448.0,9687552.0,9718656.0,9749760.0,9780864.0,9811968.0,9614976.0,9646144.0,9677312.0,9708480.0,9739648.0,9770816.0,9801984.0,9833152.0,9635712.0,9666944.0,9698176.0,9729408.0,9760640.0,9791872.0,9823104.0,9854336.0,9656448.0,9687744.0,9719040.0,9750336.0,9781632.0,9812928.0,9844224.0,9875520.0,9677184.0,9708544.0,9739904.0,9771264.0,9802624.0,9833984.0,9865344.0,9896704.0,9697920.0,9729344.0,9760768.0,9792192.0,9823616.0,9855040.0,9886464.0,9917888.0,9718656.0,9750144.0,9781632.0,9813120.0,9844608.0,9876096.0,9907584.0,9939072.0,9739392.0,9770944.0,9802496.0,9834048.0,9865600.0,9897152.0,9928704.0,9960256.0,9760128.0,9791744.0,9823360.0,9854976.0,9886592.0,9918208.0,9949824.0,9981440.0,9780864.0,9812544.0,9844224.0,9875904.0,9907584.0,9939264.0,9970944.0,10002624.0,9801600.0,9833344.0,9865088.0,9896832.0,9928576.0,9960320.0,9992064.0,10023808.0,9822336.0,9854144.0,9885952.0,9917760.0,9949568.0,9981376.0,10013184.0,10044992.0,9843072.0,9874944.0,9906816.0,9938688.0,9970560.0,10002432.0,10034304.0,10066176.0,9863808.0,9895744.0,9927680.0,9959616.0,9991552.0,10023488.0,10055424.0,10087360.0,9884544.0,9916544.0,9948544.0,9980544.0,10012544.0,10044544.0,10076544.0,10108544.0,9905280.0,9937344.0,9969408.0,10001472.0,10033536.0,10065600.0,10097664.0,10129728.0,9926016.0,9958144.0,9990272.0,10022400.0,10054528.0,10086656.0,10118784.0,10150912.0,9946752.0,9978944.0,10011136.0,10043328.0,10075520.0,10107712.0,10139904.0,10172096.0,9967488.0,9999744.0,10032000.0,10064256.0,10096512.0,10128768.0,10161024.0,10193280.0,9988224.0,10020544.0,10052864.0,10085184.0,10117504.0,10149824.0,10182144.0,10214464.0,10008960.0,10041344.0,10073728.0,10106112.0,10138496.0,10170880.0,10203264.0,10235648.0,10029696.0,10062144.0,10094592.0,10127040.0,10159488.0,10191936.0,10224384.0,10256832.0,10050432.0,10082944.0,10115456.0,10147968.0,10180480.0,10212992.0,10245504.0,10278016.0,10071168.0,10103744.0,10136320.0,10168896.0,10201472.0,10234048.0,10266624.0,10299200.0,10091904.0,10124544.0,10157184.0,10189824.0,10222464.0,10255104.0,10287744.0,10320384.0,10112640.0,10145344.0,10178048.0,10210752.0,10243456.0,10276160.0,10308864.0,10341568.0,10133376.0,10166144.0,10198912.0,10231680.0,10264448.0,10297216.0,10329984.0,10362752.0,10154112.0,10186944.0,10219776.0,10252608.0,10285440.0,10318272.0,10351104.0,10383936.0,10174848.0,10207744.0,10240640.0,10273536.0,10306432.0,10339328.0,10372224.0,10405120.0,10195584.0,10228544.0,10261504.0,10294464.0,10327424.0,10360384.0,10393344.0,10426304.0,10216320.0,10249344.0,10282368.0,10315392.0,10348416.0,10381440.0,10414464.0,10447488.0,10237056.0,10270144.0,10303232.0,10336320.0,10369408.0,10402496.0,10435584.0,10468672.0,10257792.0,10290944.0,10324096.0,10357248.0,10390400.0,10423552.0,10456704.0,10489856.0,10278528.0,10311744.0,10344960.0,10378176.0,10411392.0,10444608.0,10477824.0,10511040.0,10299264.0,10332544.0,10365824.0,10399104.0,10432384.0,10465664.0,10498944.0,10532224.0,10254464.0,10286784.0,10319104.0,10351424.0,10383744.0,10416064.0,10448384.0,10480704.0,10201472.0,10232832.0,10264192.0,10295552.0,10326912.0,10358272.0,10389632.0,10420992.0,10222208.0,10253632.0,10285056.0,10316480.0,10347904.0,10379328.0,10410752.0,10442176.0,10242944.0,10274432.0,10305920.0,10337408.0,10368896.0,10400384.0,10431872.0,10463360.0,10263680.0,10295232.0,10326784.0,10358336.0,10389888.0,10421440.0,10452992.0,10484544.0,10284416.0,10316032.0,10347648.0,10379264.0,10410880.0,10442496.0,10474112.0,10505728.0,10305152.0,10336832.0,10368512.0,10400192.0,10431872.0,10463552.0,10495232.0,10526912.0,10325888.0,10357632.0,10389376.0,10421120.0,10452864.0,10484608.0,10516352.0,10548096.0,10346624.0,10378432.0,10410240.0,10442048.0,10473856.0,10505664.0,10537472.0,10569280.0,10367360.0,10399232.0,10431104.0,10462976.0,10494848.0,10526720.0,10558592.0,10590464.0,10388096.0,10420032.0,10451968.0,10483904.0,10515840.0,10547776.0,10579712.0,10611648.0,10408832.0,10440832.0,10472832.0,10504832.0,10536832.0,10568832.0,10600832.0,10632832.0,10429568.0,10461632.0,10493696.0,10525760.0,10557824.0,10589888.0,10621952.0,10654016.0,10450304.0,10482432.0,10514560.0,10546688.0,10578816.0,10610944.0,10643072.0,10675200.0,10471040.0,10503232.0,10535424.0,10567616.0,10599808.0,10632000.0,10664192.0,10696384.0,10491776.0,10524032.0,10556288.0,10588544.0,10620800.0,10653056.0,10685312.0,10717568.0,10512512.0,10544832.0,10577152.0,10609472.0,10641792.0,10674112.0,10706432.0,10738752.0,10533248.0,10565632.0,10598016.0,10630400.0,10662784.0,10695168.0,10727552.0,10759936.0,10553984.0,10586432.0,10618880.0,10651328.0,10683776.0,10716224.0,10748672.0,10781120.0,10574720.0,10607232.0,10639744.0,10672256.0,10704768.0,10737280.0,10769792.0,10802304.0,10595456.0,10628032.0,10660608.0,10693184.0,10725760.0,10758336.0,10790912.0,10823488.0,10616192.0,10648832.0,10681472.0,10714112.0,10746752.0,10779392.0,10812032.0,10844672.0,10636928.0,10669632.0,10702336.0,10735040.0,10767744.0,10800448.0,10833152.0,10865856.0,10657664.0,10690432.0,10723200.0,10755968.0,10788736.0,10821504.0,10854272.0,10887040.0,10678400.0,10711232.0,10744064.0,10776896.0,10809728.0,10842560.0,10875392.0,10908224.0,10699136.0,10732032.0,10764928.0,10797824.0,10830720.0,10863616.0,10896512.0,10929408.0,10719872.0,10752832.0,10785792.0,10818752.0,10851712.0,10884672.0,10917632.0,10950592.0,10740608.0,10773632.0,10806656.0,10839680.0,10872704.0,10905728.0,10938752.0,10971776.0,10761344.0,10794432.0,10827520.0,10860608.0,10893696.0,10926784.0,10959872.0,10992960.0,10782080.0,10815232.0,10848384.0,10881536.0,10914688.0,10947840.0,10980992.0,11014144.0,10802816.0,10836032.0,10869248.0,10902464.0,10935680.0,10968896.0,11002112.0,11035328.0,10823552.0,10856832.0,10890112.0,10923392.0,10956672.0,10989952.0,11023232.0,11056512.0,10844288.0,10877632.0,10910976.0,10944320.0,10977664.0,11011008.0,11044352.0,11077696.0,10865024.0,10898432.0,10931840.0,10965248.0,10998656.0,11032064.0,11065472.0,11098880.0,10885760.0,10919232.0,10952704.0,10986176.0,11019648.0,11053120.0,11086592.0,11120064.0,10906496.0,10940032.0,10973568.0,11007104.0,11040640.0,11074176.0,11107712.0,11141248.0,10927232.0,10960832.0,10994432.0,11028032.0,11061632.0,11095232.0,11128832.0,11162432.0,10947968.0,10981632.0,11015296.0,11048960.0,11082624.0,11116288.0,11149952.0,11183616.0,10968704.0,11002432.0,11036160.0,11069888.0,11103616.0,11137344.0,11171072.0,11204800.0,10399616.0,10432384.0,10465152.0,10497920.0,10530688.0,10563456.0,10596224.0,10628992.0,9822336.0,9854144.0,9885952.0,9917760.0,9949568.0,9981376.0,10013184.0,10044992.0,4980096.0,4996016.0,5011936.0,5027856.0,5043776.0,5059696.0,5075616.0,5091536.0,5588736.0,5604528.0,5620320.0,5636112.0,5651904.0,5667696.0,5683488.0,5699280.0,11314560.0,11346176.0,11377792.0,11409408.0,11441024.0,11472640.0,11504256.0,11535872.0,11336320.0,11368000.0,11399680.0,11431360.0,11463040.0,11494720.0,11526400.0,11558080.0,11358080.0,11389824.0,11421568.0,11453312.0,11485056.0,11516800.0,11548544.0,11580288.0,11379840.0,11411648.0,11443456.0,11475264.0,11507072.0,11538880.0,11570688.0,11602496.0,11401600.0,11433472.0,11465344.0,11497216.0,11529088.0,11560960.0,11592832.0,11624704.0,11423360.0,11455296.0,11487232.0,11519168.0,11551104.0,11583040.0,11614976.0,11646912.0,11445120.0,11477120.0,11509120.0,11541120.0,11573120.0,11605120.0,11637120.0,11669120.0,11466880.0,11498944.0,11531008.0,11563072.0,11595136.0,11627200.0,11659264.0,11691328.0,11488640.0,11520768.0,11552896.0,11585024.0,11617152.0,11649280.0,11681408.0,11713536.0,11510400.0,11542592.0,11574784.0,11606976.0,11639168.0,11671360.0,11703552.0,11735744.0,11532160.0,11564416.0,11596672.0,11628928.0,11661184.0,11693440.0,11725696.0,11757952.0,11553920.0,11586240.0,11618560.0,11650880.0,11683200.0,11715520.0,11747840.0,11780160.0,11575680.0,11608064.0,11640448.0,11672832.0,11705216.0,11737600.0,11769984.0,11802368.0,11597440.0,11629888.0,11662336.0,11694784.0,11727232.0,11759680.0,11792128.0,11824576.0,11619200.0,11651712.0,11684224.0,11716736.0,11749248.0,11781760.0,11814272.0,11846784.0,11640960.0,11673536.0,11706112.0,11738688.0,11771264.0,11803840.0,11836416.0,11868992.0,11662720.0,11695360.0,11728000.0,11760640.0,11793280.0,11825920.0,11858560.0,11891200.0,11684480.0,11717184.0,11749888.0,11782592.0,11815296.0,11848000.0,11880704.0,11913408.0,11706240.0,11739008.0,11771776.0,11804544.0,11837312.0,11870080.0,11902848.0,11935616.0,11728000.0,11760832.0,11793664.0,11826496.0,11859328.0,11892160.0,11924992.0,11957824.0,11749760.0,11782656.0,11815552.0,11848448.0,11881344.0,11914240.0,11947136.0,11980032.0,11771520.0,11804480.0,11837440.0,11870400.0,11903360.0,11936320.0,11969280.0,12002240.0,11793280.0,11826304.0,11859328.0,11892352.0,11925376.0,11958400.0,11991424.0,12024448.0,11815040.0,11848128.0,11881216.0,11914304.0,11947392.0,11980480.0,12013568.0,12046656.0,11836800.0,11869952.0,11903104.0,11936256.0,11969408.0,12002560.0,12035712.0,12068864.0,11858560.0,11891776.0,11924992.0,11958208.0,11991424.0,12024640.0,12057856.0,12091072.0,11880320.0,11913600.0,11946880.0,11980160.0,12013440.0,12046720.0,12080000.0,12113280.0,11902080.0,11935424.0,11968768.0,12002112.0,12035456.0,12068800.0,12102144.0,12135488.0,11923840.0,11957248.0,11990656.0,12024064.0,12057472.0,12090880.0,12124288.0,12157696.0,11945600.0,11979072.0,12012544.0,12046016.0,12079488.0,12112960.0,12146432.0,12179904.0,11967360.0,12000896.0,12034432.0,12067968.0,12101504.0,12135040.0,12168576.0,12202112.0,11989120.0,12022720.0,12056320.0,12089920.0,12123520.0,12157120.0,12190720.0,12224320.0,12010880.0,12044544.0,12078208.0,12111872.0,12145536.0,12179200.0,12212864.0,12246528.0,12032640.0,12066368.0,12100096.0,12133824.0,12167552.0,12201280.0,12235008.0,12268736.0,12054400.0,12088192.0,12121984.0,12155776.0,12189568.0,12223360.0,12257152.0,12290944.0,12076160.0,12110016.0,12143872.0,12177728.0,12211584.0,12245440.0,12279296.0,12313152.0,11491712.0,11524608.0,11557504.0,11590400.0,11623296.0,11656192.0,11689088.0,11721984.0,10899072.0,10931008.0,10962944.0,10994880.0,11026816.0,11058752.0,11090688.0,11122624.0,10314624.0,10345600.0,10376576.0,10407552.0,10438528.0,10469504.0,10500480.0,10531456.0,9721984.0,9752000.0,9782016.0,9812032.0,9842048.0,9872064.0,9902080.0,9932096.0,9743744.0,9773824.0,9803904.0,9833984.0,9864064.0,9894144.0,9924224.0,9954304.0,9765504.0,9795648.0,9825792.0,9855936.0,9886080.0,9916224.0,9946368.0,9976512.0,9787264.0,9817472.0,9847680.0,9877888.0,9908096.0,9938304.0,9968512.0,9998720.0,9809024.0,9839296.0,9869568.0,9899840.0,9930112.0,9960384.0,9990656.0,10020928.0,9830784.0,9861120.0,9891456.0,9921792.0,9952128.0,9982464.0,10012800.0,10043136.0,9852544.0,9882944.0,9913344.0,9943744.0,9974144.0,10004544.0,10034944.0,10065344.0,9874304.0,9904768.0,9935232.0,9965696.0,9996160.0,10026624.0,10057088.0,10087552.0,9896064.0,9926592.0,9957120.0,9987648.0,10018176.0,10048704.0,10079232.0,10109760.0,9917824.0,9948416.0,9979008.0,10009600.0,10040192.0,10070784.0,10101376.0,10131968.0,9939584.0,9970240.0,10000896.0,10031552.0,10062208.0,10092864.0,10123520.0,10154176.0,9961344.0,9992064.0,10022784.0,10053504.0,10084224.0,10114944.0,10145664.0,10176384.0,9983104.0,10013888.0,10044672.0,10075456.0,10106240.0,10137024.0,10167808.0,10198592.0,10004864.0,10035712.0,10066560.0,10097408.0,10128256.0,10159104.0,10189952.0,10220800.0,10026624.0,10057536.0,10088448.0,10119360.0,10150272.0,10181184.0,10212096.0,10243008.0,10048384.0,10079360.0,10110336.0,10141312.0,10172288.0,10203264.0,10234240.0,10265216.0,10070144.0,10101184.0,10132224.0,10163264.0,10194304.0,10225344.0,10256384.0,10287424.0,10091904.0,10123008.0,10154112.0,10185216.0,10216320.0,10247424.0,10278528.0,10309632.0,10113664.0,10144832.0,10176000.0,10207168.0,10238336.0,10269504.0,10300672.0,10331840.0,10135424.0,10166656.0,10197888.0,10229120.0,10260352.0,10291584.0,10322816.0,10354048.0,10157184.0,10188480.0,10219776.0,10251072.0,10282368.0,10313664.0,10344960.0,10376256.0,10178944.0,10210304.0,10241664.0,10273024.0,10304384.0,10335744.0,10367104.0,10398464.0,10200704.0,10232128.0,10263552.0,10294976.0,10326400.0,10357824.0,10389248.0,10420672.0,10222464.0,10253952.0,10285440.0,10316928.0,10348416.0,10379904.0,10411392.0,10442880.0,10244224.0,10275776.0,10307328.0,10338880.0,10370432.0,10401984.0,10433536.0,10465088.0,10265984.0,10297600.0,10329216.0,10360832.0,10392448.0,10424064.0,10455680.0,10487296.0,10287744.0,10319424.0,10351104.0,10382784.0,10414464.0,10446144.0,10477824.0,10509504.0,10309504.0,10341248.0,10372992.0,10404736.0,10436480.0,10468224.0,10499968.0,10531712.0,10331264.0,10363072.0,10394880.0,10426688.0,10458496.0,10490304.0,10522112.0,10553920.0,10353024.0,10384896.0,10416768.0,10448640.0,10480512.0,10512384.0,10544256.0,10576128.0,10374784.0,10406720.0,10438656.0,10470592.0,10502528.0,10534464.0,10566400.0,10598336.0,10396544.0,10428544.0,10460544.0,10492544.0,10524544.0,10556544.0,10588544.0,10620544.0,10418304.0,10450368.0,10482432.0,10514496.0,10546560.0,10578624.0,10610688.0,10642752.0,10440064.0,10472192.0,10504320.0,10536448.0,10568576.0,10600704.0,10632832.0,10664960.0,10461824.0,10494016.0,10526208.0,10558400.0,10590592.0,10622784.0,10654976.0,10687168.0,10483584.0,10515840.0,10548096.0,10580352.0,10612608.0,10644864.0,10677120.0,10709376.0,10505344.0,10537664.0,10569984.0,10602304.0,10634624.0,10666944.0,10699264.0,10731584.0,10527104.0,10559488.0,10591872.0,10624256.0,10656640.0,10689024.0,10721408.0,10753792.0,10466944.0,10498368.0,10529792.0,10561216.0,10592640.0,10624064.0,10655488.0,10686912.0,10398592.0,10429056.0,10459520.0,10489984.0,10520448.0,10550912.0,10581376.0,10611840.0,10420352.0,10450880.0,10481408.0,10511936.0,10542464.0,10572992.0,10603520.0,10634048.0,10442112.0,10472704.0,10503296.0,10533888.0,10564480.0,10595072.0,10625664.0,10656256.0,10463872.0,10494528.0,10525184.0,10555840.0,10586496.0,10617152.0,10647808.0,10678464.0,10485632.0,10516352.0,10547072.0,10577792.0,10608512.0,10639232.0,10669952.0,10700672.0,10507392.0,10538176.0,10568960.0,10599744.0,10630528.0,10661312.0,10692096.0,10722880.0,10529152.0,10560000.0,10590848.0,10621696.0,10652544.0,10683392.0,10714240.0,10745088.0,10550912.0,10581824.0,10612736.0,10643648.0,10674560.0,10705472.0,10736384.0,10767296.0,10572672.0,10603648.0,10634624.0,10665600.0,10696576.0,10727552.0,10758528.0,10789504.0,10594432.0,10625472.0,10656512.0,10687552.0,10718592.0,10749632.0,10780672.0,10811712.0,10616192.0,10647296.0,10678400.0,10709504.0,10740608.0,10771712.0,10802816.0,10833920.0,10637952.0,10669120.0,10700288.0,10731456.0,10762624.0,10793792.0,10824960.0,10856128.0,10659712.0,10690944.0,10722176.0,10753408.0,10784640.0,10815872.0,10847104.0,10878336.0,10681472.0,10712768.0,10744064.0,10775360.0,10806656.0,10837952.0,10869248.0,10900544.0,10703232.0,10734592.0,10765952.0,10797312.0,10828672.0,10860032.0,10891392.0,10922752.0,10724992.0,10756416.0,10787840.0,10819264.0,10850688.0,10882112.0,10913536.0,10944960.0,10746752.0,10778240.0,10809728.0,10841216.0,10872704.0,10904192.0,10935680.0,10967168.0,10768512.0,10800064.0,10831616.0,10863168.0,10894720.0,10926272.0,10957824.0,10989376.0,10790272.0,10821888.0,10853504.0,10885120.0,10916736.0,10948352.0,10979968.0,11011584.0,10812032.0,10843712.0,10875392.0,10907072.0,10938752.0,10970432.0,11002112.0,11033792.0,10833792.0,10865536.0,10897280.0,10929024.0,10960768.0,10992512.0,11024256.0,11056000.0,10855552.0,10887360.0,10919168.0,10950976.0,10982784.0,11014592.0,11046400.0,11078208.0,10877312.0,10909184.0,10941056.0,10972928.0,11004800.0,11036672.0,11068544.0,11100416.0,10899072.0,10931008.0,10962944.0,10994880.0,11026816.0,11058752.0,11090688.0,11122624.0,10920832.0,10952832.0,10984832.0,11016832.0,11048832.0,11080832.0,11112832.0,11144832.0,10942592.0,10974656.0,11006720.0,11038784.0,11070848.0,11102912.0,11134976.0,11167040.0,10964352.0,10996480.0,11028608.0,11060736.0,11092864.0,11124992.0,11157120.0,11189248.0,10986112.0,11018304.0,11050496.0,11082688.0,11114880.0,11147072.0,11179264.0,11211456.0,11007872.0,11040128.0,11072384.0,11104640.0,11136896.0,11169152.0,11201408.0,11233664.0,11029632.0,11061952.0,11094272.0,11126592.0,11158912.0,11191232.0,11223552.0,11255872.0,11051392.0,11083776.0,11116160.0,11148544.0,11180928.0,11213312.0,11245696.0,11278080.0,11073152.0,11105600.0,11138048.0,11170496.0,11202944.0,11235392.0,11267840.0,11300288.0,11094912.0,11127424.0,11159936.0,11192448.0,11224960.0,11257472.0,11289984.0,11322496.0,11116672.0,11149248.0,11181824.0,11214400.0,11246976.0,11279552.0,11312128.0,11344704.0,11138432.0,11171072.0,11203712.0,11236352.0,11268992.0,11301632.0,11334272.0,11366912.0,11160192.0,11192896.0,11225600.0,11258304.0,11291008.0,11323712.0,11356416.0,11389120.0,11181952.0,11214720.0,11247488.0,11280256.0,11313024.0,11345792.0,11378560.0,11411328.0,11203712.0,11236544.0,11269376.0,11302208.0,11335040.0,11367872.0,11400704.0,11433536.0,10619264.0,10651136.0,10683008.0,10714880.0,10746752.0,10778624.0,10810496.0,10842368.0,10026624.0,10057536.0,10088448.0,10119360.0,10150272.0,10181184.0,10212096.0,10243008.0,10048384.0,10079360.0,10110336.0,10141312.0,10172288.0,10203264.0,10234240.0,10265216.0,10070144.0,10101184.0,10132224.0,10163264.0,10194304.0,10225344.0,10256384.0,10287424.0,10091904.0,10123008.0,10154112.0,10185216.0,10216320.0,10247424.0,10278528.0,10309632.0,10113664.0,10144832.0,10176000.0,10207168.0,10238336.0,10269504.0,10300672.0,10331840.0,10135424.0,10166656.0,10197888.0,10229120.0,10260352.0,10291584.0,10322816.0,10354048.0,10157184.0,10188480.0,10219776.0,10251072.0,10282368.0,10313664.0,10344960.0,10376256.0,10178944.0,10210304.0,10241664.0,10273024.0,10304384.0,10335744.0,10367104.0,10398464.0,10200704.0,10232128.0,10263552.0,10294976.0,10326400.0,10357824.0,10389248.0,10420672.0,10222464.0,10253952.0,10285440.0,10316928.0,10348416.0,10379904.0,10411392.0,10442880.0,10244224.0,10275776.0,10307328.0,10338880.0,10370432.0,10401984.0,10433536.0,10465088.0,10265984.0,10297600.0,10329216.0,10360832.0,10392448.0,10424064.0,10455680.0,10487296.0,10287744.0,10319424.0,10351104.0,10382784.0,10414464.0,10446144.0,10477824.0,10509504.0,10309504.0,10341248.0,10372992.0,10404736.0,10436480.0,10468224.0,10499968.0,10531712.0,10331264.0,10363072.0,10394880.0,10426688.0,10458496.0,10490304.0,10522112.0,10553920.0,10353024.0,10384896.0,10416768.0,10448640.0,10480512.0,10512384.0,10544256.0,10576128.0,10374784.0,10406720.0,10438656.0,10470592.0,10502528.0,10534464.0,10566400.0,10598336.0,10396544.0,10428544.0,10460544.0,10492544.0,10524544.0,10556544.0,10588544.0,10620544.0,10418304.0,10450368.0,10482432.0,10514496.0,10546560.0,10578624.0,10610688.0,10642752.0,10440064.0,10472192.0,10504320.0,10536448.0,10568576.0,10600704.0,10632832.0,10664960.0,10461824.0,10494016.0,10526208.0,10558400.0,10590592.0,10622784.0,10654976.0,10687168.0,10483584.0,10515840.0,10548096.0,10580352.0,10612608.0,10644864.0,10677120.0,10709376.0,10505344.0,10537664.0,10569984.0,10602304.0,10634624.0,10666944.0,10699264.0,10731584.0,10527104.0,10559488.0,10591872.0,10624256.0,10656640.0,10689024.0,10721408.0,10753792.0,10548864.0,10581312.0,10613760.0,10646208.0,10678656.0,10711104.0,10743552.0,10776000.0,10570624.0,10603136.0,10635648.0,10668160.0,10700672.0,10733184.0,10765696.0,10798208.0,10592384.0,10624960.0,10657536.0,10690112.0,10722688.0,10755264.0,10787840.0,10820416.0,10614144.0,10646784.0,10679424.0,10712064.0,10744704.0,10777344.0,10809984.0,10842624.0,10635904.0,10668608.0,10701312.0,10734016.0,10766720.0,10799424.0,10832128.0,10864832.0,10657664.0,10690432.0,10723200.0,10755968.0,10788736.0,10821504.0,10854272.0,10887040.0,10679424.0,10712256.0,10745088.0,10777920.0,10810752.0,10843584.0,10876416.0,10909248.0,10701184.0,10734080.0,10766976.0,10799872.0,10832768.0,10865664.0,10898560.0,10931456.0,10722944.0,10755904.0,10788864.0,10821824.0,10854784.0,10887744.0,10920704.0,10953664.0,10744704.0,10777728.0,10810752.0,10843776.0,10876800.0,10909824.0,10942848.0,10975872.0,10766464.0,10799552.0,10832640.0,10865728.0,10898816.0,10931904.0,10964992.0,10998080.0,10788224.0,10821376.0,10854528.0,10887680.0,10920832.0,10953984.0,10987136.0,11020288.0,10809984.0,10843200.0,10876416.0,10909632.0,10942848.0,10976064.0,11009280.0,11042496.0,10831744.0,10865024.0,10898304.0,10931584.0,10964864.0,10998144.0,11031424.0,11064704.0,10771584.0,10803904.0,10836224.0,10868544.0,10900864.0,10933184.0,10965504.0,10997824.0,10703232.0,10734592.0,10765952.0,10797312.0,10828672.0,10860032.0,10891392.0,10922752.0,10724992.0,10756416.0,10787840.0,10819264.0,10850688.0,10882112.0,10913536.0,10944960.0,10746752.0,10778240.0,10809728.0,10841216.0,10872704.0,10904192.0,10935680.0,10967168.0,10768512.0,10800064.0,10831616.0,10863168.0,10894720.0,10926272.0,10957824.0,10989376.0,10790272.0,10821888.0,10853504.0,10885120.0,10916736.0,10948352.0,10979968.0,11011584.0,10812032.0,10843712.0,10875392.0,10907072.0,10938752.0,10970432.0,11002112.0,11033792.0,10833792.0,10865536.0,10897280.0,10929024.0,10960768.0,10992512.0,11024256.0,11056000.0,10855552.0,10887360.0,10919168.0,10950976.0,10982784.0,11014592.0,11046400.0,11078208.0,10877312.0,10909184.0,10941056.0,10972928.0,11004800.0,11036672.0,11068544.0,11100416.0,10899072.0,10931008.0,10962944.0,10994880.0,11026816.0,11058752.0,11090688.0,11122624.0,10920832.0,10952832.0,10984832.0,11016832.0,11048832.0,11080832.0,11112832.0,11144832.0,10942592.0,10974656.0,11006720.0,11038784.0,11070848.0,11102912.0,11134976.0,11167040.0,10964352.0,10996480.0,11028608.0,11060736.0,11092864.0,11124992.0,11157120.0,11189248.0,10986112.0,11018304.0,11050496.0,11082688.0,11114880.0,11147072.0,11179264.0,11211456.0,11007872.0,11040128.0,11072384.0,11104640.0,11136896.0,11169152.0,11201408.0,11233664.0,11029632.0,11061952.0,11094272.0,11126592.0,11158912.0,11191232.0,11223552.0,11255872.0,11051392.0,11083776.0,11116160.0,11148544.0,11180928.0,11213312.0,11245696.0,11278080.0,11073152.0,11105600.0,11138048.0,11170496.0,11202944.0,11235392.0,11267840.0,11300288.0,11094912.0,11127424.0,11159936.0,11192448.0,11224960.0,11257472.0,11289984.0,11322496.0,11116672.0,11149248.0,11181824.0,11214400.0,11246976.0,11279552.0,11312128.0,11344704.0,11138432.0,11171072.0,11203712.0,11236352.0,11268992.0,11301632.0,11334272.0,11366912.0,11160192.0,11192896.0,11225600.0,11258304.0,11291008.0,11323712.0,11356416.0,11389120.0,11181952.0,11214720.0,11247488.0,11280256.0,11313024.0,11345792.0,11378560.0,11411328.0,11203712.0,11236544.0,11269376.0,11302208.0,11335040.0,11367872.0,11400704.0,11433536.0,11225472.0,11258368.0,11291264.0,11324160.0,11357056.0,11389952.0,11422848.0,11455744.0,11247232.0,11280192.0,11313152.0,11346112.0,11379072.0,11412032.0,11444992.0,11477952.0,11268992.0,11302016.0,11335040.0,11368064.0,11401088.0,11434112.0,11467136.0,11500160.0,11290752.0,11323840.0,11356928.0,11390016.0,11423104.0,11456192.0,11489280.0,11522368.0,11312512.0,11345664.0,11378816.0,11411968.0,11445120.0,11478272.0,11511424.0,11544576.0,11334272.0,11367488.0,11400704.0,11433920.0,11467136.0,11500352.0,11533568.0,11566784.0,11356032.0,11389312.0,11422592.0,11455872.0,11489152.0,11522432.0,11555712.0,11588992.0,11377792.0,11411136.0,11444480.0,11477824.0,11511168.0,11544512.0,11577856.0,11611200.0,11399552.0,11432960.0,11466368.0,11499776.0,11533184.0,11566592.0,11600000.0,11633408.0,11421312.0,11454784.0,11488256.0,11521728.0,11555200.0,11588672.0,11622144.0,11655616.0,11443072.0,11476608.0,11510144.0,11543680.0,11577216.0,11610752.0,11644288.0,11677824.0,11464832.0,11498432.0,11532032.0,11565632.0,11599232.0,11632832.0,11666432.0,11700032.0,11486592.0,11520256.0,11553920.0,11587584.0,11621248.0,11654912.0,11688576.0,11722240.0,11508352.0,11542080.0,11575808.0,11609536.0,11643264.0,11676992.0,11710720.0,11744448.0,10923904.0,10956672.0,10989440.0,11022208.0,11054976.0,11087744.0,11120512.0,11153280.0,10331264.0,10363072.0,10394880.0,10426688.0,10458496.0,10490304.0,10522112.0,10553920.0,5234816.0,5250736.0,5266656.0,5282576.0,5298496.0,5314416.0,5330336.0,5346256.0,5841408.0,5857200.0,5872992.0,5888784.0,5904576.0,5920368.0,5936160.0,5951952.0,11820416.0,11852032.0,11883648.0,11915264.0,11946880.0,11978496.0,12010112.0,12041728.0,11843200.0,11874880.0,11906560.0,11938240.0,11969920.0,12001600.0,12033280.0,12064960.0,11865984.0,11897728.0,11929472.0,11961216.0,11992960.0,12024704.0,12056448.0,12088192.0,11888768.0,11920576.0,11952384.0,11984192.0,12016000.0,12047808.0,12079616.0,12111424.0,11911552.0,11943424.0,11975296.0,12007168.0,12039040.0,12070912.0,12102784.0,12134656.0,11934336.0,11966272.0,11998208.0,12030144.0,12062080.0,12094016.0,12125952.0,12157888.0,11957120.0,11989120.0,12021120.0,12053120.0,12085120.0,12117120.0,12149120.0,12181120.0,11979904.0,12011968.0,12044032.0,12076096.0,12108160.0,12140224.0,12172288.0,12204352.0,12002688.0,12034816.0,12066944.0,12099072.0,12131200.0,12163328.0,12195456.0,12227584.0,12025472.0,12057664.0,12089856.0,12122048.0,12154240.0,12186432.0,12218624.0,12250816.0,12048256.0,12080512.0,12112768.0,12145024.0,12177280.0,12209536.0,12241792.0,12274048.0,12071040.0,12103360.0,12135680.0,12168000.0,12200320.0,12232640.0,12264960.0,12297280.0,12093824.0,12126208.0,12158592.0,12190976.0,12223360.0,12255744.0,12288128.0,12320512.0,12116608.0,12149056.0,12181504.0,12213952.0,12246400.0,12278848.0,12311296.0,12343744.0,12139392.0,12171904.0,12204416.0,12236928.0,12269440.0,12301952.0,12334464.0,12366976.0,12162176.0,12194752.0,12227328.0,12259904.0,12292480.0,12325056.0,12357632.0,12390208.0,12184960.0,12217600.0,12250240.0,12282880.0,12315520.0,12348160.0,12380800.0,12413440.0,12207744.0,12240448.0,12273152.0,12305856.0,12338560.0,12371264.0,12403968.0,12436672.0,12230528.0,12263296.0,12296064.0,12328832.0,12361600.0,12394368.0,12427136.0,12459904.0,12253312.0,12286144.0,12318976.0,12351808.0,12384640.0,12417472.0,12450304.0,12483136.0,12276096.0,12308992.0,12341888.0,12374784.0,12407680.0,12440576.0,12473472.0,12506368.0,12298880.0,12331840.0,12364800.0,12397760.0,12430720.0,12463680.0,12496640.0,12529600.0,12321664.0,12354688.0,12387712.0,12420736.0,12453760.0,12486784.0,12519808.0,12552832.0,12344448.0,12377536.0,12410624.0,12443712.0,12476800.0,12509888.0,12542976.0,12576064.0,12367232.0,12400384.0,12433536.0,12466688.0,12499840.0,12532992.0,12566144.0,12599296.0,12390016.0,12423232.0,12456448.0,12489664.0,12522880.0,12556096.0,12589312.0,12622528.0,12412800.0,12446080.0,12479360.0,12512640.0,12545920.0,12579200.0,12612480.0,12645760.0,12435584.0,12468928.0,12502272.0,12535616.0,12568960.0,12602304.0,12635648.0,12668992.0,12458368.0,12491776.0,12525184.0,12558592.0,12592000.0,12625408.0,12658816.0,12692224.0,12481152.0,12514624.0,12548096.0,12581568.0,12615040.0,12648512.0,12681984.0,12715456.0,12503936.0,12537472.0,12571008.0,12604544.0,12638080.0,12671616.0,12705152.0,12738688.0,12526720.0,12560320.0,12593920.0,12627520.0,12661120.0,12694720.0,12728320.0,12761920.0,12549504.0,12583168.0,12616832.0,12650496.0,12684160.0,12717824.0,12751488.0,12785152.0,12572288.0,12606016.0,12639744.0,12673472.0,12707200.0,12740928.0,12774656.0,12808384.0,12595072.0,12628864.0,12662656.0,12696448.0,12730240.0,12764032.0,12797824.0,12831616.0,12617856.0,12651712.0,12685568.0,12719424.0,12753280.0,12787136.0,12820992.0,12854848.0,12018048.0,12050944.0,12083840.0,12116736.0,12149632.0,12182528.0,12215424.0,12248320.0,11410048.0,11441984.0,11473920.0,11505856.0,11537792.0,11569728.0,11601664.0,11633600.0,10810240.0,10841216.0,10872192.0,10903168.0,10934144.0,10965120.0,10996096.0,11027072.0,10202240.0,10232256.0,10262272.0,10292288.0,10322304.0,10352320.0,10382336.0,10412352.0,10225024.0,10255104.0,10285184.0,10315264.0,10345344.0,10375424.0,10405504.0,10435584.0,10247808.0,10277952.0,10308096.0,10338240.0,10368384.0,10398528.0,10428672.0,10458816.0,10270592.0,10300800.0,10331008.0,10361216.0,10391424.0,10421632.0,10451840.0,10482048.0,10293376.0,10323648.0,10353920.0,10384192.0,10414464.0,10444736.0,10475008.0,10505280.0,10316160.0,10346496.0,10376832.0,10407168.0,10437504.0,10467840.0,10498176.0,10528512.0,10338944.0,10369344.0,10399744.0,10430144.0,10460544.0,10490944.0,10521344.0,10551744.0,10361728.0,10392192.0,10422656.0,10453120.0,10483584.0,10514048.0,10544512.0,10574976.0,10384512.0,10415040.0,10445568.0,10476096.0,10506624.0,10537152.0,10567680.0,10598208.0,10407296.0,10437888.0,10468480.0,10499072.0,10529664.0,10560256.0,10590848.0,10621440.0,10430080.0,10460736.0,10491392.0,10522048.0,10552704.0,10583360.0,10614016.0,10644672.0,10452864.0,10483584.0,10514304.0,10545024.0,10575744.0,10606464.0,10637184.0,10667904.0,10475648.0,10506432.0,10537216.0,10568000.0,10598784.0,10629568.0,10660352.0,10691136.0,10498432.0,10529280.0,10560128.0,10590976.0,10621824.0,10652672.0,10683520.0,10714368.0,10521216.0,10552128.0,10583040.0,10613952.0,10644864.0,10675776.0,10706688.0,10737600.0,10544000.0,10574976.0,10605952.0,10636928.0,10667904.0,10698880.0,10729856.0,10760832.0,10566784.0,10597824.0,10628864.0,10659904.0,10690944.0,10721984.0,10753024.0,10784064.0,10589568.0,10620672.0,10651776.0,10682880.0,10713984.0,10745088.0,10776192.0,10807296.0,10612352.0,10643520.0,10674688.0,10705856.0,10737024.0,10768192.0,10799360.0,10830528.0,10635136.0,10666368.0,10697600.0,10728832.0,10760064.0,10791296.0,10822528.0,10853760.0,10657920.0,10689216.0,10720512.0,10751808.0,10783104.0,10814400.0,10845696.0,10876992.0,10680704.0,10712064.0,10743424.0,10774784.0,10806144.0,10837504.0,10868864.0,10900224.0,10703488.0,10734912.0,10766336.0,10797760.0,10829184.0,10860608.0,10892032.0,10923456.0,10726272.0,10757760.0,10789248.0,10820736.0,10852224.0,10883712.0,10915200.0,10946688.0,10749056.0,10780608.0,10812160.0,10843712.0,10875264.0,10906816.0,10938368.0,10969920.0,10771840.0,10803456.0,10835072.0,10866688.0,10898304.0,10929920.0,10961536.0,10993152.0,10794624.0,10826304.0,10857984.0,10889664.0,10921344.0,10953024.0,10984704.0,11016384.0,10817408.0,10849152.0,10880896.0,10912640.0,10944384.0,10976128.0,11007872.0,11039616.0,10840192.0,10872000.0,10903808.0,10935616.0,10967424.0,10999232.0,11031040.0,11062848.0,10862976.0,10894848.0,10926720.0,10958592.0,10990464.0,11022336.0,11054208.0,11086080.0,10885760.0,10917696.0,10949632.0,10981568.0,11013504.0,11045440.0,11077376.0,11109312.0,10908544.0,10940544.0,10972544.0,11004544.0,11036544.0,11068544.0,11100544.0,11132544.0,10931328.0,10963392.0,10995456.0,11027520.0,11059584.0,11091648.0,11123712.0,11155776.0,10954112.0,10986240.0,11018368.0,11050496.0,11082624.0,11114752.0,11146880.0,11179008.0,10976896.0,11009088.0,11041280.0,11073472.0,11105664.0,11137856.0,11170048.0,11202240.0,10999680.0,11031936.0,11064192.0,11096448.0,11128704.0,11160960.0,11193216.0,11225472.0,11022464.0,11054784.0,11087104.0,11119424.0,11151744.0,11184064.0,11216384.0,11248704.0,11045248.0,11077632.0,11110016.0,11142400.0,11174784.0,11207168.0,11239552.0,11271936.0,10969728.0,11001152.0,11032576.0,11064000.0,11095424.0,11126848.0,11158272.0,11189696.0,10886016.0,10916480.0,10946944.0,10977408.0,11007872.0,11038336.0,11068800.0,11099264.0,10908800.0,10939328.0,10969856.0,11000384.0,11030912.0,11061440.0,11091968.0,11122496.0,10931584.0,10962176.0,10992768.0,11023360.0,11053952.0,11084544.0,11115136.0,11145728.0,10954368.0,10985024.0,11015680.0,11046336.0,11076992.0,11107648.0,11138304.0,11168960.0,10977152.0,11007872.0,11038592.0,11069312.0,11100032.0,11130752.0,11161472.0,11192192.0,10999936.0,11030720.0,11061504.0,11092288.0,11123072.0,11153856.0,11184640.0,11215424.0,11022720.0,11053568.0,11084416.0,11115264.0,11146112.0,11176960.0,11207808.0,11238656.0,11045504.0,11076416.0,11107328.0,11138240.0,11169152.0,11200064.0,11230976.0,11261888.0,11068288.0,11099264.0,11130240.0,11161216.0,11192192.0,11223168.0,11254144.0,11285120.0,11091072.0,11122112.0,11153152.0,11184192.0,11215232.0,11246272.0,11277312.0,11308352.0,11113856.0,11144960.0,11176064.0,11207168.0,11238272.0,11269376.0,11300480.0,11331584.0,11136640.0,11167808.0,11198976.0,11230144.0,11261312.0,11292480.0,11323648.0,11354816.0,11159424.0,11190656.0,11221888.0,11253120.0,11284352.0,11315584.0,11346816.0,11378048.0,11182208.0,11213504.0,11244800.0,11276096.0,11307392.0,11338688.0,11369984.0,11401280.0,11204992.0,11236352.0,11267712.0,11299072.0,11330432.0,11361792.0,11393152.0,11424512.0,11227776.0,11259200.0,11290624.0,11322048.0,11353472.0,11384896.0,11416320.0,11447744.0,11250560.0,11282048.0,11313536.0,11345024.0,11376512.0,11408000.0,11439488.0,11470976.0,11273344.0,11304896.0,11336448.0,11368000.0,11399552.0,11431104.0,11462656.0,11494208.0,11296128.0,11327744.0,11359360.0,11390976.0,11422592.0,11454208.0,11485824.0,11517440.0,11318912.0,11350592.0,11382272.0,11413952.0,11445632.0,11477312.0,11508992.0,11540672.0,11341696.0,11373440.0,11405184.0,11436928.0,11468672.0,11500416.0,11532160.0,11563904.0,11364480.0,11396288.0,11428096.0,11459904.0,11491712.0,11523520.0,11555328.0,11587136.0,11387264.0,11419136.0,11451008.0,11482880.0,11514752.0,11546624.0,11578496.0,11610368.0,11410048.0,11441984.0,11473920.0,11505856.0,11537792.0,11569728.0,11601664.0,11633600.0,11432832.0,11464832.0,11496832.0,11528832.0,11560832.0,11592832.0,11624832.0,11656832.0,11455616.0,11487680.0,11519744.0,11551808.0,11583872.0,11615936.0,11648000.0,11680064.0,11478400.0,11510528.0,11542656.0,11574784.0,11606912.0,11639040.0,11671168.0,11703296.0,11501184.0,11533376.0,11565568.0,11597760.0,11629952.0,11662144.0,11694336.0,11726528.0,11523968.0,11556224.0,11588480.0,11620736.0,11652992.0,11685248.0,11717504.0,11749760.0,11546752.0,11579072.0,11611392.0,11643712.0,11676032.0,11708352.0,11740672.0,11772992.0,11569536.0,11601920.0,11634304.0,11666688.0,11699072.0,11731456.0,11763840.0,11796224.0,11592320.0,11624768.0,11657216.0,11689664.0,11722112.0,11754560.0,11787008.0,11819456.0,11615104.0,11647616.0,11680128.0,11712640.0,11745152.0,11777664.0,11810176.0,11842688.0,11637888.0,11670464.0,11703040.0,11735616.0,11768192.0,11800768.0,11833344.0,11865920.0,11660672.0,11693312.0,11725952.0,11758592.0,11791232.0,11823872.0,11856512.0,11889152.0,11683456.0,11716160.0,11748864.0,11781568.0,11814272.0,11846976.0,11879680.0,11912384.0,11706240.0,11739008.0,11771776.0,11804544.0,11837312.0,11870080.0,11902848.0,11935616.0,11729024.0,11761856.0,11794688.0,11827520.0,11860352.0,11893184.0,11926016.0,11958848.0,11129216.0,11161088.0,11192960.0,11224832.0,11256704.0,11288576.0,11320448.0,11352320.0,10521216.0,10552128.0,10583040.0,10613952.0,10644864.0,10675776.0,10706688.0,10737600.0,10544000.0,10574976.0,10605952.0,10636928.0,10667904.0,10698880.0,10729856.0,10760832.0,10566784.0,10597824.0,10628864.0,10659904.0,10690944.0,10721984.0,10753024.0,10784064.0,10589568.0,10620672.0,10651776.0,10682880.0,10713984.0,10745088.0,10776192.0,10807296.0,10612352.0,10643520.0,10674688.0,10705856.0,10737024.0,10768192.0,10799360.0,10830528.0,10635136.0,10666368.0,10697600.0,10728832.0,10760064.0,10791296.0,10822528.0,10853760.0,10657920.0,10689216.0,10720512.0,10751808.0,10783104.0,10814400.0,10845696.0,10876992.0,10680704.0,10712064.0,10743424.0,10774784.0,10806144.0,10837504.0,10868864.0,10900224.0,10703488.0,10734912.0,10766336.0,10797760.0,10829184.0,10860608.0,10892032.0,10923456.0,10726272.0,10757760.0,10789248.0,10820736.0,10852224.0,10883712.0,10915200.0,10946688.0,10749056.0,10780608.0,10812160.0,10843712.0,10875264.0,10906816.0,10938368.0,10969920.0,10771840.0,10803456.0,10835072.0,10866688.0,10898304.0,10929920.0,10961536.0,10993152.0,10794624.0,10826304.0,10857984.0,10889664.0,10921344.0,10953024.0,10984704.0,11016384.0,10817408.0,10849152.0,10880896.0,10912640.0,10944384.0,10976128.0,11007872.0,11039616.0,10840192.0,10872000.0,10903808.0,10935616.0,10967424.0,10999232.0,11031040.0,11062848.0,10862976.0,10894848.0,10926720.0,10958592.0,10990464.0,11022336.0,11054208.0,11086080.0,10885760.0,10917696.0,10949632.0,10981568.0,11013504.0,11045440.0,11077376.0,11109312.0,10908544.0,10940544.0,10972544.0,11004544.0,11036544.0,11068544.0,11100544.0,11132544.0,10931328.0,10963392.0,10995456.0,11027520.0,11059584.0,11091648.0,11123712.0,11155776.0,10954112.0,10986240.0,11018368.0,11050496.0,11082624.0,11114752.0,11146880.0,11179008.0,10976896.0,11009088.0,11041280.0,11073472.0,11105664.0,11137856.0,11170048.0,11202240.0,10999680.0,11031936.0,11064192.0,11096448.0,11128704.0,11160960.0,11193216.0,11225472.0,11022464.0,11054784.0,11087104.0,11119424.0,11151744.0,11184064.0,11216384.0,11248704.0,11045248.0,11077632.0,11110016.0,11142400.0,11174784.0,11207168.0,11239552.0,11271936.0,11068032.0,11100480.0,11132928.0,11165376.0,11197824.0,11230272.0,11262720.0,11295168.0,11090816.0,11123328.0,11155840.0,11188352.0,11220864.0,11253376.0,11285888.0,11318400.0,11113600.0,11146176.0,11178752.0,11211328.0,11243904.0,11276480.0,11309056.0,11341632.0,11136384.0,11169024.0,11201664.0,11234304.0,11266944.0,11299584.0,11332224.0,11364864.0,11159168.0,11191872.0,11224576.0,11257280.0,11289984.0,11322688.0,11355392.0,11388096.0,11181952.0,11214720.0,11247488.0,11280256.0,11313024.0,11345792.0,11378560.0,11411328.0,11204736.0,11237568.0,11270400.0,11303232.0,11336064.0,11368896.0,11401728.0,11434560.0,11227520.0,11260416.0,11293312.0,11326208.0,11359104.0,11392000.0,11424896.0,11457792.0,11250304.0,11283264.0,11316224.0,11349184.0,11382144.0,11415104.0,11448064.0,11481024.0,11273088.0,11306112.0,11339136.0,11372160.0,11405184.0,11438208.0,11471232.0,11504256.0,11295872.0,11328960.0,11362048.0,11395136.0,11428224.0,11461312.0,11494400.0,11527488.0,11318656.0,11351808.0,11384960.0,11418112.0,11451264.0,11484416.0,11517568.0,11550720.0,11341440.0,11374656.0,11407872.0,11441088.0,11474304.0,11507520.0,11540736.0,11573952.0,11364224.0,11397504.0,11430784.0,11464064.0,11497344.0,11530624.0,11563904.0,11597184.0,11288704.0,11321024.0,11353344.0,11385664.0,11417984.0,11450304.0,11482624.0,11514944.0,11204992.0,11236352.0,11267712.0,11299072.0,11330432.0,11361792.0,11393152.0,11424512.0,11227776.0,11259200.0,11290624.0,11322048.0,11353472.0,11384896.0,11416320.0,11447744.0,11250560.0,11282048.0,11313536.0,11345024.0,11376512.0,11408000.0,11439488.0,11470976.0,11273344.0,11304896.0,11336448.0,11368000.0,11399552.0,11431104.0,11462656.0,11494208.0,11296128.0,11327744.0,11359360.0,11390976.0,11422592.0,11454208.0,11485824.0,11517440.0,11318912.0,11350592.0,11382272.0,11413952.0,11445632.0,11477312.0,11508992.0,11540672.0,11341696.0,11373440.0,11405184.0,11436928.0,11468672.0,11500416.0,11532160.0,11563904.0,11364480.0,11396288.0,11428096.0,11459904.0,11491712.0,11523520.0,11555328.0,11587136.0,11387264.0,11419136.0,11451008.0,11482880.0,11514752.0,11546624.0,11578496.0,11610368.0,11410048.0,11441984.0,11473920.0,11505856.0,11537792.0,11569728.0,11601664.0,11633600.0,11432832.0,11464832.0,11496832.0,11528832.0,11560832.0,11592832.0,11624832.0,11656832.0,11455616.0,11487680.0,11519744.0,11551808.0,11583872.0,11615936.0,11648000.0,11680064.0,11478400.0,11510528.0,11542656.0,11574784.0,11606912.0,11639040.0,11671168.0,11703296.0,11501184.0,11533376.0,11565568.0,11597760.0,11629952.0,11662144.0,11694336.0,11726528.0,11523968.0,11556224.0,11588480.0,11620736.0,11652992.0,11685248.0,11717504.0,11749760.0,11546752.0,11579072.0,11611392.0,11643712.0,11676032.0,11708352.0,11740672.0,11772992.0,11569536.0,11601920.0,11634304.0,11666688.0,11699072.0,11731456.0,11763840.0,11796224.0,11592320.0,11624768.0,11657216.0,11689664.0,11722112.0,11754560.0,11787008.0,11819456.0,11615104.0,11647616.0,11680128.0,11712640.0,11745152.0,11777664.0,11810176.0,11842688.0,11637888.0,11670464.0,11703040.0,11735616.0,11768192.0,11800768.0,11833344.0,11865920.0,11660672.0,11693312.0,11725952.0,11758592.0,11791232.0,11823872.0,11856512.0,11889152.0,11683456.0,11716160.0,11748864.0,11781568.0,11814272.0,11846976.0,11879680.0,11912384.0,11706240.0,11739008.0,11771776.0,11804544.0,11837312.0,11870080.0,11902848.0,11935616.0,11729024.0,11761856.0,11794688.0,11827520.0,11860352.0,11893184.0,11926016.0,11958848.0,11751808.0,11784704.0,11817600.0,11850496.0,11883392.0,11916288.0,11949184.0,11982080.0,11774592.0,11807552.0,11840512.0,11873472.0,11906432.0,11939392.0,11972352.0,12005312.0,11797376.0,11830400.0,11863424.0,11896448.0,11929472.0,11962496.0,11995520.0,12028544.0,11820160.0,11853248.0,11886336.0,11919424.0,11952512.0,11985600.0,12018688.0,12051776.0,11842944.0,11876096.0,11909248.0,11942400.0,11975552.0,12008704.0,12041856.0,12075008.0,11865728.0,11898944.0,11932160.0,11965376.0,11998592.0,12031808.0,12065024.0,12098240.0,11888512.0,11921792.0,11955072.0,11988352.0,12021632.0,12054912.0,12088192.0,12121472.0,11911296.0,11944640.0,11977984.0,12011328.0,12044672.0,12078016.0,12111360.0,12144704.0,11934080.0,11967488.0,12000896.0,12034304.0,12067712.0,12101120.0,12134528.0,12167936.0,11956864.0,11990336.0,12023808.0,12057280.0,12090752.0,12124224.0,12157696.0,12191168.0,11979648.0,12013184.0,12046720.0,12080256.0,12113792.0,12147328.0,12180864.0,12214400.0,12002432.0,12036032.0,12069632.0,12103232.0,12136832.0,12170432.0,12204032.0,12237632.0,12025216.0,12058880.0,12092544.0,12126208.0,12159872.0,12193536.0,12227200.0,12260864.0,12048000.0,12081728.0,12115456.0,12149184.0,12182912.0,12216640.0,12250368.0,12284096.0,11448192.0,11480960.0,11513728.0,11546496.0,11579264.0,11612032.0,11644800.0,11677568.0,10840192.0,10872000.0,10903808.0,10935616.0,10967424.0,10999232.0,11031040.0,11062848.0,5489536.0,5505456.0,5521376.0,5537296.0,5553216.0,5569136.0,5585056.0,5600976.0,6094080.0,6109872.0,6125664.0,6141456.0,6157248.0,6173040.0,6188832.0,6204624.0,12326272.0,12357888.0,12389504.0,12421120.0,12452736.0,12484352.0,12515968.0,12547584.0,12350080.0,12381760.0,12413440.0,12445120.0,12476800.0,12508480.0,12540160.0,12571840.0,12373888.0,12405632.0,12437376.0,12469120.0,12500864.0,12532608.0,12564352.0,12596096.0,12397696.0,12429504.0,12461312.0,12493120.0,12524928.0,12556736.0,12588544.0,12620352.0,12421504.0,12453376.0,12485248.0,12517120.0,12548992.0,12580864.0,12612736.0,12644608.0,12445312.0,12477248.0,12509184.0,12541120.0,12573056.0,12604992.0,12636928.0,12668864.0,12469120.0,12501120.0,12533120.0,12565120.0,12597120.0,12629120.0,12661120.0,12693120.0,12492928.0,12524992.0,12557056.0,12589120.0,12621184.0,12653248.0,12685312.0,12717376.0,12516736.0,12548864.0,12580992.0,12613120.0,12645248.0,12677376.0,12709504.0,12741632.0,12540544.0,12572736.0,12604928.0,12637120.0,12669312.0,12701504.0,12733696.0,12765888.0,12564352.0,12596608.0,12628864.0,12661120.0,12693376.0,12725632.0,12757888.0,12790144.0,12588160.0,12620480.0,12652800.0,12685120.0,12717440.0,12749760.0,12782080.0,12814400.0,12611968.0,12644352.0,12676736.0,12709120.0,12741504.0,12773888.0,12806272.0,12838656.0,12635776.0,12668224.0,12700672.0,12733120.0,12765568.0,12798016.0,12830464.0,12862912.0,12659584.0,12692096.0,12724608.0,12757120.0,12789632.0,12822144.0,12854656.0,12887168.0,12683392.0,12715968.0,12748544.0,12781120.0,12813696.0,12846272.0,12878848.0,12911424.0,12707200.0,12739840.0,12772480.0,12805120.0,12837760.0,12870400.0,12903040.0,12935680.0,12731008.0,12763712.0,12796416.0,12829120.0,12861824.0,12894528.0,12927232.0,12959936.0,12754816.0,12787584.0,12820352.0,12853120.0,12885888.0,12918656.0,12951424.0,12984192.0,12778624.0,12811456.0,12844288.0,12877120.0,12909952.0,12942784.0,12975616.0,13008448.0,12802432.0,12835328.0,12868224.0,12901120.0,12934016.0,12966912.0,12999808.0,13032704.0,12826240.0,12859200.0,12892160.0,12925120.0,12958080.0,12991040.0,13024000.0,13056960.0,12850048.0,12883072.0,12916096.0,12949120.0,12982144.0,13015168.0,13048192.0,13081216.0,12873856.0,12906944.0,12940032.0,12973120.0,13006208.0,13039296.0,13072384.0,13105472.0,12897664.0,12930816.0,12963968.0,12997120.0,13030272.0,13063424.0,13096576.0,13129728.0,12921472.0,12954688.0,12987904.0,13021120.0,13054336.0,13087552.0,13120768.0,13153984.0,12945280.0,12978560.0,13011840.0,13045120.0,13078400.0,13111680.0,13144960.0,13178240.0,12969088.0,13002432.0,13035776.0,13069120.0,13102464.0,13135808.0,13169152.0,13202496.0,12992896.0,13026304.0,13059712.0,13093120.0,13126528.0,13159936.0,13193344.0,13226752.0,13016704.0,13050176.0,13083648.0,13117120.0,13150592.0,13184064.0,13217536.0,13251008.0,13040512.0,13074048.0,13107584.0,13141120.0,13174656.0,13208192.0,13241728.0,13275264.0,13064320.0,13097920.0,13131520.0,13165120.0,13198720.0,13232320.0,13265920.0,13299520.0,13088128.0,13121792.0,13155456.0,13189120.0,13222784.0,13256448.0,13290112.0,13323776.0,13111936.0,13145664.0,13179392.0,13213120.0,13246848.0,13280576.0,13314304.0,13348032.0,13135744.0,13169536.0,13203328.0,13237120.0,13270912.0,13304704.0,13338496.0,13372288.0,13159552.0,13193408.0,13227264.0,13261120.0,13294976.0,13328832.0,13362688.0,13396544.0,12544384.0,12577280.0,12610176.0,12643072.0,12675968.0,12708864.0,12741760.0,12774656.0,11921024.0,11952960.0,11984896.0,12016832.0,12048768.0,12080704.0,12112640.0,12144576.0,11305856.0,11336832.0,11367808.0,11398784.0,11429760.0,11460736.0,11491712.0,11522688.0,10682496.0,10712512.0,10742528.0,10772544.0,10802560.0,10832576.0,10862592.0,10892608.0,10706304.0,10736384.0,10766464.0,10796544.0,10826624.0,10856704.0,10886784.0,10916864.0,10730112.0,10760256.0,10790400.0,10820544.0,10850688.0,10880832.0,10910976.0,10941120.0,10753920.0,10784128.0,10814336.0,10844544.0,10874752.0,10904960.0,10935168.0,10965376.0,10777728.0,10808000.0,10838272.0,10868544.0,10898816.0,10929088.0,10959360.0,10989632.0,10801536.0,10831872.0,10862208.0,10892544.0,10922880.0,10953216.0,10983552.0,11013888.0,10825344.0,10855744.0,10886144.0,10916544.0,10946944.0,10977344.0,11007744.0,11038144.0,10849152.0,10879616.0,10910080.0,10940544.0,10971008.0,11001472.0,11031936.0,11062400.0,10872960.0,10903488.0,10934016.0,10964544.0,10995072.0,11025600.0,11056128.0,11086656.0,10896768.0,10927360.0,10957952.0,10988544.0,11019136.0,11049728.0,11080320.0,11110912.0,10920576.0,10951232.0,10981888.0,11012544.0,11043200.0,11073856.0,11104512.0,11135168.0,10944384.0,10975104.0,11005824.0,11036544.0,11067264.0,11097984.0,11128704.0,11159424.0,10968192.0,10998976.0,11029760.0,11060544.0,11091328.0,11122112.0,11152896.0,11183680.0,10992000.0,11022848.0,11053696.0,11084544.0,11115392.0,11146240.0,11177088.0,11207936.0,11015808.0,11046720.0,11077632.0,11108544.0,11139456.0,11170368.0,11201280.0,11232192.0,11039616.0,11070592.0,11101568.0,11132544.0,11163520.0,11194496.0,11225472.0,11256448.0,11063424.0,11094464.0,11125504.0,11156544.0,11187584.0,11218624.0,11249664.0,11280704.0,11087232.0,11118336.0,11149440.0,11180544.0,11211648.0,11242752.0,11273856.0,11304960.0,11111040.0,11142208.0,11173376.0,11204544.0,11235712.0,11266880.0,11298048.0,11329216.0,11134848.0,11166080.0,11197312.0,11228544.0,11259776.0,11291008.0,11322240.0,11353472.0,11158656.0,11189952.0,11221248.0,11252544.0,11283840.0,11315136.0,11346432.0,11377728.0,11182464.0,11213824.0,11245184.0,11276544.0,11307904.0,11339264.0,11370624.0,11401984.0,11206272.0,11237696.0,11269120.0,11300544.0,11331968.0,11363392.0,11394816.0,11426240.0,11230080.0,11261568.0,11293056.0,11324544.0,11356032.0,11387520.0,11419008.0,11450496.0,11253888.0,11285440.0,11316992.0,11348544.0,11380096.0,11411648.0,11443200.0,11474752.0,11277696.0,11309312.0,11340928.0,11372544.0,11404160.0,11435776.0,11467392.0,11499008.0,11301504.0,11333184.0,11364864.0,11396544.0,11428224.0,11459904.0,11491584.0,11523264.0,11325312.0,11357056.0,11388800.0,11420544.0,11452288.0,11484032.0,11515776.0,11547520.0,11349120.0,11380928.0,11412736.0,11444544.0,11476352.0,11508160.0,11539968.0,11571776.0,11372928.0,11404800.0,11436672.0,11468544.0,11500416.0,11532288.0,11564160.0,11596032.0,11396736.0,11428672.0,11460608.0,11492544.0,11524480.0,11556416.0,11588352.0,11620288.0,11420544.0,11452544.0,11484544.0,11516544.0,11548544.0,11580544.0,11612544.0,11644544.0,11444352.0,11476416.0,11508480.0,11540544.0,11572608.0,11604672.0,11636736.0,11668800.0,11468160.0,11500288.0,11532416.0,11564544.0,11596672.0,11628800.0,11660928.0,11693056.0,11491968.0,11524160.0,11556352.0,11588544.0,11620736.0,11652928.0,11685120.0,11717312.0,11515776.0,11548032.0,11580288.0,11612544.0,11644800.0,11677056.0,11709312.0,11741568.0,11539584.0,11571904.0,11604224.0,11636544.0,11668864.0,11701184.0,11733504.0,11765824.0,11563392.0,11595776.0,11628160.0,11660544.0,11692928.0,11725312.0,11757696.0,11790080.0,11472512.0,11503936.0,11535360.0,11566784.0,11598208.0,11629632.0,11661056.0,11692480.0,11373440.0,11403904.0,11434368.0,11464832.0,11495296.0,11525760.0,11556224.0,11586688.0,11397248.0,11427776.0,11458304.0,11488832.0,11519360.0,11549888.0,11580416.0,11610944.0,11421056.0,11451648.0,11482240.0,11512832.0,11543424.0,11574016.0,11604608.0,11635200.0,11444864.0,11475520.0,11506176.0,11536832.0,11567488.0,11598144.0,11628800.0,11659456.0,11468672.0,11499392.0,11530112.0,11560832.0,11591552.0,11622272.0,11652992.0,11683712.0,11492480.0,11523264.0,11554048.0,11584832.0,11615616.0,11646400.0,11677184.0,11707968.0,11516288.0,11547136.0,11577984.0,11608832.0,11639680.0,11670528.0,11701376.0,11732224.0,11540096.0,11571008.0,11601920.0,11632832.0,11663744.0,11694656.0,11725568.0,11756480.0,11563904.0,11594880.0,11625856.0,11656832.0,11687808.0,11718784.0,11749760.0,11780736.0,11587712.0,11618752.0,11649792.0,11680832.0,11711872.0,11742912.0,11773952.0,11804992.0,11611520.0,11642624.0,11673728.0,11704832.0,11735936.0,11767040.0,11798144.0,11829248.0,11635328.0,11666496.0,11697664.0,11728832.0,11760000.0,11791168.0,11822336.0,11853504.0,11659136.0,11690368.0,11721600.0,11752832.0,11784064.0,11815296.0,11846528.0,11877760.0,11682944.0,11714240.0,11745536.0,11776832.0,11808128.0,11839424.0,11870720.0,11902016.0,11706752.0,11738112.0,11769472.0,11800832.0,11832192.0,11863552.0,11894912.0,11926272.0,11730560.0,11761984.0,11793408.0,11824832.0,11856256.0,11887680.0,11919104.0,11950528.0,11754368.0,11785856.0,11817344.0,11848832.0,11880320.0,11911808.0,11943296.0,11974784.0,11778176.0,11809728.0,11841280.0,11872832.0,11904384.0,11935936.0,11967488.0,11999040.0,11801984.0,11833600.0,11865216.0,11896832.0,11928448.0,11960064.0,11991680.0,12023296.0,11825792.0,11857472.0,11889152.0,11920832.0,11952512.0,11984192.0,12015872.0,12047552.0,11849600.0,11881344.0,11913088.0,11944832.0,11976576.0,12008320.0,12040064.0,12071808.0,11873408.0,11905216.0,11937024.0,11968832.0,12000640.0,12032448.0,12064256.0,12096064.0,11897216.0,11929088.0,11960960.0,11992832.0,12024704.0,12056576.0,12088448.0,12120320.0,11921024.0,11952960.0,11984896.0,12016832.0,12048768.0,12080704.0,12112640.0,12144576.0,11944832.0,11976832.0,12008832.0,12040832.0,12072832.0,12104832.0,12136832.0,12168832.0,11968640.0,12000704.0,12032768.0,12064832.0,12096896.0,12128960.0,12161024.0,12193088.0,11992448.0,12024576.0,12056704.0,12088832.0,12120960.0,12153088.0,12185216.0,12217344.0,12016256.0,12048448.0,12080640.0,12112832.0,12145024.0,12177216.0,12209408.0,12241600.0,12040064.0,12072320.0,12104576.0,12136832.0,12169088.0,12201344.0,12233600.0,12265856.0,12063872.0,12096192.0,12128512.0,12160832.0,12193152.0,12225472.0,12257792.0,12290112.0,12087680.0,12120064.0,12152448.0,12184832.0,12217216.0,12249600.0,12281984.0,12314368.0,12111488.0,12143936.0,12176384.0,12208832.0,12241280.0,12273728.0,12306176.0,12338624.0,12135296.0,12167808.0,12200320.0,12232832.0,12265344.0,12297856.0,12330368.0,12362880.0,12159104.0,12191680.0,12224256.0,12256832.0,12289408.0,12321984.0,12354560.0,12387136.0,12182912.0,12215552.0,12248192.0,12280832.0,12313472.0,12346112.0,12378752.0,12411392.0,12206720.0,12239424.0,12272128.0,12304832.0,12337536.0,12370240.0,12402944.0,12435648.0,12230528.0,12263296.0,12296064.0,12328832.0,12361600.0,12394368.0,12427136.0,12459904.0,12254336.0,12287168.0,12320000.0,12352832.0,12385664.0,12418496.0,12451328.0,12484160.0,11639168.0,11671040.0,11702912.0,11734784.0,11766656.0,11798528.0,11830400.0,11862272.0,11015808.0,11046720.0,11077632.0,11108544.0,11139456.0,11170368.0,11201280.0,11232192.0,11039616.0,11070592.0,11101568.0,11132544.0,11163520.0,11194496.0,11225472.0,11256448.0,11063424.0,11094464.0,11125504.0,11156544.0,11187584.0,11218624.0,11249664.0,11280704.0,11087232.0,11118336.0,11149440.0,11180544.0,11211648.0,11242752.0,11273856.0,11304960.0,11111040.0,11142208.0,11173376.0,11204544.0,11235712.0,11266880.0,11298048.0,11329216.0,11134848.0,11166080.0,11197312.0,11228544.0,11259776.0,11291008.0,11322240.0,11353472.0,11158656.0,11189952.0,11221248.0,11252544.0,11283840.0,11315136.0,11346432.0,11377728.0,11182464.0,11213824.0,11245184.0,11276544.0,11307904.0,11339264.0,11370624.0,11401984.0,11206272.0,11237696.0,11269120.0,11300544.0,11331968.0,11363392.0,11394816.0,11426240.0,11230080.0,11261568.0,11293056.0,11324544.0,11356032.0,11387520.0,11419008.0,11450496.0,11253888.0,11285440.0,11316992.0,11348544.0,11380096.0,11411648.0,11443200.0,11474752.0,11277696.0,11309312.0,11340928.0,11372544.0,11404160.0,11435776.0,11467392.0,11499008.0,11301504.0,11333184.0,11364864.0,11396544.0,11428224.0,11459904.0,11491584.0,11523264.0,11325312.0,11357056.0,11388800.0,11420544.0,11452288.0,11484032.0,11515776.0,11547520.0,11349120.0,11380928.0,11412736.0,11444544.0,11476352.0,11508160.0,11539968.0,11571776.0,11372928.0,11404800.0,11436672.0,11468544.0,11500416.0,11532288.0,11564160.0,11596032.0,11396736.0,11428672.0,11460608.0,11492544.0,11524480.0,11556416.0,11588352.0,11620288.0,11420544.0,11452544.0,11484544.0,11516544.0,11548544.0,11580544.0,11612544.0,11644544.0,11444352.0,11476416.0,11508480.0,11540544.0,11572608.0,11604672.0,11636736.0,11668800.0,11468160.0,11500288.0,11532416.0,11564544.0,11596672.0,11628800.0,11660928.0,11693056.0,11491968.0,11524160.0,11556352.0,11588544.0,11620736.0,11652928.0,11685120.0,11717312.0,11515776.0,11548032.0,11580288.0,11612544.0,11644800.0,11677056.0,11709312.0,11741568.0,11539584.0,11571904.0,11604224.0,11636544.0,11668864.0,11701184.0,11733504.0,11765824.0,11563392.0,11595776.0,11628160.0,11660544.0,11692928.0,11725312.0,11757696.0,11790080.0,11587200.0,11619648.0,11652096.0,11684544.0,11716992.0,11749440.0,11781888.0,11814336.0,11611008.0,11643520.0,11676032.0,11708544.0,11741056.0,11773568.0,11806080.0,11838592.0,11634816.0,11667392.0,11699968.0,11732544.0,11765120.0,11797696.0,11830272.0,11862848.0,11658624.0,11691264.0,11723904.0,11756544.0,11789184.0,11821824.0,11854464.0,11887104.0,11682432.0,11715136.0,11747840.0,11780544.0,11813248.0,11845952.0,11878656.0,11911360.0,11706240.0,11739008.0,11771776.0,11804544.0,11837312.0,11870080.0,11902848.0,11935616.0,11730048.0,11762880.0,11795712.0,11828544.0,11861376.0,11894208.0,11927040.0,11959872.0,11753856.0,11786752.0,11819648.0,11852544.0,11885440.0,11918336.0,11951232.0,11984128.0,11777664.0,11810624.0,11843584.0,11876544.0,11909504.0,11942464.0,11975424.0,12008384.0,11801472.0,11834496.0,11867520.0,11900544.0,11933568.0,11966592.0,11999616.0,12032640.0,11825280.0,11858368.0,11891456.0,11924544.0,11957632.0,11990720.0,12023808.0,12056896.0,11849088.0,11882240.0,11915392.0,11948544.0,11981696.0,12014848.0,12048000.0,12081152.0,11872896.0,11906112.0,11939328.0,11972544.0,12005760.0,12038976.0,12072192.0,12105408.0,11896704.0,11929984.0,11963264.0,11996544.0,12029824.0,12063104.0,12096384.0,12129664.0,11805824.0,11838144.0,11870464.0,11902784.0,11935104.0,11967424.0,11999744.0,12032064.0,11706752.0,11738112.0,11769472.0,11800832.0,11832192.0,11863552.0,11894912.0,11926272.0,11730560.0,11761984.0,11793408.0,11824832.0,11856256.0,11887680.0,11919104.0,11950528.0,11754368.0,11785856.0,11817344.0,11848832.0,11880320.0,11911808.0,11943296.0,11974784.0,11778176.0,11809728.0,11841280.0,11872832.0,11904384.0,11935936.0,11967488.0,11999040.0,11801984.0,11833600.0,11865216.0,11896832.0,11928448.0,11960064.0,11991680.0,12023296.0,11825792.0,11857472.0,11889152.0,11920832.0,11952512.0,11984192.0,12015872.0,12047552.0,11849600.0,11881344.0,11913088.0,11944832.0,11976576.0,12008320.0,12040064.0,12071808.0,11873408.0,11905216.0,11937024.0,11968832.0,12000640.0,12032448.0,12064256.0,12096064.0,11897216.0,11929088.0,11960960.0,11992832.0,12024704.0,12056576.0,12088448.0,12120320.0,11921024.0,11952960.0,11984896.0,12016832.0,12048768.0,12080704.0,12112640.0,12144576.0,11944832.0,11976832.0,12008832.0,12040832.0,12072832.0,12104832.0,12136832.0,12168832.0,11968640.0,12000704.0,12032768.0,12064832.0,12096896.0,12128960.0,12161024.0,12193088.0,11992448.0,12024576.0,12056704.0,12088832.0,12120960.0,12153088.0,12185216.0,12217344.0,12016256.0,12048448.0,12080640.0,12112832.0,12145024.0,12177216.0,12209408.0,12241600.0,12040064.0,12072320.0,12104576.0,12136832.0,12169088.0,12201344.0,12233600.0,12265856.0,12063872.0,12096192.0,12128512.0,12160832.0,12193152.0,12225472.0,12257792.0,12290112.0,12087680.0,12120064.0,12152448.0,12184832.0,12217216.0,12249600.0,12281984.0,12314368.0,12111488.0,12143936.0,12176384.0,12208832.0,12241280.0,12273728.0,12306176.0,12338624.0,12135296.0,12167808.0,12200320.0,12232832.0,12265344.0,12297856.0,12330368.0,12362880.0,12159104.0,12191680.0,12224256.0,12256832.0,12289408.0,12321984.0,12354560.0,12387136.0,12182912.0,12215552.0,12248192.0,12280832.0,12313472.0,12346112.0,12378752.0,12411392.0,12206720.0,12239424.0,12272128.0,12304832.0,12337536.0,12370240.0,12402944.0,12435648.0,12230528.0,12263296.0,12296064.0,12328832.0,12361600.0,12394368.0,12427136.0,12459904.0,12254336.0,12287168.0,12320000.0,12352832.0,12385664.0,12418496.0,12451328.0,12484160.0,12278144.0,12311040.0,12343936.0,12376832.0,12409728.0,12442624.0,12475520.0,12508416.0,12301952.0,12334912.0,12367872.0,12400832.0,12433792.0,12466752.0,12499712.0,12532672.0,12325760.0,12358784.0,12391808.0,12424832.0,12457856.0,12490880.0,12523904.0,12556928.0,12349568.0,12382656.0,12415744.0,12448832.0,12481920.0,12515008.0,12548096.0,12581184.0,12373376.0,12406528.0,12439680.0,12472832.0,12505984.0,12539136.0,12572288.0,12605440.0,12397184.0,12430400.0,12463616.0,12496832.0,12530048.0,12563264.0,12596480.0,12629696.0,12420992.0,12454272.0,12487552.0,12520832.0,12554112.0,12587392.0,12620672.0,12653952.0,12444800.0,12478144.0,12511488.0,12544832.0,12578176.0,12611520.0,12644864.0,12678208.0,12468608.0,12502016.0,12535424.0,12568832.0,12602240.0,12635648.0,12669056.0,12702464.0,12492416.0,12525888.0,12559360.0,12592832.0,12626304.0,12659776.0,12693248.0,12726720.0,12516224.0,12549760.0,12583296.0,12616832.0,12650368.0,12683904.0,12717440.0,12750976.0,12540032.0,12573632.0,12607232.0,12640832.0,12674432.0,12708032.0,12741632.0,12775232.0,12563840.0,12597504.0,12631168.0,12664832.0,12698496.0,12732160.0,12765824.0,12799488.0,12587648.0,12621376.0,12655104.0,12688832.0,12722560.0,12756288.0,12790016.0,12823744.0,11972480.0,12005248.0,12038016.0,12070784.0,12103552.0,12136320.0,12169088.0,12201856.0,11349120.0,11380928.0,11412736.0,11444544.0,11476352.0,11508160.0,11539968.0,11571776.0,5744256.0,5760176.0,5776096.0,5792016.0,5807936.0,5823856.0,5839776.0,5855696.0,6346752.0,6362544.0,6378336.0,6394128.0,6409920.0,6425712.0,6441504.0,6457296.0,12832128.0,12863744.0,12895360.0,12926976.0,12958592.0,12990208.0,13021824.0,13053440.0,12856960.0,12888640.0,12920320.0,12952000.0,12983680.0,13015360.0,13047040.0,13078720.0,12881792.0,12913536.0,12945280.0,12977024.0,13008768.0,13040512.0,13072256.0,13104000.0,12906624.0,12938432.0,12970240.0,13002048.0,13033856.0,13065664.0,13097472.0,13129280.0,12931456.0,12963328.0,12995200.0,13027072.0,13058944.0,13090816.0,13122688.0,13154560.0,12956288.0,12988224.0,13020160.0,13052096.0,13084032.0,13115968.0,13147904.0,13179840.0,12981120.0,13013120.0,13045120.0,13077120.0,13109120.0,13141120.0,13173120.0,13205120.0,13005952.0,13038016.0,13070080.0,13102144.0,13134208.0,13166272.0,13198336.0,13230400.0,13030784.0,13062912.0,13095040.0,13127168.0,13159296.0,13191424.0,13223552.0,13255680.0,13055616.0,13087808.0,13120000.0,13152192.0,13184384.0,13216576.0,13248768.0,13280960.0,13080448.0,13112704.0,13144960.0,13177216.0,13209472.0,13241728.0,13273984.0,13306240.0,13105280.0,13137600.0,13169920.0,13202240.0,13234560.0,13266880.0,13299200.0,13331520.0,13130112.0,13162496.0,13194880.0,13227264.0,13259648.0,13292032.0,13324416.0,13356800.0,13154944.0,13187392.0,13219840.0,13252288.0,13284736.0,13317184.0,13349632.0,13382080.0,13179776.0,13212288.0,13244800.0,13277312.0,13309824.0,13342336.0,13374848.0,13407360.0,13204608.0,13237184.0,13269760.0,13302336.0,13334912.0,13367488.0,13400064.0,13432640.0,13229440.0,13262080.0,13294720.0,13327360.0,13360000.0,13392640.0,13425280.0,13457920.0,13254272.0,13286976.0,13319680.0,13352384.0,13385088.0,13417792.0,13450496.0,13483200.0,13279104.0,13311872.0,13344640.0,13377408.0,13410176.0,13442944.0,13475712.0,13508480.0,13303936.0,13336768.0,13369600.0,13402432.0,13435264.0,13468096.0,13500928.0,13533760.0,13328768.0,13361664.0,13394560.0,13427456.0,13460352.0,13493248.0,13526144.0,13559040.0,13353600.0,13386560.0,13419520.0,13452480.0,13485440.0,13518400.0,13551360.0,13584320.0,13378432.0,13411456.0,13444480.0,13477504.0,13510528.0,13543552.0,13576576.0,13609600.0,13403264.0,13436352.0,13469440.0,13502528.0,13535616.0,13568704.0,13601792.0,13634880.0,13428096.0,13461248.0,13494400.0,13527552.0,13560704.0,13593856.0,13627008.0,13660160.0,13452928.0,13486144.0,13519360.0,13552576.0,13585792.0,13619008.0,13652224.0,13685440.0,13477760.0,13511040.0,13544320.0,13577600.0,13610880.0,13644160.0,13677440.0,13710720.0,13502592.0,13535936.0,13569280.0,13602624.0,13635968.0,13669312.0,13702656.0,13736000.0,13527424.0,13560832.0,13594240.0,13627648.0,13661056.0,13694464.0,13727872.0,13761280.0,13552256.0,13585728.0,13619200.0,13652672.0,13686144.0,13719616.0,13753088.0,13786560.0,13577088.0,13610624.0,13644160.0,13677696.0,13711232.0,13744768.0,13778304.0,13811840.0,13601920.0,13635520.0,13669120.0,13702720.0,13736320.0,13769920.0,13803520.0,13837120.0,13626752.0,13660416.0,13694080.0,13727744.0,13761408.0,13795072.0,13828736.0,13862400.0,13651584.0,13685312.0,13719040.0,13752768.0,13786496.0,13820224.0,13853952.0,13887680.0,13676416.0,13710208.0,13744000.0,13777792.0,13811584.0,13845376.0,13879168.0,13912960.0,13701248.0,13735104.0,13768960.0,13802816.0,13836672.0,13870528.0,13904384.0,13938240.0,13070720.0,13103616.0,13136512.0,13169408.0,13202304.0,13235200.0,13268096.0,13300992.0,12432000.0,12463936.0,12495872.0,12527808.0,12559744.0,12591680.0,12623616.0,12655552.0,11801472.0,11832448.0,11863424.0,11894400.0,11925376.0,11956352.0,11987328.0,12018304.0,11162752.0,11192768.0,11222784.0,11252800.0,11282816.0,11312832.0,11342848.0,11372864.0,11187584.0,11217664.0,11247744.0,11277824.0,11307904.0,11337984.0,11368064.0,11398144.0,11212416.0,11242560.0,11272704.0,11302848.0,11332992.0,11363136.0,11393280.0,11423424.0,11237248.0,11267456.0,11297664.0,11327872.0,11358080.0,11388288.0,11418496.0,11448704.0,11262080.0,11292352.0,11322624.0,11352896.0,11383168.0,11413440.0,11443712.0,11473984.0,11286912.0,11317248.0,11347584.0,11377920.0,11408256.0,11438592.0,11468928.0,11499264.0,11311744.0,11342144.0,11372544.0,11402944.0,11433344.0,11463744.0,11494144.0,11524544.0,11336576.0,11367040.0,11397504.0,11427968.0,11458432.0,11488896.0,11519360.0,11549824.0,11361408.0,11391936.0,11422464.0,11452992.0,11483520.0,11514048.0,11544576.0,11575104.0,11386240.0,11416832.0,11447424.0,11478016.0,11508608.0,11539200.0,11569792.0,11600384.0,11411072.0,11441728.0,11472384.0,11503040.0,11533696.0,11564352.0,11595008.0,11625664.0,11435904.0,11466624.0,11497344.0,11528064.0,11558784.0,11589504.0,11620224.0,11650944.0,11460736.0,11491520.0,11522304.0,11553088.0,11583872.0,11614656.0,11645440.0,11676224.0,11485568.0,11516416.0,11547264.0,11578112.0,11608960.0,11639808.0,11670656.0,11701504.0,11510400.0,11541312.0,11572224.0,11603136.0,11634048.0,11664960.0,11695872.0,11726784.0,11535232.0,11566208.0,11597184.0,11628160.0,11659136.0,11690112.0,11721088.0,11752064.0,11560064.0,11591104.0,11622144.0,11653184.0,11684224.0,11715264.0,11746304.0,11777344.0,11584896.0,11616000.0,11647104.0,11678208.0,11709312.0,11740416.0,11771520.0,11802624.0,11609728.0,11640896.0,11672064.0,11703232.0,11734400.0,11765568.0,11796736.0,11827904.0,11634560.0,11665792.0,11697024.0,11728256.0,11759488.0,11790720.0,11821952.0,11853184.0,11659392.0,11690688.0,11721984.0,11753280.0,11784576.0,11815872.0,11847168.0,11878464.0,11684224.0,11715584.0,11746944.0,11778304.0,11809664.0,11841024.0,11872384.0,11903744.0,11709056.0,11740480.0,11771904.0,11803328.0,11834752.0,11866176.0,11897600.0,11929024.0,11733888.0,11765376.0,11796864.0,11828352.0,11859840.0,11891328.0,11922816.0,11954304.0,11758720.0,11790272.0,11821824.0,11853376.0,11884928.0,11916480.0,11948032.0,11979584.0,11783552.0,11815168.0,11846784.0,11878400.0,11910016.0,11941632.0,11973248.0,12004864.0,11808384.0,11840064.0,11871744.0,11903424.0,11935104.0,11966784.0,11998464.0,12030144.0,11833216.0,11864960.0,11896704.0,11928448.0,11960192.0,11991936.0,12023680.0,12055424.0,11858048.0,11889856.0,11921664.0,11953472.0,11985280.0,12017088.0,12048896.0,12080704.0,11882880.0,11914752.0,11946624.0,11978496.0,12010368.0,12042240.0,12074112.0,12105984.0,11907712.0,11939648.0,11971584.0,12003520.0,12035456.0,12067392.0,12099328.0,12131264.0,11932544.0,11964544.0,11996544.0,12028544.0,12060544.0,12092544.0,12124544.0,12156544.0,11957376.0,11989440.0,12021504.0,12053568.0,12085632.0,12117696.0,12149760.0,12181824.0,11982208.0,12014336.0,12046464.0,12078592.0,12110720.0,12142848.0,12174976.0,12207104.0,12007040.0,12039232.0,12071424.0,12103616.0,12135808.0,12168000.0,12200192.0,12232384.0,12031872.0,12064128.0,12096384.0,12128640.0,12160896.0,12193152.0,12225408.0,12257664.0,12056704.0,12089024.0,12121344.0,12153664.0,12185984.0,12218304.0,12250624.0,12282944.0,12081536.0,12113920.0,12146304.0,12178688.0,12211072.0,12243456.0,12275840.0,12308224.0,11975296.0,12006720.0,12038144.0,12069568.0,12100992.0,12132416.0,12163840.0,12195264.0,11860864.0,11891328.0,11921792.0,11952256.0,11982720.0,12013184.0,12043648.0,12074112.0,11885696.0,11916224.0,11946752.0,11977280.0,12007808.0,12038336.0,12068864.0,12099392.0,11910528.0,11941120.0,11971712.0,12002304.0,12032896.0,12063488.0,12094080.0,12124672.0,11935360.0,11966016.0,11996672.0,12027328.0,12057984.0,12088640.0,12119296.0,12149952.0,11960192.0,11990912.0,12021632.0,12052352.0,12083072.0,12113792.0,12144512.0,12175232.0,11985024.0,12015808.0,12046592.0,12077376.0,12108160.0,12138944.0,12169728.0,12200512.0,12009856.0,12040704.0,12071552.0,12102400.0,12133248.0,12164096.0,12194944.0,12225792.0,12034688.0,12065600.0,12096512.0,12127424.0,12158336.0,12189248.0,12220160.0,12251072.0,12059520.0,12090496.0,12121472.0,12152448.0,12183424.0,12214400.0,12245376.0,12276352.0,12084352.0,12115392.0,12146432.0,12177472.0,12208512.0,12239552.0,12270592.0,12301632.0,12109184.0,12140288.0,12171392.0,12202496.0,12233600.0,12264704.0,12295808.0,12326912.0,12134016.0,12165184.0,12196352.0,12227520.0,12258688.0,12289856.0,12321024.0,12352192.0,12158848.0,12190080.0,12221312.0,12252544.0,12283776.0,12315008.0,12346240.0,12377472.0,12183680.0,12214976.0,12246272.0,12277568.0,12308864.0,12340160.0,12371456.0,12402752.0,12208512.0,12239872.0,12271232.0,12302592.0,12333952.0,12365312.0,12396672.0,12428032.0,12233344.0,12264768.0,12296192.0,12327616.0,12359040.0,12390464.0,12421888.0,12453312.0,12258176.0,12289664.0,12321152.0,12352640.0,12384128.0,12415616.0,12447104.0,12478592.0,12283008.0,12314560.0,12346112.0,12377664.0,12409216.0,12440768.0,12472320.0,12503872.0,12307840.0,12339456.0,12371072.0,12402688.0,12434304.0,12465920.0,12497536.0,12529152.0,12332672.0,12364352.0,12396032.0,12427712.0,12459392.0,12491072.0,12522752.0,12554432.0,12357504.0,12389248.0,12420992.0,12452736.0,12484480.0,12516224.0,12547968.0,12579712.0,12382336.0,12414144.0,12445952.0,12477760.0,12509568.0,12541376.0,12573184.0,12604992.0,12407168.0,12439040.0,12470912.0,12502784.0,12534656.0,12566528.0,12598400.0,12630272.0,12432000.0,12463936.0,12495872.0,12527808.0,12559744.0,12591680.0,12623616.0,12655552.0,12456832.0,12488832.0,12520832.0,12552832.0,12584832.0,12616832.0,12648832.0,12680832.0,12481664.0,12513728.0,12545792.0,12577856.0,12609920.0,12641984.0,12674048.0,12706112.0,12506496.0,12538624.0,12570752.0,12602880.0,12635008.0,12667136.0,12699264.0,12731392.0,12531328.0,12563520.0,12595712.0,12627904.0,12660096.0,12692288.0,12724480.0,12756672.0,12556160.0,12588416.0,12620672.0,12652928.0,12685184.0,12717440.0,12749696.0,12781952.0,12580992.0,12613312.0,12645632.0,12677952.0,12710272.0,12742592.0,12774912.0,12807232.0,12605824.0,12638208.0,12670592.0,12702976.0,12735360.0,12767744.0,12800128.0,12832512.0,12630656.0,12663104.0,12695552.0,12728000.0,12760448.0,12792896.0,12825344.0,12857792.0,12655488.0,12688000.0,12720512.0,12753024.0,12785536.0,12818048.0,12850560.0,12883072.0,12680320.0,12712896.0,12745472.0,12778048.0,12810624.0,12843200.0,12875776.0,12908352.0,12705152.0,12737792.0,12770432.0,12803072.0,12835712.0,12868352.0,12900992.0,12933632.0,12729984.0,12762688.0,12795392.0,12828096.0,12860800.0,12893504.0,12926208.0,12958912.0,12754816.0,12787584.0,12820352.0,12853120.0,12885888.0,12918656.0,12951424.0,12984192.0,12779648.0,12812480.0,12845312.0,12878144.0,12910976.0,12943808.0,12976640.0,13009472.0,12149120.0,12180992.0,12212864.0,12244736.0,12276608.0,12308480.0,12340352.0,12372224.0,11510400.0,11541312.0,11572224.0,11603136.0,11634048.0,11664960.0,11695872.0,11726784.0,11535232.0,11566208.0,11597184.0,11628160.0,11659136.0,11690112.0,11721088.0,11752064.0,11560064.0,11591104.0,11622144.0,11653184.0,11684224.0,11715264.0,11746304.0,11777344.0,11584896.0,11616000.0,11647104.0,11678208.0,11709312.0,11740416.0,11771520.0,11802624.0,11609728.0,11640896.0,11672064.0,11703232.0,11734400.0,11765568.0,11796736.0,11827904.0,11634560.0,11665792.0,11697024.0,11728256.0,11759488.0,11790720.0,11821952.0,11853184.0,11659392.0,11690688.0,11721984.0,11753280.0,11784576.0,11815872.0,11847168.0,11878464.0,11684224.0,11715584.0,11746944.0,11778304.0,11809664.0,11841024.0,11872384.0,11903744.0,11709056.0,11740480.0,11771904.0,11803328.0,11834752.0,11866176.0,11897600.0,11929024.0,11733888.0,11765376.0,11796864.0,11828352.0,11859840.0,11891328.0,11922816.0,11954304.0,11758720.0,11790272.0,11821824.0,11853376.0,11884928.0,11916480.0,11948032.0,11979584.0,11783552.0,11815168.0,11846784.0,11878400.0,11910016.0,11941632.0,11973248.0,12004864.0,11808384.0,11840064.0,11871744.0,11903424.0,11935104.0,11966784.0,11998464.0,12030144.0,11833216.0,11864960.0,11896704.0,11928448.0,11960192.0,11991936.0,12023680.0,12055424.0,11858048.0,11889856.0,11921664.0,11953472.0,11985280.0,12017088.0,12048896.0,12080704.0,11882880.0,11914752.0,11946624.0,11978496.0,12010368.0,12042240.0,12074112.0,12105984.0,11907712.0,11939648.0,11971584.0,12003520.0,12035456.0,12067392.0,12099328.0,12131264.0,11932544.0,11964544.0,11996544.0,12028544.0,12060544.0,12092544.0,12124544.0,12156544.0,11957376.0,11989440.0,12021504.0,12053568.0,12085632.0,12117696.0,12149760.0,12181824.0,11982208.0,12014336.0,12046464.0,12078592.0,12110720.0,12142848.0,12174976.0,12207104.0,12007040.0,12039232.0,12071424.0,12103616.0,12135808.0,12168000.0,12200192.0,12232384.0,12031872.0,12064128.0,12096384.0,12128640.0,12160896.0,12193152.0,12225408.0,12257664.0,12056704.0,12089024.0,12121344.0,12153664.0,12185984.0,12218304.0,12250624.0,12282944.0,12081536.0,12113920.0,12146304.0,12178688.0,12211072.0,12243456.0,12275840.0,12308224.0,12106368.0,12138816.0,12171264.0,12203712.0,12236160.0,12268608.0,12301056.0,12333504.0,12131200.0,12163712.0,12196224.0,12228736.0,12261248.0,12293760.0,12326272.0,12358784.0,12156032.0,12188608.0,12221184.0,12253760.0,12286336.0,12318912.0,12351488.0,12384064.0,12180864.0,12213504.0,12246144.0,12278784.0,12311424.0,12344064.0,12376704.0,12409344.0,12205696.0,12238400.0,12271104.0,12303808.0,12336512.0,12369216.0,12401920.0,12434624.0,12230528.0,12263296.0,12296064.0,12328832.0,12361600.0,12394368.0,12427136.0,12459904.0,12255360.0,12288192.0,12321024.0,12353856.0,12386688.0,12419520.0,12452352.0,12485184.0,12280192.0,12313088.0,12345984.0,12378880.0,12411776.0,12444672.0,12477568.0,12510464.0,12305024.0,12337984.0,12370944.0,12403904.0,12436864.0,12469824.0,12502784.0,12535744.0,12329856.0,12362880.0,12395904.0,12428928.0,12461952.0,12494976.0,12528000.0,12561024.0,12354688.0,12387776.0,12420864.0,12453952.0,12487040.0,12520128.0,12553216.0,12586304.0,12379520.0,12412672.0,12445824.0,12478976.0,12512128.0,12545280.0,12578432.0,12611584.0,12404352.0,12437568.0,12470784.0,12504000.0,12537216.0,12570432.0,12603648.0,12636864.0,12429184.0,12462464.0,12495744.0,12529024.0,12562304.0,12595584.0,12628864.0,12662144.0,12322944.0,12355264.0,12387584.0,12419904.0,12452224.0,12484544.0,12516864.0,12549184.0,12208512.0,12239872.0,12271232.0,12302592.0,12333952.0,12365312.0,12396672.0,12428032.0,12233344.0,12264768.0,12296192.0,12327616.0,12359040.0,12390464.0,12421888.0,12453312.0,12258176.0,12289664.0,12321152.0,12352640.0,12384128.0,12415616.0,12447104.0,12478592.0,12283008.0,12314560.0,12346112.0,12377664.0,12409216.0,12440768.0,12472320.0,12503872.0,12307840.0,12339456.0,12371072.0,12402688.0,12434304.0,12465920.0,12497536.0,12529152.0,12332672.0,12364352.0,12396032.0,12427712.0,12459392.0,12491072.0,12522752.0,12554432.0,12357504.0,12389248.0,12420992.0,12452736.0,12484480.0,12516224.0,12547968.0,12579712.0,12382336.0,12414144.0,12445952.0,12477760.0,12509568.0,12541376.0,12573184.0,12604992.0,12407168.0,12439040.0,12470912.0,12502784.0,12534656.0,12566528.0,12598400.0,12630272.0,12432000.0,12463936.0,12495872.0,12527808.0,12559744.0,12591680.0,12623616.0,12655552.0,12456832.0,12488832.0,12520832.0,12552832.0,12584832.0,12616832.0,12648832.0,12680832.0,12481664.0,12513728.0,12545792.0,12577856.0,12609920.0,12641984.0,12674048.0,12706112.0,12506496.0,12538624.0,12570752.0,12602880.0,12635008.0,12667136.0,12699264.0,12731392.0,12531328.0,12563520.0,12595712.0,12627904.0,12660096.0,12692288.0,12724480.0,12756672.0,12556160.0,12588416.0,12620672.0,12652928.0,12685184.0,12717440.0,12749696.0,12781952.0,12580992.0,12613312.0,12645632.0,12677952.0,12710272.0,12742592.0,12774912.0,12807232.0,12605824.0,12638208.0,12670592.0,12702976.0,12735360.0,12767744.0,12800128.0,12832512.0,12630656.0,12663104.0,12695552.0,12728000.0,12760448.0,12792896.0,12825344.0,12857792.0,12655488.0,12688000.0,12720512.0,12753024.0,12785536.0,12818048.0,12850560.0,12883072.0,12680320.0,12712896.0,12745472.0,12778048.0,12810624.0,12843200.0,12875776.0,12908352.0,12705152.0,12737792.0,12770432.0,12803072.0,12835712.0,12868352.0,12900992.0,12933632.0,12729984.0,12762688.0,12795392.0,12828096.0,12860800.0,12893504.0,12926208.0,12958912.0,12754816.0,12787584.0,12820352.0,12853120.0,12885888.0,12918656.0,12951424.0,12984192.0,12779648.0,12812480.0,12845312.0,12878144.0,12910976.0,12943808.0,12976640.0,13009472.0,12804480.0,12837376.0,12870272.0,12903168.0,12936064.0,12968960.0,13001856.0,13034752.0,12829312.0,12862272.0,12895232.0,12928192.0,12961152.0,12994112.0,13027072.0,13060032.0,12854144.0,12887168.0,12920192.0,12953216.0,12986240.0,13019264.0,13052288.0,13085312.0,12878976.0,12912064.0,12945152.0,12978240.0,13011328.0,13044416.0,13077504.0,13110592.0,12903808.0,12936960.0,12970112.0,13003264.0,13036416.0,13069568.0,13102720.0,13135872.0,12928640.0,12961856.0,12995072.0,13028288.0,13061504.0,13094720.0,13127936.0,13161152.0,12953472.0,12986752.0,13020032.0,13053312.0,13086592.0,13119872.0,13153152.0,13186432.0,12978304.0,13011648.0,13044992.0,13078336.0,13111680.0,13145024.0,13178368.0,13211712.0,13003136.0,13036544.0,13069952.0,13103360.0,13136768.0,13170176.0,13203584.0,13236992.0,13027968.0,13061440.0,13094912.0,13128384.0,13161856.0,13195328.0,13228800.0,13262272.0,13052800.0,13086336.0,13119872.0,13153408.0,13186944.0,13220480.0,13254016.0,13287552.0,13077632.0,13111232.0,13144832.0,13178432.0,13212032.0,13245632.0,13279232.0,13312832.0,13102464.0,13136128.0,13169792.0,13203456.0,13237120.0,13270784.0,13304448.0,13338112.0,13127296.0,13161024.0,13194752.0,13228480.0,13262208.0,13295936.0,13329664.0,13363392.0,12496768.0,12529536.0,12562304.0,12595072.0,12627840.0,12660608.0,12693376.0,12726144.0,11858048.0,11889856.0,11921664.0,11953472.0,11985280.0,12017088.0,12048896.0,12080704.0,5998976.0,6014896.0,6030816.0,6046736.0,6062656.0,6078576.0,6094496.0,6110416.0,6599424.0,6615216.0,6631008.0,6646800.0,6662592.0,6678384.0,6694176.0,6709968.0,13337984.0,13369600.0,13401216.0,13432832.0,13464448.0,13496064.0,13527680.0,13559296.0,13363840.0,13395520.0,13427200.0,13458880.0,13490560.0,13522240.0,13553920.0,13585600.0,13389696.0,13421440.0,13453184.0,13484928.0,13516672.0,13548416.0,13580160.0,13611904.0,13415552.0,13447360.0,13479168.0,13510976.0,13542784.0,13574592.0,13606400.0,13638208.0,13441408.0,13473280.0,13505152.0,13537024.0,13568896.0,13600768.0,13632640.0,13664512.0,13467264.0,13499200.0,13531136.0,13563072.0,13595008.0,13626944.0,13658880.0,13690816.0,13493120.0,13525120.0,13557120.0,13589120.0,13621120.0,13653120.0,13685120.0,13717120.0,13518976.0,13551040.0,13583104.0,13615168.0,13647232.0,13679296.0,13711360.0,13743424.0,13544832.0,13576960.0,13609088.0,13641216.0,13673344.0,13705472.0,13737600.0,13769728.0,13570688.0,13602880.0,13635072.0,13667264.0,13699456.0,13731648.0,13763840.0,13796032.0,13596544.0,13628800.0,13661056.0,13693312.0,13725568.0,13757824.0,13790080.0,13822336.0,13622400.0,13654720.0,13687040.0,13719360.0,13751680.0,13784000.0,13816320.0,13848640.0,13648256.0,13680640.0,13713024.0,13745408.0,13777792.0,13810176.0,13842560.0,13874944.0,13674112.0,13706560.0,13739008.0,13771456.0,13803904.0,13836352.0,13868800.0,13901248.0,13699968.0,13732480.0,13764992.0,13797504.0,13830016.0,13862528.0,13895040.0,13927552.0,13725824.0,13758400.0,13790976.0,13823552.0,13856128.0,13888704.0,13921280.0,13953856.0,13751680.0,13784320.0,13816960.0,13849600.0,13882240.0,13914880.0,13947520.0,13980160.0,13777536.0,13810240.0,13842944.0,13875648.0,13908352.0,13941056.0,13973760.0,14006464.0,13803392.0,13836160.0,13868928.0,13901696.0,13934464.0,13967232.0,14000000.0,14032768.0,13829248.0,13862080.0,13894912.0,13927744.0,13960576.0,13993408.0,14026240.0,14059072.0,13855104.0,13888000.0,13920896.0,13953792.0,13986688.0,14019584.0,14052480.0,14085376.0,13880960.0,13913920.0,13946880.0,13979840.0,14012800.0,14045760.0,14078720.0,14111680.0,13906816.0,13939840.0,13972864.0,14005888.0,14038912.0,14071936.0,14104960.0,14137984.0,13932672.0,13965760.0,13998848.0,14031936.0,14065024.0,14098112.0,14131200.0,14164288.0,13958528.0,13991680.0,14024832.0,14057984.0,14091136.0,14124288.0,14157440.0,14190592.0,13984384.0,14017600.0,14050816.0,14084032.0,14117248.0,14150464.0,14183680.0,14216896.0,14010240.0,14043520.0,14076800.0,14110080.0,14143360.0,14176640.0,14209920.0,14243200.0,14036096.0,14069440.0,14102784.0,14136128.0,14169472.0,14202816.0,14236160.0,14269504.0,14061952.0,14095360.0,14128768.0,14162176.0,14195584.0,14228992.0,14262400.0,14295808.0,14087808.0,14121280.0,14154752.0,14188224.0,14221696.0,14255168.0,14288640.0,14322112.0,14113664.0,14147200.0,14180736.0,14214272.0,14247808.0,14281344.0,14314880.0,14348416.0,14139520.0,14173120.0,14206720.0,14240320.0,14273920.0,14307520.0,14341120.0,14374720.0,14165376.0,14199040.0,14232704.0,14266368.0,14300032.0,14333696.0,14367360.0,14401024.0,14191232.0,14224960.0,14258688.0,14292416.0,14326144.0,14359872.0,14393600.0,14427328.0,14217088.0,14250880.0,14284672.0,14318464.0,14352256.0,14386048.0,14419840.0,14453632.0,14242944.0,14276800.0,14310656.0,14344512.0,14378368.0,14412224.0,14446080.0,14479936.0,13597056.0,13629952.0,13662848.0,13695744.0,13728640.0,13761536.0,13794432.0,13827328.0,12942976.0,12974912.0,13006848.0,13038784.0,13070720.0,13102656.0,13134592.0,13166528.0,12297088.0,12328064.0,12359040.0,12390016.0,12420992.0,12451968.0,12482944.0,12513920.0,11643008.0,11673024.0,11703040.0,11733056.0,11763072.0,11793088.0,11823104.0,11853120.0,11668864.0,11698944.0,11729024.0,11759104.0,11789184.0,11819264.0,11849344.0,11879424.0,11694720.0,11724864.0,11755008.0,11785152.0,11815296.0,11845440.0,11875584.0,11905728.0,11720576.0,11750784.0,11780992.0,11811200.0,11841408.0,11871616.0,11901824.0,11932032.0,11746432.0,11776704.0,11806976.0,11837248.0,11867520.0,11897792.0,11928064.0,11958336.0,11772288.0,11802624.0,11832960.0,11863296.0,11893632.0,11923968.0,11954304.0,11984640.0,11798144.0,11828544.0,11858944.0,11889344.0,11919744.0,11950144.0,11980544.0,12010944.0,11824000.0,11854464.0,11884928.0,11915392.0,11945856.0,11976320.0,12006784.0,12037248.0,11849856.0,11880384.0,11910912.0,11941440.0,11971968.0,12002496.0,12033024.0,12063552.0,11875712.0,11906304.0,11936896.0,11967488.0,11998080.0,12028672.0,12059264.0,12089856.0,11901568.0,11932224.0,11962880.0,11993536.0,12024192.0,12054848.0,12085504.0,12116160.0,11927424.0,11958144.0,11988864.0,12019584.0,12050304.0,12081024.0,12111744.0,12142464.0,11953280.0,11984064.0,12014848.0,12045632.0,12076416.0,12107200.0,12137984.0,12168768.0,11979136.0,12009984.0,12040832.0,12071680.0,12102528.0,12133376.0,12164224.0,12195072.0,12004992.0,12035904.0,12066816.0,12097728.0,12128640.0,12159552.0,12190464.0,12221376.0,12030848.0,12061824.0,12092800.0,12123776.0,12154752.0,12185728.0,12216704.0,12247680.0,12056704.0,12087744.0,12118784.0,12149824.0,12180864.0,12211904.0,12242944.0,12273984.0,12082560.0,12113664.0,12144768.0,12175872.0,12206976.0,12238080.0,12269184.0,12300288.0,12108416.0,12139584.0,12170752.0,12201920.0,12233088.0,12264256.0,12295424.0,12326592.0,12134272.0,12165504.0,12196736.0,12227968.0,12259200.0,12290432.0,12321664.0,12352896.0,12160128.0,12191424.0,12222720.0,12254016.0,12285312.0,12316608.0,12347904.0,12379200.0,12185984.0,12217344.0,12248704.0,12280064.0,12311424.0,12342784.0,12374144.0,12405504.0,12211840.0,12243264.0,12274688.0,12306112.0,12337536.0,12368960.0,12400384.0,12431808.0,12237696.0,12269184.0,12300672.0,12332160.0,12363648.0,12395136.0,12426624.0,12458112.0,12263552.0,12295104.0,12326656.0,12358208.0,12389760.0,12421312.0,12452864.0,12484416.0,12289408.0,12321024.0,12352640.0,12384256.0,12415872.0,12447488.0,12479104.0,12510720.0,12315264.0,12346944.0,12378624.0,12410304.0,12441984.0,12473664.0,12505344.0,12537024.0,12341120.0,12372864.0,12404608.0,12436352.0,12468096.0,12499840.0,12531584.0,12563328.0,12366976.0,12398784.0,12430592.0,12462400.0,12494208.0,12526016.0,12557824.0,12589632.0,12392832.0,12424704.0,12456576.0,12488448.0,12520320.0,12552192.0,12584064.0,12615936.0,12418688.0,12450624.0,12482560.0,12514496.0,12546432.0,12578368.0,12610304.0,12642240.0,12444544.0,12476544.0,12508544.0,12540544.0,12572544.0,12604544.0,12636544.0,12668544.0,12470400.0,12502464.0,12534528.0,12566592.0,12598656.0,12630720.0,12662784.0,12694848.0,12496256.0,12528384.0,12560512.0,12592640.0,12624768.0,12656896.0,12689024.0,12721152.0,12522112.0,12554304.0,12586496.0,12618688.0,12650880.0,12683072.0,12715264.0,12747456.0,12547968.0,12580224.0,12612480.0,12644736.0,12676992.0,12709248.0,12741504.0,12773760.0,12573824.0,12606144.0,12638464.0,12670784.0,12703104.0,12735424.0,12767744.0,12800064.0,12599680.0,12632064.0,12664448.0,12696832.0,12729216.0,12761600.0,12793984.0,12826368.0,12478080.0,12509504.0,12540928.0,12572352.0,12603776.0,12635200.0,12666624.0,12698048.0,12348288.0,12378752.0,12409216.0,12439680.0,12470144.0,12500608.0,12531072.0,12561536.0,12374144.0,12404672.0,12435200.0,12465728.0,12496256.0,12526784.0,12557312.0,12587840.0,12400000.0,12430592.0,12461184.0,12491776.0,12522368.0,12552960.0,12583552.0,12614144.0,12425856.0,12456512.0,12487168.0,12517824.0,12548480.0,12579136.0,12609792.0,12640448.0,12451712.0,12482432.0,12513152.0,12543872.0,12574592.0,12605312.0,12636032.0,12666752.0,12477568.0,12508352.0,12539136.0,12569920.0,12600704.0,12631488.0,12662272.0,12693056.0,12503424.0,12534272.0,12565120.0,12595968.0,12626816.0,12657664.0,12688512.0,12719360.0,12529280.0,12560192.0,12591104.0,12622016.0,12652928.0,12683840.0,12714752.0,12745664.0,12555136.0,12586112.0,12617088.0,12648064.0,12679040.0,12710016.0,12740992.0,12771968.0,12580992.0,12612032.0,12643072.0,12674112.0,12705152.0,12736192.0,12767232.0,12798272.0,12606848.0,12637952.0,12669056.0,12700160.0,12731264.0,12762368.0,12793472.0,12824576.0,12632704.0,12663872.0,12695040.0,12726208.0,12757376.0,12788544.0,12819712.0,12850880.0,12658560.0,12689792.0,12721024.0,12752256.0,12783488.0,12814720.0,12845952.0,12877184.0,12684416.0,12715712.0,12747008.0,12778304.0,12809600.0,12840896.0,12872192.0,12903488.0,12710272.0,12741632.0,12772992.0,12804352.0,12835712.0,12867072.0,12898432.0,12929792.0,12736128.0,12767552.0,12798976.0,12830400.0,12861824.0,12893248.0,12924672.0,12956096.0,12761984.0,12793472.0,12824960.0,12856448.0,12887936.0,12919424.0,12950912.0,12982400.0,12787840.0,12819392.0,12850944.0,12882496.0,12914048.0,12945600.0,12977152.0,13008704.0,12813696.0,12845312.0,12876928.0,12908544.0,12940160.0,12971776.0,13003392.0,13035008.0,12839552.0,12871232.0,12902912.0,12934592.0,12966272.0,12997952.0,13029632.0,13061312.0,12865408.0,12897152.0,12928896.0,12960640.0,12992384.0,13024128.0,13055872.0,13087616.0,12891264.0,12923072.0,12954880.0,12986688.0,13018496.0,13050304.0,13082112.0,13113920.0,12917120.0,12948992.0,12980864.0,13012736.0,13044608.0,13076480.0,13108352.0,13140224.0,12942976.0,12974912.0,13006848.0,13038784.0,13070720.0,13102656.0,13134592.0,13166528.0,12968832.0,13000832.0,13032832.0,13064832.0,13096832.0,13128832.0,13160832.0,13192832.0,12994688.0,13026752.0,13058816.0,13090880.0,13122944.0,13155008.0,13187072.0,13219136.0,13020544.0,13052672.0,13084800.0,13116928.0,13149056.0,13181184.0,13213312.0,13245440.0,13046400.0,13078592.0,13110784.0,13142976.0,13175168.0,13207360.0,13239552.0,13271744.0,13072256.0,13104512.0,13136768.0,13169024.0,13201280.0,13233536.0,13265792.0,13298048.0,13098112.0,13130432.0,13162752.0,13195072.0,13227392.0,13259712.0,13292032.0,13324352.0,13123968.0,13156352.0,13188736.0,13221120.0,13253504.0,13285888.0,13318272.0,13350656.0,13149824.0,13182272.0,13214720.0,13247168.0,13279616.0,13312064.0,13344512.0,13376960.0,13175680.0,13208192.0,13240704.0,13273216.0,13305728.0,13338240.0,13370752.0,13403264.0,13201536.0,13234112.0,13266688.0,13299264.0,13331840.0,13364416.0,13396992.0,13429568.0,13227392.0,13260032.0,13292672.0,13325312.0,13357952.0,13390592.0,13423232.0,13455872.0,13253248.0,13285952.0,13318656.0,13351360.0,13384064.0,13416768.0,13449472.0,13482176.0,13279104.0,13311872.0,13344640.0,13377408.0,13410176.0,13442944.0,13475712.0,13508480.0,13304960.0,13337792.0,13370624.0,13403456.0,13436288.0,13469120.0,13501952.0,13534784.0,12659072.0,12690944.0,12722816.0,12754688.0,12786560.0,12818432.0,12850304.0,12882176.0,12004992.0,12035904.0,12066816.0,12097728.0,12128640.0,12159552.0,12190464.0,12221376.0,12030848.0,12061824.0,12092800.0,12123776.0,12154752.0,12185728.0,12216704.0,12247680.0,12056704.0,12087744.0,12118784.0,12149824.0,12180864.0,12211904.0,12242944.0,12273984.0,12082560.0,12113664.0,12144768.0,12175872.0,12206976.0,12238080.0,12269184.0,12300288.0,12108416.0,12139584.0,12170752.0,12201920.0,12233088.0,12264256.0,12295424.0,12326592.0,12134272.0,12165504.0,12196736.0,12227968.0,12259200.0,12290432.0,12321664.0,12352896.0,12160128.0,12191424.0,12222720.0,12254016.0,12285312.0,12316608.0,12347904.0,12379200.0,12185984.0,12217344.0,12248704.0,12280064.0,12311424.0,12342784.0,12374144.0,12405504.0,12211840.0,12243264.0,12274688.0,12306112.0,12337536.0,12368960.0,12400384.0,12431808.0,12237696.0,12269184.0,12300672.0,12332160.0,12363648.0,12395136.0,12426624.0,12458112.0,12263552.0,12295104.0,12326656.0,12358208.0,12389760.0,12421312.0,12452864.0,12484416.0,12289408.0,12321024.0,12352640.0,12384256.0,12415872.0,12447488.0,12479104.0,12510720.0,12315264.0,12346944.0,12378624.0,12410304.0,12441984.0,12473664.0,12505344.0,12537024.0,12341120.0,12372864.0,12404608.0,12436352.0,12468096.0,12499840.0,12531584.0,12563328.0,12366976.0,12398784.0,12430592.0,12462400.0,12494208.0,12526016.0,12557824.0,12589632.0,12392832.0,12424704.0,12456576.0,12488448.0,12520320.0,12552192.0,12584064.0,12615936.0,12418688.0,12450624.0,12482560.0,12514496.0,12546432.0,12578368.0,12610304.0,12642240.0,12444544.0,12476544.0,12508544.0,12540544.0,12572544.0,12604544.0,12636544.0,12668544.0,12470400.0,12502464.0,12534528.0,12566592.0,12598656.0,12630720.0,12662784.0,12694848.0,12496256.0,12528384.0,12560512.0,12592640.0,12624768.0,12656896.0,12689024.0,12721152.0,12522112.0,12554304.0,12586496.0,12618688.0,12650880.0,12683072.0,12715264.0,12747456.0,12547968.0,12580224.0,12612480.0,12644736.0,12676992.0,12709248.0,12741504.0,12773760.0,12573824.0,12606144.0,12638464.0,12670784.0,12703104.0,12735424.0,12767744.0,12800064.0,12599680.0,12632064.0,12664448.0,12696832.0,12729216.0,12761600.0,12793984.0,12826368.0,12625536.0,12657984.0,12690432.0,12722880.0,12755328.0,12787776.0,12820224.0,12852672.0,12651392.0,12683904.0,12716416.0,12748928.0,12781440.0,12813952.0,12846464.0,12878976.0,12677248.0,12709824.0,12742400.0,12774976.0,12807552.0,12840128.0,12872704.0,12905280.0,12703104.0,12735744.0,12768384.0,12801024.0,12833664.0,12866304.0,12898944.0,12931584.0,12728960.0,12761664.0,12794368.0,12827072.0,12859776.0,12892480.0,12925184.0,12957888.0,12754816.0,12787584.0,12820352.0,12853120.0,12885888.0,12918656.0,12951424.0,12984192.0,12780672.0,12813504.0,12846336.0,12879168.0,12912000.0,12944832.0,12977664.0,13010496.0,12806528.0,12839424.0,12872320.0,12905216.0,12938112.0,12971008.0,13003904.0,13036800.0,12832384.0,12865344.0,12898304.0,12931264.0,12964224.0,12997184.0,13030144.0,13063104.0,12858240.0,12891264.0,12924288.0,12957312.0,12990336.0,13023360.0,13056384.0,13089408.0,12884096.0,12917184.0,12950272.0,12983360.0,13016448.0,13049536.0,13082624.0,13115712.0,12909952.0,12943104.0,12976256.0,13009408.0,13042560.0,13075712.0,13108864.0,13142016.0,12935808.0,12969024.0,13002240.0,13035456.0,13068672.0,13101888.0,13135104.0,13168320.0,12961664.0,12994944.0,13028224.0,13061504.0,13094784.0,13128064.0,13161344.0,13194624.0,12840064.0,12872384.0,12904704.0,12937024.0,12969344.0,13001664.0,13033984.0,13066304.0,12710272.0,12741632.0,12772992.0,12804352.0,12835712.0,12867072.0,12898432.0,12929792.0,12736128.0,12767552.0,12798976.0,12830400.0,12861824.0,12893248.0,12924672.0,12956096.0,12761984.0,12793472.0,12824960.0,12856448.0,12887936.0,12919424.0,12950912.0,12982400.0,12787840.0,12819392.0,12850944.0,12882496.0,12914048.0,12945600.0,12977152.0,13008704.0,12813696.0,12845312.0,12876928.0,12908544.0,12940160.0,12971776.0,13003392.0,13035008.0,12839552.0,12871232.0,12902912.0,12934592.0,12966272.0,12997952.0,13029632.0,13061312.0,12865408.0,12897152.0,12928896.0,12960640.0,12992384.0,13024128.0,13055872.0,13087616.0,12891264.0,12923072.0,12954880.0,12986688.0,13018496.0,13050304.0,13082112.0,13113920.0,12917120.0,12948992.0,12980864.0,13012736.0,13044608.0,13076480.0,13108352.0,13140224.0,12942976.0,12974912.0,13006848.0,13038784.0,13070720.0,13102656.0,13134592.0,13166528.0,12968832.0,13000832.0,13032832.0,13064832.0,13096832.0,13128832.0,13160832.0,13192832.0,12994688.0,13026752.0,13058816.0,13090880.0,13122944.0,13155008.0,13187072.0,13219136.0,13020544.0,13052672.0,13084800.0,13116928.0,13149056.0,13181184.0,13213312.0,13245440.0,13046400.0,13078592.0,13110784.0,13142976.0,13175168.0,13207360.0,13239552.0,13271744.0,13072256.0,13104512.0,13136768.0,13169024.0,13201280.0,13233536.0,13265792.0,13298048.0,13098112.0,13130432.0,13162752.0,13195072.0,13227392.0,13259712.0,13292032.0,13324352.0,13123968.0,13156352.0,13188736.0,13221120.0,13253504.0,13285888.0,13318272.0,13350656.0,13149824.0,13182272.0,13214720.0,13247168.0,13279616.0,13312064.0,13344512.0,13376960.0,13175680.0,13208192.0,13240704.0,13273216.0,13305728.0,13338240.0,13370752.0,13403264.0,13201536.0,13234112.0,13266688.0,13299264.0,13331840.0,13364416.0,13396992.0,13429568.0,13227392.0,13260032.0,13292672.0,13325312.0,13357952.0,13390592.0,13423232.0,13455872.0,13253248.0,13285952.0,13318656.0,13351360.0,13384064.0,13416768.0,13449472.0,13482176.0,13279104.0,13311872.0,13344640.0,13377408.0,13410176.0,13442944.0,13475712.0,13508480.0,13304960.0,13337792.0,13370624.0,13403456.0,13436288.0,13469120.0,13501952.0,13534784.0,13330816.0,13363712.0,13396608.0,13429504.0,13462400.0,13495296.0,13528192.0,13561088.0,13356672.0,13389632.0,13422592.0,13455552.0,13488512.0,13521472.0,13554432.0,13587392.0,13382528.0,13415552.0,13448576.0,13481600.0,13514624.0,13547648.0,13580672.0,13613696.0,13408384.0,13441472.0,13474560.0,13507648.0,13540736.0,13573824.0,13606912.0,13640000.0,13434240.0,13467392.0,13500544.0,13533696.0,13566848.0,13600000.0,13633152.0,13666304.0,13460096.0,13493312.0,13526528.0,13559744.0,13592960.0,13626176.0,13659392.0,13692608.0,13485952.0,13519232.0,13552512.0,13585792.0,13619072.0,13652352.0,13685632.0,13718912.0,13511808.0,13545152.0,13578496.0,13611840.0,13645184.0,13678528.0,13711872.0,13745216.0,13537664.0,13571072.0,13604480.0,13637888.0,13671296.0,13704704.0,13738112.0,13771520.0,13563520.0,13596992.0,13630464.0,13663936.0,13697408.0,13730880.0,13764352.0,13797824.0,13589376.0,13622912.0,13656448.0,13689984.0,13723520.0,13757056.0,13790592.0,13824128.0,13615232.0,13648832.0,13682432.0,13716032.0,13749632.0,13783232.0,13816832.0,13850432.0,13641088.0,13674752.0,13708416.0,13742080.0,13775744.0,13809408.0,13843072.0,13876736.0,13666944.0,13700672.0,13734400.0,13768128.0,13801856.0,13835584.0,13869312.0,13903040.0,13021056.0,13053824.0,13086592.0,13119360.0,13152128.0,13184896.0,13217664.0,13250432.0,12366976.0,12398784.0,12430592.0,12462400.0,12494208.0,12526016.0,12557824.0,12589632.0,6253696.0,6269616.0,6285536.0,6301456.0,6317376.0,6333296.0,6349216.0,6365136.0,6852096.0,6867888.0,6883680.0,6899472.0,6915264.0,6931056.0,6946848.0,6962640.0,13843840.0,13875456.0,13907072.0,13938688.0,13970304.0,14001920.0,14033536.0,14065152.0,13870720.0,13902400.0,13934080.0,13965760.0,13997440.0,14029120.0,14060800.0,14092480.0,13897600.0,13929344.0,13961088.0,13992832.0,14024576.0,14056320.0,14088064.0,14119808.0,13924480.0,13956288.0,13988096.0,14019904.0,14051712.0,14083520.0,14115328.0,14147136.0,13951360.0,13983232.0,14015104.0,14046976.0,14078848.0,14110720.0,14142592.0,14174464.0,13978240.0,14010176.0,14042112.0,14074048.0,14105984.0,14137920.0,14169856.0,14201792.0,14005120.0,14037120.0,14069120.0,14101120.0,14133120.0,14165120.0,14197120.0,14229120.0,14032000.0,14064064.0,14096128.0,14128192.0,14160256.0,14192320.0,14224384.0,14256448.0,14058880.0,14091008.0,14123136.0,14155264.0,14187392.0,14219520.0,14251648.0,14283776.0,14085760.0,14117952.0,14150144.0,14182336.0,14214528.0,14246720.0,14278912.0,14311104.0,14112640.0,14144896.0,14177152.0,14209408.0,14241664.0,14273920.0,14306176.0,14338432.0,14139520.0,14171840.0,14204160.0,14236480.0,14268800.0,14301120.0,14333440.0,14365760.0,14166400.0,14198784.0,14231168.0,14263552.0,14295936.0,14328320.0,14360704.0,14393088.0,14193280.0,14225728.0,14258176.0,14290624.0,14323072.0,14355520.0,14387968.0,14420416.0,14220160.0,14252672.0,14285184.0,14317696.0,14350208.0,14382720.0,14415232.0,14447744.0,14247040.0,14279616.0,14312192.0,14344768.0,14377344.0,14409920.0,14442496.0,14475072.0,14273920.0,14306560.0,14339200.0,14371840.0,14404480.0,14437120.0,14469760.0,14502400.0,14300800.0,14333504.0,14366208.0,14398912.0,14431616.0,14464320.0,14497024.0,14529728.0,14327680.0,14360448.0,14393216.0,14425984.0,14458752.0,14491520.0,14524288.0,14557056.0,14354560.0,14387392.0,14420224.0,14453056.0,14485888.0,14518720.0,14551552.0,14584384.0,14381440.0,14414336.0,14447232.0,14480128.0,14513024.0,14545920.0,14578816.0,14611712.0,14408320.0,14441280.0,14474240.0,14507200.0,14540160.0,14573120.0,14606080.0,14639040.0,14435200.0,14468224.0,14501248.0,14534272.0,14567296.0,14600320.0,14633344.0,14666368.0,14462080.0,14495168.0,14528256.0,14561344.0,14594432.0,14627520.0,14660608.0,14693696.0,14488960.0,14522112.0,14555264.0,14588416.0,14621568.0,14654720.0,14687872.0,14721024.0,14515840.0,14549056.0,14582272.0,14615488.0,14648704.0,14681920.0,14715136.0,14748352.0,14542720.0,14576000.0,14609280.0,14642560.0,14675840.0,14709120.0,14742400.0,14775680.0,14569600.0,14602944.0,14636288.0,14669632.0,14702976.0,14736320.0,14769664.0,14803008.0,14596480.0,14629888.0,14663296.0,14696704.0,14730112.0,14763520.0,14796928.0,14830336.0,14623360.0,14656832.0,14690304.0,14723776.0,14757248.0,14790720.0,14824192.0,14857664.0,14650240.0,14683776.0,14717312.0,14750848.0,14784384.0,14817920.0,14851456.0,14884992.0,14677120.0,14710720.0,14744320.0,14777920.0,14811520.0,14845120.0,14878720.0,14912320.0,14704000.0,14737664.0,14771328.0,14804992.0,14838656.0,14872320.0,14905984.0,14939648.0,14730880.0,14764608.0,14798336.0,14832064.0,14865792.0,14899520.0,14933248.0,14966976.0,14757760.0,14791552.0,14825344.0,14859136.0,14892928.0,14926720.0,14960512.0,14994304.0,14784640.0,14818496.0,14852352.0,14886208.0,14920064.0,14953920.0,14987776.0,15021632.0,14123392.0,14156288.0,14189184.0,14222080.0,14254976.0,14287872.0,14320768.0,14353664.0,13453952.0,13485888.0,13517824.0,13549760.0,13581696.0,13613632.0,13645568.0,13677504.0,12792704.0,12823680.0,12854656.0,12885632.0,12916608.0,12947584.0,12978560.0,13009536.0,12123264.0,12153280.0,12183296.0,12213312.0,12243328.0,12273344.0,12303360.0,12333376.0,12150144.0,12180224.0,12210304.0,12240384.0,12270464.0,12300544.0,12330624.0,12360704.0,12177024.0,12207168.0,12237312.0,12267456.0,12297600.0,12327744.0,12357888.0,12388032.0,12203904.0,12234112.0,12264320.0,12294528.0,12324736.0,12354944.0,12385152.0,12415360.0,12230784.0,12261056.0,12291328.0,12321600.0,12351872.0,12382144.0,12412416.0,12442688.0,12257664.0,12288000.0,12318336.0,12348672.0,12379008.0,12409344.0,12439680.0,12470016.0,12284544.0,12314944.0,12345344.0,12375744.0,12406144.0,12436544.0,12466944.0,12497344.0,12311424.0,12341888.0,12372352.0,12402816.0,12433280.0,12463744.0,12494208.0,12524672.0,12338304.0,12368832.0,12399360.0,12429888.0,12460416.0,12490944.0,12521472.0,12552000.0,12365184.0,12395776.0,12426368.0,12456960.0,12487552.0,12518144.0,12548736.0,12579328.0,12392064.0,12422720.0,12453376.0,12484032.0,12514688.0,12545344.0,12576000.0,12606656.0,12418944.0,12449664.0,12480384.0,12511104.0,12541824.0,12572544.0,12603264.0,12633984.0,12445824.0,12476608.0,12507392.0,12538176.0,12568960.0,12599744.0,12630528.0,12661312.0,12472704.0,12503552.0,12534400.0,12565248.0,12596096.0,12626944.0,12657792.0,12688640.0,12499584.0,12530496.0,12561408.0,12592320.0,12623232.0,12654144.0,12685056.0,12715968.0,12526464.0,12557440.0,12588416.0,12619392.0,12650368.0,12681344.0,12712320.0,12743296.0,12553344.0,12584384.0,12615424.0,12646464.0,12677504.0,12708544.0,12739584.0,12770624.0,12580224.0,12611328.0,12642432.0,12673536.0,12704640.0,12735744.0,12766848.0,12797952.0,12607104.0,12638272.0,12669440.0,12700608.0,12731776.0,12762944.0,12794112.0,12825280.0,12633984.0,12665216.0,12696448.0,12727680.0,12758912.0,12790144.0,12821376.0,12852608.0,12660864.0,12692160.0,12723456.0,12754752.0,12786048.0,12817344.0,12848640.0,12879936.0,12687744.0,12719104.0,12750464.0,12781824.0,12813184.0,12844544.0,12875904.0,12907264.0,12714624.0,12746048.0,12777472.0,12808896.0,12840320.0,12871744.0,12903168.0,12934592.0,12741504.0,12772992.0,12804480.0,12835968.0,12867456.0,12898944.0,12930432.0,12961920.0,12768384.0,12799936.0,12831488.0,12863040.0,12894592.0,12926144.0,12957696.0,12989248.0,12795264.0,12826880.0,12858496.0,12890112.0,12921728.0,12953344.0,12984960.0,13016576.0,12822144.0,12853824.0,12885504.0,12917184.0,12948864.0,12980544.0,13012224.0,13043904.0,12849024.0,12880768.0,12912512.0,12944256.0,12976000.0,13007744.0,13039488.0,13071232.0,12875904.0,12907712.0,12939520.0,12971328.0,13003136.0,13034944.0,13066752.0,13098560.0,12902784.0,12934656.0,12966528.0,12998400.0,13030272.0,13062144.0,13094016.0,13125888.0,12929664.0,12961600.0,12993536.0,13025472.0,13057408.0,13089344.0,13121280.0,13153216.0,12956544.0,12988544.0,13020544.0,13052544.0,13084544.0,13116544.0,13148544.0,13180544.0,12983424.0,13015488.0,13047552.0,13079616.0,13111680.0,13143744.0,13175808.0,13207872.0,13010304.0,13042432.0,13074560.0,13106688.0,13138816.0,13170944.0,13203072.0,13235200.0,13037184.0,13069376.0,13101568.0,13133760.0,13165952.0,13198144.0,13230336.0,13262528.0,13064064.0,13096320.0,13128576.0,13160832.0,13193088.0,13225344.0,13257600.0,13289856.0,13090944.0,13123264.0,13155584.0,13187904.0,13220224.0,13252544.0,13284864.0,13317184.0,13117824.0,13150208.0,13182592.0,13214976.0,13247360.0,13279744.0,13312128.0,13344512.0,12980864.0,13012288.0,13043712.0,13075136.0,13106560.0,13137984.0,13169408.0,13200832.0,12835712.0,12866176.0,12896640.0,12927104.0,12957568.0,12988032.0,13018496.0,13048960.0,12862592.0,12893120.0,12923648.0,12954176.0,12984704.0,13015232.0,13045760.0,13076288.0,12889472.0,12920064.0,12950656.0,12981248.0,13011840.0,13042432.0,13073024.0,13103616.0,12916352.0,12947008.0,12977664.0,13008320.0,13038976.0,13069632.0,13100288.0,13130944.0,12943232.0,12973952.0,13004672.0,13035392.0,13066112.0,13096832.0,13127552.0,13158272.0,12970112.0,13000896.0,13031680.0,13062464.0,13093248.0,13124032.0,13154816.0,13185600.0,12996992.0,13027840.0,13058688.0,13089536.0,13120384.0,13151232.0,13182080.0,13212928.0,13023872.0,13054784.0,13085696.0,13116608.0,13147520.0,13178432.0,13209344.0,13240256.0,13050752.0,13081728.0,13112704.0,13143680.0,13174656.0,13205632.0,13236608.0,13267584.0,13077632.0,13108672.0,13139712.0,13170752.0,13201792.0,13232832.0,13263872.0,13294912.0,13104512.0,13135616.0,13166720.0,13197824.0,13228928.0,13260032.0,13291136.0,13322240.0,13131392.0,13162560.0,13193728.0,13224896.0,13256064.0,13287232.0,13318400.0,13349568.0,13158272.0,13189504.0,13220736.0,13251968.0,13283200.0,13314432.0,13345664.0,13376896.0,13185152.0,13216448.0,13247744.0,13279040.0,13310336.0,13341632.0,13372928.0,13404224.0,13212032.0,13243392.0,13274752.0,13306112.0,13337472.0,13368832.0,13400192.0,13431552.0,13238912.0,13270336.0,13301760.0,13333184.0,13364608.0,13396032.0,13427456.0,13458880.0,13265792.0,13297280.0,13328768.0,13360256.0,13391744.0,13423232.0,13454720.0,13486208.0,13292672.0,13324224.0,13355776.0,13387328.0,13418880.0,13450432.0,13481984.0,13513536.0,13319552.0,13351168.0,13382784.0,13414400.0,13446016.0,13477632.0,13509248.0,13540864.0,13346432.0,13378112.0,13409792.0,13441472.0,13473152.0,13504832.0,13536512.0,13568192.0,13373312.0,13405056.0,13436800.0,13468544.0,13500288.0,13532032.0,13563776.0,13595520.0,13400192.0,13432000.0,13463808.0,13495616.0,13527424.0,13559232.0,13591040.0,13622848.0,13427072.0,13458944.0,13490816.0,13522688.0,13554560.0,13586432.0,13618304.0,13650176.0,13453952.0,13485888.0,13517824.0,13549760.0,13581696.0,13613632.0,13645568.0,13677504.0,13480832.0,13512832.0,13544832.0,13576832.0,13608832.0,13640832.0,13672832.0,13704832.0,13507712.0,13539776.0,13571840.0,13603904.0,13635968.0,13668032.0,13700096.0,13732160.0,13534592.0,13566720.0,13598848.0,13630976.0,13663104.0,13695232.0,13727360.0,13759488.0,13561472.0,13593664.0,13625856.0,13658048.0,13690240.0,13722432.0,13754624.0,13786816.0,13588352.0,13620608.0,13652864.0,13685120.0,13717376.0,13749632.0,13781888.0,13814144.0,13615232.0,13647552.0,13679872.0,13712192.0,13744512.0,13776832.0,13809152.0,13841472.0,13642112.0,13674496.0,13706880.0,13739264.0,13771648.0,13804032.0,13836416.0,13868800.0,13668992.0,13701440.0,13733888.0,13766336.0,13798784.0,13831232.0,13863680.0,13896128.0,13695872.0,13728384.0,13760896.0,13793408.0,13825920.0,13858432.0,13890944.0,13923456.0,13722752.0,13755328.0,13787904.0,13820480.0,13853056.0,13885632.0,13918208.0,13950784.0,13749632.0,13782272.0,13814912.0,13847552.0,13880192.0,13912832.0,13945472.0,13978112.0,13776512.0,13809216.0,13841920.0,13874624.0,13907328.0,13940032.0,13972736.0,14005440.0,13803392.0,13836160.0,13868928.0,13901696.0,13934464.0,13967232.0,14000000.0,14032768.0,13830272.0,13863104.0,13895936.0,13928768.0,13961600.0,13994432.0,14027264.0,14060096.0,13169024.0,13200896.0,13232768.0,13264640.0,13296512.0,13328384.0,13360256.0,13392128.0,12499584.0,12530496.0,12561408.0,12592320.0,12623232.0,12654144.0,12685056.0,12715968.0,12526464.0,12557440.0,12588416.0,12619392.0,12650368.0,12681344.0,12712320.0,12743296.0,12553344.0,12584384.0,12615424.0,12646464.0,12677504.0,12708544.0,12739584.0,12770624.0,12580224.0,12611328.0,12642432.0,12673536.0,12704640.0,12735744.0,12766848.0,12797952.0,12607104.0,12638272.0,12669440.0,12700608.0,12731776.0,12762944.0,12794112.0,12825280.0,12633984.0,12665216.0,12696448.0,12727680.0,12758912.0,12790144.0,12821376.0,12852608.0,12660864.0,12692160.0,12723456.0,12754752.0,12786048.0,12817344.0,12848640.0,12879936.0,12687744.0,12719104.0,12750464.0,12781824.0,12813184.0,12844544.0,12875904.0,12907264.0,12714624.0,12746048.0,12777472.0,12808896.0,12840320.0,12871744.0,12903168.0,12934592.0,12741504.0,12772992.0,12804480.0,12835968.0,12867456.0,12898944.0,12930432.0,12961920.0,12768384.0,12799936.0,12831488.0,12863040.0,12894592.0,12926144.0,12957696.0,12989248.0,12795264.0,12826880.0,12858496.0,12890112.0,12921728.0,12953344.0,12984960.0,13016576.0,12822144.0,12853824.0,12885504.0,12917184.0,12948864.0,12980544.0,13012224.0,13043904.0,12849024.0,12880768.0,12912512.0,12944256.0,12976000.0,13007744.0,13039488.0,13071232.0,12875904.0,12907712.0,12939520.0,12971328.0,13003136.0,13034944.0,13066752.0,13098560.0,12902784.0,12934656.0,12966528.0,12998400.0,13030272.0,13062144.0,13094016.0,13125888.0,12929664.0,12961600.0,12993536.0,13025472.0,13057408.0,13089344.0,13121280.0,13153216.0,12956544.0,12988544.0,13020544.0,13052544.0,13084544.0,13116544.0,13148544.0,13180544.0,12983424.0,13015488.0,13047552.0,13079616.0,13111680.0,13143744.0,13175808.0,13207872.0,13010304.0,13042432.0,13074560.0,13106688.0,13138816.0,13170944.0,13203072.0,13235200.0,13037184.0,13069376.0,13101568.0,13133760.0,13165952.0,13198144.0,13230336.0,13262528.0,13064064.0,13096320.0,13128576.0,13160832.0,13193088.0,13225344.0,13257600.0,13289856.0,13090944.0,13123264.0,13155584.0,13187904.0,13220224.0,13252544.0,13284864.0,13317184.0,13117824.0,13150208.0,13182592.0,13214976.0,13247360.0,13279744.0,13312128.0,13344512.0,13144704.0,13177152.0,13209600.0,13242048.0,13274496.0,13306944.0,13339392.0,13371840.0,13171584.0,13204096.0,13236608.0,13269120.0,13301632.0,13334144.0,13366656.0,13399168.0,13198464.0,13231040.0,13263616.0,13296192.0,13328768.0,13361344.0,13393920.0,13426496.0,13225344.0,13257984.0,13290624.0,13323264.0,13355904.0,13388544.0,13421184.0,13453824.0,13252224.0,13284928.0,13317632.0,13350336.0,13383040.0,13415744.0,13448448.0,13481152.0,13279104.0,13311872.0,13344640.0,13377408.0,13410176.0,13442944.0,13475712.0,13508480.0,13305984.0,13338816.0,13371648.0,13404480.0,13437312.0,13470144.0,13502976.0,13535808.0,13332864.0,13365760.0,13398656.0,13431552.0,13464448.0,13497344.0,13530240.0,13563136.0,13359744.0,13392704.0,13425664.0,13458624.0,13491584.0,13524544.0,13557504.0,13590464.0,13386624.0,13419648.0,13452672.0,13485696.0,13518720.0,13551744.0,13584768.0,13617792.0,13413504.0,13446592.0,13479680.0,13512768.0,13545856.0,13578944.0,13612032.0,13645120.0,13440384.0,13473536.0,13506688.0,13539840.0,13572992.0,13606144.0,13639296.0,13672448.0,13467264.0,13500480.0,13533696.0,13566912.0,13600128.0,13633344.0,13666560.0,13699776.0,13494144.0,13527424.0,13560704.0,13593984.0,13627264.0,13660544.0,13693824.0,13727104.0,13357184.0,13389504.0,13421824.0,13454144.0,13486464.0,13518784.0,13551104.0,13583424.0,13212032.0,13243392.0,13274752.0,13306112.0,13337472.0,13368832.0,13400192.0,13431552.0,13238912.0,13270336.0,13301760.0,13333184.0,13364608.0,13396032.0,13427456.0,13458880.0,13265792.0,13297280.0,13328768.0,13360256.0,13391744.0,13423232.0,13454720.0,13486208.0,13292672.0,13324224.0,13355776.0,13387328.0,13418880.0,13450432.0,13481984.0,13513536.0,13319552.0,13351168.0,13382784.0,13414400.0,13446016.0,13477632.0,13509248.0,13540864.0,13346432.0,13378112.0,13409792.0,13441472.0,13473152.0,13504832.0,13536512.0,13568192.0,13373312.0,13405056.0,13436800.0,13468544.0,13500288.0,13532032.0,13563776.0,13595520.0,13400192.0,13432000.0,13463808.0,13495616.0,13527424.0,13559232.0,13591040.0,13622848.0,13427072.0,13458944.0,13490816.0,13522688.0,13554560.0,13586432.0,13618304.0,13650176.0,13453952.0,13485888.0,13517824.0,13549760.0,13581696.0,13613632.0,13645568.0,13677504.0,13480832.0,13512832.0,13544832.0,13576832.0,13608832.0,13640832.0,13672832.0,13704832.0,13507712.0,13539776.0,13571840.0,13603904.0,13635968.0,13668032.0,13700096.0,13732160.0,13534592.0,13566720.0,13598848.0,13630976.0,13663104.0,13695232.0,13727360.0,13759488.0,13561472.0,13593664.0,13625856.0,13658048.0,13690240.0,13722432.0,13754624.0,13786816.0,13588352.0,13620608.0,13652864.0,13685120.0,13717376.0,13749632.0,13781888.0,13814144.0,13615232.0,13647552.0,13679872.0,13712192.0,13744512.0,13776832.0,13809152.0,13841472.0,13642112.0,13674496.0,13706880.0,13739264.0,13771648.0,13804032.0,13836416.0,13868800.0,13668992.0,13701440.0,13733888.0,13766336.0,13798784.0,13831232.0,13863680.0,13896128.0,13695872.0,13728384.0,13760896.0,13793408.0,13825920.0,13858432.0,13890944.0,13923456.0,13722752.0,13755328.0,13787904.0,13820480.0,13853056.0,13885632.0,13918208.0,13950784.0,13749632.0,13782272.0,13814912.0,13847552.0,13880192.0,13912832.0,13945472.0,13978112.0,13776512.0,13809216.0,13841920.0,13874624.0,13907328.0,13940032.0,13972736.0,14005440.0,13803392.0,13836160.0,13868928.0,13901696.0,13934464.0,13967232.0,14000000.0,14032768.0,13830272.0,13863104.0,13895936.0,13928768.0,13961600.0,13994432.0,14027264.0,14060096.0,13857152.0,13890048.0,13922944.0,13955840.0,13988736.0,14021632.0,14054528.0,14087424.0,13884032.0,13916992.0,13949952.0,13982912.0,14015872.0,14048832.0,14081792.0,14114752.0,13910912.0,13943936.0,13976960.0,14009984.0,14043008.0,14076032.0,14109056.0,14142080.0,13937792.0,13970880.0,14003968.0,14037056.0,14070144.0,14103232.0,14136320.0,14169408.0,13964672.0,13997824.0,14030976.0,14064128.0,14097280.0,14130432.0,14163584.0,14196736.0,13991552.0,14024768.0,14057984.0,14091200.0,14124416.0,14157632.0,14190848.0,14224064.0,14018432.0,14051712.0,14084992.0,14118272.0,14151552.0,14184832.0,14218112.0,14251392.0,14045312.0,14078656.0,14112000.0,14145344.0,14178688.0,14212032.0,14245376.0,14278720.0,14072192.0,14105600.0,14139008.0,14172416.0,14205824.0,14239232.0,14272640.0,14306048.0,14099072.0,14132544.0,14166016.0,14199488.0,14232960.0,14266432.0,14299904.0,14333376.0,14125952.0,14159488.0,14193024.0,14226560.0,14260096.0,14293632.0,14327168.0,14360704.0,14152832.0,14186432.0,14220032.0,14253632.0,14287232.0,14320832.0,14354432.0,14388032.0,14179712.0,14213376.0,14247040.0,14280704.0,14314368.0,14348032.0,14381696.0,14415360.0,14206592.0,14240320.0,14274048.0,14307776.0,14341504.0,14375232.0,14408960.0,14442688.0,13545344.0,13578112.0,13610880.0,13643648.0,13676416.0,13709184.0,13741952.0,13774720.0,12875904.0,12907712.0,12939520.0,12971328.0,13003136.0,13034944.0,13066752.0,13098560.0,6508416.0,6524336.0,6540256.0,6556176.0,6572096.0,6588016.0,6603936.0,6619856.0,7104768.0,7120560.0,7136352.0,7152144.0,7167936.0,7183728.0,7199520.0,7215312.0,14349696.0,14381312.0,14412928.0,14444544.0,14476160.0,14507776.0,14539392.0,14571008.0,14377600.0,14409280.0,14440960.0,14472640.0,14504320.0,14536000.0,14567680.0,14599360.0,14405504.0,14437248.0,14468992.0,14500736.0,14532480.0,14564224.0,14595968.0,14627712.0,14433408.0,14465216.0,14497024.0,14528832.0,14560640.0,14592448.0,14624256.0,14656064.0,14461312.0,14493184.0,14525056.0,14556928.0,14588800.0,14620672.0,14652544.0,14684416.0,14489216.0,14521152.0,14553088.0,14585024.0,14616960.0,14648896.0,14680832.0,14712768.0,14517120.0,14549120.0,14581120.0,14613120.0,14645120.0,14677120.0,14709120.0,14741120.0,14545024.0,14577088.0,14609152.0,14641216.0,14673280.0,14705344.0,14737408.0,14769472.0,14572928.0,14605056.0,14637184.0,14669312.0,14701440.0,14733568.0,14765696.0,14797824.0,14600832.0,14633024.0,14665216.0,14697408.0,14729600.0,14761792.0,14793984.0,14826176.0,14628736.0,14660992.0,14693248.0,14725504.0,14757760.0,14790016.0,14822272.0,14854528.0,14656640.0,14688960.0,14721280.0,14753600.0,14785920.0,14818240.0,14850560.0,14882880.0,14684544.0,14716928.0,14749312.0,14781696.0,14814080.0,14846464.0,14878848.0,14911232.0,14712448.0,14744896.0,14777344.0,14809792.0,14842240.0,14874688.0,14907136.0,14939584.0,14740352.0,14772864.0,14805376.0,14837888.0,14870400.0,14902912.0,14935424.0,14967936.0,14768256.0,14800832.0,14833408.0,14865984.0,14898560.0,14931136.0,14963712.0,14996288.0,14796160.0,14828800.0,14861440.0,14894080.0,14926720.0,14959360.0,14992000.0,15024640.0,14824064.0,14856768.0,14889472.0,14922176.0,14954880.0,14987584.0,15020288.0,15052992.0,14851968.0,14884736.0,14917504.0,14950272.0,14983040.0,15015808.0,15048576.0,15081344.0,14879872.0,14912704.0,14945536.0,14978368.0,15011200.0,15044032.0,15076864.0,15109696.0,14907776.0,14940672.0,14973568.0,15006464.0,15039360.0,15072256.0,15105152.0,15138048.0,14935680.0,14968640.0,15001600.0,15034560.0,15067520.0,15100480.0,15133440.0,15166400.0,14963584.0,14996608.0,15029632.0,15062656.0,15095680.0,15128704.0,15161728.0,15194752.0,14991488.0,15024576.0,15057664.0,15090752.0,15123840.0,15156928.0,15190016.0,15223104.0,15019392.0,15052544.0,15085696.0,15118848.0,15152000.0,15185152.0,15218304.0,15251456.0,15047296.0,15080512.0,15113728.0,15146944.0,15180160.0,15213376.0,15246592.0,15279808.0,15075200.0,15108480.0,15141760.0,15175040.0,15208320.0,15241600.0,15274880.0,15308160.0,15103104.0,15136448.0,15169792.0,15203136.0,15236480.0,15269824.0,15303168.0,15336512.0,15131008.0,15164416.0,15197824.0,15231232.0,15264640.0,15298048.0,15331456.0,15364864.0,15158912.0,15192384.0,15225856.0,15259328.0,15292800.0,15326272.0,15359744.0,15393216.0,15186816.0,15220352.0,15253888.0,15287424.0,15320960.0,15354496.0,15388032.0,15421568.0,15214720.0,15248320.0,15281920.0,15315520.0,15349120.0,15382720.0,15416320.0,15449920.0,15242624.0,15276288.0,15309952.0,15343616.0,15377280.0,15410944.0,15444608.0,15478272.0,15270528.0,15304256.0,15337984.0,15371712.0,15405440.0,15439168.0,15472896.0,15506624.0,15298432.0,15332224.0,15366016.0,15399808.0,15433600.0,15467392.0,15501184.0,15534976.0,15326336.0,15360192.0,15394048.0,15427904.0,15461760.0,15495616.0,15529472.0,15563328.0,14649728.0,14682624.0,14715520.0,14748416.0,14781312.0,14814208.0,14847104.0,14880000.0,13964928.0,13996864.0,14028800.0,14060736.0,14092672.0,14124608.0,14156544.0,14188480.0,13288320.0,13319296.0,13350272.0,13381248.0,13412224.0,13443200.0,13474176.0,13505152.0,12603520.0,12633536.0,12663552.0,12693568.0,12723584.0,12753600.0,12783616.0,12813632.0,12631424.0,12661504.0,12691584.0,12721664.0,12751744.0,12781824.0,12811904.0,12841984.0,12659328.0,12689472.0,12719616.0,12749760.0,12779904.0,12810048.0,12840192.0,12870336.0,12687232.0,12717440.0,12747648.0,12777856.0,12808064.0,12838272.0,12868480.0,12898688.0,12715136.0,12745408.0,12775680.0,12805952.0,12836224.0,12866496.0,12896768.0,12927040.0,12743040.0,12773376.0,12803712.0,12834048.0,12864384.0,12894720.0,12925056.0,12955392.0,12770944.0,12801344.0,12831744.0,12862144.0,12892544.0,12922944.0,12953344.0,12983744.0,12798848.0,12829312.0,12859776.0,12890240.0,12920704.0,12951168.0,12981632.0,13012096.0,12826752.0,12857280.0,12887808.0,12918336.0,12948864.0,12979392.0,13009920.0,13040448.0,12854656.0,12885248.0,12915840.0,12946432.0,12977024.0,13007616.0,13038208.0,13068800.0,12882560.0,12913216.0,12943872.0,12974528.0,13005184.0,13035840.0,13066496.0,13097152.0,12910464.0,12941184.0,12971904.0,13002624.0,13033344.0,13064064.0,13094784.0,13125504.0,12938368.0,12969152.0,12999936.0,13030720.0,13061504.0,13092288.0,13123072.0,13153856.0,12966272.0,12997120.0,13027968.0,13058816.0,13089664.0,13120512.0,13151360.0,13182208.0,12994176.0,13025088.0,13056000.0,13086912.0,13117824.0,13148736.0,13179648.0,13210560.0,13022080.0,13053056.0,13084032.0,13115008.0,13145984.0,13176960.0,13207936.0,13238912.0,13049984.0,13081024.0,13112064.0,13143104.0,13174144.0,13205184.0,13236224.0,13267264.0,13077888.0,13108992.0,13140096.0,13171200.0,13202304.0,13233408.0,13264512.0,13295616.0,13105792.0,13136960.0,13168128.0,13199296.0,13230464.0,13261632.0,13292800.0,13323968.0,13133696.0,13164928.0,13196160.0,13227392.0,13258624.0,13289856.0,13321088.0,13352320.0,13161600.0,13192896.0,13224192.0,13255488.0,13286784.0,13318080.0,13349376.0,13380672.0,13189504.0,13220864.0,13252224.0,13283584.0,13314944.0,13346304.0,13377664.0,13409024.0,13217408.0,13248832.0,13280256.0,13311680.0,13343104.0,13374528.0,13405952.0,13437376.0,13245312.0,13276800.0,13308288.0,13339776.0,13371264.0,13402752.0,13434240.0,13465728.0,13273216.0,13304768.0,13336320.0,13367872.0,13399424.0,13430976.0,13462528.0,13494080.0,13301120.0,13332736.0,13364352.0,13395968.0,13427584.0,13459200.0,13490816.0,13522432.0,13329024.0,13360704.0,13392384.0,13424064.0,13455744.0,13487424.0,13519104.0,13550784.0,13356928.0,13388672.0,13420416.0,13452160.0,13483904.0,13515648.0,13547392.0,13579136.0,13384832.0,13416640.0,13448448.0,13480256.0,13512064.0,13543872.0,13575680.0,13607488.0,13412736.0,13444608.0,13476480.0,13508352.0,13540224.0,13572096.0,13603968.0,13635840.0,13440640.0,13472576.0,13504512.0,13536448.0,13568384.0,13600320.0,13632256.0,13664192.0,13468544.0,13500544.0,13532544.0,13564544.0,13596544.0,13628544.0,13660544.0,13692544.0,13496448.0,13528512.0,13560576.0,13592640.0,13624704.0,13656768.0,13688832.0,13720896.0,13524352.0,13556480.0,13588608.0,13620736.0,13652864.0,13684992.0,13717120.0,13749248.0,13552256.0,13584448.0,13616640.0,13648832.0,13681024.0,13713216.0,13745408.0,13777600.0,13580160.0,13612416.0,13644672.0,13676928.0,13709184.0,13741440.0,13773696.0,13805952.0,13608064.0,13640384.0,13672704.0,13705024.0,13737344.0,13769664.0,13801984.0,13834304.0,13635968.0,13668352.0,13700736.0,13733120.0,13765504.0,13797888.0,13830272.0,13862656.0,13483648.0,13515072.0,13546496.0,13577920.0,13609344.0,13640768.0,13672192.0,13703616.0,13323136.0,13353600.0,13384064.0,13414528.0,13444992.0,13475456.0,13505920.0,13536384.0,13351040.0,13381568.0,13412096.0,13442624.0,13473152.0,13503680.0,13534208.0,13564736.0,13378944.0,13409536.0,13440128.0,13470720.0,13501312.0,13531904.0,13562496.0,13593088.0,13406848.0,13437504.0,13468160.0,13498816.0,13529472.0,13560128.0,13590784.0,13621440.0,13434752.0,13465472.0,13496192.0,13526912.0,13557632.0,13588352.0,13619072.0,13649792.0,13462656.0,13493440.0,13524224.0,13555008.0,13585792.0,13616576.0,13647360.0,13678144.0,13490560.0,13521408.0,13552256.0,13583104.0,13613952.0,13644800.0,13675648.0,13706496.0,13518464.0,13549376.0,13580288.0,13611200.0,13642112.0,13673024.0,13703936.0,13734848.0,13546368.0,13577344.0,13608320.0,13639296.0,13670272.0,13701248.0,13732224.0,13763200.0,13574272.0,13605312.0,13636352.0,13667392.0,13698432.0,13729472.0,13760512.0,13791552.0,13602176.0,13633280.0,13664384.0,13695488.0,13726592.0,13757696.0,13788800.0,13819904.0,13630080.0,13661248.0,13692416.0,13723584.0,13754752.0,13785920.0,13817088.0,13848256.0,13657984.0,13689216.0,13720448.0,13751680.0,13782912.0,13814144.0,13845376.0,13876608.0,13685888.0,13717184.0,13748480.0,13779776.0,13811072.0,13842368.0,13873664.0,13904960.0,13713792.0,13745152.0,13776512.0,13807872.0,13839232.0,13870592.0,13901952.0,13933312.0,13741696.0,13773120.0,13804544.0,13835968.0,13867392.0,13898816.0,13930240.0,13961664.0,13769600.0,13801088.0,13832576.0,13864064.0,13895552.0,13927040.0,13958528.0,13990016.0,13797504.0,13829056.0,13860608.0,13892160.0,13923712.0,13955264.0,13986816.0,14018368.0,13825408.0,13857024.0,13888640.0,13920256.0,13951872.0,13983488.0,14015104.0,14046720.0,13853312.0,13884992.0,13916672.0,13948352.0,13980032.0,14011712.0,14043392.0,14075072.0,13881216.0,13912960.0,13944704.0,13976448.0,14008192.0,14039936.0,14071680.0,14103424.0,13909120.0,13940928.0,13972736.0,14004544.0,14036352.0,14068160.0,14099968.0,14131776.0,13937024.0,13968896.0,14000768.0,14032640.0,14064512.0,14096384.0,14128256.0,14160128.0,13964928.0,13996864.0,14028800.0,14060736.0,14092672.0,14124608.0,14156544.0,14188480.0,13992832.0,14024832.0,14056832.0,14088832.0,14120832.0,14152832.0,14184832.0,14216832.0,14020736.0,14052800.0,14084864.0,14116928.0,14148992.0,14181056.0,14213120.0,14245184.0,14048640.0,14080768.0,14112896.0,14145024.0,14177152.0,14209280.0,14241408.0,14273536.0,14076544.0,14108736.0,14140928.0,14173120.0,14205312.0,14237504.0,14269696.0,14301888.0,14104448.0,14136704.0,14168960.0,14201216.0,14233472.0,14265728.0,14297984.0,14330240.0,14132352.0,14164672.0,14196992.0,14229312.0,14261632.0,14293952.0,14326272.0,14358592.0,14160256.0,14192640.0,14225024.0,14257408.0,14289792.0,14322176.0,14354560.0,14386944.0,14188160.0,14220608.0,14253056.0,14285504.0,14317952.0,14350400.0,14382848.0,14415296.0,14216064.0,14248576.0,14281088.0,14313600.0,14346112.0,14378624.0,14411136.0,14443648.0,14243968.0,14276544.0,14309120.0,14341696.0,14374272.0,14406848.0,14439424.0,14472000.0,14271872.0,14304512.0,14337152.0,14369792.0,14402432.0,14435072.0,14467712.0,14500352.0,14299776.0,14332480.0,14365184.0,14397888.0,14430592.0,14463296.0,14496000.0,14528704.0,14327680.0,14360448.0,14393216.0,14425984.0,14458752.0,14491520.0,14524288.0,14557056.0,14355584.0,14388416.0,14421248.0,14454080.0,14486912.0,14519744.0,14552576.0,14585408.0,13678976.0,13710848.0,13742720.0,13774592.0,13806464.0,13838336.0,13870208.0,13902080.0,12994176.0,13025088.0,13056000.0,13086912.0,13117824.0,13148736.0,13179648.0,13210560.0,13022080.0,13053056.0,13084032.0,13115008.0,13145984.0,13176960.0,13207936.0,13238912.0,13049984.0,13081024.0,13112064.0,13143104.0,13174144.0,13205184.0,13236224.0,13267264.0,13077888.0,13108992.0,13140096.0,13171200.0,13202304.0,13233408.0,13264512.0,13295616.0,13105792.0,13136960.0,13168128.0,13199296.0,13230464.0,13261632.0,13292800.0,13323968.0,13133696.0,13164928.0,13196160.0,13227392.0,13258624.0,13289856.0,13321088.0,13352320.0,13161600.0,13192896.0,13224192.0,13255488.0,13286784.0,13318080.0,13349376.0,13380672.0,13189504.0,13220864.0,13252224.0,13283584.0,13314944.0,13346304.0,13377664.0,13409024.0,13217408.0,13248832.0,13280256.0,13311680.0,13343104.0,13374528.0,13405952.0,13437376.0,13245312.0,13276800.0,13308288.0,13339776.0,13371264.0,13402752.0,13434240.0,13465728.0,13273216.0,13304768.0,13336320.0,13367872.0,13399424.0,13430976.0,13462528.0,13494080.0,13301120.0,13332736.0,13364352.0,13395968.0,13427584.0,13459200.0,13490816.0,13522432.0,13329024.0,13360704.0,13392384.0,13424064.0,13455744.0,13487424.0,13519104.0,13550784.0,13356928.0,13388672.0,13420416.0,13452160.0,13483904.0,13515648.0,13547392.0,13579136.0,13384832.0,13416640.0,13448448.0,13480256.0,13512064.0,13543872.0,13575680.0,13607488.0,13412736.0,13444608.0,13476480.0,13508352.0,13540224.0,13572096.0,13603968.0,13635840.0,13440640.0,13472576.0,13504512.0,13536448.0,13568384.0,13600320.0,13632256.0,13664192.0,13468544.0,13500544.0,13532544.0,13564544.0,13596544.0,13628544.0,13660544.0,13692544.0,13496448.0,13528512.0,13560576.0,13592640.0,13624704.0,13656768.0,13688832.0,13720896.0,13524352.0,13556480.0,13588608.0,13620736.0,13652864.0,13684992.0,13717120.0,13749248.0,13552256.0,13584448.0,13616640.0,13648832.0,13681024.0,13713216.0,13745408.0,13777600.0,13580160.0,13612416.0,13644672.0,13676928.0,13709184.0,13741440.0,13773696.0,13805952.0,13608064.0,13640384.0,13672704.0,13705024.0,13737344.0,13769664.0,13801984.0,13834304.0,13635968.0,13668352.0,13700736.0,13733120.0,13765504.0,13797888.0,13830272.0,13862656.0,13663872.0,13696320.0,13728768.0,13761216.0,13793664.0,13826112.0,13858560.0,13891008.0,13691776.0,13724288.0,13756800.0,13789312.0,13821824.0,13854336.0,13886848.0,13919360.0,13719680.0,13752256.0,13784832.0,13817408.0,13849984.0,13882560.0,13915136.0,13947712.0,13747584.0,13780224.0,13812864.0,13845504.0,13878144.0,13910784.0,13943424.0,13976064.0,13775488.0,13808192.0,13840896.0,13873600.0,13906304.0,13939008.0,13971712.0,14004416.0,13803392.0,13836160.0,13868928.0,13901696.0,13934464.0,13967232.0,14000000.0,14032768.0,13831296.0,13864128.0,13896960.0,13929792.0,13962624.0,13995456.0,14028288.0,14061120.0,13859200.0,13892096.0,13924992.0,13957888.0,13990784.0,14023680.0,14056576.0,14089472.0,13887104.0,13920064.0,13953024.0,13985984.0,14018944.0,14051904.0,14084864.0,14117824.0,13915008.0,13948032.0,13981056.0,14014080.0,14047104.0,14080128.0,14113152.0,14146176.0,13942912.0,13976000.0,14009088.0,14042176.0,14075264.0,14108352.0,14141440.0,14174528.0,13970816.0,14003968.0,14037120.0,14070272.0,14103424.0,14136576.0,14169728.0,14202880.0,13998720.0,14031936.0,14065152.0,14098368.0,14131584.0,14164800.0,14198016.0,14231232.0,14026624.0,14059904.0,14093184.0,14126464.0,14159744.0,14193024.0,14226304.0,14259584.0,13874304.0,13906624.0,13938944.0,13971264.0,14003584.0,14035904.0,14068224.0,14100544.0,13713792.0,13745152.0,13776512.0,13807872.0,13839232.0,13870592.0,13901952.0,13933312.0,13741696.0,13773120.0,13804544.0,13835968.0,13867392.0,13898816.0,13930240.0,13961664.0,13769600.0,13801088.0,13832576.0,13864064.0,13895552.0,13927040.0,13958528.0,13990016.0,13797504.0,13829056.0,13860608.0,13892160.0,13923712.0,13955264.0,13986816.0,14018368.0,13825408.0,13857024.0,13888640.0,13920256.0,13951872.0,13983488.0,14015104.0,14046720.0,13853312.0,13884992.0,13916672.0,13948352.0,13980032.0,14011712.0,14043392.0,14075072.0,13881216.0,13912960.0,13944704.0,13976448.0,14008192.0,14039936.0,14071680.0,14103424.0,13909120.0,13940928.0,13972736.0,14004544.0,14036352.0,14068160.0,14099968.0,14131776.0,13937024.0,13968896.0,14000768.0,14032640.0,14064512.0,14096384.0,14128256.0,14160128.0,13964928.0,13996864.0,14028800.0,14060736.0,14092672.0,14124608.0,14156544.0,14188480.0,13992832.0,14024832.0,14056832.0,14088832.0,14120832.0,14152832.0,14184832.0,14216832.0,14020736.0,14052800.0,14084864.0,14116928.0,14148992.0,14181056.0,14213120.0,14245184.0,14048640.0,14080768.0,14112896.0,14145024.0,14177152.0,14209280.0,14241408.0,14273536.0,14076544.0,14108736.0,14140928.0,14173120.0,14205312.0,14237504.0,14269696.0,14301888.0,14104448.0,14136704.0,14168960.0,14201216.0,14233472.0,14265728.0,14297984.0,14330240.0,14132352.0,14164672.0,14196992.0,14229312.0,14261632.0,14293952.0,14326272.0,14358592.0,14160256.0,14192640.0,14225024.0,14257408.0,14289792.0,14322176.0,14354560.0,14386944.0,14188160.0,14220608.0,14253056.0,14285504.0,14317952.0,14350400.0,14382848.0,14415296.0,14216064.0,14248576.0,14281088.0,14313600.0,14346112.0,14378624.0,14411136.0,14443648.0,14243968.0,14276544.0,14309120.0,14341696.0,14374272.0,14406848.0,14439424.0,14472000.0,14271872.0,14304512.0,14337152.0,14369792.0,14402432.0,14435072.0,14467712.0,14500352.0,14299776.0,14332480.0,14365184.0,14397888.0,14430592.0,14463296.0,14496000.0,14528704.0,14327680.0,14360448.0,14393216.0,14425984.0,14458752.0,14491520.0,14524288.0,14557056.0,14355584.0,14388416.0,14421248.0,14454080.0,14486912.0,14519744.0,14552576.0,14585408.0,14383488.0,14416384.0,14449280.0,14482176.0,14515072.0,14547968.0,14580864.0,14613760.0,14411392.0,14444352.0,14477312.0,14510272.0,14543232.0,14576192.0,14609152.0,14642112.0,14439296.0,14472320.0,14505344.0,14538368.0,14571392.0,14604416.0,14637440.0,14670464.0,14467200.0,14500288.0,14533376.0,14566464.0,14599552.0,14632640.0,14665728.0,14698816.0,14495104.0,14528256.0,14561408.0,14594560.0,14627712.0,14660864.0,14694016.0,14727168.0,14523008.0,14556224.0,14589440.0,14622656.0,14655872.0,14689088.0,14722304.0,14755520.0,14550912.0,14584192.0,14617472.0,14650752.0,14684032.0,14717312.0,14750592.0,14783872.0,14578816.0,14612160.0,14645504.0,14678848.0,14712192.0,14745536.0,14778880.0,14812224.0,14606720.0,14640128.0,14673536.0,14706944.0,14740352.0,14773760.0,14807168.0,14840576.0,14634624.0,14668096.0,14701568.0,14735040.0,14768512.0,14801984.0,14835456.0,14868928.0,14662528.0,14696064.0,14729600.0,14763136.0,14796672.0,14830208.0,14863744.0,14897280.0,14690432.0,14724032.0,14757632.0,14791232.0,14824832.0,14858432.0,14892032.0,14925632.0,14718336.0,14752000.0,14785664.0,14819328.0,14852992.0,14886656.0,14920320.0,14953984.0,14746240.0,14779968.0,14813696.0,14847424.0,14881152.0,14914880.0,14948608.0,14982336.0,14069632.0,14102400.0,14135168.0,14167936.0,14200704.0,14233472.0,14266240.0,14299008.0,13384832.0,13416640.0,13448448.0,13480256.0,13512064.0,13543872.0,13575680.0,13607488.0,6763136.0,6779056.0,6794976.0,6810896.0,6826816.0,6842736.0,6858656.0,6874576.0,7357440.0,7373232.0,7389024.0,7404816.0,7420608.0,7436400.0,7452192.0,7467984.0,14855552.0,14887168.0,14918784.0,14950400.0,14982016.0,15013632.0,15045248.0,15076864.0,14884480.0,14916160.0,14947840.0,14979520.0,15011200.0,15042880.0,15074560.0,15106240.0,14913408.0,14945152.0,14976896.0,15008640.0,15040384.0,15072128.0,15103872.0,15135616.0,14942336.0,14974144.0,15005952.0,15037760.0,15069568.0,15101376.0,15133184.0,15164992.0,14971264.0,15003136.0,15035008.0,15066880.0,15098752.0,15130624.0,15162496.0,15194368.0,15000192.0,15032128.0,15064064.0,15096000.0,15127936.0,15159872.0,15191808.0,15223744.0,15029120.0,15061120.0,15093120.0,15125120.0,15157120.0,15189120.0,15221120.0,15253120.0,15058048.0,15090112.0,15122176.0,15154240.0,15186304.0,15218368.0,15250432.0,15282496.0,15086976.0,15119104.0,15151232.0,15183360.0,15215488.0,15247616.0,15279744.0,15311872.0,15115904.0,15148096.0,15180288.0,15212480.0,15244672.0,15276864.0,15309056.0,15341248.0,15144832.0,15177088.0,15209344.0,15241600.0,15273856.0,15306112.0,15338368.0,15370624.0,15173760.0,15206080.0,15238400.0,15270720.0,15303040.0,15335360.0,15367680.0,15400000.0,15202688.0,15235072.0,15267456.0,15299840.0,15332224.0,15364608.0,15396992.0,15429376.0,15231616.0,15264064.0,15296512.0,15328960.0,15361408.0,15393856.0,15426304.0,15458752.0,15260544.0,15293056.0,15325568.0,15358080.0,15390592.0,15423104.0,15455616.0,15488128.0,15289472.0,15322048.0,15354624.0,15387200.0,15419776.0,15452352.0,15484928.0,15517504.0,15318400.0,15351040.0,15383680.0,15416320.0,15448960.0,15481600.0,15514240.0,15546880.0,15347328.0,15380032.0,15412736.0,15445440.0,15478144.0,15510848.0,15543552.0,15576256.0,15376256.0,15409024.0,15441792.0,15474560.0,15507328.0,15540096.0,15572864.0,15605632.0,15405184.0,15438016.0,15470848.0,15503680.0,15536512.0,15569344.0,15602176.0,15635008.0,15434112.0,15467008.0,15499904.0,15532800.0,15565696.0,15598592.0,15631488.0,15664384.0,15463040.0,15496000.0,15528960.0,15561920.0,15594880.0,15627840.0,15660800.0,15693760.0,15491968.0,15524992.0,15558016.0,15591040.0,15624064.0,15657088.0,15690112.0,15723136.0,15520896.0,15553984.0,15587072.0,15620160.0,15653248.0,15686336.0,15719424.0,15752512.0,15549824.0,15582976.0,15616128.0,15649280.0,15682432.0,15715584.0,15748736.0,15781888.0,15578752.0,15611968.0,15645184.0,15678400.0,15711616.0,15744832.0,15778048.0,15811264.0,15607680.0,15640960.0,15674240.0,15707520.0,15740800.0,15774080.0,15807360.0,15840640.0,15636608.0,15669952.0,15703296.0,15736640.0,15769984.0,15803328.0,15836672.0,15870016.0,15665536.0,15698944.0,15732352.0,15765760.0,15799168.0,15832576.0,15865984.0,15899392.0,15694464.0,15727936.0,15761408.0,15794880.0,15828352.0,15861824.0,15895296.0,15928768.0,15723392.0,15756928.0,15790464.0,15824000.0,15857536.0,15891072.0,15924608.0,15958144.0,15752320.0,15785920.0,15819520.0,15853120.0,15886720.0,15920320.0,15953920.0,15987520.0,15781248.0,15814912.0,15848576.0,15882240.0,15915904.0,15949568.0,15983232.0,16016896.0,15810176.0,15843904.0,15877632.0,15911360.0,15945088.0,15978816.0,16012544.0,16046272.0,15839104.0,15872896.0,15906688.0,15940480.0,15974272.0,16008064.0,16041856.0,16075648.0,15868032.0,15901888.0,15935744.0,15969600.0,16003456.0,16037312.0,16071168.0,16105024.0,15176064.0,15208960.0,15241856.0,15274752.0,15307648.0,15340544.0,15373440.0,15406336.0,14475904.0,14507840.0,14539776.0,14571712.0,14603648.0,14635584.0,14667520.0,14699456.0,13783936.0,13814912.0,13845888.0,13876864.0,13907840.0,13938816.0,13969792.0,14000768.0,13083776.0,13113792.0,13143808.0,13173824.0,13203840.0,13233856.0,13263872.0,13293888.0,13112704.0,13142784.0,13172864.0,13202944.0,13233024.0,13263104.0,13293184.0,13323264.0,13141632.0,13171776.0,13201920.0,13232064.0,13262208.0,13292352.0,13322496.0,13352640.0,13170560.0,13200768.0,13230976.0,13261184.0,13291392.0,13321600.0,13351808.0,13382016.0,13199488.0,13229760.0,13260032.0,13290304.0,13320576.0,13350848.0,13381120.0,13411392.0,13228416.0,13258752.0,13289088.0,13319424.0,13349760.0,13380096.0,13410432.0,13440768.0,13257344.0,13287744.0,13318144.0,13348544.0,13378944.0,13409344.0,13439744.0,13470144.0,13286272.0,13316736.0,13347200.0,13377664.0,13408128.0,13438592.0,13469056.0,13499520.0,13315200.0,13345728.0,13376256.0,13406784.0,13437312.0,13467840.0,13498368.0,13528896.0,13344128.0,13374720.0,13405312.0,13435904.0,13466496.0,13497088.0,13527680.0,13558272.0,13373056.0,13403712.0,13434368.0,13465024.0,13495680.0,13526336.0,13556992.0,13587648.0,13401984.0,13432704.0,13463424.0,13494144.0,13524864.0,13555584.0,13586304.0,13617024.0,13430912.0,13461696.0,13492480.0,13523264.0,13554048.0,13584832.0,13615616.0,13646400.0,13459840.0,13490688.0,13521536.0,13552384.0,13583232.0,13614080.0,13644928.0,13675776.0,13488768.0,13519680.0,13550592.0,13581504.0,13612416.0,13643328.0,13674240.0,13705152.0,13517696.0,13548672.0,13579648.0,13610624.0,13641600.0,13672576.0,13703552.0,13734528.0,13546624.0,13577664.0,13608704.0,13639744.0,13670784.0,13701824.0,13732864.0,13763904.0,13575552.0,13606656.0,13637760.0,13668864.0,13699968.0,13731072.0,13762176.0,13793280.0,13604480.0,13635648.0,13666816.0,13697984.0,13729152.0,13760320.0,13791488.0,13822656.0,13633408.0,13664640.0,13695872.0,13727104.0,13758336.0,13789568.0,13820800.0,13852032.0,13662336.0,13693632.0,13724928.0,13756224.0,13787520.0,13818816.0,13850112.0,13881408.0,13691264.0,13722624.0,13753984.0,13785344.0,13816704.0,13848064.0,13879424.0,13910784.0,13720192.0,13751616.0,13783040.0,13814464.0,13845888.0,13877312.0,13908736.0,13940160.0,13749120.0,13780608.0,13812096.0,13843584.0,13875072.0,13906560.0,13938048.0,13969536.0,13778048.0,13809600.0,13841152.0,13872704.0,13904256.0,13935808.0,13967360.0,13998912.0,13806976.0,13838592.0,13870208.0,13901824.0,13933440.0,13965056.0,13996672.0,14028288.0,13835904.0,13867584.0,13899264.0,13930944.0,13962624.0,13994304.0,14025984.0,14057664.0,13864832.0,13896576.0,13928320.0,13960064.0,13991808.0,14023552.0,14055296.0,14087040.0,13893760.0,13925568.0,13957376.0,13989184.0,14020992.0,14052800.0,14084608.0,14116416.0,13922688.0,13954560.0,13986432.0,14018304.0,14050176.0,14082048.0,14113920.0,14145792.0,13951616.0,13983552.0,14015488.0,14047424.0,14079360.0,14111296.0,14143232.0,14175168.0,13980544.0,14012544.0,14044544.0,14076544.0,14108544.0,14140544.0,14172544.0,14204544.0,14009472.0,14041536.0,14073600.0,14105664.0,14137728.0,14169792.0,14201856.0,14233920.0,14038400.0,14070528.0,14102656.0,14134784.0,14166912.0,14199040.0,14231168.0,14263296.0,14067328.0,14099520.0,14131712.0,14163904.0,14196096.0,14228288.0,14260480.0,14292672.0,14096256.0,14128512.0,14160768.0,14193024.0,14225280.0,14257536.0,14289792.0,14322048.0,14125184.0,14157504.0,14189824.0,14222144.0,14254464.0,14286784.0,14319104.0,14351424.0,14154112.0,14186496.0,14218880.0,14251264.0,14283648.0,14316032.0,14348416.0,14380800.0,13986432.0,14017856.0,14049280.0,14080704.0,14112128.0,14143552.0,14174976.0,14206400.0,13810560.0,13841024.0,13871488.0,13901952.0,13932416.0,13962880.0,13993344.0,14023808.0,13839488.0,13870016.0,13900544.0,13931072.0,13961600.0,13992128.0,14022656.0,14053184.0,13868416.0,13899008.0,13929600.0,13960192.0,13990784.0,14021376.0,14051968.0,14082560.0,13897344.0,13928000.0,13958656.0,13989312.0,14019968.0,14050624.0,14081280.0,14111936.0,13926272.0,13956992.0,13987712.0,14018432.0,14049152.0,14079872.0,14110592.0,14141312.0,13955200.0,13985984.0,14016768.0,14047552.0,14078336.0,14109120.0,14139904.0,14170688.0,13984128.0,14014976.0,14045824.0,14076672.0,14107520.0,14138368.0,14169216.0,14200064.0,14013056.0,14043968.0,14074880.0,14105792.0,14136704.0,14167616.0,14198528.0,14229440.0,14041984.0,14072960.0,14103936.0,14134912.0,14165888.0,14196864.0,14227840.0,14258816.0,14070912.0,14101952.0,14132992.0,14164032.0,14195072.0,14226112.0,14257152.0,14288192.0,14099840.0,14130944.0,14162048.0,14193152.0,14224256.0,14255360.0,14286464.0,14317568.0,14128768.0,14159936.0,14191104.0,14222272.0,14253440.0,14284608.0,14315776.0,14346944.0,14157696.0,14188928.0,14220160.0,14251392.0,14282624.0,14313856.0,14345088.0,14376320.0,14186624.0,14217920.0,14249216.0,14280512.0,14311808.0,14343104.0,14374400.0,14405696.0,14215552.0,14246912.0,14278272.0,14309632.0,14340992.0,14372352.0,14403712.0,14435072.0,14244480.0,14275904.0,14307328.0,14338752.0,14370176.0,14401600.0,14433024.0,14464448.0,14273408.0,14304896.0,14336384.0,14367872.0,14399360.0,14430848.0,14462336.0,14493824.0,14302336.0,14333888.0,14365440.0,14396992.0,14428544.0,14460096.0,14491648.0,14523200.0,14331264.0,14362880.0,14394496.0,14426112.0,14457728.0,14489344.0,14520960.0,14552576.0,14360192.0,14391872.0,14423552.0,14455232.0,14486912.0,14518592.0,14550272.0,14581952.0,14389120.0,14420864.0,14452608.0,14484352.0,14516096.0,14547840.0,14579584.0,14611328.0,14418048.0,14449856.0,14481664.0,14513472.0,14545280.0,14577088.0,14608896.0,14640704.0,14446976.0,14478848.0,14510720.0,14542592.0,14574464.0,14606336.0,14638208.0,14670080.0,14475904.0,14507840.0,14539776.0,14571712.0,14603648.0,14635584.0,14667520.0,14699456.0,14504832.0,14536832.0,14568832.0,14600832.0,14632832.0,14664832.0,14696832.0,14728832.0,14533760.0,14565824.0,14597888.0,14629952.0,14662016.0,14694080.0,14726144.0,14758208.0,14562688.0,14594816.0,14626944.0,14659072.0,14691200.0,14723328.0,14755456.0,14787584.0,14591616.0,14623808.0,14656000.0,14688192.0,14720384.0,14752576.0,14784768.0,14816960.0,14620544.0,14652800.0,14685056.0,14717312.0,14749568.0,14781824.0,14814080.0,14846336.0,14649472.0,14681792.0,14714112.0,14746432.0,14778752.0,14811072.0,14843392.0,14875712.0,14678400.0,14710784.0,14743168.0,14775552.0,14807936.0,14840320.0,14872704.0,14905088.0,14707328.0,14739776.0,14772224.0,14804672.0,14837120.0,14869568.0,14902016.0,14934464.0,14736256.0,14768768.0,14801280.0,14833792.0,14866304.0,14898816.0,14931328.0,14963840.0,14765184.0,14797760.0,14830336.0,14862912.0,14895488.0,14928064.0,14960640.0,14993216.0,14794112.0,14826752.0,14859392.0,14892032.0,14924672.0,14957312.0,14989952.0,15022592.0,14823040.0,14855744.0,14888448.0,14921152.0,14953856.0,14986560.0,15019264.0,15051968.0,14851968.0,14884736.0,14917504.0,14950272.0,14983040.0,15015808.0,15048576.0,15081344.0,14880896.0,14913728.0,14946560.0,14979392.0,15012224.0,15045056.0,15077888.0,15110720.0,14188928.0,14220800.0,14252672.0,14284544.0,14316416.0,14348288.0,14380160.0,14412032.0,13488768.0,13519680.0,13550592.0,13581504.0,13612416.0,13643328.0,13674240.0,13705152.0,13517696.0,13548672.0,13579648.0,13610624.0,13641600.0,13672576.0,13703552.0,13734528.0,13546624.0,13577664.0,13608704.0,13639744.0,13670784.0,13701824.0,13732864.0,13763904.0,13575552.0,13606656.0,13637760.0,13668864.0,13699968.0,13731072.0,13762176.0,13793280.0,13604480.0,13635648.0,13666816.0,13697984.0,13729152.0,13760320.0,13791488.0,13822656.0,13633408.0,13664640.0,13695872.0,13727104.0,13758336.0,13789568.0,13820800.0,13852032.0,13662336.0,13693632.0,13724928.0,13756224.0,13787520.0,13818816.0,13850112.0,13881408.0,13691264.0,13722624.0,13753984.0,13785344.0,13816704.0,13848064.0,13879424.0,13910784.0,13720192.0,13751616.0,13783040.0,13814464.0,13845888.0,13877312.0,13908736.0,13940160.0,13749120.0,13780608.0,13812096.0,13843584.0,13875072.0,13906560.0,13938048.0,13969536.0,13778048.0,13809600.0,13841152.0,13872704.0,13904256.0,13935808.0,13967360.0,13998912.0,13806976.0,13838592.0,13870208.0,13901824.0,13933440.0,13965056.0,13996672.0,14028288.0,13835904.0,13867584.0,13899264.0,13930944.0,13962624.0,13994304.0,14025984.0,14057664.0,13864832.0,13896576.0,13928320.0,13960064.0,13991808.0,14023552.0,14055296.0,14087040.0,13893760.0,13925568.0,13957376.0,13989184.0,14020992.0,14052800.0,14084608.0,14116416.0,13922688.0,13954560.0,13986432.0,14018304.0,14050176.0,14082048.0,14113920.0,14145792.0,13951616.0,13983552.0,14015488.0,14047424.0,14079360.0,14111296.0,14143232.0,14175168.0,13980544.0,14012544.0,14044544.0,14076544.0,14108544.0,14140544.0,14172544.0,14204544.0,14009472.0,14041536.0,14073600.0,14105664.0,14137728.0,14169792.0,14201856.0,14233920.0,14038400.0,14070528.0,14102656.0,14134784.0,14166912.0,14199040.0,14231168.0,14263296.0,14067328.0,14099520.0,14131712.0,14163904.0,14196096.0,14228288.0,14260480.0,14292672.0,14096256.0,14128512.0,14160768.0,14193024.0,14225280.0,14257536.0,14289792.0,14322048.0,14125184.0,14157504.0,14189824.0,14222144.0,14254464.0,14286784.0,14319104.0,14351424.0,14154112.0,14186496.0,14218880.0,14251264.0,14283648.0,14316032.0,14348416.0,14380800.0,14183040.0,14215488.0,14247936.0,14280384.0,14312832.0,14345280.0,14377728.0,14410176.0,14211968.0,14244480.0,14276992.0,14309504.0,14342016.0,14374528.0,14407040.0,14439552.0,14240896.0,14273472.0,14306048.0,14338624.0,14371200.0,14403776.0,14436352.0,14468928.0,14269824.0,14302464.0,14335104.0,14367744.0,14400384.0,14433024.0,14465664.0,14498304.0,14298752.0,14331456.0,14364160.0,14396864.0,14429568.0,14462272.0,14494976.0,14527680.0,14327680.0,14360448.0,14393216.0,14425984.0,14458752.0,14491520.0,14524288.0,14557056.0,14356608.0,14389440.0,14422272.0,14455104.0,14487936.0,14520768.0,14553600.0,14586432.0,14385536.0,14418432.0,14451328.0,14484224.0,14517120.0,14550016.0,14582912.0,14615808.0,14414464.0,14447424.0,14480384.0,14513344.0,14546304.0,14579264.0,14612224.0,14645184.0,14443392.0,14476416.0,14509440.0,14542464.0,14575488.0,14608512.0,14641536.0,14674560.0,14472320.0,14505408.0,14538496.0,14571584.0,14604672.0,14637760.0,14670848.0,14703936.0,14501248.0,14534400.0,14567552.0,14600704.0,14633856.0,14667008.0,14700160.0,14733312.0,14530176.0,14563392.0,14596608.0,14629824.0,14663040.0,14696256.0,14729472.0,14762688.0,14559104.0,14592384.0,14625664.0,14658944.0,14692224.0,14725504.0,14758784.0,14792064.0,14391424.0,14423744.0,14456064.0,14488384.0,14520704.0,14553024.0,14585344.0,14617664.0,14215552.0,14246912.0,14278272.0,14309632.0,14340992.0,14372352.0,14403712.0,14435072.0,14244480.0,14275904.0,14307328.0,14338752.0,14370176.0,14401600.0,14433024.0,14464448.0,14273408.0,14304896.0,14336384.0,14367872.0,14399360.0,14430848.0,14462336.0,14493824.0,14302336.0,14333888.0,14365440.0,14396992.0,14428544.0,14460096.0,14491648.0,14523200.0,14331264.0,14362880.0,14394496.0,14426112.0,14457728.0,14489344.0,14520960.0,14552576.0,14360192.0,14391872.0,14423552.0,14455232.0,14486912.0,14518592.0,14550272.0,14581952.0,14389120.0,14420864.0,14452608.0,14484352.0,14516096.0,14547840.0,14579584.0,14611328.0,14418048.0,14449856.0,14481664.0,14513472.0,14545280.0,14577088.0,14608896.0,14640704.0,14446976.0,14478848.0,14510720.0,14542592.0,14574464.0,14606336.0,14638208.0,14670080.0,14475904.0,14507840.0,14539776.0,14571712.0,14603648.0,14635584.0,14667520.0,14699456.0,14504832.0,14536832.0,14568832.0,14600832.0,14632832.0,14664832.0,14696832.0,14728832.0,14533760.0,14565824.0,14597888.0,14629952.0,14662016.0,14694080.0,14726144.0,14758208.0,14562688.0,14594816.0,14626944.0,14659072.0,14691200.0,14723328.0,14755456.0,14787584.0,14591616.0,14623808.0,14656000.0,14688192.0,14720384.0,14752576.0,14784768.0,14816960.0,14620544.0,14652800.0,14685056.0,14717312.0,14749568.0,14781824.0,14814080.0,14846336.0,14649472.0,14681792.0,14714112.0,14746432.0,14778752.0,14811072.0,14843392.0,14875712.0,14678400.0,14710784.0,14743168.0,14775552.0,14807936.0,14840320.0,14872704.0,14905088.0,14707328.0,14739776.0,14772224.0,14804672.0,14837120.0,14869568.0,14902016.0,14934464.0,14736256.0,14768768.0,14801280.0,14833792.0,14866304.0,14898816.0,14931328.0,14963840.0,14765184.0,14797760.0,14830336.0,14862912.0,14895488.0,14928064.0,14960640.0,14993216.0,14794112.0,14826752.0,14859392.0,14892032.0,14924672.0,14957312.0,14989952.0,15022592.0,14823040.0,14855744.0,14888448.0,14921152.0,14953856.0,14986560.0,15019264.0,15051968.0,14851968.0,14884736.0,14917504.0,14950272.0,14983040.0,15015808.0,15048576.0,15081344.0,14880896.0,14913728.0,14946560.0,14979392.0,15012224.0,15045056.0,15077888.0,15110720.0,14909824.0,14942720.0,14975616.0,15008512.0,15041408.0,15074304.0,15107200.0,15140096.0,14938752.0,14971712.0,15004672.0,15037632.0,15070592.0,15103552.0,15136512.0,15169472.0,14967680.0,15000704.0,15033728.0,15066752.0,15099776.0,15132800.0,15165824.0,15198848.0,14996608.0,15029696.0,15062784.0,15095872.0,15128960.0,15162048.0,15195136.0,15228224.0,15025536.0,15058688.0,15091840.0,15124992.0,15158144.0,15191296.0,15224448.0,15257600.0,15054464.0,15087680.0,15120896.0,15154112.0,15187328.0,15220544.0,15253760.0,15286976.0,15083392.0,15116672.0,15149952.0,15183232.0,15216512.0,15249792.0,15283072.0,15316352.0,15112320.0,15145664.0,15179008.0,15212352.0,15245696.0,15279040.0,15312384.0,15345728.0,15141248.0,15174656.0,15208064.0,15241472.0,15274880.0,15308288.0,15341696.0,15375104.0,15170176.0,15203648.0,15237120.0,15270592.0,15304064.0,15337536.0,15371008.0,15404480.0,15199104.0,15232640.0,15266176.0,15299712.0,15333248.0,15366784.0,15400320.0,15433856.0,15228032.0,15261632.0,15295232.0,15328832.0,15362432.0,15396032.0,15429632.0,15463232.0,15256960.0,15290624.0,15324288.0,15357952.0,15391616.0,15425280.0,15458944.0,15492608.0,15285888.0,15319616.0,15353344.0,15387072.0,15420800.0,15454528.0,15488256.0,15521984.0,14593920.0,14626688.0,14659456.0,14692224.0,14724992.0,14757760.0,14790528.0,14823296.0,13893760.0,13925568.0,13957376.0,13989184.0,14020992.0,14052800.0,14084608.0,14116416.0,7017856.0,7033776.0,7049696.0,7065616.0,7081536.0,7097456.0,7113376.0,7129296.0,7610112.0,7625904.0,7641696.0,7657488.0,7673280.0,7689072.0,7704864.0,7720656.0,15361408.0,15393024.0,15424640.0,15456256.0,15487872.0,15519488.0,15551104.0,15582720.0,15391360.0,15423040.0,15454720.0,15486400.0,15518080.0,15549760.0,15581440.0,15613120.0,15421312.0,15453056.0,15484800.0,15516544.0,15548288.0,15580032.0,15611776.0,15643520.0,15451264.0,15483072.0,15514880.0,15546688.0,15578496.0,15610304.0,15642112.0,15673920.0,15481216.0,15513088.0,15544960.0,15576832.0,15608704.0,15640576.0,15672448.0,15704320.0,15511168.0,15543104.0,15575040.0,15606976.0,15638912.0,15670848.0,15702784.0,15734720.0,15541120.0,15573120.0,15605120.0,15637120.0,15669120.0,15701120.0,15733120.0,15765120.0,15571072.0,15603136.0,15635200.0,15667264.0,15699328.0,15731392.0,15763456.0,15795520.0,15601024.0,15633152.0,15665280.0,15697408.0,15729536.0,15761664.0,15793792.0,15825920.0,15630976.0,15663168.0,15695360.0,15727552.0,15759744.0,15791936.0,15824128.0,15856320.0,15660928.0,15693184.0,15725440.0,15757696.0,15789952.0,15822208.0,15854464.0,15886720.0,15690880.0,15723200.0,15755520.0,15787840.0,15820160.0,15852480.0,15884800.0,15917120.0,15720832.0,15753216.0,15785600.0,15817984.0,15850368.0,15882752.0,15915136.0,15947520.0,15750784.0,15783232.0,15815680.0,15848128.0,15880576.0,15913024.0,15945472.0,15977920.0,15780736.0,15813248.0,15845760.0,15878272.0,15910784.0,15943296.0,15975808.0,16008320.0,15810688.0,15843264.0,15875840.0,15908416.0,15940992.0,15973568.0,16006144.0,16038720.0,15840640.0,15873280.0,15905920.0,15938560.0,15971200.0,16003840.0,16036480.0,16069120.0,15870592.0,15903296.0,15936000.0,15968704.0,16001408.0,16034112.0,16066816.0,16099520.0,15900544.0,15933312.0,15966080.0,15998848.0,16031616.0,16064384.0,16097152.0,16129920.0,15930496.0,15963328.0,15996160.0,16028992.0,16061824.0,16094656.0,16127488.0,16160320.0,15960448.0,15993344.0,16026240.0,16059136.0,16092032.0,16124928.0,16157824.0,16190720.0,15990400.0,16023360.0,16056320.0,16089280.0,16122240.0,16155200.0,16188160.0,16221120.0,16020352.0,16053376.0,16086400.0,16119424.0,16152448.0,16185472.0,16218496.0,16251520.0,16050304.0,16083392.0,16116480.0,16149568.0,16182656.0,16215744.0,16248832.0,16281920.0,16080256.0,16113408.0,16146560.0,16179712.0,16212864.0,16246016.0,16279168.0,16312320.0,16110208.0,16143424.0,16176640.0,16209856.0,16243072.0,16276288.0,16309504.0,16342720.0,16140160.0,16173440.0,16206720.0,16240000.0,16273280.0,16306560.0,16339840.0,16373120.0,16170112.0,16203456.0,16236800.0,16270144.0,16303488.0,16336832.0,16370176.0,16403520.0,16200064.0,16233472.0,16266880.0,16300288.0,16333696.0,16367104.0,16400512.0,16433920.0,16230016.0,16263488.0,16296960.0,16330432.0,16363904.0,16397376.0,16430848.0,16464320.0,16259968.0,16293504.0,16327040.0,16360576.0,16394112.0,16427648.0,16461184.0,16494720.0,16289920.0,16323520.0,16357120.0,16390720.0,16424320.0,16457920.0,16491520.0,16525120.0,16319872.0,16353536.0,16387200.0,16420864.0,16454528.0,16488192.0,16521856.0,16555520.0,16349824.0,16383552.0,16417280.0,16451008.0,16484736.0,16518464.0,16552192.0,16585920.0,16379776.0,16413568.0,16447360.0,16481152.0,16514944.0,16548736.0,16582528.0,16616320.0,16409728.0,16443584.0,16477440.0,16511296.0,16545152.0,16579008.0,16612864.0,16646720.0,15702400.0,15735296.0,15768192.0,15801088.0,15833984.0,15866880.0,15899776.0,15932672.0,14986880.0,15018816.0,15050752.0,15082688.0,15114624.0,15146560.0,15178496.0,15210432.0,14279552.0,14310528.0,14341504.0,14372480.0,14403456.0,14434432.0,14465408.0,14496384.0,13564032.0,13594048.0,13624064.0,13654080.0,13684096.0,13714112.0,13744128.0,13774144.0,13593984.0,13624064.0,13654144.0,13684224.0,13714304.0,13744384.0,13774464.0,13804544.0,13623936.0,13654080.0,13684224.0,13714368.0,13744512.0,13774656.0,13804800.0,13834944.0,13653888.0,13684096.0,13714304.0,13744512.0,13774720.0,13804928.0,13835136.0,13865344.0,13683840.0,13714112.0,13744384.0,13774656.0,13804928.0,13835200.0,13865472.0,13895744.0,13713792.0,13744128.0,13774464.0,13804800.0,13835136.0,13865472.0,13895808.0,13926144.0,13743744.0,13774144.0,13804544.0,13834944.0,13865344.0,13895744.0,13926144.0,13956544.0,13773696.0,13804160.0,13834624.0,13865088.0,13895552.0,13926016.0,13956480.0,13986944.0,13803648.0,13834176.0,13864704.0,13895232.0,13925760.0,13956288.0,13986816.0,14017344.0,13833600.0,13864192.0,13894784.0,13925376.0,13955968.0,13986560.0,14017152.0,14047744.0,13863552.0,13894208.0,13924864.0,13955520.0,13986176.0,14016832.0,14047488.0,14078144.0,13893504.0,13924224.0,13954944.0,13985664.0,14016384.0,14047104.0,14077824.0,14108544.0,13923456.0,13954240.0,13985024.0,14015808.0,14046592.0,14077376.0,14108160.0,14138944.0,13953408.0,13984256.0,14015104.0,14045952.0,14076800.0,14107648.0,14138496.0,14169344.0,13983360.0,14014272.0,14045184.0,14076096.0,14107008.0,14137920.0,14168832.0,14199744.0,14013312.0,14044288.0,14075264.0,14106240.0,14137216.0,14168192.0,14199168.0,14230144.0,14043264.0,14074304.0,14105344.0,14136384.0,14167424.0,14198464.0,14229504.0,14260544.0,14073216.0,14104320.0,14135424.0,14166528.0,14197632.0,14228736.0,14259840.0,14290944.0,14103168.0,14134336.0,14165504.0,14196672.0,14227840.0,14259008.0,14290176.0,14321344.0,14133120.0,14164352.0,14195584.0,14226816.0,14258048.0,14289280.0,14320512.0,14351744.0,14163072.0,14194368.0,14225664.0,14256960.0,14288256.0,14319552.0,14350848.0,14382144.0,14193024.0,14224384.0,14255744.0,14287104.0,14318464.0,14349824.0,14381184.0,14412544.0,14222976.0,14254400.0,14285824.0,14317248.0,14348672.0,14380096.0,14411520.0,14442944.0,14252928.0,14284416.0,14315904.0,14347392.0,14378880.0,14410368.0,14441856.0,14473344.0,14282880.0,14314432.0,14345984.0,14377536.0,14409088.0,14440640.0,14472192.0,14503744.0,14312832.0,14344448.0,14376064.0,14407680.0,14439296.0,14470912.0,14502528.0,14534144.0,14342784.0,14374464.0,14406144.0,14437824.0,14469504.0,14501184.0,14532864.0,14564544.0,14372736.0,14404480.0,14436224.0,14467968.0,14499712.0,14531456.0,14563200.0,14594944.0,14402688.0,14434496.0,14466304.0,14498112.0,14529920.0,14561728.0,14593536.0,14625344.0,14432640.0,14464512.0,14496384.0,14528256.0,14560128.0,14592000.0,14623872.0,14655744.0,14462592.0,14494528.0,14526464.0,14558400.0,14590336.0,14622272.0,14654208.0,14686144.0,14492544.0,14524544.0,14556544.0,14588544.0,14620544.0,14652544.0,14684544.0,14716544.0,14522496.0,14554560.0,14586624.0,14618688.0,14650752.0,14682816.0,14714880.0,14746944.0,14552448.0,14584576.0,14616704.0,14648832.0,14680960.0,14713088.0,14745216.0,14777344.0,14582400.0,14614592.0,14646784.0,14678976.0,14711168.0,14743360.0,14775552.0,14807744.0,14612352.0,14644608.0,14676864.0,14709120.0,14741376.0,14773632.0,14805888.0,14838144.0,14642304.0,14674624.0,14706944.0,14739264.0,14771584.0,14803904.0,14836224.0,14868544.0,14672256.0,14704640.0,14737024.0,14769408.0,14801792.0,14834176.0,14866560.0,14898944.0,14489216.0,14520640.0,14552064.0,14583488.0,14614912.0,14646336.0,14677760.0,14709184.0,14297984.0,14328448.0,14358912.0,14389376.0,14419840.0,14450304.0,14480768.0,14511232.0,14327936.0,14358464.0,14388992.0,14419520.0,14450048.0,14480576.0,14511104.0,14541632.0,14357888.0,14388480.0,14419072.0,14449664.0,14480256.0,14510848.0,14541440.0,14572032.0,14387840.0,14418496.0,14449152.0,14479808.0,14510464.0,14541120.0,14571776.0,14602432.0,14417792.0,14448512.0,14479232.0,14509952.0,14540672.0,14571392.0,14602112.0,14632832.0,14447744.0,14478528.0,14509312.0,14540096.0,14570880.0,14601664.0,14632448.0,14663232.0,14477696.0,14508544.0,14539392.0,14570240.0,14601088.0,14631936.0,14662784.0,14693632.0,14507648.0,14538560.0,14569472.0,14600384.0,14631296.0,14662208.0,14693120.0,14724032.0,14537600.0,14568576.0,14599552.0,14630528.0,14661504.0,14692480.0,14723456.0,14754432.0,14567552.0,14598592.0,14629632.0,14660672.0,14691712.0,14722752.0,14753792.0,14784832.0,14597504.0,14628608.0,14659712.0,14690816.0,14721920.0,14753024.0,14784128.0,14815232.0,14627456.0,14658624.0,14689792.0,14720960.0,14752128.0,14783296.0,14814464.0,14845632.0,14657408.0,14688640.0,14719872.0,14751104.0,14782336.0,14813568.0,14844800.0,14876032.0,14687360.0,14718656.0,14749952.0,14781248.0,14812544.0,14843840.0,14875136.0,14906432.0,14717312.0,14748672.0,14780032.0,14811392.0,14842752.0,14874112.0,14905472.0,14936832.0,14747264.0,14778688.0,14810112.0,14841536.0,14872960.0,14904384.0,14935808.0,14967232.0,14777216.0,14808704.0,14840192.0,14871680.0,14903168.0,14934656.0,14966144.0,14997632.0,14807168.0,14838720.0,14870272.0,14901824.0,14933376.0,14964928.0,14996480.0,15028032.0,14837120.0,14868736.0,14900352.0,14931968.0,14963584.0,14995200.0,15026816.0,15058432.0,14867072.0,14898752.0,14930432.0,14962112.0,14993792.0,15025472.0,15057152.0,15088832.0,14897024.0,14928768.0,14960512.0,14992256.0,15024000.0,15055744.0,15087488.0,15119232.0,14926976.0,14958784.0,14990592.0,15022400.0,15054208.0,15086016.0,15117824.0,15149632.0,14956928.0,14988800.0,15020672.0,15052544.0,15084416.0,15116288.0,15148160.0,15180032.0,14986880.0,15018816.0,15050752.0,15082688.0,15114624.0,15146560.0,15178496.0,15210432.0,15016832.0,15048832.0,15080832.0,15112832.0,15144832.0,15176832.0,15208832.0,15240832.0,15046784.0,15078848.0,15110912.0,15142976.0,15175040.0,15207104.0,15239168.0,15271232.0,15076736.0,15108864.0,15140992.0,15173120.0,15205248.0,15237376.0,15269504.0,15301632.0,15106688.0,15138880.0,15171072.0,15203264.0,15235456.0,15267648.0,15299840.0,15332032.0,15136640.0,15168896.0,15201152.0,15233408.0,15265664.0,15297920.0,15330176.0,15362432.0,15166592.0,15198912.0,15231232.0,15263552.0,15295872.0,15328192.0,15360512.0,15392832.0,15196544.0,15228928.0,15261312.0,15293696.0,15326080.0,15358464.0,15390848.0,15423232.0,15226496.0,15258944.0,15291392.0,15323840.0,15356288.0,15388736.0,15421184.0,15453632.0,15256448.0,15288960.0,15321472.0,15353984.0,15386496.0,15419008.0,15451520.0,15484032.0,15286400.0,15318976.0,15351552.0,15384128.0,15416704.0,15449280.0,15481856.0,15514432.0,15316352.0,15348992.0,15381632.0,15414272.0,15446912.0,15479552.0,15512192.0,15544832.0,15346304.0,15379008.0,15411712.0,15444416.0,15477120.0,15509824.0,15542528.0,15575232.0,15376256.0,15409024.0,15441792.0,15474560.0,15507328.0,15540096.0,15572864.0,15605632.0,15406208.0,15439040.0,15471872.0,15504704.0,15537536.0,15570368.0,15603200.0,15636032.0,14698880.0,14730752.0,14762624.0,14794496.0,14826368.0,14858240.0,14890112.0,14921984.0,13983360.0,14014272.0,14045184.0,14076096.0,14107008.0,14137920.0,14168832.0,14199744.0,14013312.0,14044288.0,14075264.0,14106240.0,14137216.0,14168192.0,14199168.0,14230144.0,14043264.0,14074304.0,14105344.0,14136384.0,14167424.0,14198464.0,14229504.0,14260544.0,14073216.0,14104320.0,14135424.0,14166528.0,14197632.0,14228736.0,14259840.0,14290944.0,14103168.0,14134336.0,14165504.0,14196672.0,14227840.0,14259008.0,14290176.0,14321344.0,14133120.0,14164352.0,14195584.0,14226816.0,14258048.0,14289280.0,14320512.0,14351744.0,14163072.0,14194368.0,14225664.0,14256960.0,14288256.0,14319552.0,14350848.0,14382144.0,14193024.0,14224384.0,14255744.0,14287104.0,14318464.0,14349824.0,14381184.0,14412544.0,14222976.0,14254400.0,14285824.0,14317248.0,14348672.0,14380096.0,14411520.0,14442944.0,14252928.0,14284416.0,14315904.0,14347392.0,14378880.0,14410368.0,14441856.0,14473344.0,14282880.0,14314432.0,14345984.0,14377536.0,14409088.0,14440640.0,14472192.0,14503744.0,14312832.0,14344448.0,14376064.0,14407680.0,14439296.0,14470912.0,14502528.0,14534144.0,14342784.0,14374464.0,14406144.0,14437824.0,14469504.0,14501184.0,14532864.0,14564544.0,14372736.0,14404480.0,14436224.0,14467968.0,14499712.0,14531456.0,14563200.0,14594944.0,14402688.0,14434496.0,14466304.0,14498112.0,14529920.0,14561728.0,14593536.0,14625344.0,14432640.0,14464512.0,14496384.0,14528256.0,14560128.0,14592000.0,14623872.0,14655744.0,14462592.0,14494528.0,14526464.0,14558400.0,14590336.0,14622272.0,14654208.0,14686144.0,14492544.0,14524544.0,14556544.0,14588544.0,14620544.0,14652544.0,14684544.0,14716544.0,14522496.0,14554560.0,14586624.0,14618688.0,14650752.0,14682816.0,14714880.0,14746944.0,14552448.0,14584576.0,14616704.0,14648832.0,14680960.0,14713088.0,14745216.0,14777344.0,14582400.0,14614592.0,14646784.0,14678976.0,14711168.0,14743360.0,14775552.0,14807744.0,14612352.0,14644608.0,14676864.0,14709120.0,14741376.0,14773632.0,14805888.0,14838144.0,14642304.0,14674624.0,14706944.0,14739264.0,14771584.0,14803904.0,14836224.0,14868544.0,14672256.0,14704640.0,14737024.0,14769408.0,14801792.0,14834176.0,14866560.0,14898944.0,14702208.0,14734656.0,14767104.0,14799552.0,14832000.0,14864448.0,14896896.0,14929344.0,14732160.0,14764672.0,14797184.0,14829696.0,14862208.0,14894720.0,14927232.0,14959744.0,14762112.0,14794688.0,14827264.0,14859840.0,14892416.0,14924992.0,14957568.0,14990144.0,14792064.0,14824704.0,14857344.0,14889984.0,14922624.0,14955264.0,14987904.0,15020544.0,14822016.0,14854720.0,14887424.0,14920128.0,14952832.0,14985536.0,15018240.0,15050944.0,14851968.0,14884736.0,14917504.0,14950272.0,14983040.0,15015808.0,15048576.0,15081344.0,14881920.0,14914752.0,14947584.0,14980416.0,15013248.0,15046080.0,15078912.0,15111744.0,14911872.0,14944768.0,14977664.0,15010560.0,15043456.0,15076352.0,15109248.0,15142144.0,14941824.0,14974784.0,15007744.0,15040704.0,15073664.0,15106624.0,15139584.0,15172544.0,14971776.0,15004800.0,15037824.0,15070848.0,15103872.0,15136896.0,15169920.0,15202944.0,15001728.0,15034816.0,15067904.0,15100992.0,15134080.0,15167168.0,15200256.0,15233344.0,15031680.0,15064832.0,15097984.0,15131136.0,15164288.0,15197440.0,15230592.0,15263744.0,15061632.0,15094848.0,15128064.0,15161280.0,15194496.0,15227712.0,15260928.0,15294144.0,15091584.0,15124864.0,15158144.0,15191424.0,15224704.0,15257984.0,15291264.0,15324544.0,14908544.0,14940864.0,14973184.0,15005504.0,15037824.0,15070144.0,15102464.0,15134784.0,14717312.0,14748672.0,14780032.0,14811392.0,14842752.0,14874112.0,14905472.0,14936832.0,14747264.0,14778688.0,14810112.0,14841536.0,14872960.0,14904384.0,14935808.0,14967232.0,14777216.0,14808704.0,14840192.0,14871680.0,14903168.0,14934656.0,14966144.0,14997632.0,14807168.0,14838720.0,14870272.0,14901824.0,14933376.0,14964928.0,14996480.0,15028032.0,14837120.0,14868736.0,14900352.0,14931968.0,14963584.0,14995200.0,15026816.0,15058432.0,14867072.0,14898752.0,14930432.0,14962112.0,14993792.0,15025472.0,15057152.0,15088832.0,14897024.0,14928768.0,14960512.0,14992256.0,15024000.0,15055744.0,15087488.0,15119232.0,14926976.0,14958784.0,14990592.0,15022400.0,15054208.0,15086016.0,15117824.0,15149632.0,14956928.0,14988800.0,15020672.0,15052544.0,15084416.0,15116288.0,15148160.0,15180032.0,14986880.0,15018816.0,15050752.0,15082688.0,15114624.0,15146560.0,15178496.0,15210432.0,15016832.0,15048832.0,15080832.0,15112832.0,15144832.0,15176832.0,15208832.0,15240832.0,15046784.0,15078848.0,15110912.0,15142976.0,15175040.0,15207104.0,15239168.0,15271232.0,15076736.0,15108864.0,15140992.0,15173120.0,15205248.0,15237376.0,15269504.0,15301632.0,15106688.0,15138880.0,15171072.0,15203264.0,15235456.0,15267648.0,15299840.0,15332032.0,15136640.0,15168896.0,15201152.0,15233408.0,15265664.0,15297920.0,15330176.0,15362432.0,15166592.0,15198912.0,15231232.0,15263552.0,15295872.0,15328192.0,15360512.0,15392832.0,15196544.0,15228928.0,15261312.0,15293696.0,15326080.0,15358464.0,15390848.0,15423232.0,15226496.0,15258944.0,15291392.0,15323840.0,15356288.0,15388736.0,15421184.0,15453632.0,15256448.0,15288960.0,15321472.0,15353984.0,15386496.0,15419008.0,15451520.0,15484032.0,15286400.0,15318976.0,15351552.0,15384128.0,15416704.0,15449280.0,15481856.0,15514432.0,15316352.0,15348992.0,15381632.0,15414272.0,15446912.0,15479552.0,15512192.0,15544832.0,15346304.0,15379008.0,15411712.0,15444416.0,15477120.0,15509824.0,15542528.0,15575232.0,15376256.0,15409024.0,15441792.0,15474560.0,15507328.0,15540096.0,15572864.0,15605632.0,15406208.0,15439040.0,15471872.0,15504704.0,15537536.0,15570368.0,15603200.0,15636032.0,15436160.0,15469056.0,15501952.0,15534848.0,15567744.0,15600640.0,15633536.0,15666432.0,15466112.0,15499072.0,15532032.0,15564992.0,15597952.0,15630912.0,15663872.0,15696832.0,15496064.0,15529088.0,15562112.0,15595136.0,15628160.0,15661184.0,15694208.0,15727232.0,15526016.0,15559104.0,15592192.0,15625280.0,15658368.0,15691456.0,15724544.0,15757632.0,15555968.0,15589120.0,15622272.0,15655424.0,15688576.0,15721728.0,15754880.0,15788032.0,15585920.0,15619136.0,15652352.0,15685568.0,15718784.0,15752000.0,15785216.0,15818432.0,15615872.0,15649152.0,15682432.0,15715712.0,15748992.0,15782272.0,15815552.0,15848832.0,15645824.0,15679168.0,15712512.0,15745856.0,15779200.0,15812544.0,15845888.0,15879232.0,15675776.0,15709184.0,15742592.0,15776000.0,15809408.0,15842816.0,15876224.0,15909632.0,15705728.0,15739200.0,15772672.0,15806144.0,15839616.0,15873088.0,15906560.0,15940032.0,15735680.0,15769216.0,15802752.0,15836288.0,15869824.0,15903360.0,15936896.0,15970432.0,15765632.0,15799232.0,15832832.0,15866432.0,15900032.0,15933632.0,15967232.0,16000832.0,15795584.0,15829248.0,15862912.0,15896576.0,15930240.0,15963904.0,15997568.0,16031232.0,15825536.0,15859264.0,15892992.0,15926720.0,15960448.0,15994176.0,16027904.0,16061632.0,15118208.0,15150976.0,15183744.0,15216512.0,15249280.0,15282048.0,15314816.0,15347584.0,14402688.0,14434496.0,14466304.0,14498112.0,14529920.0,14561728.0,14593536.0,14625344.0,7272576.0,7288496.0,7304416.0,7320336.0,7336256.0,7352176.0,7368096.0,7384016.0,7862784.0,7878576.0,7894368.0,7910160.0,7925952.0,7941744.0,7957536.0,7973328.0,15867264.0,15898880.0,15930496.0,15962112.0,15993728.0,16025344.0,16056960.0,16088576.0,15898240.0,15929920.0,15961600.0,15993280.0,16024960.0,16056640.0,16088320.0,16120000.0,15929216.0,15960960.0,15992704.0,16024448.0,16056192.0,16087936.0,16119680.0,16151424.0,15960192.0,15992000.0,16023808.0,16055616.0,16087424.0,16119232.0,16151040.0,16182848.0,15991168.0,16023040.0,16054912.0,16086784.0,16118656.0,16150528.0,16182400.0,16214272.0,16022144.0,16054080.0,16086016.0,16117952.0,16149888.0,16181824.0,16213760.0,16245696.0,16053120.0,16085120.0,16117120.0,16149120.0,16181120.0,16213120.0,16245120.0,16277120.0,16084096.0,16116160.0,16148224.0,16180288.0,16212352.0,16244416.0,16276480.0,16308544.0,16115072.0,16147200.0,16179328.0,16211456.0,16243584.0,16275712.0,16307840.0,16339968.0,16146048.0,16178240.0,16210432.0,16242624.0,16274816.0,16307008.0,16339200.0,16371392.0,16177024.0,16209280.0,16241536.0,16273792.0,16306048.0,16338304.0,16370560.0,16402816.0,16208000.0,16240320.0,16272640.0,16304960.0,16337280.0,16369600.0,16401920.0,16434240.0,16238976.0,16271360.0,16303744.0,16336128.0,16368512.0,16400896.0,16433280.0,16465664.0,16269952.0,16302400.0,16334848.0,16367296.0,16399744.0,16432192.0,16464640.0,16497088.0,16300928.0,16333440.0,16365952.0,16398464.0,16430976.0,16463488.0,16496000.0,16528512.0,16331904.0,16364480.0,16397056.0,16429632.0,16462208.0,16494784.0,16527360.0,16559936.0,16362880.0,16395520.0,16428160.0,16460800.0,16493440.0,16526080.0,16558720.0,16591360.0,16393856.0,16426560.0,16459264.0,16491968.0,16524672.0,16557376.0,16590080.0,16622784.0,16424832.0,16457600.0,16490368.0,16523136.0,16555904.0,16588672.0,16621440.0,16654208.0,16455808.0,16488640.0,16521472.0,16554304.0,16587136.0,16619968.0,16652800.0,16685632.0,16486784.0,16519680.0,16552576.0,16585472.0,16618368.0,16651264.0,16684160.0,16717056.0,16517760.0,16550720.0,16583680.0,16616640.0,16649600.0,16682560.0,16715520.0,16748480.0,16548736.0,16581760.0,16614784.0,16647808.0,16680832.0,16713856.0,16746880.0,16779904.0,16579712.0,16612800.0,16645888.0,16678976.0,16712064.0,16745152.0,16778240.0,16811328.0,16610688.0,16643840.0,16676992.0,16710144.0,16743296.0,16776448.0,16809600.0,16842752.0,16641664.0,16674880.0,16708096.0,16741312.0,16774528.0,16807744.0,16840960.0,16874176.0,16672640.0,16705920.0,16739200.0,16772480.0,16805760.0,16839040.0,16872320.0,16905600.0,16703616.0,16736960.0,16770304.0,16803648.0,16836992.0,16870336.0,16903680.0,16937024.0,16734592.0,16768000.0,16801408.0,16834816.0,16868224.0,16901632.0,16935040.0,16968448.0,16765568.0,16799040.0,16832512.0,16865984.0,16899456.0,16932928.0,16966400.0,16999872.0,16796544.0,16830080.0,16863616.0,16897152.0,16930688.0,16964224.0,16997760.0,17031296.0,16827520.0,16861120.0,16894720.0,16928320.0,16961920.0,16995520.0,17029120.0,17062720.0,16858496.0,16892160.0,16925824.0,16959488.0,16993152.0,17026816.0,17060480.0,17094144.0,16889472.0,16923200.0,16956928.0,16990656.0,17024384.0,17058112.0,17091840.0,17125568.0,16920448.0,16954240.0,16988032.0,17021824.0,17055616.0,17089408.0,17123200.0,17156992.0,16951424.0,16985280.0,17019136.0,17052992.0,17086848.0,17120704.0,17154560.0,17188416.0,16228736.0,16261632.0,16294528.0,16327424.0,16360320.0,16393216.0,16426112.0,16459008.0,15497856.0,15529792.0,15561728.0,15593664.0,15625600.0,15657536.0,15689472.0,15721408.0,14775168.0,14806144.0,14837120.0,14868096.0,14899072.0,14930048.0,14961024.0,14992000.0,14044288.0,14074304.0,14104320.0,14134336.0,14164352.0,14194368.0,14224384.0,14254400.0,14075264.0,14105344.0,14135424.0,14165504.0,14195584.0,14225664.0,14255744.0,14285824.0,14106240.0,14136384.0,14166528.0,14196672.0,14226816.0,14256960.0,14287104.0,14317248.0,14137216.0,14167424.0,14197632.0,14227840.0,14258048.0,14288256.0,14318464.0,14348672.0,14168192.0,14198464.0,14228736.0,14259008.0,14289280.0,14319552.0,14349824.0,14380096.0,14199168.0,14229504.0,14259840.0,14290176.0,14320512.0,14350848.0,14381184.0,14411520.0,14230144.0,14260544.0,14290944.0,14321344.0,14351744.0,14382144.0,14412544.0,14442944.0,14261120.0,14291584.0,14322048.0,14352512.0,14382976.0,14413440.0,14443904.0,14474368.0,14292096.0,14322624.0,14353152.0,14383680.0,14414208.0,14444736.0,14475264.0,14505792.0,14323072.0,14353664.0,14384256.0,14414848.0,14445440.0,14476032.0,14506624.0,14537216.0,14354048.0,14384704.0,14415360.0,14446016.0,14476672.0,14507328.0,14537984.0,14568640.0,14385024.0,14415744.0,14446464.0,14477184.0,14507904.0,14538624.0,14569344.0,14600064.0,14416000.0,14446784.0,14477568.0,14508352.0,14539136.0,14569920.0,14600704.0,14631488.0,14446976.0,14477824.0,14508672.0,14539520.0,14570368.0,14601216.0,14632064.0,14662912.0,14477952.0,14508864.0,14539776.0,14570688.0,14601600.0,14632512.0,14663424.0,14694336.0,14508928.0,14539904.0,14570880.0,14601856.0,14632832.0,14663808.0,14694784.0,14725760.0,14539904.0,14570944.0,14601984.0,14633024.0,14664064.0,14695104.0,14726144.0,14757184.0,14570880.0,14601984.0,14633088.0,14664192.0,14695296.0,14726400.0,14757504.0,14788608.0,14601856.0,14633024.0,14664192.0,14695360.0,14726528.0,14757696.0,14788864.0,14820032.0,14632832.0,14664064.0,14695296.0,14726528.0,14757760.0,14788992.0,14820224.0,14851456.0,14663808.0,14695104.0,14726400.0,14757696.0,14788992.0,14820288.0,14851584.0,14882880.0,14694784.0,14726144.0,14757504.0,14788864.0,14820224.0,14851584.0,14882944.0,14914304.0,14725760.0,14757184.0,14788608.0,14820032.0,14851456.0,14882880.0,14914304.0,14945728.0,14756736.0,14788224.0,14819712.0,14851200.0,14882688.0,14914176.0,14945664.0,14977152.0,14787712.0,14819264.0,14850816.0,14882368.0,14913920.0,14945472.0,14977024.0,15008576.0,14818688.0,14850304.0,14881920.0,14913536.0,14945152.0,14976768.0,15008384.0,15040000.0,14849664.0,14881344.0,14913024.0,14944704.0,14976384.0,15008064.0,15039744.0,15071424.0,14880640.0,14912384.0,14944128.0,14975872.0,15007616.0,15039360.0,15071104.0,15102848.0,14911616.0,14943424.0,14975232.0,15007040.0,15038848.0,15070656.0,15102464.0,15134272.0,14942592.0,14974464.0,15006336.0,15038208.0,15070080.0,15101952.0,15133824.0,15165696.0,14973568.0,15005504.0,15037440.0,15069376.0,15101312.0,15133248.0,15165184.0,15197120.0,15004544.0,15036544.0,15068544.0,15100544.0,15132544.0,15164544.0,15196544.0,15228544.0,15035520.0,15067584.0,15099648.0,15131712.0,15163776.0,15195840.0,15227904.0,15259968.0,15066496.0,15098624.0,15130752.0,15162880.0,15195008.0,15227136.0,15259264.0,15291392.0,15097472.0,15129664.0,15161856.0,15194048.0,15226240.0,15258432.0,15290624.0,15322816.0,15128448.0,15160704.0,15192960.0,15225216.0,15257472.0,15289728.0,15321984.0,15354240.0,15159424.0,15191744.0,15224064.0,15256384.0,15288704.0,15321024.0,15353344.0,15385664.0,15190400.0,15222784.0,15255168.0,15287552.0,15319936.0,15352320.0,15384704.0,15417088.0,14992000.0,15023424.0,15054848.0,15086272.0,15117696.0,15149120.0,15180544.0,15211968.0,14785408.0,14815872.0,14846336.0,14876800.0,14907264.0,14937728.0,14968192.0,14998656.0,14816384.0,14846912.0,14877440.0,14907968.0,14938496.0,14969024.0,14999552.0,15030080.0,14847360.0,14877952.0,14908544.0,14939136.0,14969728.0,15000320.0,15030912.0,15061504.0,14878336.0,14908992.0,14939648.0,14970304.0,15000960.0,15031616.0,15062272.0,15092928.0,14909312.0,14940032.0,14970752.0,15001472.0,15032192.0,15062912.0,15093632.0,15124352.0,14940288.0,14971072.0,15001856.0,15032640.0,15063424.0,15094208.0,15124992.0,15155776.0,14971264.0,15002112.0,15032960.0,15063808.0,15094656.0,15125504.0,15156352.0,15187200.0,15002240.0,15033152.0,15064064.0,15094976.0,15125888.0,15156800.0,15187712.0,15218624.0,15033216.0,15064192.0,15095168.0,15126144.0,15157120.0,15188096.0,15219072.0,15250048.0,15064192.0,15095232.0,15126272.0,15157312.0,15188352.0,15219392.0,15250432.0,15281472.0,15095168.0,15126272.0,15157376.0,15188480.0,15219584.0,15250688.0,15281792.0,15312896.0,15126144.0,15157312.0,15188480.0,15219648.0,15250816.0,15281984.0,15313152.0,15344320.0,15157120.0,15188352.0,15219584.0,15250816.0,15282048.0,15313280.0,15344512.0,15375744.0,15188096.0,15219392.0,15250688.0,15281984.0,15313280.0,15344576.0,15375872.0,15407168.0,15219072.0,15250432.0,15281792.0,15313152.0,15344512.0,15375872.0,15407232.0,15438592.0,15250048.0,15281472.0,15312896.0,15344320.0,15375744.0,15407168.0,15438592.0,15470016.0,15281024.0,15312512.0,15344000.0,15375488.0,15406976.0,15438464.0,15469952.0,15501440.0,15312000.0,15343552.0,15375104.0,15406656.0,15438208.0,15469760.0,15501312.0,15532864.0,15342976.0,15374592.0,15406208.0,15437824.0,15469440.0,15501056.0,15532672.0,15564288.0,15373952.0,15405632.0,15437312.0,15468992.0,15500672.0,15532352.0,15564032.0,15595712.0,15404928.0,15436672.0,15468416.0,15500160.0,15531904.0,15563648.0,15595392.0,15627136.0,15435904.0,15467712.0,15499520.0,15531328.0,15563136.0,15594944.0,15626752.0,15658560.0,15466880.0,15498752.0,15530624.0,15562496.0,15594368.0,15626240.0,15658112.0,15689984.0,15497856.0,15529792.0,15561728.0,15593664.0,15625600.0,15657536.0,15689472.0,15721408.0,15528832.0,15560832.0,15592832.0,15624832.0,15656832.0,15688832.0,15720832.0,15752832.0,15559808.0,15591872.0,15623936.0,15656000.0,15688064.0,15720128.0,15752192.0,15784256.0,15590784.0,15622912.0,15655040.0,15687168.0,15719296.0,15751424.0,15783552.0,15815680.0,15621760.0,15653952.0,15686144.0,15718336.0,15750528.0,15782720.0,15814912.0,15847104.0,15652736.0,15684992.0,15717248.0,15749504.0,15781760.0,15814016.0,15846272.0,15878528.0,15683712.0,15716032.0,15748352.0,15780672.0,15812992.0,15845312.0,15877632.0,15909952.0,15714688.0,15747072.0,15779456.0,15811840.0,15844224.0,15876608.0,15908992.0,15941376.0,15745664.0,15778112.0,15810560.0,15843008.0,15875456.0,15907904.0,15940352.0,15972800.0,15776640.0,15809152.0,15841664.0,15874176.0,15906688.0,15939200.0,15971712.0,16004224.0,15807616.0,15840192.0,15872768.0,15905344.0,15937920.0,15970496.0,16003072.0,16035648.0,15838592.0,15871232.0,15903872.0,15936512.0,15969152.0,16001792.0,16034432.0,16067072.0,15869568.0,15902272.0,15934976.0,15967680.0,16000384.0,16033088.0,16065792.0,16098496.0,15900544.0,15933312.0,15966080.0,15998848.0,16031616.0,16064384.0,16097152.0,16129920.0,15931520.0,15964352.0,15997184.0,16030016.0,16062848.0,16095680.0,16128512.0,16161344.0,15208832.0,15240704.0,15272576.0,15304448.0,15336320.0,15368192.0,15400064.0,15431936.0,14477952.0,14508864.0,14539776.0,14570688.0,14601600.0,14632512.0,14663424.0,14694336.0,14508928.0,14539904.0,14570880.0,14601856.0,14632832.0,14663808.0,14694784.0,14725760.0,14539904.0,14570944.0,14601984.0,14633024.0,14664064.0,14695104.0,14726144.0,14757184.0,14570880.0,14601984.0,14633088.0,14664192.0,14695296.0,14726400.0,14757504.0,14788608.0,14601856.0,14633024.0,14664192.0,14695360.0,14726528.0,14757696.0,14788864.0,14820032.0,14632832.0,14664064.0,14695296.0,14726528.0,14757760.0,14788992.0,14820224.0,14851456.0,14663808.0,14695104.0,14726400.0,14757696.0,14788992.0,14820288.0,14851584.0,14882880.0,14694784.0,14726144.0,14757504.0,14788864.0,14820224.0,14851584.0,14882944.0,14914304.0,14725760.0,14757184.0,14788608.0,14820032.0,14851456.0,14882880.0,14914304.0,14945728.0,14756736.0,14788224.0,14819712.0,14851200.0,14882688.0,14914176.0,14945664.0,14977152.0,14787712.0,14819264.0,14850816.0,14882368.0,14913920.0,14945472.0,14977024.0,15008576.0,14818688.0,14850304.0,14881920.0,14913536.0,14945152.0,14976768.0,15008384.0,15040000.0,14849664.0,14881344.0,14913024.0,14944704.0,14976384.0,15008064.0,15039744.0,15071424.0,14880640.0,14912384.0,14944128.0,14975872.0,15007616.0,15039360.0,15071104.0,15102848.0,14911616.0,14943424.0,14975232.0,15007040.0,15038848.0,15070656.0,15102464.0,15134272.0,14942592.0,14974464.0,15006336.0,15038208.0,15070080.0,15101952.0,15133824.0,15165696.0,14973568.0,15005504.0,15037440.0,15069376.0,15101312.0,15133248.0,15165184.0,15197120.0,15004544.0,15036544.0,15068544.0,15100544.0,15132544.0,15164544.0,15196544.0,15228544.0,15035520.0,15067584.0,15099648.0,15131712.0,15163776.0,15195840.0,15227904.0,15259968.0,15066496.0,15098624.0,15130752.0,15162880.0,15195008.0,15227136.0,15259264.0,15291392.0,15097472.0,15129664.0,15161856.0,15194048.0,15226240.0,15258432.0,15290624.0,15322816.0,15128448.0,15160704.0,15192960.0,15225216.0,15257472.0,15289728.0,15321984.0,15354240.0,15159424.0,15191744.0,15224064.0,15256384.0,15288704.0,15321024.0,15353344.0,15385664.0,15190400.0,15222784.0,15255168.0,15287552.0,15319936.0,15352320.0,15384704.0,15417088.0,15221376.0,15253824.0,15286272.0,15318720.0,15351168.0,15383616.0,15416064.0,15448512.0,15252352.0,15284864.0,15317376.0,15349888.0,15382400.0,15414912.0,15447424.0,15479936.0,15283328.0,15315904.0,15348480.0,15381056.0,15413632.0,15446208.0,15478784.0,15511360.0,15314304.0,15346944.0,15379584.0,15412224.0,15444864.0,15477504.0,15510144.0,15542784.0,15345280.0,15377984.0,15410688.0,15443392.0,15476096.0,15508800.0,15541504.0,15574208.0,15376256.0,15409024.0,15441792.0,15474560.0,15507328.0,15540096.0,15572864.0,15605632.0,15407232.0,15440064.0,15472896.0,15505728.0,15538560.0,15571392.0,15604224.0,15637056.0,15438208.0,15471104.0,15504000.0,15536896.0,15569792.0,15602688.0,15635584.0,15668480.0,15469184.0,15502144.0,15535104.0,15568064.0,15601024.0,15633984.0,15666944.0,15699904.0,15500160.0,15533184.0,15566208.0,15599232.0,15632256.0,15665280.0,15698304.0,15731328.0,15531136.0,15564224.0,15597312.0,15630400.0,15663488.0,15696576.0,15729664.0,15762752.0,15562112.0,15595264.0,15628416.0,15661568.0,15694720.0,15727872.0,15761024.0,15794176.0,15593088.0,15626304.0,15659520.0,15692736.0,15725952.0,15759168.0,15792384.0,15825600.0,15624064.0,15657344.0,15690624.0,15723904.0,15757184.0,15790464.0,15823744.0,15857024.0,15425664.0,15457984.0,15490304.0,15522624.0,15554944.0,15587264.0,15619584.0,15651904.0,15219072.0,15250432.0,15281792.0,15313152.0,15344512.0,15375872.0,15407232.0,15438592.0,15250048.0,15281472.0,15312896.0,15344320.0,15375744.0,15407168.0,15438592.0,15470016.0,15281024.0,15312512.0,15344000.0,15375488.0,15406976.0,15438464.0,15469952.0,15501440.0,15312000.0,15343552.0,15375104.0,15406656.0,15438208.0,15469760.0,15501312.0,15532864.0,15342976.0,15374592.0,15406208.0,15437824.0,15469440.0,15501056.0,15532672.0,15564288.0,15373952.0,15405632.0,15437312.0,15468992.0,15500672.0,15532352.0,15564032.0,15595712.0,15404928.0,15436672.0,15468416.0,15500160.0,15531904.0,15563648.0,15595392.0,15627136.0,15435904.0,15467712.0,15499520.0,15531328.0,15563136.0,15594944.0,15626752.0,15658560.0,15466880.0,15498752.0,15530624.0,15562496.0,15594368.0,15626240.0,15658112.0,15689984.0,15497856.0,15529792.0,15561728.0,15593664.0,15625600.0,15657536.0,15689472.0,15721408.0,15528832.0,15560832.0,15592832.0,15624832.0,15656832.0,15688832.0,15720832.0,15752832.0,15559808.0,15591872.0,15623936.0,15656000.0,15688064.0,15720128.0,15752192.0,15784256.0,15590784.0,15622912.0,15655040.0,15687168.0,15719296.0,15751424.0,15783552.0,15815680.0,15621760.0,15653952.0,15686144.0,15718336.0,15750528.0,15782720.0,15814912.0,15847104.0,15652736.0,15684992.0,15717248.0,15749504.0,15781760.0,15814016.0,15846272.0,15878528.0,15683712.0,15716032.0,15748352.0,15780672.0,15812992.0,15845312.0,15877632.0,15909952.0,15714688.0,15747072.0,15779456.0,15811840.0,15844224.0,15876608.0,15908992.0,15941376.0,15745664.0,15778112.0,15810560.0,15843008.0,15875456.0,15907904.0,15940352.0,15972800.0,15776640.0,15809152.0,15841664.0,15874176.0,15906688.0,15939200.0,15971712.0,16004224.0,15807616.0,15840192.0,15872768.0,15905344.0,15937920.0,15970496.0,16003072.0,16035648.0,15838592.0,15871232.0,15903872.0,15936512.0,15969152.0,16001792.0,16034432.0,16067072.0,15869568.0,15902272.0,15934976.0,15967680.0,16000384.0,16033088.0,16065792.0,16098496.0,15900544.0,15933312.0,15966080.0,15998848.0,16031616.0,16064384.0,16097152.0,16129920.0,15931520.0,15964352.0,15997184.0,16030016.0,16062848.0,16095680.0,16128512.0,16161344.0,15962496.0,15995392.0,16028288.0,16061184.0,16094080.0,16126976.0,16159872.0,16192768.0,15993472.0,16026432.0,16059392.0,16092352.0,16125312.0,16158272.0,16191232.0,16224192.0,16024448.0,16057472.0,16090496.0,16123520.0,16156544.0,16189568.0,16222592.0,16255616.0,16055424.0,16088512.0,16121600.0,16154688.0,16187776.0,16220864.0,16253952.0,16287040.0,16086400.0,16119552.0,16152704.0,16185856.0,16219008.0,16252160.0,16285312.0,16318464.0,16117376.0,16150592.0,16183808.0,16217024.0,16250240.0,16283456.0,16316672.0,16349888.0,16148352.0,16181632.0,16214912.0,16248192.0,16281472.0,16314752.0,16348032.0,16381312.0,16179328.0,16212672.0,16246016.0,16279360.0,16312704.0,16346048.0,16379392.0,16412736.0,16210304.0,16243712.0,16277120.0,16310528.0,16343936.0,16377344.0,16410752.0,16444160.0,16241280.0,16274752.0,16308224.0,16341696.0,16375168.0,16408640.0,16442112.0,16475584.0,16272256.0,16305792.0,16339328.0,16372864.0,16406400.0,16439936.0,16473472.0,16507008.0,16303232.0,16336832.0,16370432.0,16404032.0,16437632.0,16471232.0,16504832.0,16538432.0,16334208.0,16367872.0,16401536.0,16435200.0,16468864.0,16502528.0,16536192.0,16569856.0,16365184.0,16398912.0,16432640.0,16466368.0,16500096.0,16533824.0,16567552.0,16601280.0,15642496.0,15675264.0,15708032.0,15740800.0,15773568.0,15806336.0,15839104.0,15871872.0,14911616.0,14943424.0,14975232.0,15007040.0,15038848.0,15070656.0,15102464.0,15134272.0,7527296.0,7543216.0,7559136.0,7575056.0,7590976.0,7606896.0,7622816.0,7638736.0,8115456.0,8131248.0,8147040.0,8162832.0,8178624.0,8194416.0,8210208.0,8226000.0,16373120.0,16404736.0,16436352.0,16467968.0,16499584.0,16531200.0,16562816.0,16594432.0,16405120.0,16436800.0,16468480.0,16500160.0,16531840.0,16563520.0,16595200.0,16626880.0,16437120.0,16468864.0,16500608.0,16532352.0,16564096.0,16595840.0,16627584.0,16659328.0,16469120.0,16500928.0,16532736.0,16564544.0,16596352.0,16628160.0,16659968.0,16691776.0,16501120.0,16532992.0,16564864.0,16596736.0,16628608.0,16660480.0,16692352.0,16724224.0,16533120.0,16565056.0,16596992.0,16628928.0,16660864.0,16692800.0,16724736.0,16756672.0,16565120.0,16597120.0,16629120.0,16661120.0,16693120.0,16725120.0,16757120.0,16789120.0,16597120.0,16629184.0,16661248.0,16693312.0,16725376.0,16757440.0,16789504.0,16821568.0,16629120.0,16661248.0,16693376.0,16725504.0,16757632.0,16789760.0,16821888.0,16854016.0,16661120.0,16693312.0,16725504.0,16757696.0,16789888.0,16822080.0,16854272.0,16886464.0,16693120.0,16725376.0,16757632.0,16789888.0,16822144.0,16854400.0,16886656.0,16918912.0,16725120.0,16757440.0,16789760.0,16822080.0,16854400.0,16886720.0,16919040.0,16951360.0,16757120.0,16789504.0,16821888.0,16854272.0,16886656.0,16919040.0,16951424.0,16983808.0,16789120.0,16821568.0,16854016.0,16886464.0,16918912.0,16951360.0,16983808.0,17016256.0,16821120.0,16853632.0,16886144.0,16918656.0,16951168.0,16983680.0,17016192.0,17048704.0,16853120.0,16885696.0,16918272.0,16950848.0,16983424.0,17016000.0,17048576.0,17081152.0,16885120.0,16917760.0,16950400.0,16983040.0,17015680.0,17048320.0,17080960.0,17113600.0,16917120.0,16949824.0,16982528.0,17015232.0,17047936.0,17080640.0,17113344.0,17146048.0,16949120.0,16981888.0,17014656.0,17047424.0,17080192.0,17112960.0,17145728.0,17178496.0,16981120.0,17013952.0,17046784.0,17079616.0,17112448.0,17145280.0,17178112.0,17210944.0,17013120.0,17046016.0,17078912.0,17111808.0,17144704.0,17177600.0,17210496.0,17243392.0,17045120.0,17078080.0,17111040.0,17144000.0,17176960.0,17209920.0,17242880.0,17275840.0,17077120.0,17110144.0,17143168.0,17176192.0,17209216.0,17242240.0,17275264.0,17308288.0,17109120.0,17142208.0,17175296.0,17208384.0,17241472.0,17274560.0,17307648.0,17340736.0,17141120.0,17174272.0,17207424.0,17240576.0,17273728.0,17306880.0,17340032.0,17373184.0,17173120.0,17206336.0,17239552.0,17272768.0,17305984.0,17339200.0,17372416.0,17405632.0,17205120.0,17238400.0,17271680.0,17304960.0,17338240.0,17371520.0,17404800.0,17438080.0,17237120.0,17270464.0,17303808.0,17337152.0,17370496.0,17403840.0,17437184.0,17470528.0,17269120.0,17302528.0,17335936.0,17369344.0,17402752.0,17436160.0,17469568.0,17502976.0,17301120.0,17334592.0,17368064.0,17401536.0,17435008.0,17468480.0,17501952.0,17535424.0,17333120.0,17366656.0,17400192.0,17433728.0,17467264.0,17500800.0,17534336.0,17567872.0,17365120.0,17398720.0,17432320.0,17465920.0,17499520.0,17533120.0,17566720.0,17600320.0,17397120.0,17430784.0,17464448.0,17498112.0,17531776.0,17565440.0,17599104.0,17632768.0,17429120.0,17462848.0,17496576.0,17530304.0,17564032.0,17597760.0,17631488.0,17665216.0,17461120.0,17494912.0,17528704.0,17562496.0,17596288.0,17630080.0,17663872.0,17697664.0,17493120.0,17526976.0,17560832.0,17594688.0,17628544.0,17662400.0,17696256.0,17730112.0,16755072.0,16787968.0,16820864.0,16853760.0,16886656.0,16919552.0,16952448.0,16985344.0,16008832.0,16040768.0,16072704.0,16104640.0,16136576.0,16168512.0,16200448.0,16232384.0,15270784.0,15301760.0,15332736.0,15363712.0,15394688.0,15425664.0,15456640.0,15487616.0,14524544.0,14554560.0,14584576.0,14614592.0,14644608.0,14674624.0,14704640.0,14734656.0,14556544.0,14586624.0,14616704.0,14646784.0,14676864.0,14706944.0,14737024.0,14767104.0,14588544.0,14618688.0,14648832.0,14678976.0,14709120.0,14739264.0,14769408.0,14799552.0,14620544.0,14650752.0,14680960.0,14711168.0,14741376.0,14771584.0,14801792.0,14832000.0,14652544.0,14682816.0,14713088.0,14743360.0,14773632.0,14803904.0,14834176.0,14864448.0,14684544.0,14714880.0,14745216.0,14775552.0,14805888.0,14836224.0,14866560.0,14896896.0,14716544.0,14746944.0,14777344.0,14807744.0,14838144.0,14868544.0,14898944.0,14929344.0,14748544.0,14779008.0,14809472.0,14839936.0,14870400.0,14900864.0,14931328.0,14961792.0,14780544.0,14811072.0,14841600.0,14872128.0,14902656.0,14933184.0,14963712.0,14994240.0,14812544.0,14843136.0,14873728.0,14904320.0,14934912.0,14965504.0,14996096.0,15026688.0,14844544.0,14875200.0,14905856.0,14936512.0,14967168.0,14997824.0,15028480.0,15059136.0,14876544.0,14907264.0,14937984.0,14968704.0,14999424.0,15030144.0,15060864.0,15091584.0,14908544.0,14939328.0,14970112.0,15000896.0,15031680.0,15062464.0,15093248.0,15124032.0,14940544.0,14971392.0,15002240.0,15033088.0,15063936.0,15094784.0,15125632.0,15156480.0,14972544.0,15003456.0,15034368.0,15065280.0,15096192.0,15127104.0,15158016.0,15188928.0,15004544.0,15035520.0,15066496.0,15097472.0,15128448.0,15159424.0,15190400.0,15221376.0,15036544.0,15067584.0,15098624.0,15129664.0,15160704.0,15191744.0,15222784.0,15253824.0,15068544.0,15099648.0,15130752.0,15161856.0,15192960.0,15224064.0,15255168.0,15286272.0,15100544.0,15131712.0,15162880.0,15194048.0,15225216.0,15256384.0,15287552.0,15318720.0,15132544.0,15163776.0,15195008.0,15226240.0,15257472.0,15288704.0,15319936.0,15351168.0,15164544.0,15195840.0,15227136.0,15258432.0,15289728.0,15321024.0,15352320.0,15383616.0,15196544.0,15227904.0,15259264.0,15290624.0,15321984.0,15353344.0,15384704.0,15416064.0,15228544.0,15259968.0,15291392.0,15322816.0,15354240.0,15385664.0,15417088.0,15448512.0,15260544.0,15292032.0,15323520.0,15355008.0,15386496.0,15417984.0,15449472.0,15480960.0,15292544.0,15324096.0,15355648.0,15387200.0,15418752.0,15450304.0,15481856.0,15513408.0,15324544.0,15356160.0,15387776.0,15419392.0,15451008.0,15482624.0,15514240.0,15545856.0,15356544.0,15388224.0,15419904.0,15451584.0,15483264.0,15514944.0,15546624.0,15578304.0,15388544.0,15420288.0,15452032.0,15483776.0,15515520.0,15547264.0,15579008.0,15610752.0,15420544.0,15452352.0,15484160.0,15515968.0,15547776.0,15579584.0,15611392.0,15643200.0,15452544.0,15484416.0,15516288.0,15548160.0,15580032.0,15611904.0,15643776.0,15675648.0,15484544.0,15516480.0,15548416.0,15580352.0,15612288.0,15644224.0,15676160.0,15708096.0,15516544.0,15548544.0,15580544.0,15612544.0,15644544.0,15676544.0,15708544.0,15740544.0,15548544.0,15580608.0,15612672.0,15644736.0,15676800.0,15708864.0,15740928.0,15772992.0,15580544.0,15612672.0,15644800.0,15676928.0,15709056.0,15741184.0,15773312.0,15805440.0,15612544.0,15644736.0,15676928.0,15709120.0,15741312.0,15773504.0,15805696.0,15837888.0,15644544.0,15676800.0,15709056.0,15741312.0,15773568.0,15805824.0,15838080.0,15870336.0,15676544.0,15708864.0,15741184.0,15773504.0,15805824.0,15838144.0,15870464.0,15902784.0,15708544.0,15740928.0,15773312.0,15805696.0,15838080.0,15870464.0,15902848.0,15935232.0,15494784.0,15526208.0,15557632.0,15589056.0,15620480.0,15651904.0,15683328.0,15714752.0,15272832.0,15303296.0,15333760.0,15364224.0,15394688.0,15425152.0,15455616.0,15486080.0,15304832.0,15335360.0,15365888.0,15396416.0,15426944.0,15457472.0,15488000.0,15518528.0,15336832.0,15367424.0,15398016.0,15428608.0,15459200.0,15489792.0,15520384.0,15550976.0,15368832.0,15399488.0,15430144.0,15460800.0,15491456.0,15522112.0,15552768.0,15583424.0,15400832.0,15431552.0,15462272.0,15492992.0,15523712.0,15554432.0,15585152.0,15615872.0,15432832.0,15463616.0,15494400.0,15525184.0,15555968.0,15586752.0,15617536.0,15648320.0,15464832.0,15495680.0,15526528.0,15557376.0,15588224.0,15619072.0,15649920.0,15680768.0,15496832.0,15527744.0,15558656.0,15589568.0,15620480.0,15651392.0,15682304.0,15713216.0,15528832.0,15559808.0,15590784.0,15621760.0,15652736.0,15683712.0,15714688.0,15745664.0,15560832.0,15591872.0,15622912.0,15653952.0,15684992.0,15716032.0,15747072.0,15778112.0,15592832.0,15623936.0,15655040.0,15686144.0,15717248.0,15748352.0,15779456.0,15810560.0,15624832.0,15656000.0,15687168.0,15718336.0,15749504.0,15780672.0,15811840.0,15843008.0,15656832.0,15688064.0,15719296.0,15750528.0,15781760.0,15812992.0,15844224.0,15875456.0,15688832.0,15720128.0,15751424.0,15782720.0,15814016.0,15845312.0,15876608.0,15907904.0,15720832.0,15752192.0,15783552.0,15814912.0,15846272.0,15877632.0,15908992.0,15940352.0,15752832.0,15784256.0,15815680.0,15847104.0,15878528.0,15909952.0,15941376.0,15972800.0,15784832.0,15816320.0,15847808.0,15879296.0,15910784.0,15942272.0,15973760.0,16005248.0,15816832.0,15848384.0,15879936.0,15911488.0,15943040.0,15974592.0,16006144.0,16037696.0,15848832.0,15880448.0,15912064.0,15943680.0,15975296.0,16006912.0,16038528.0,16070144.0,15880832.0,15912512.0,15944192.0,15975872.0,16007552.0,16039232.0,16070912.0,16102592.0,15912832.0,15944576.0,15976320.0,16008064.0,16039808.0,16071552.0,16103296.0,16135040.0,15944832.0,15976640.0,16008448.0,16040256.0,16072064.0,16103872.0,16135680.0,16167488.0,15976832.0,16008704.0,16040576.0,16072448.0,16104320.0,16136192.0,16168064.0,16199936.0,16008832.0,16040768.0,16072704.0,16104640.0,16136576.0,16168512.0,16200448.0,16232384.0,16040832.0,16072832.0,16104832.0,16136832.0,16168832.0,16200832.0,16232832.0,16264832.0,16072832.0,16104896.0,16136960.0,16169024.0,16201088.0,16233152.0,16265216.0,16297280.0,16104832.0,16136960.0,16169088.0,16201216.0,16233344.0,16265472.0,16297600.0,16329728.0,16136832.0,16169024.0,16201216.0,16233408.0,16265600.0,16297792.0,16329984.0,16362176.0,16168832.0,16201088.0,16233344.0,16265600.0,16297856.0,16330112.0,16362368.0,16394624.0,16200832.0,16233152.0,16265472.0,16297792.0,16330112.0,16362432.0,16394752.0,16427072.0,16232832.0,16265216.0,16297600.0,16329984.0,16362368.0,16394752.0,16427136.0,16459520.0,16264832.0,16297280.0,16329728.0,16362176.0,16394624.0,16427072.0,16459520.0,16491968.0,16296832.0,16329344.0,16361856.0,16394368.0,16426880.0,16459392.0,16491904.0,16524416.0,16328832.0,16361408.0,16393984.0,16426560.0,16459136.0,16491712.0,16524288.0,16556864.0,16360832.0,16393472.0,16426112.0,16458752.0,16491392.0,16524032.0,16556672.0,16589312.0,16392832.0,16425536.0,16458240.0,16490944.0,16523648.0,16556352.0,16589056.0,16621760.0,16424832.0,16457600.0,16490368.0,16523136.0,16555904.0,16588672.0,16621440.0,16654208.0,16456832.0,16489664.0,16522496.0,16555328.0,16588160.0,16620992.0,16653824.0,16686656.0,15718784.0,15750656.0,15782528.0,15814400.0,15846272.0,15878144.0,15910016.0,15941888.0,14972544.0,15003456.0,15034368.0,15065280.0,15096192.0,15127104.0,15158016.0,15188928.0,15004544.0,15035520.0,15066496.0,15097472.0,15128448.0,15159424.0,15190400.0,15221376.0,15036544.0,15067584.0,15098624.0,15129664.0,15160704.0,15191744.0,15222784.0,15253824.0,15068544.0,15099648.0,15130752.0,15161856.0,15192960.0,15224064.0,15255168.0,15286272.0,15100544.0,15131712.0,15162880.0,15194048.0,15225216.0,15256384.0,15287552.0,15318720.0,15132544.0,15163776.0,15195008.0,15226240.0,15257472.0,15288704.0,15319936.0,15351168.0,15164544.0,15195840.0,15227136.0,15258432.0,15289728.0,15321024.0,15352320.0,15383616.0,15196544.0,15227904.0,15259264.0,15290624.0,15321984.0,15353344.0,15384704.0,15416064.0,15228544.0,15259968.0,15291392.0,15322816.0,15354240.0,15385664.0,15417088.0,15448512.0,15260544.0,15292032.0,15323520.0,15355008.0,15386496.0,15417984.0,15449472.0,15480960.0,15292544.0,15324096.0,15355648.0,15387200.0,15418752.0,15450304.0,15481856.0,15513408.0,15324544.0,15356160.0,15387776.0,15419392.0,15451008.0,15482624.0,15514240.0,15545856.0,15356544.0,15388224.0,15419904.0,15451584.0,15483264.0,15514944.0,15546624.0,15578304.0,15388544.0,15420288.0,15452032.0,15483776.0,15515520.0,15547264.0,15579008.0,15610752.0,15420544.0,15452352.0,15484160.0,15515968.0,15547776.0,15579584.0,15611392.0,15643200.0,15452544.0,15484416.0,15516288.0,15548160.0,15580032.0,15611904.0,15643776.0,15675648.0,15484544.0,15516480.0,15548416.0,15580352.0,15612288.0,15644224.0,15676160.0,15708096.0,15516544.0,15548544.0,15580544.0,15612544.0,15644544.0,15676544.0,15708544.0,15740544.0,15548544.0,15580608.0,15612672.0,15644736.0,15676800.0,15708864.0,15740928.0,15772992.0,15580544.0,15612672.0,15644800.0,15676928.0,15709056.0,15741184.0,15773312.0,15805440.0,15612544.0,15644736.0,15676928.0,15709120.0,15741312.0,15773504.0,15805696.0,15837888.0,15644544.0,15676800.0,15709056.0,15741312.0,15773568.0,15805824.0,15838080.0,15870336.0,15676544.0,15708864.0,15741184.0,15773504.0,15805824.0,15838144.0,15870464.0,15902784.0,15708544.0,15740928.0,15773312.0,15805696.0,15838080.0,15870464.0,15902848.0,15935232.0,15740544.0,15772992.0,15805440.0,15837888.0,15870336.0,15902784.0,15935232.0,15967680.0,15772544.0,15805056.0,15837568.0,15870080.0,15902592.0,15935104.0,15967616.0,16000128.0,15804544.0,15837120.0,15869696.0,15902272.0,15934848.0,15967424.0,16000000.0,16032576.0,15836544.0,15869184.0,15901824.0,15934464.0,15967104.0,15999744.0,16032384.0,16065024.0,15868544.0,15901248.0,15933952.0,15966656.0,15999360.0,16032064.0,16064768.0,16097472.0,15900544.0,15933312.0,15966080.0,15998848.0,16031616.0,16064384.0,16097152.0,16129920.0,15932544.0,15965376.0,15998208.0,16031040.0,16063872.0,16096704.0,16129536.0,16162368.0,15964544.0,15997440.0,16030336.0,16063232.0,16096128.0,16129024.0,16161920.0,16194816.0,15996544.0,16029504.0,16062464.0,16095424.0,16128384.0,16161344.0,16194304.0,16227264.0,16028544.0,16061568.0,16094592.0,16127616.0,16160640.0,16193664.0,16226688.0,16259712.0,16060544.0,16093632.0,16126720.0,16159808.0,16192896.0,16225984.0,16259072.0,16292160.0,16092544.0,16125696.0,16158848.0,16192000.0,16225152.0,16258304.0,16291456.0,16324608.0,16124544.0,16157760.0,16190976.0,16224192.0,16257408.0,16290624.0,16323840.0,16357056.0,16156544.0,16189824.0,16223104.0,16256384.0,16289664.0,16322944.0,16356224.0,16389504.0,15942784.0,15975104.0,16007424.0,16039744.0,16072064.0,16104384.0,16136704.0,16169024.0,15720832.0,15752192.0,15783552.0,15814912.0,15846272.0,15877632.0,15908992.0,15940352.0,15752832.0,15784256.0,15815680.0,15847104.0,15878528.0,15909952.0,15941376.0,15972800.0,15784832.0,15816320.0,15847808.0,15879296.0,15910784.0,15942272.0,15973760.0,16005248.0,15816832.0,15848384.0,15879936.0,15911488.0,15943040.0,15974592.0,16006144.0,16037696.0,15848832.0,15880448.0,15912064.0,15943680.0,15975296.0,16006912.0,16038528.0,16070144.0,15880832.0,15912512.0,15944192.0,15975872.0,16007552.0,16039232.0,16070912.0,16102592.0,15912832.0,15944576.0,15976320.0,16008064.0,16039808.0,16071552.0,16103296.0,16135040.0,15944832.0,15976640.0,16008448.0,16040256.0,16072064.0,16103872.0,16135680.0,16167488.0,15976832.0,16008704.0,16040576.0,16072448.0,16104320.0,16136192.0,16168064.0,16199936.0,16008832.0,16040768.0,16072704.0,16104640.0,16136576.0,16168512.0,16200448.0,16232384.0,16040832.0,16072832.0,16104832.0,16136832.0,16168832.0,16200832.0,16232832.0,16264832.0,16072832.0,16104896.0,16136960.0,16169024.0,16201088.0,16233152.0,16265216.0,16297280.0,16104832.0,16136960.0,16169088.0,16201216.0,16233344.0,16265472.0,16297600.0,16329728.0,16136832.0,16169024.0,16201216.0,16233408.0,16265600.0,16297792.0,16329984.0,16362176.0,16168832.0,16201088.0,16233344.0,16265600.0,16297856.0,16330112.0,16362368.0,16394624.0,16200832.0,16233152.0,16265472.0,16297792.0,16330112.0,16362432.0,16394752.0,16427072.0,16232832.0,16265216.0,16297600.0,16329984.0,16362368.0,16394752.0,16427136.0,16459520.0,16264832.0,16297280.0,16329728.0,16362176.0,16394624.0,16427072.0,16459520.0,16491968.0,16296832.0,16329344.0,16361856.0,16394368.0,16426880.0,16459392.0,16491904.0,16524416.0,16328832.0,16361408.0,16393984.0,16426560.0,16459136.0,16491712.0,16524288.0,16556864.0,16360832.0,16393472.0,16426112.0,16458752.0,16491392.0,16524032.0,16556672.0,16589312.0,16392832.0,16425536.0,16458240.0,16490944.0,16523648.0,16556352.0,16589056.0,16621760.0,16424832.0,16457600.0,16490368.0,16523136.0,16555904.0,16588672.0,16621440.0,16654208.0,16456832.0,16489664.0,16522496.0,16555328.0,16588160.0,16620992.0,16653824.0,16686656.0,16488832.0,16521728.0,16554624.0,16587520.0,16620416.0,16653312.0,16686208.0,16719104.0,16520832.0,16553792.0,16586752.0,16619712.0,16652672.0,16685632.0,16718592.0,16751552.0,16552832.0,16585856.0,16618880.0,16651904.0,16684928.0,16717952.0,16750976.0,16784000.0,16584832.0,16617920.0,16651008.0,16684096.0,16717184.0,16750272.0,16783360.0,16816448.0,16616832.0,16649984.0,16683136.0,16716288.0,16749440.0,16782592.0,16815744.0,16848896.0,16648832.0,16682048.0,16715264.0,16748480.0,16781696.0,16814912.0,16848128.0,16881344.0,16680832.0,16714112.0,16747392.0,16780672.0,16813952.0,16847232.0,16880512.0,16913792.0,16712832.0,16746176.0,16779520.0,16812864.0,16846208.0,16879552.0,16912896.0,16946240.0,16744832.0,16778240.0,16811648.0,16845056.0,16878464.0,16911872.0,16945280.0,16978688.0,16776832.0,16810304.0,16843776.0,16877248.0,16910720.0,16944192.0,16977664.0,17011136.0,16808832.0,16842368.0,16875904.0,16909440.0,16942976.0,16976512.0,17010048.0,17043584.0,16840832.0,16874432.0,16908032.0,16941632.0,16975232.0,17008832.0,17042432.0,17076032.0,16872832.0,16906496.0,16940160.0,16973824.0,17007488.0,17041152.0,17074816.0,17108480.0,16904832.0,16938560.0,16972288.0,17006016.0,17039744.0,17073472.0,17107200.0,17140928.0,16166784.0,16199552.0,16232320.0,16265088.0,16297856.0,16330624.0,16363392.0,16396160.0,15420544.0,15452352.0,15484160.0,15515968.0,15547776.0,15579584.0,15611392.0,15643200.0,7782016.0,7797936.0,7813856.0,7829776.0,7845696.0,7861616.0,7877536.0,7893456.0,8368128.0,8383920.0,8399712.0,8415504.0,8431296.0,8447088.0,8462880.0,8478672.0,16878976.0,16910592.0,16942208.0,16973824.0,17005440.0,17037056.0,17068672.0,17100288.0,16912000.0,16943680.0,16975360.0,17007040.0,17038720.0,17070400.0,17102080.0,17133760.0,16945024.0,16976768.0,17008512.0,17040256.0,17072000.0,17103744.0,17135488.0,17167232.0,16978048.0,17009856.0,17041664.0,17073472.0,17105280.0,17137088.0,17168896.0,17200704.0,17011072.0,17042944.0,17074816.0,17106688.0,17138560.0,17170432.0,17202304.0,17234176.0,17044096.0,17076032.0,17107968.0,17139904.0,17171840.0,17203776.0,17235712.0,17267648.0,17077120.0,17109120.0,17141120.0,17173120.0,17205120.0,17237120.0,17269120.0,17301120.0,17110144.0,17142208.0,17174272.0,17206336.0,17238400.0,17270464.0,17302528.0,17334592.0,17143168.0,17175296.0,17207424.0,17239552.0,17271680.0,17303808.0,17335936.0,17368064.0,17176192.0,17208384.0,17240576.0,17272768.0,17304960.0,17337152.0,17369344.0,17401536.0,17209216.0,17241472.0,17273728.0,17305984.0,17338240.0,17370496.0,17402752.0,17435008.0,17242240.0,17274560.0,17306880.0,17339200.0,17371520.0,17403840.0,17436160.0,17468480.0,17275264.0,17307648.0,17340032.0,17372416.0,17404800.0,17437184.0,17469568.0,17501952.0,17308288.0,17340736.0,17373184.0,17405632.0,17438080.0,17470528.0,17502976.0,17535424.0,17341312.0,17373824.0,17406336.0,17438848.0,17471360.0,17503872.0,17536384.0,17568896.0,17374336.0,17406912.0,17439488.0,17472064.0,17504640.0,17537216.0,17569792.0,17602368.0,17407360.0,17440000.0,17472640.0,17505280.0,17537920.0,17570560.0,17603200.0,17635840.0,17440384.0,17473088.0,17505792.0,17538496.0,17571200.0,17603904.0,17636608.0,17669312.0,17473408.0,17506176.0,17538944.0,17571712.0,17604480.0,17637248.0,17670016.0,17702784.0,17506432.0,17539264.0,17572096.0,17604928.0,17637760.0,17670592.0,17703424.0,17736256.0,17539456.0,17572352.0,17605248.0,17638144.0,17671040.0,17703936.0,17736832.0,17769728.0,17572480.0,17605440.0,17638400.0,17671360.0,17704320.0,17737280.0,17770240.0,17803200.0,17605504.0,17638528.0,17671552.0,17704576.0,17737600.0,17770624.0,17803648.0,17836672.0,17638528.0,17671616.0,17704704.0,17737792.0,17770880.0,17803968.0,17837056.0,17870144.0,17671552.0,17704704.0,17737856.0,17771008.0,17804160.0,17837312.0,17870464.0,17903616.0,17704576.0,17737792.0,17771008.0,17804224.0,17837440.0,17870656.0,17903872.0,17937088.0,17737600.0,17770880.0,17804160.0,17837440.0,17870720.0,17904000.0,17937280.0,17970560.0,17770624.0,17803968.0,17837312.0,17870656.0,17904000.0,17937344.0,17970688.0,18004032.0,17803648.0,17837056.0,17870464.0,17903872.0,17937280.0,17970688.0,18004096.0,18037504.0,17836672.0,17870144.0,17903616.0,17937088.0,17970560.0,18004032.0,18037504.0,18070976.0,17869696.0,17903232.0,17936768.0,17970304.0,18003840.0,18037376.0,18070912.0,18104448.0,17902720.0,17936320.0,17969920.0,18003520.0,18037120.0,18070720.0,18104320.0,18137920.0,17935744.0,17969408.0,18003072.0,18036736.0,18070400.0,18104064.0,18137728.0,18171392.0,17968768.0,18002496.0,18036224.0,18069952.0,18103680.0,18137408.0,18171136.0,18204864.0,18001792.0,18035584.0,18069376.0,18103168.0,18136960.0,18170752.0,18204544.0,18238336.0,18034816.0,18068672.0,18102528.0,18136384.0,18170240.0,18204096.0,18237952.0,18271808.0,17281408.0,17314304.0,17347200.0,17380096.0,17412992.0,17445888.0,17478784.0,17511680.0,16519808.0,16551744.0,16583680.0,16615616.0,16647552.0,16679488.0,16711424.0,16743360.0,15766400.0,15797376.0,15828352.0,15859328.0,15890304.0,15921280.0,15952256.0,15983232.0,15004800.0,15034816.0,15064832.0,15094848.0,15124864.0,15154880.0,15184896.0,15214912.0,15037824.0,15067904.0,15097984.0,15128064.0,15158144.0,15188224.0,15218304.0,15248384.0,15070848.0,15100992.0,15131136.0,15161280.0,15191424.0,15221568.0,15251712.0,15281856.0,15103872.0,15134080.0,15164288.0,15194496.0,15224704.0,15254912.0,15285120.0,15315328.0,15136896.0,15167168.0,15197440.0,15227712.0,15257984.0,15288256.0,15318528.0,15348800.0,15169920.0,15200256.0,15230592.0,15260928.0,15291264.0,15321600.0,15351936.0,15382272.0,15202944.0,15233344.0,15263744.0,15294144.0,15324544.0,15354944.0,15385344.0,15415744.0,15235968.0,15266432.0,15296896.0,15327360.0,15357824.0,15388288.0,15418752.0,15449216.0,15268992.0,15299520.0,15330048.0,15360576.0,15391104.0,15421632.0,15452160.0,15482688.0,15302016.0,15332608.0,15363200.0,15393792.0,15424384.0,15454976.0,15485568.0,15516160.0,15335040.0,15365696.0,15396352.0,15427008.0,15457664.0,15488320.0,15518976.0,15549632.0,15368064.0,15398784.0,15429504.0,15460224.0,15490944.0,15521664.0,15552384.0,15583104.0,15401088.0,15431872.0,15462656.0,15493440.0,15524224.0,15555008.0,15585792.0,15616576.0,15434112.0,15464960.0,15495808.0,15526656.0,15557504.0,15588352.0,15619200.0,15650048.0,15467136.0,15498048.0,15528960.0,15559872.0,15590784.0,15621696.0,15652608.0,15683520.0,15500160.0,15531136.0,15562112.0,15593088.0,15624064.0,15655040.0,15686016.0,15716992.0,15533184.0,15564224.0,15595264.0,15626304.0,15657344.0,15688384.0,15719424.0,15750464.0,15566208.0,15597312.0,15628416.0,15659520.0,15690624.0,15721728.0,15752832.0,15783936.0,15599232.0,15630400.0,15661568.0,15692736.0,15723904.0,15755072.0,15786240.0,15817408.0,15632256.0,15663488.0,15694720.0,15725952.0,15757184.0,15788416.0,15819648.0,15850880.0,15665280.0,15696576.0,15727872.0,15759168.0,15790464.0,15821760.0,15853056.0,15884352.0,15698304.0,15729664.0,15761024.0,15792384.0,15823744.0,15855104.0,15886464.0,15917824.0,15731328.0,15762752.0,15794176.0,15825600.0,15857024.0,15888448.0,15919872.0,15951296.0,15764352.0,15795840.0,15827328.0,15858816.0,15890304.0,15921792.0,15953280.0,15984768.0,15797376.0,15828928.0,15860480.0,15892032.0,15923584.0,15955136.0,15986688.0,16018240.0,15830400.0,15862016.0,15893632.0,15925248.0,15956864.0,15988480.0,16020096.0,16051712.0,15863424.0,15895104.0,15926784.0,15958464.0,15990144.0,16021824.0,16053504.0,16085184.0,15896448.0,15928192.0,15959936.0,15991680.0,16023424.0,16055168.0,16086912.0,16118656.0,15929472.0,15961280.0,15993088.0,16024896.0,16056704.0,16088512.0,16120320.0,16152128.0,15962496.0,15994368.0,16026240.0,16058112.0,16089984.0,16121856.0,16153728.0,16185600.0,15995520.0,16027456.0,16059392.0,16091328.0,16123264.0,16155200.0,16187136.0,16219072.0,16028544.0,16060544.0,16092544.0,16124544.0,16156544.0,16188544.0,16220544.0,16252544.0,16061568.0,16093632.0,16125696.0,16157760.0,16189824.0,16221888.0,16253952.0,16286016.0,16094592.0,16126720.0,16158848.0,16190976.0,16223104.0,16255232.0,16287360.0,16319488.0,16127616.0,16159808.0,16192000.0,16224192.0,16256384.0,16288576.0,16320768.0,16352960.0,16160640.0,16192896.0,16225152.0,16257408.0,16289664.0,16321920.0,16354176.0,16386432.0,16193664.0,16225984.0,16258304.0,16290624.0,16322944.0,16355264.0,16387584.0,16419904.0,16226688.0,16259072.0,16291456.0,16323840.0,16356224.0,16388608.0,16420992.0,16453376.0,15997568.0,16028992.0,16060416.0,16091840.0,16123264.0,16154688.0,16186112.0,16217536.0,15760256.0,15790720.0,15821184.0,15851648.0,15882112.0,15912576.0,15943040.0,15973504.0,15793280.0,15823808.0,15854336.0,15884864.0,15915392.0,15945920.0,15976448.0,16006976.0,15826304.0,15856896.0,15887488.0,15918080.0,15948672.0,15979264.0,16009856.0,16040448.0,15859328.0,15889984.0,15920640.0,15951296.0,15981952.0,16012608.0,16043264.0,16073920.0,15892352.0,15923072.0,15953792.0,15984512.0,16015232.0,16045952.0,16076672.0,16107392.0,15925376.0,15956160.0,15986944.0,16017728.0,16048512.0,16079296.0,16110080.0,16140864.0,15958400.0,15989248.0,16020096.0,16050944.0,16081792.0,16112640.0,16143488.0,16174336.0,15991424.0,16022336.0,16053248.0,16084160.0,16115072.0,16145984.0,16176896.0,16207808.0,16024448.0,16055424.0,16086400.0,16117376.0,16148352.0,16179328.0,16210304.0,16241280.0,16057472.0,16088512.0,16119552.0,16150592.0,16181632.0,16212672.0,16243712.0,16274752.0,16090496.0,16121600.0,16152704.0,16183808.0,16214912.0,16246016.0,16277120.0,16308224.0,16123520.0,16154688.0,16185856.0,16217024.0,16248192.0,16279360.0,16310528.0,16341696.0,16156544.0,16187776.0,16219008.0,16250240.0,16281472.0,16312704.0,16343936.0,16375168.0,16189568.0,16220864.0,16252160.0,16283456.0,16314752.0,16346048.0,16377344.0,16408640.0,16222592.0,16253952.0,16285312.0,16316672.0,16348032.0,16379392.0,16410752.0,16442112.0,16255616.0,16287040.0,16318464.0,16349888.0,16381312.0,16412736.0,16444160.0,16475584.0,16288640.0,16320128.0,16351616.0,16383104.0,16414592.0,16446080.0,16477568.0,16509056.0,16321664.0,16353216.0,16384768.0,16416320.0,16447872.0,16479424.0,16510976.0,16542528.0,16354688.0,16386304.0,16417920.0,16449536.0,16481152.0,16512768.0,16544384.0,16576000.0,16387712.0,16419392.0,16451072.0,16482752.0,16514432.0,16546112.0,16577792.0,16609472.0,16420736.0,16452480.0,16484224.0,16515968.0,16547712.0,16579456.0,16611200.0,16642944.0,16453760.0,16485568.0,16517376.0,16549184.0,16580992.0,16612800.0,16644608.0,16676416.0,16486784.0,16518656.0,16550528.0,16582400.0,16614272.0,16646144.0,16678016.0,16709888.0,16519808.0,16551744.0,16583680.0,16615616.0,16647552.0,16679488.0,16711424.0,16743360.0,16552832.0,16584832.0,16616832.0,16648832.0,16680832.0,16712832.0,16744832.0,16776832.0,16585856.0,16617920.0,16649984.0,16682048.0,16714112.0,16746176.0,16778240.0,16810304.0,16618880.0,16651008.0,16683136.0,16715264.0,16747392.0,16779520.0,16811648.0,16843776.0,16651904.0,16684096.0,16716288.0,16748480.0,16780672.0,16812864.0,16845056.0,16877248.0,16684928.0,16717184.0,16749440.0,16781696.0,16813952.0,16846208.0,16878464.0,16910720.0,16717952.0,16750272.0,16782592.0,16814912.0,16847232.0,16879552.0,16911872.0,16944192.0,16750976.0,16783360.0,16815744.0,16848128.0,16880512.0,16912896.0,16945280.0,16977664.0,16784000.0,16816448.0,16848896.0,16881344.0,16913792.0,16946240.0,16978688.0,17011136.0,16817024.0,16849536.0,16882048.0,16914560.0,16947072.0,16979584.0,17012096.0,17044608.0,16850048.0,16882624.0,16915200.0,16947776.0,16980352.0,17012928.0,17045504.0,17078080.0,16883072.0,16915712.0,16948352.0,16980992.0,17013632.0,17046272.0,17078912.0,17111552.0,16916096.0,16948800.0,16981504.0,17014208.0,17046912.0,17079616.0,17112320.0,17145024.0,16949120.0,16981888.0,17014656.0,17047424.0,17080192.0,17112960.0,17145728.0,17178496.0,16982144.0,17014976.0,17047808.0,17080640.0,17113472.0,17146304.0,17179136.0,17211968.0,16228736.0,16260608.0,16292480.0,16324352.0,16356224.0,16388096.0,16419968.0,16451840.0,15467136.0,15498048.0,15528960.0,15559872.0,15590784.0,15621696.0,15652608.0,15683520.0,15500160.0,15531136.0,15562112.0,15593088.0,15624064.0,15655040.0,15686016.0,15716992.0,15533184.0,15564224.0,15595264.0,15626304.0,15657344.0,15688384.0,15719424.0,15750464.0,15566208.0,15597312.0,15628416.0,15659520.0,15690624.0,15721728.0,15752832.0,15783936.0,15599232.0,15630400.0,15661568.0,15692736.0,15723904.0,15755072.0,15786240.0,15817408.0,15632256.0,15663488.0,15694720.0,15725952.0,15757184.0,15788416.0,15819648.0,15850880.0,15665280.0,15696576.0,15727872.0,15759168.0,15790464.0,15821760.0,15853056.0,15884352.0,15698304.0,15729664.0,15761024.0,15792384.0,15823744.0,15855104.0,15886464.0,15917824.0,15731328.0,15762752.0,15794176.0,15825600.0,15857024.0,15888448.0,15919872.0,15951296.0,15764352.0,15795840.0,15827328.0,15858816.0,15890304.0,15921792.0,15953280.0,15984768.0,15797376.0,15828928.0,15860480.0,15892032.0,15923584.0,15955136.0,15986688.0,16018240.0,15830400.0,15862016.0,15893632.0,15925248.0,15956864.0,15988480.0,16020096.0,16051712.0,15863424.0,15895104.0,15926784.0,15958464.0,15990144.0,16021824.0,16053504.0,16085184.0,15896448.0,15928192.0,15959936.0,15991680.0,16023424.0,16055168.0,16086912.0,16118656.0,15929472.0,15961280.0,15993088.0,16024896.0,16056704.0,16088512.0,16120320.0,16152128.0,15962496.0,15994368.0,16026240.0,16058112.0,16089984.0,16121856.0,16153728.0,16185600.0,15995520.0,16027456.0,16059392.0,16091328.0,16123264.0,16155200.0,16187136.0,16219072.0,16028544.0,16060544.0,16092544.0,16124544.0,16156544.0,16188544.0,16220544.0,16252544.0,16061568.0,16093632.0,16125696.0,16157760.0,16189824.0,16221888.0,16253952.0,16286016.0,16094592.0,16126720.0,16158848.0,16190976.0,16223104.0,16255232.0,16287360.0,16319488.0,16127616.0,16159808.0,16192000.0,16224192.0,16256384.0,16288576.0,16320768.0,16352960.0,16160640.0,16192896.0,16225152.0,16257408.0,16289664.0,16321920.0,16354176.0,16386432.0,16193664.0,16225984.0,16258304.0,16290624.0,16322944.0,16355264.0,16387584.0,16419904.0,16226688.0,16259072.0,16291456.0,16323840.0,16356224.0,16388608.0,16420992.0,16453376.0,16259712.0,16292160.0,16324608.0,16357056.0,16389504.0,16421952.0,16454400.0,16486848.0,16292736.0,16325248.0,16357760.0,16390272.0,16422784.0,16455296.0,16487808.0,16520320.0,16325760.0,16358336.0,16390912.0,16423488.0,16456064.0,16488640.0,16521216.0,16553792.0,16358784.0,16391424.0,16424064.0,16456704.0,16489344.0,16521984.0,16554624.0,16587264.0,16391808.0,16424512.0,16457216.0,16489920.0,16522624.0,16555328.0,16588032.0,16620736.0,16424832.0,16457600.0,16490368.0,16523136.0,16555904.0,16588672.0,16621440.0,16654208.0,16457856.0,16490688.0,16523520.0,16556352.0,16589184.0,16622016.0,16654848.0,16687680.0,16490880.0,16523776.0,16556672.0,16589568.0,16622464.0,16655360.0,16688256.0,16721152.0,16523904.0,16556864.0,16589824.0,16622784.0,16655744.0,16688704.0,16721664.0,16754624.0,16556928.0,16589952.0,16622976.0,16656000.0,16689024.0,16722048.0,16755072.0,16788096.0,16589952.0,16623040.0,16656128.0,16689216.0,16722304.0,16755392.0,16788480.0,16821568.0,16622976.0,16656128.0,16689280.0,16722432.0,16755584.0,16788736.0,16821888.0,16855040.0,16656000.0,16689216.0,16722432.0,16755648.0,16788864.0,16822080.0,16855296.0,16888512.0,16689024.0,16722304.0,16755584.0,16788864.0,16822144.0,16855424.0,16888704.0,16921984.0,16459904.0,16492224.0,16524544.0,16556864.0,16589184.0,16621504.0,16653824.0,16686144.0,16222592.0,16253952.0,16285312.0,16316672.0,16348032.0,16379392.0,16410752.0,16442112.0,16255616.0,16287040.0,16318464.0,16349888.0,16381312.0,16412736.0,16444160.0,16475584.0,16288640.0,16320128.0,16351616.0,16383104.0,16414592.0,16446080.0,16477568.0,16509056.0,16321664.0,16353216.0,16384768.0,16416320.0,16447872.0,16479424.0,16510976.0,16542528.0,16354688.0,16386304.0,16417920.0,16449536.0,16481152.0,16512768.0,16544384.0,16576000.0,16387712.0,16419392.0,16451072.0,16482752.0,16514432.0,16546112.0,16577792.0,16609472.0,16420736.0,16452480.0,16484224.0,16515968.0,16547712.0,16579456.0,16611200.0,16642944.0,16453760.0,16485568.0,16517376.0,16549184.0,16580992.0,16612800.0,16644608.0,16676416.0,16486784.0,16518656.0,16550528.0,16582400.0,16614272.0,16646144.0,16678016.0,16709888.0,16519808.0,16551744.0,16583680.0,16615616.0,16647552.0,16679488.0,16711424.0,16743360.0,16552832.0,16584832.0,16616832.0,16648832.0,16680832.0,16712832.0,16744832.0,16776832.0,16585856.0,16617920.0,16649984.0,16682048.0,16714112.0,16746176.0,16778240.0,16810304.0,16618880.0,16651008.0,16683136.0,16715264.0,16747392.0,16779520.0,16811648.0,16843776.0,16651904.0,16684096.0,16716288.0,16748480.0,16780672.0,16812864.0,16845056.0,16877248.0,16684928.0,16717184.0,16749440.0,16781696.0,16813952.0,16846208.0,16878464.0,16910720.0,16717952.0,16750272.0,16782592.0,16814912.0,16847232.0,16879552.0,16911872.0,16944192.0,16750976.0,16783360.0,16815744.0,16848128.0,16880512.0,16912896.0,16945280.0,16977664.0,16784000.0,16816448.0,16848896.0,16881344.0,16913792.0,16946240.0,16978688.0,17011136.0,16817024.0,16849536.0,16882048.0,16914560.0,16947072.0,16979584.0,17012096.0,17044608.0,16850048.0,16882624.0,16915200.0,16947776.0,16980352.0,17012928.0,17045504.0,17078080.0,16883072.0,16915712.0,16948352.0,16980992.0,17013632.0,17046272.0,17078912.0,17111552.0,16916096.0,16948800.0,16981504.0,17014208.0,17046912.0,17079616.0,17112320.0,17145024.0,16949120.0,16981888.0,17014656.0,17047424.0,17080192.0,17112960.0,17145728.0,17178496.0,16982144.0,17014976.0,17047808.0,17080640.0,17113472.0,17146304.0,17179136.0,17211968.0,17015168.0,17048064.0,17080960.0,17113856.0,17146752.0,17179648.0,17212544.0,17245440.0,17048192.0,17081152.0,17114112.0,17147072.0,17180032.0,17212992.0,17245952.0,17278912.0,17081216.0,17114240.0,17147264.0,17180288.0,17213312.0,17246336.0,17279360.0,17312384.0,17114240.0,17147328.0,17180416.0,17213504.0,17246592.0,17279680.0,17312768.0,17345856.0,17147264.0,17180416.0,17213568.0,17246720.0,17279872.0,17313024.0,17346176.0,17379328.0,17180288.0,17213504.0,17246720.0,17279936.0,17313152.0,17346368.0,17379584.0,17412800.0,17213312.0,17246592.0,17279872.0,17313152.0,17346432.0,17379712.0,17412992.0,17446272.0,17246336.0,17279680.0,17313024.0,17346368.0,17379712.0,17413056.0,17446400.0,17479744.0,17279360.0,17312768.0,17346176.0,17379584.0,17412992.0,17446400.0,17479808.0,17513216.0,17312384.0,17345856.0,17379328.0,17412800.0,17446272.0,17479744.0,17513216.0,17546688.0,17345408.0,17378944.0,17412480.0,17446016.0,17479552.0,17513088.0,17546624.0,17580160.0,17378432.0,17412032.0,17445632.0,17479232.0,17512832.0,17546432.0,17580032.0,17613632.0,17411456.0,17445120.0,17478784.0,17512448.0,17546112.0,17579776.0,17613440.0,17647104.0,17444480.0,17478208.0,17511936.0,17545664.0,17579392.0,17613120.0,17646848.0,17680576.0,16691072.0,16723840.0,16756608.0,16789376.0,16822144.0,16854912.0,16887680.0,16920448.0,15929472.0,15961280.0,15993088.0,16024896.0,16056704.0,16088512.0,16120320.0,16152128.0,8036736.0,8052656.0,8068576.0,8084496.0,8100416.0,8116336.0,8132256.0,8148176.0,8620800.0,8636592.0,8652384.0,8668176.0,8683968.0,8699760.0,8715552.0,8731344.0,17384832.0,17416448.0,17448064.0,17479680.0,17511296.0,17542912.0,17574528.0,17606144.0,17418880.0,17450560.0,17482240.0,17513920.0,17545600.0,17577280.0,17608960.0,17640640.0,17452928.0,17484672.0,17516416.0,17548160.0,17579904.0,17611648.0,17643392.0,17675136.0,17486976.0,17518784.0,17550592.0,17582400.0,17614208.0,17646016.0,17677824.0,17709632.0,17521024.0,17552896.0,17584768.0,17616640.0,17648512.0,17680384.0,17712256.0,17744128.0,17555072.0,17587008.0,17618944.0,17650880.0,17682816.0,17714752.0,17746688.0,17778624.0,17589120.0,17621120.0,17653120.0,17685120.0,17717120.0,17749120.0,17781120.0,17813120.0,17623168.0,17655232.0,17687296.0,17719360.0,17751424.0,17783488.0,17815552.0,17847616.0,17657216.0,17689344.0,17721472.0,17753600.0,17785728.0,17817856.0,17849984.0,17882112.0,17691264.0,17723456.0,17755648.0,17787840.0,17820032.0,17852224.0,17884416.0,17916608.0,17725312.0,17757568.0,17789824.0,17822080.0,17854336.0,17886592.0,17918848.0,17951104.0,17759360.0,17791680.0,17824000.0,17856320.0,17888640.0,17920960.0,17953280.0,17985600.0,17793408.0,17825792.0,17858176.0,17890560.0,17922944.0,17955328.0,17987712.0,18020096.0,17827456.0,17859904.0,17892352.0,17924800.0,17957248.0,17989696.0,18022144.0,18054592.0,17861504.0,17894016.0,17926528.0,17959040.0,17991552.0,18024064.0,18056576.0,18089088.0,17895552.0,17928128.0,17960704.0,17993280.0,18025856.0,18058432.0,18091008.0,18123584.0,17929600.0,17962240.0,17994880.0,18027520.0,18060160.0,18092800.0,18125440.0,18158080.0,17963648.0,17996352.0,18029056.0,18061760.0,18094464.0,18127168.0,18159872.0,18192576.0,17997696.0,18030464.0,18063232.0,18096000.0,18128768.0,18161536.0,18194304.0,18227072.0,18031744.0,18064576.0,18097408.0,18130240.0,18163072.0,18195904.0,18228736.0,18261568.0,18065792.0,18098688.0,18131584.0,18164480.0,18197376.0,18230272.0,18263168.0,18296064.0,18099840.0,18132800.0,18165760.0,18198720.0,18231680.0,18264640.0,18297600.0,18330560.0,18133888.0,18166912.0,18199936.0,18232960.0,18265984.0,18299008.0,18332032.0,18365056.0,18167936.0,18201024.0,18234112.0,18267200.0,18300288.0,18333376.0,18366464.0,18399552.0,18201984.0,18235136.0,18268288.0,18301440.0,18334592.0,18367744.0,18400896.0,18434048.0,18236032.0,18269248.0,18302464.0,18335680.0,18368896.0,18402112.0,18435328.0,18468544.0,18270080.0,18303360.0,18336640.0,18369920.0,18403200.0,18436480.0,18469760.0,18503040.0,18304128.0,18337472.0,18370816.0,18404160.0,18437504.0,18470848.0,18504192.0,18537536.0,18338176.0,18371584.0,18404992.0,18438400.0,18471808.0,18505216.0,18538624.0,18572032.0,18372224.0,18405696.0,18439168.0,18472640.0,18506112.0,18539584.0,18573056.0,18606528.0,18406272.0,18439808.0,18473344.0,18506880.0,18540416.0,18573952.0,18607488.0,18641024.0,18440320.0,18473920.0,18507520.0,18541120.0,18574720.0,18608320.0,18641920.0,18675520.0,18474368.0,18508032.0,18541696.0,18575360.0,18609024.0,18642688.0,18676352.0,18710016.0,18508416.0,18542144.0,18575872.0,18609600.0,18643328.0,18677056.0,18710784.0,18744512.0,18542464.0,18576256.0,18610048.0,18643840.0,18677632.0,18711424.0,18745216.0,18779008.0,18576512.0,18610368.0,18644224.0,18678080.0,18711936.0,18745792.0,18779648.0,18813504.0,17807744.0,17840640.0,17873536.0,17906432.0,17939328.0,17972224.0,18005120.0,18038016.0,17030784.0,17062720.0,17094656.0,17126592.0,17158528.0,17190464.0,17222400.0,17254336.0,16262016.0,16292992.0,16323968.0,16354944.0,16385920.0,16416896.0,16447872.0,16478848.0,15485056.0,15515072.0,15545088.0,15575104.0,15605120.0,15635136.0,15665152.0,15695168.0,15519104.0,15549184.0,15579264.0,15609344.0,15639424.0,15669504.0,15699584.0,15729664.0,15553152.0,15583296.0,15613440.0,15643584.0,15673728.0,15703872.0,15734016.0,15764160.0,15587200.0,15617408.0,15647616.0,15677824.0,15708032.0,15738240.0,15768448.0,15798656.0,15621248.0,15651520.0,15681792.0,15712064.0,15742336.0,15772608.0,15802880.0,15833152.0,15655296.0,15685632.0,15715968.0,15746304.0,15776640.0,15806976.0,15837312.0,15867648.0,15689344.0,15719744.0,15750144.0,15780544.0,15810944.0,15841344.0,15871744.0,15902144.0,15723392.0,15753856.0,15784320.0,15814784.0,15845248.0,15875712.0,15906176.0,15936640.0,15757440.0,15787968.0,15818496.0,15849024.0,15879552.0,15910080.0,15940608.0,15971136.0,15791488.0,15822080.0,15852672.0,15883264.0,15913856.0,15944448.0,15975040.0,16005632.0,15825536.0,15856192.0,15886848.0,15917504.0,15948160.0,15978816.0,16009472.0,16040128.0,15859584.0,15890304.0,15921024.0,15951744.0,15982464.0,16013184.0,16043904.0,16074624.0,15893632.0,15924416.0,15955200.0,15985984.0,16016768.0,16047552.0,16078336.0,16109120.0,15927680.0,15958528.0,15989376.0,16020224.0,16051072.0,16081920.0,16112768.0,16143616.0,15961728.0,15992640.0,16023552.0,16054464.0,16085376.0,16116288.0,16147200.0,16178112.0,15995776.0,16026752.0,16057728.0,16088704.0,16119680.0,16150656.0,16181632.0,16212608.0,16029824.0,16060864.0,16091904.0,16122944.0,16153984.0,16185024.0,16216064.0,16247104.0,16063872.0,16094976.0,16126080.0,16157184.0,16188288.0,16219392.0,16250496.0,16281600.0,16097920.0,16129088.0,16160256.0,16191424.0,16222592.0,16253760.0,16284928.0,16316096.0,16131968.0,16163200.0,16194432.0,16225664.0,16256896.0,16288128.0,16319360.0,16350592.0,16166016.0,16197312.0,16228608.0,16259904.0,16291200.0,16322496.0,16353792.0,16385088.0,16200064.0,16231424.0,16262784.0,16294144.0,16325504.0,16356864.0,16388224.0,16419584.0,16234112.0,16265536.0,16296960.0,16328384.0,16359808.0,16391232.0,16422656.0,16454080.0,16268160.0,16299648.0,16331136.0,16362624.0,16394112.0,16425600.0,16457088.0,16488576.0,16302208.0,16333760.0,16365312.0,16396864.0,16428416.0,16459968.0,16491520.0,16523072.0,16336256.0,16367872.0,16399488.0,16431104.0,16462720.0,16494336.0,16525952.0,16557568.0,16370304.0,16401984.0,16433664.0,16465344.0,16497024.0,16528704.0,16560384.0,16592064.0,16404352.0,16436096.0,16467840.0,16499584.0,16531328.0,16563072.0,16594816.0,16626560.0,16438400.0,16470208.0,16502016.0,16533824.0,16565632.0,16597440.0,16629248.0,16661056.0,16472448.0,16504320.0,16536192.0,16568064.0,16599936.0,16631808.0,16663680.0,16695552.0,16506496.0,16538432.0,16570368.0,16602304.0,16634240.0,16666176.0,16698112.0,16730048.0,16540544.0,16572544.0,16604544.0,16636544.0,16668544.0,16700544.0,16732544.0,16764544.0,16574592.0,16606656.0,16638720.0,16670784.0,16702848.0,16734912.0,16766976.0,16799040.0,16608640.0,16640768.0,16672896.0,16705024.0,16737152.0,16769280.0,16801408.0,16833536.0,16642688.0,16674880.0,16707072.0,16739264.0,16771456.0,16803648.0,16835840.0,16868032.0,16676736.0,16708992.0,16741248.0,16773504.0,16805760.0,16838016.0,16870272.0,16902528.0,16710784.0,16743104.0,16775424.0,16807744.0,16840064.0,16872384.0,16904704.0,16937024.0,16744832.0,16777216.0,16809600.0,16841984.0,16874368.0,16906752.0,16939136.0,16971520.0,16500352.0,16531776.0,16563200.0,16594624.0,16626048.0,16657472.0,16688896.0,16720320.0,16247680.0,16278144.0,16308608.0,16339072.0,16369536.0,16400000.0,16430464.0,16460928.0,16281728.0,16312256.0,16342784.0,16373312.0,16403840.0,16434368.0,16464896.0,16495424.0,16315776.0,16346368.0,16376960.0,16407552.0,16438144.0,16468736.0,16499328.0,16529920.0,16349824.0,16380480.0,16411136.0,16441792.0,16472448.0,16503104.0,16533760.0,16564416.0,16383872.0,16414592.0,16445312.0,16476032.0,16506752.0,16537472.0,16568192.0,16598912.0,16417920.0,16448704.0,16479488.0,16510272.0,16541056.0,16571840.0,16602624.0,16633408.0,16451968.0,16482816.0,16513664.0,16544512.0,16575360.0,16606208.0,16637056.0,16667904.0,16486016.0,16516928.0,16547840.0,16578752.0,16609664.0,16640576.0,16671488.0,16702400.0,16520064.0,16551040.0,16582016.0,16612992.0,16643968.0,16674944.0,16705920.0,16736896.0,16554112.0,16585152.0,16616192.0,16647232.0,16678272.0,16709312.0,16740352.0,16771392.0,16588160.0,16619264.0,16650368.0,16681472.0,16712576.0,16743680.0,16774784.0,16805888.0,16622208.0,16653376.0,16684544.0,16715712.0,16746880.0,16778048.0,16809216.0,16840384.0,16656256.0,16687488.0,16718720.0,16749952.0,16781184.0,16812416.0,16843648.0,16874880.0,16690304.0,16721600.0,16752896.0,16784192.0,16815488.0,16846784.0,16878080.0,16909376.0,16724352.0,16755712.0,16787072.0,16818432.0,16849792.0,16881152.0,16912512.0,16943872.0,16758400.0,16789824.0,16821248.0,16852672.0,16884096.0,16915520.0,16946944.0,16978368.0,16792448.0,16823936.0,16855424.0,16886912.0,16918400.0,16949888.0,16981376.0,17012864.0,16826496.0,16858048.0,16889600.0,16921152.0,16952704.0,16984256.0,17015808.0,17047360.0,16860544.0,16892160.0,16923776.0,16955392.0,16987008.0,17018624.0,17050240.0,17081856.0,16894592.0,16926272.0,16957952.0,16989632.0,17021312.0,17052992.0,17084672.0,17116352.0,16928640.0,16960384.0,16992128.0,17023872.0,17055616.0,17087360.0,17119104.0,17150848.0,16962688.0,16994496.0,17026304.0,17058112.0,17089920.0,17121728.0,17153536.0,17185344.0,16996736.0,17028608.0,17060480.0,17092352.0,17124224.0,17156096.0,17187968.0,17219840.0,17030784.0,17062720.0,17094656.0,17126592.0,17158528.0,17190464.0,17222400.0,17254336.0,17064832.0,17096832.0,17128832.0,17160832.0,17192832.0,17224832.0,17256832.0,17288832.0,17098880.0,17130944.0,17163008.0,17195072.0,17227136.0,17259200.0,17291264.0,17323328.0,17132928.0,17165056.0,17197184.0,17229312.0,17261440.0,17293568.0,17325696.0,17357824.0,17166976.0,17199168.0,17231360.0,17263552.0,17295744.0,17327936.0,17360128.0,17392320.0,17201024.0,17233280.0,17265536.0,17297792.0,17330048.0,17362304.0,17394560.0,17426816.0,17235072.0,17267392.0,17299712.0,17332032.0,17364352.0,17396672.0,17428992.0,17461312.0,17269120.0,17301504.0,17333888.0,17366272.0,17398656.0,17431040.0,17463424.0,17495808.0,17303168.0,17335616.0,17368064.0,17400512.0,17432960.0,17465408.0,17497856.0,17530304.0,17337216.0,17369728.0,17402240.0,17434752.0,17467264.0,17499776.0,17532288.0,17564800.0,17371264.0,17403840.0,17436416.0,17468992.0,17501568.0,17534144.0,17566720.0,17599296.0,17405312.0,17437952.0,17470592.0,17503232.0,17535872.0,17568512.0,17601152.0,17633792.0,17439360.0,17472064.0,17504768.0,17537472.0,17570176.0,17602880.0,17635584.0,17668288.0,17473408.0,17506176.0,17538944.0,17571712.0,17604480.0,17637248.0,17670016.0,17702784.0,17507456.0,17540288.0,17573120.0,17605952.0,17638784.0,17671616.0,17704448.0,17737280.0,16738688.0,16770560.0,16802432.0,16834304.0,16866176.0,16898048.0,16929920.0,16961792.0,15961728.0,15992640.0,16023552.0,16054464.0,16085376.0,16116288.0,16147200.0,16178112.0,15995776.0,16026752.0,16057728.0,16088704.0,16119680.0,16150656.0,16181632.0,16212608.0,16029824.0,16060864.0,16091904.0,16122944.0,16153984.0,16185024.0,16216064.0,16247104.0,16063872.0,16094976.0,16126080.0,16157184.0,16188288.0,16219392.0,16250496.0,16281600.0,16097920.0,16129088.0,16160256.0,16191424.0,16222592.0,16253760.0,16284928.0,16316096.0,16131968.0,16163200.0,16194432.0,16225664.0,16256896.0,16288128.0,16319360.0,16350592.0,16166016.0,16197312.0,16228608.0,16259904.0,16291200.0,16322496.0,16353792.0,16385088.0,16200064.0,16231424.0,16262784.0,16294144.0,16325504.0,16356864.0,16388224.0,16419584.0,16234112.0,16265536.0,16296960.0,16328384.0,16359808.0,16391232.0,16422656.0,16454080.0,16268160.0,16299648.0,16331136.0,16362624.0,16394112.0,16425600.0,16457088.0,16488576.0,16302208.0,16333760.0,16365312.0,16396864.0,16428416.0,16459968.0,16491520.0,16523072.0,16336256.0,16367872.0,16399488.0,16431104.0,16462720.0,16494336.0,16525952.0,16557568.0,16370304.0,16401984.0,16433664.0,16465344.0,16497024.0,16528704.0,16560384.0,16592064.0,16404352.0,16436096.0,16467840.0,16499584.0,16531328.0,16563072.0,16594816.0,16626560.0,16438400.0,16470208.0,16502016.0,16533824.0,16565632.0,16597440.0,16629248.0,16661056.0,16472448.0,16504320.0,16536192.0,16568064.0,16599936.0,16631808.0,16663680.0,16695552.0,16506496.0,16538432.0,16570368.0,16602304.0,16634240.0,16666176.0,16698112.0,16730048.0,16540544.0,16572544.0,16604544.0,16636544.0,16668544.0,16700544.0,16732544.0,16764544.0,16574592.0,16606656.0,16638720.0,16670784.0,16702848.0,16734912.0,16766976.0,16799040.0,16608640.0,16640768.0,16672896.0,16705024.0,16737152.0,16769280.0,16801408.0,16833536.0,16642688.0,16674880.0,16707072.0,16739264.0,16771456.0,16803648.0,16835840.0,16868032.0,16676736.0,16708992.0,16741248.0,16773504.0,16805760.0,16838016.0,16870272.0,16902528.0,16710784.0,16743104.0,16775424.0,16807744.0,16840064.0,16872384.0,16904704.0,16937024.0,16744832.0,16777216.0,16809600.0,16841984.0,16874368.0,16906752.0,16939136.0,16971520.0,16778880.0,16811328.0,16843776.0,16876224.0,16908672.0,16941120.0,16973568.0,17006016.0,16812928.0,16845440.0,16877952.0,16910464.0,16942976.0,16975488.0,17008000.0,17040512.0,16846976.0,16879552.0,16912128.0,16944704.0,16977280.0,17009856.0,17042432.0,17075008.0,16881024.0,16913664.0,16946304.0,16978944.0,17011584.0,17044224.0,17076864.0,17109504.0,16915072.0,16947776.0,16980480.0,17013184.0,17045888.0,17078592.0,17111296.0,17144000.0,16949120.0,16981888.0,17014656.0,17047424.0,17080192.0,17112960.0,17145728.0,17178496.0,16983168.0,17016000.0,17048832.0,17081664.0,17114496.0,17147328.0,17180160.0,17212992.0,17017216.0,17050112.0,17083008.0,17115904.0,17148800.0,17181696.0,17214592.0,17247488.0,17051264.0,17084224.0,17117184.0,17150144.0,17183104.0,17216064.0,17249024.0,17281984.0,17085312.0,17118336.0,17151360.0,17184384.0,17217408.0,17250432.0,17283456.0,17316480.0,17119360.0,17152448.0,17185536.0,17218624.0,17251712.0,17284800.0,17317888.0,17350976.0,17153408.0,17186560.0,17219712.0,17252864.0,17286016.0,17319168.0,17352320.0,17385472.0,17187456.0,17220672.0,17253888.0,17287104.0,17320320.0,17353536.0,17386752.0,17419968.0,17221504.0,17254784.0,17288064.0,17321344.0,17354624.0,17387904.0,17421184.0,17454464.0,16977024.0,17009344.0,17041664.0,17073984.0,17106304.0,17138624.0,17170944.0,17203264.0,16724352.0,16755712.0,16787072.0,16818432.0,16849792.0,16881152.0,16912512.0,16943872.0,16758400.0,16789824.0,16821248.0,16852672.0,16884096.0,16915520.0,16946944.0,16978368.0,16792448.0,16823936.0,16855424.0,16886912.0,16918400.0,16949888.0,16981376.0,17012864.0,16826496.0,16858048.0,16889600.0,16921152.0,16952704.0,16984256.0,17015808.0,17047360.0,16860544.0,16892160.0,16923776.0,16955392.0,16987008.0,17018624.0,17050240.0,17081856.0,16894592.0,16926272.0,16957952.0,16989632.0,17021312.0,17052992.0,17084672.0,17116352.0,16928640.0,16960384.0,16992128.0,17023872.0,17055616.0,17087360.0,17119104.0,17150848.0,16962688.0,16994496.0,17026304.0,17058112.0,17089920.0,17121728.0,17153536.0,17185344.0,16996736.0,17028608.0,17060480.0,17092352.0,17124224.0,17156096.0,17187968.0,17219840.0,17030784.0,17062720.0,17094656.0,17126592.0,17158528.0,17190464.0,17222400.0,17254336.0,17064832.0,17096832.0,17128832.0,17160832.0,17192832.0,17224832.0,17256832.0,17288832.0,17098880.0,17130944.0,17163008.0,17195072.0,17227136.0,17259200.0,17291264.0,17323328.0,17132928.0,17165056.0,17197184.0,17229312.0,17261440.0,17293568.0,17325696.0,17357824.0,17166976.0,17199168.0,17231360.0,17263552.0,17295744.0,17327936.0,17360128.0,17392320.0,17201024.0,17233280.0,17265536.0,17297792.0,17330048.0,17362304.0,17394560.0,17426816.0,17235072.0,17267392.0,17299712.0,17332032.0,17364352.0,17396672.0,17428992.0,17461312.0,17269120.0,17301504.0,17333888.0,17366272.0,17398656.0,17431040.0,17463424.0,17495808.0,17303168.0,17335616.0,17368064.0,17400512.0,17432960.0,17465408.0,17497856.0,17530304.0,17337216.0,17369728.0,17402240.0,17434752.0,17467264.0,17499776.0,17532288.0,17564800.0,17371264.0,17403840.0,17436416.0,17468992.0,17501568.0,17534144.0,17566720.0,17599296.0,17405312.0,17437952.0,17470592.0,17503232.0,17535872.0,17568512.0,17601152.0,17633792.0,17439360.0,17472064.0,17504768.0,17537472.0,17570176.0,17602880.0,17635584.0,17668288.0,17473408.0,17506176.0,17538944.0,17571712.0,17604480.0,17637248.0,17670016.0,17702784.0,17507456.0,17540288.0,17573120.0,17605952.0,17638784.0,17671616.0,17704448.0,17737280.0,17541504.0,17574400.0,17607296.0,17640192.0,17673088.0,17705984.0,17738880.0,17771776.0,17575552.0,17608512.0,17641472.0,17674432.0,17707392.0,17740352.0,17773312.0,17806272.0,17609600.0,17642624.0,17675648.0,17708672.0,17741696.0,17774720.0,17807744.0,17840768.0,17643648.0,17676736.0,17709824.0,17742912.0,17776000.0,17809088.0,17842176.0,17875264.0,17677696.0,17710848.0,17744000.0,17777152.0,17810304.0,17843456.0,17876608.0,17909760.0,17711744.0,17744960.0,17778176.0,17811392.0,17844608.0,17877824.0,17911040.0,17944256.0,17745792.0,17779072.0,17812352.0,17845632.0,17878912.0,17912192.0,17945472.0,17978752.0,17779840.0,17813184.0,17846528.0,17879872.0,17913216.0,17946560.0,17979904.0,18013248.0,17813888.0,17847296.0,17880704.0,17914112.0,17947520.0,17980928.0,18014336.0,18047744.0,17847936.0,17881408.0,17914880.0,17948352.0,17981824.0,18015296.0,18048768.0,18082240.0,17881984.0,17915520.0,17949056.0,17982592.0,18016128.0,18049664.0,18083200.0,18116736.0,17916032.0,17949632.0,17983232.0,18016832.0,18050432.0,18084032.0,18117632.0,18151232.0,17950080.0,17983744.0,18017408.0,18051072.0,18084736.0,18118400.0,18152064.0,18185728.0,17984128.0,18017856.0,18051584.0,18085312.0,18119040.0,18152768.0,18186496.0,18220224.0,17215360.0,17248128.0,17280896.0,17313664.0,17346432.0,17379200.0,17411968.0,17444736.0,16438400.0,16470208.0,16502016.0,16533824.0,16565632.0,16597440.0,16629248.0,16661056.0,8291456.0,8307376.0,8323296.0,8339216.0,8355136.0,8371056.0,8386976.0,8402896.0,8873472.0,8889264.0,8905056.0,8920848.0,8936640.0,8952432.0,8968224.0,8984016.0,17890688.0,17922304.0,17953920.0,17985536.0,18017152.0,18048768.0,18080384.0,18112000.0,17925760.0,17957440.0,17989120.0,18020800.0,18052480.0,18084160.0,18115840.0,18147520.0,17960832.0,17992576.0,18024320.0,18056064.0,18087808.0,18119552.0,18151296.0,18183040.0,17995904.0,18027712.0,18059520.0,18091328.0,18123136.0,18154944.0,18186752.0,18218560.0,18030976.0,18062848.0,18094720.0,18126592.0,18158464.0,18190336.0,18222208.0,18254080.0,18066048.0,18097984.0,18129920.0,18161856.0,18193792.0,18225728.0,18257664.0,18289600.0,18101120.0,18133120.0,18165120.0,18197120.0,18229120.0,18261120.0,18293120.0,18325120.0,18136192.0,18168256.0,18200320.0,18232384.0,18264448.0,18296512.0,18328576.0,18360640.0,18171264.0,18203392.0,18235520.0,18267648.0,18299776.0,18331904.0,18364032.0,18396160.0,18206336.0,18238528.0,18270720.0,18302912.0,18335104.0,18367296.0,18399488.0,18431680.0,18241408.0,18273664.0,18305920.0,18338176.0,18370432.0,18402688.0,18434944.0,18467200.0,18276480.0,18308800.0,18341120.0,18373440.0,18405760.0,18438080.0,18470400.0,18502720.0,18311552.0,18343936.0,18376320.0,18408704.0,18441088.0,18473472.0,18505856.0,18538240.0,18346624.0,18379072.0,18411520.0,18443968.0,18476416.0,18508864.0,18541312.0,18573760.0,18381696.0,18414208.0,18446720.0,18479232.0,18511744.0,18544256.0,18576768.0,18609280.0,18416768.0,18449344.0,18481920.0,18514496.0,18547072.0,18579648.0,18612224.0,18644800.0,18451840.0,18484480.0,18517120.0,18549760.0,18582400.0,18615040.0,18647680.0,18680320.0,18486912.0,18519616.0,18552320.0,18585024.0,18617728.0,18650432.0,18683136.0,18715840.0,18521984.0,18554752.0,18587520.0,18620288.0,18653056.0,18685824.0,18718592.0,18751360.0,18557056.0,18589888.0,18622720.0,18655552.0,18688384.0,18721216.0,18754048.0,18786880.0,18592128.0,18625024.0,18657920.0,18690816.0,18723712.0,18756608.0,18789504.0,18822400.0,18627200.0,18660160.0,18693120.0,18726080.0,18759040.0,18792000.0,18824960.0,18857920.0,18662272.0,18695296.0,18728320.0,18761344.0,18794368.0,18827392.0,18860416.0,18893440.0,18697344.0,18730432.0,18763520.0,18796608.0,18829696.0,18862784.0,18895872.0,18928960.0,18732416.0,18765568.0,18798720.0,18831872.0,18865024.0,18898176.0,18931328.0,18964480.0,18767488.0,18800704.0,18833920.0,18867136.0,18900352.0,18933568.0,18966784.0,19000000.0,18802560.0,18835840.0,18869120.0,18902400.0,18935680.0,18968960.0,19002240.0,19035520.0,18837632.0,18870976.0,18904320.0,18937664.0,18971008.0,19004352.0,19037696.0,19071040.0,18872704.0,18906112.0,18939520.0,18972928.0,19006336.0,19039744.0,19073152.0,19106560.0,18907776.0,18941248.0,18974720.0,19008192.0,19041664.0,19075136.0,19108608.0,19142080.0,18942848.0,18976384.0,19009920.0,19043456.0,19076992.0,19110528.0,19144064.0,19177600.0,18977920.0,19011520.0,19045120.0,19078720.0,19112320.0,19145920.0,19179520.0,19213120.0,19012992.0,19046656.0,19080320.0,19113984.0,19147648.0,19181312.0,19214976.0,19248640.0,19048064.0,19081792.0,19115520.0,19149248.0,19182976.0,19216704.0,19250432.0,19284160.0,19083136.0,19116928.0,19150720.0,19184512.0,19218304.0,19252096.0,19285888.0,19319680.0,19118208.0,19152064.0,19185920.0,19219776.0,19253632.0,19287488.0,19321344.0,19355200.0,18334080.0,18366976.0,18399872.0,18432768.0,18465664.0,18498560.0,18531456.0,18564352.0,17541760.0,17573696.0,17605632.0,17637568.0,17669504.0,17701440.0,17733376.0,17765312.0,16757632.0,16788608.0,16819584.0,16850560.0,16881536.0,16912512.0,16943488.0,16974464.0,15965312.0,15995328.0,16025344.0,16055360.0,16085376.0,16115392.0,16145408.0,16175424.0,16000384.0,16030464.0,16060544.0,16090624.0,16120704.0,16150784.0,16180864.0,16210944.0,16035456.0,16065600.0,16095744.0,16125888.0,16156032.0,16186176.0,16216320.0,16246464.0,16070528.0,16100736.0,16130944.0,16161152.0,16191360.0,16221568.0,16251776.0,16281984.0,16105600.0,16135872.0,16166144.0,16196416.0,16226688.0,16256960.0,16287232.0,16317504.0,16140672.0,16171008.0,16201344.0,16231680.0,16262016.0,16292352.0,16322688.0,16353024.0,16175744.0,16206144.0,16236544.0,16266944.0,16297344.0,16327744.0,16358144.0,16388544.0,16210816.0,16241280.0,16271744.0,16302208.0,16332672.0,16363136.0,16393600.0,16424064.0,16245888.0,16276416.0,16306944.0,16337472.0,16368000.0,16398528.0,16429056.0,16459584.0,16280960.0,16311552.0,16342144.0,16372736.0,16403328.0,16433920.0,16464512.0,16495104.0,16316032.0,16346688.0,16377344.0,16408000.0,16438656.0,16469312.0,16499968.0,16530624.0,16351104.0,16381824.0,16412544.0,16443264.0,16473984.0,16504704.0,16535424.0,16566144.0,16386176.0,16416960.0,16447744.0,16478528.0,16509312.0,16540096.0,16570880.0,16601664.0,16421248.0,16452096.0,16482944.0,16513792.0,16544640.0,16575488.0,16606336.0,16637184.0,16456320.0,16487232.0,16518144.0,16549056.0,16579968.0,16610880.0,16641792.0,16672704.0,16491392.0,16522368.0,16553344.0,16584320.0,16615296.0,16646272.0,16677248.0,16708224.0,16526464.0,16557504.0,16588544.0,16619584.0,16650624.0,16681664.0,16712704.0,16743744.0,16561536.0,16592640.0,16623744.0,16654848.0,16685952.0,16717056.0,16748160.0,16779264.0,16596608.0,16627776.0,16658944.0,16690112.0,16721280.0,16752448.0,16783616.0,16814784.0,16631680.0,16662912.0,16694144.0,16725376.0,16756608.0,16787840.0,16819072.0,16850304.0,16666752.0,16698048.0,16729344.0,16760640.0,16791936.0,16823232.0,16854528.0,16885824.0,16701824.0,16733184.0,16764544.0,16795904.0,16827264.0,16858624.0,16889984.0,16921344.0,16736896.0,16768320.0,16799744.0,16831168.0,16862592.0,16894016.0,16925440.0,16956864.0,16771968.0,16803456.0,16834944.0,16866432.0,16897920.0,16929408.0,16960896.0,16992384.0,16807040.0,16838592.0,16870144.0,16901696.0,16933248.0,16964800.0,16996352.0,17027904.0,16842112.0,16873728.0,16905344.0,16936960.0,16968576.0,17000192.0,17031808.0,17063424.0,16877184.0,16908864.0,16940544.0,16972224.0,17003904.0,17035584.0,17067264.0,17098944.0,16912256.0,16944000.0,16975744.0,17007488.0,17039232.0,17070976.0,17102720.0,17134464.0,16947328.0,16979136.0,17010944.0,17042752.0,17074560.0,17106368.0,17138176.0,17169984.0,16982400.0,17014272.0,17046144.0,17078016.0,17109888.0,17141760.0,17173632.0,17205504.0,17017472.0,17049408.0,17081344.0,17113280.0,17145216.0,17177152.0,17209088.0,17241024.0,17052544.0,17084544.0,17116544.0,17148544.0,17180544.0,17212544.0,17244544.0,17276544.0,17087616.0,17119680.0,17151744.0,17183808.0,17215872.0,17247936.0,17280000.0,17312064.0,17122688.0,17154816.0,17186944.0,17219072.0,17251200.0,17283328.0,17315456.0,17347584.0,17157760.0,17189952.0,17222144.0,17254336.0,17286528.0,17318720.0,17350912.0,17383104.0,17192832.0,17225088.0,17257344.0,17289600.0,17321856.0,17354112.0,17386368.0,17418624.0,17227904.0,17260224.0,17292544.0,17324864.0,17357184.0,17389504.0,17421824.0,17454144.0,17262976.0,17295360.0,17327744.0,17360128.0,17392512.0,17424896.0,17457280.0,17489664.0,17003136.0,17034560.0,17065984.0,17097408.0,17128832.0,17160256.0,17191680.0,17223104.0,16735104.0,16765568.0,16796032.0,16826496.0,16856960.0,16887424.0,16917888.0,16948352.0,16770176.0,16800704.0,16831232.0,16861760.0,16892288.0,16922816.0,16953344.0,16983872.0,16805248.0,16835840.0,16866432.0,16897024.0,16927616.0,16958208.0,16988800.0,17019392.0,16840320.0,16870976.0,16901632.0,16932288.0,16962944.0,16993600.0,17024256.0,17054912.0,16875392.0,16906112.0,16936832.0,16967552.0,16998272.0,17028992.0,17059712.0,17090432.0,16910464.0,16941248.0,16972032.0,17002816.0,17033600.0,17064384.0,17095168.0,17125952.0,16945536.0,16976384.0,17007232.0,17038080.0,17068928.0,17099776.0,17130624.0,17161472.0,16980608.0,17011520.0,17042432.0,17073344.0,17104256.0,17135168.0,17166080.0,17196992.0,17015680.0,17046656.0,17077632.0,17108608.0,17139584.0,17170560.0,17201536.0,17232512.0,17050752.0,17081792.0,17112832.0,17143872.0,17174912.0,17205952.0,17236992.0,17268032.0,17085824.0,17116928.0,17148032.0,17179136.0,17210240.0,17241344.0,17272448.0,17303552.0,17120896.0,17152064.0,17183232.0,17214400.0,17245568.0,17276736.0,17307904.0,17339072.0,17155968.0,17187200.0,17218432.0,17249664.0,17280896.0,17312128.0,17343360.0,17374592.0,17191040.0,17222336.0,17253632.0,17284928.0,17316224.0,17347520.0,17378816.0,17410112.0,17226112.0,17257472.0,17288832.0,17320192.0,17351552.0,17382912.0,17414272.0,17445632.0,17261184.0,17292608.0,17324032.0,17355456.0,17386880.0,17418304.0,17449728.0,17481152.0,17296256.0,17327744.0,17359232.0,17390720.0,17422208.0,17453696.0,17485184.0,17516672.0,17331328.0,17362880.0,17394432.0,17425984.0,17457536.0,17489088.0,17520640.0,17552192.0,17366400.0,17398016.0,17429632.0,17461248.0,17492864.0,17524480.0,17556096.0,17587712.0,17401472.0,17433152.0,17464832.0,17496512.0,17528192.0,17559872.0,17591552.0,17623232.0,17436544.0,17468288.0,17500032.0,17531776.0,17563520.0,17595264.0,17627008.0,17658752.0,17471616.0,17503424.0,17535232.0,17567040.0,17598848.0,17630656.0,17662464.0,17694272.0,17506688.0,17538560.0,17570432.0,17602304.0,17634176.0,17666048.0,17697920.0,17729792.0,17541760.0,17573696.0,17605632.0,17637568.0,17669504.0,17701440.0,17733376.0,17765312.0,17576832.0,17608832.0,17640832.0,17672832.0,17704832.0,17736832.0,17768832.0,17800832.0,17611904.0,17643968.0,17676032.0,17708096.0,17740160.0,17772224.0,17804288.0,17836352.0,17646976.0,17679104.0,17711232.0,17743360.0,17775488.0,17807616.0,17839744.0,17871872.0,17682048.0,17714240.0,17746432.0,17778624.0,17810816.0,17843008.0,17875200.0,17907392.0,17717120.0,17749376.0,17781632.0,17813888.0,17846144.0,17878400.0,17910656.0,17942912.0,17752192.0,17784512.0,17816832.0,17849152.0,17881472.0,17913792.0,17946112.0,17978432.0,17787264.0,17819648.0,17852032.0,17884416.0,17916800.0,17949184.0,17981568.0,18013952.0,17822336.0,17854784.0,17887232.0,17919680.0,17952128.0,17984576.0,18017024.0,18049472.0,17857408.0,17889920.0,17922432.0,17954944.0,17987456.0,18019968.0,18052480.0,18084992.0,17892480.0,17925056.0,17957632.0,17990208.0,18022784.0,18055360.0,18087936.0,18120512.0,17927552.0,17960192.0,17992832.0,18025472.0,18058112.0,18090752.0,18123392.0,18156032.0,17962624.0,17995328.0,18028032.0,18060736.0,18093440.0,18126144.0,18158848.0,18191552.0,17997696.0,18030464.0,18063232.0,18096000.0,18128768.0,18161536.0,18194304.0,18227072.0,18032768.0,18065600.0,18098432.0,18131264.0,18164096.0,18196928.0,18229760.0,18262592.0,17248640.0,17280512.0,17312384.0,17344256.0,17376128.0,17408000.0,17439872.0,17471744.0,16456320.0,16487232.0,16518144.0,16549056.0,16579968.0,16610880.0,16641792.0,16672704.0,16491392.0,16522368.0,16553344.0,16584320.0,16615296.0,16646272.0,16677248.0,16708224.0,16526464.0,16557504.0,16588544.0,16619584.0,16650624.0,16681664.0,16712704.0,16743744.0,16561536.0,16592640.0,16623744.0,16654848.0,16685952.0,16717056.0,16748160.0,16779264.0,16596608.0,16627776.0,16658944.0,16690112.0,16721280.0,16752448.0,16783616.0,16814784.0,16631680.0,16662912.0,16694144.0,16725376.0,16756608.0,16787840.0,16819072.0,16850304.0,16666752.0,16698048.0,16729344.0,16760640.0,16791936.0,16823232.0,16854528.0,16885824.0,16701824.0,16733184.0,16764544.0,16795904.0,16827264.0,16858624.0,16889984.0,16921344.0,16736896.0,16768320.0,16799744.0,16831168.0,16862592.0,16894016.0,16925440.0,16956864.0,16771968.0,16803456.0,16834944.0,16866432.0,16897920.0,16929408.0,16960896.0,16992384.0,16807040.0,16838592.0,16870144.0,16901696.0,16933248.0,16964800.0,16996352.0,17027904.0,16842112.0,16873728.0,16905344.0,16936960.0,16968576.0,17000192.0,17031808.0,17063424.0,16877184.0,16908864.0,16940544.0,16972224.0,17003904.0,17035584.0,17067264.0,17098944.0,16912256.0,16944000.0,16975744.0,17007488.0,17039232.0,17070976.0,17102720.0,17134464.0,16947328.0,16979136.0,17010944.0,17042752.0,17074560.0,17106368.0,17138176.0,17169984.0,16982400.0,17014272.0,17046144.0,17078016.0,17109888.0,17141760.0,17173632.0,17205504.0,17017472.0,17049408.0,17081344.0,17113280.0,17145216.0,17177152.0,17209088.0,17241024.0,17052544.0,17084544.0,17116544.0,17148544.0,17180544.0,17212544.0,17244544.0,17276544.0,17087616.0,17119680.0,17151744.0,17183808.0,17215872.0,17247936.0,17280000.0,17312064.0,17122688.0,17154816.0,17186944.0,17219072.0,17251200.0,17283328.0,17315456.0,17347584.0,17157760.0,17189952.0,17222144.0,17254336.0,17286528.0,17318720.0,17350912.0,17383104.0,17192832.0,17225088.0,17257344.0,17289600.0,17321856.0,17354112.0,17386368.0,17418624.0,17227904.0,17260224.0,17292544.0,17324864.0,17357184.0,17389504.0,17421824.0,17454144.0,17262976.0,17295360.0,17327744.0,17360128.0,17392512.0,17424896.0,17457280.0,17489664.0,17298048.0,17330496.0,17362944.0,17395392.0,17427840.0,17460288.0,17492736.0,17525184.0,17333120.0,17365632.0,17398144.0,17430656.0,17463168.0,17495680.0,17528192.0,17560704.0,17368192.0,17400768.0,17433344.0,17465920.0,17498496.0,17531072.0,17563648.0,17596224.0,17403264.0,17435904.0,17468544.0,17501184.0,17533824.0,17566464.0,17599104.0,17631744.0,17438336.0,17471040.0,17503744.0,17536448.0,17569152.0,17601856.0,17634560.0,17667264.0,17473408.0,17506176.0,17538944.0,17571712.0,17604480.0,17637248.0,17670016.0,17702784.0,17508480.0,17541312.0,17574144.0,17606976.0,17639808.0,17672640.0,17705472.0,17738304.0,17543552.0,17576448.0,17609344.0,17642240.0,17675136.0,17708032.0,17740928.0,17773824.0,17578624.0,17611584.0,17644544.0,17677504.0,17710464.0,17743424.0,17776384.0,17809344.0,17613696.0,17646720.0,17679744.0,17712768.0,17745792.0,17778816.0,17811840.0,17844864.0,17648768.0,17681856.0,17714944.0,17748032.0,17781120.0,17814208.0,17847296.0,17880384.0,17683840.0,17716992.0,17750144.0,17783296.0,17816448.0,17849600.0,17882752.0,17915904.0,17718912.0,17752128.0,17785344.0,17818560.0,17851776.0,17884992.0,17918208.0,17951424.0,17753984.0,17787264.0,17820544.0,17853824.0,17887104.0,17920384.0,17953664.0,17986944.0,17494144.0,17526464.0,17558784.0,17591104.0,17623424.0,17655744.0,17688064.0,17720384.0,17226112.0,17257472.0,17288832.0,17320192.0,17351552.0,17382912.0,17414272.0,17445632.0,17261184.0,17292608.0,17324032.0,17355456.0,17386880.0,17418304.0,17449728.0,17481152.0,17296256.0,17327744.0,17359232.0,17390720.0,17422208.0,17453696.0,17485184.0,17516672.0,17331328.0,17362880.0,17394432.0,17425984.0,17457536.0,17489088.0,17520640.0,17552192.0,17366400.0,17398016.0,17429632.0,17461248.0,17492864.0,17524480.0,17556096.0,17587712.0,17401472.0,17433152.0,17464832.0,17496512.0,17528192.0,17559872.0,17591552.0,17623232.0,17436544.0,17468288.0,17500032.0,17531776.0,17563520.0,17595264.0,17627008.0,17658752.0,17471616.0,17503424.0,17535232.0,17567040.0,17598848.0,17630656.0,17662464.0,17694272.0,17506688.0,17538560.0,17570432.0,17602304.0,17634176.0,17666048.0,17697920.0,17729792.0,17541760.0,17573696.0,17605632.0,17637568.0,17669504.0,17701440.0,17733376.0,17765312.0,17576832.0,17608832.0,17640832.0,17672832.0,17704832.0,17736832.0,17768832.0,17800832.0,17611904.0,17643968.0,17676032.0,17708096.0,17740160.0,17772224.0,17804288.0,17836352.0,17646976.0,17679104.0,17711232.0,17743360.0,17775488.0,17807616.0,17839744.0,17871872.0,17682048.0,17714240.0,17746432.0,17778624.0,17810816.0,17843008.0,17875200.0,17907392.0,17717120.0,17749376.0,17781632.0,17813888.0,17846144.0,17878400.0,17910656.0,17942912.0,17752192.0,17784512.0,17816832.0,17849152.0,17881472.0,17913792.0,17946112.0,17978432.0,17787264.0,17819648.0,17852032.0,17884416.0,17916800.0,17949184.0,17981568.0,18013952.0,17822336.0,17854784.0,17887232.0,17919680.0,17952128.0,17984576.0,18017024.0,18049472.0,17857408.0,17889920.0,17922432.0,17954944.0,17987456.0,18019968.0,18052480.0,18084992.0,17892480.0,17925056.0,17957632.0,17990208.0,18022784.0,18055360.0,18087936.0,18120512.0,17927552.0,17960192.0,17992832.0,18025472.0,18058112.0,18090752.0,18123392.0,18156032.0,17962624.0,17995328.0,18028032.0,18060736.0,18093440.0,18126144.0,18158848.0,18191552.0,17997696.0,18030464.0,18063232.0,18096000.0,18128768.0,18161536.0,18194304.0,18227072.0,18032768.0,18065600.0,18098432.0,18131264.0,18164096.0,18196928.0,18229760.0,18262592.0,18067840.0,18100736.0,18133632.0,18166528.0,18199424.0,18232320.0,18265216.0,18298112.0,18102912.0,18135872.0,18168832.0,18201792.0,18234752.0,18267712.0,18300672.0,18333632.0,18137984.0,18171008.0,18204032.0,18237056.0,18270080.0,18303104.0,18336128.0,18369152.0,18173056.0,18206144.0,18239232.0,18272320.0,18305408.0,18338496.0,18371584.0,18404672.0,18208128.0,18241280.0,18274432.0,18307584.0,18340736.0,18373888.0,18407040.0,18440192.0,18243200.0,18276416.0,18309632.0,18342848.0,18376064.0,18409280.0,18442496.0,18475712.0,18278272.0,18311552.0,18344832.0,18378112.0,18411392.0,18444672.0,18477952.0,18511232.0,18313344.0,18346688.0,18380032.0,18413376.0,18446720.0,18480064.0,18513408.0,18546752.0,18348416.0,18381824.0,18415232.0,18448640.0,18482048.0,18515456.0,18548864.0,18582272.0,18383488.0,18416960.0,18450432.0,18483904.0,18517376.0,18550848.0,18584320.0,18617792.0,18418560.0,18452096.0,18485632.0,18519168.0,18552704.0,18586240.0,18619776.0,18653312.0,18453632.0,18487232.0,18520832.0,18554432.0,18588032.0,18621632.0,18655232.0,18688832.0,18488704.0,18522368.0,18556032.0,18589696.0,18623360.0,18657024.0,18690688.0,18724352.0,18523776.0,18557504.0,18591232.0,18624960.0,18658688.0,18692416.0,18726144.0,18759872.0,17739648.0,17772416.0,17805184.0,17837952.0,17870720.0,17903488.0,17936256.0,17969024.0,16947328.0,16979136.0,17010944.0,17042752.0,17074560.0,17106368.0,17138176.0,17169984.0,8546176.0,8562096.0,8578016.0,8593936.0,8609856.0,8625776.0,8641696.0,8657616.0,9126144.0,9141936.0,9157728.0,9173520.0,9189312.0,9205104.0,9220896.0,9236688.0,18396544.0,18428160.0,18459776.0,18491392.0,18523008.0,18554624.0,18586240.0,18617856.0,18432640.0,18464320.0,18496000.0,18527680.0,18559360.0,18591040.0,18622720.0,18654400.0,18468736.0,18500480.0,18532224.0,18563968.0,18595712.0,18627456.0,18659200.0,18690944.0,18504832.0,18536640.0,18568448.0,18600256.0,18632064.0,18663872.0,18695680.0,18727488.0,18540928.0,18572800.0,18604672.0,18636544.0,18668416.0,18700288.0,18732160.0,18764032.0,18577024.0,18608960.0,18640896.0,18672832.0,18704768.0,18736704.0,18768640.0,18800576.0,18613120.0,18645120.0,18677120.0,18709120.0,18741120.0,18773120.0,18805120.0,18837120.0,18649216.0,18681280.0,18713344.0,18745408.0,18777472.0,18809536.0,18841600.0,18873664.0,18685312.0,18717440.0,18749568.0,18781696.0,18813824.0,18845952.0,18878080.0,18910208.0,18721408.0,18753600.0,18785792.0,18817984.0,18850176.0,18882368.0,18914560.0,18946752.0,18757504.0,18789760.0,18822016.0,18854272.0,18886528.0,18918784.0,18951040.0,18983296.0,18793600.0,18825920.0,18858240.0,18890560.0,18922880.0,18955200.0,18987520.0,19019840.0,18829696.0,18862080.0,18894464.0,18926848.0,18959232.0,18991616.0,19024000.0,19056384.0,18865792.0,18898240.0,18930688.0,18963136.0,18995584.0,19028032.0,19060480.0,19092928.0,18901888.0,18934400.0,18966912.0,18999424.0,19031936.0,19064448.0,19096960.0,19129472.0,18937984.0,18970560.0,19003136.0,19035712.0,19068288.0,19100864.0,19133440.0,19166016.0,18974080.0,19006720.0,19039360.0,19072000.0,19104640.0,19137280.0,19169920.0,19202560.0,19010176.0,19042880.0,19075584.0,19108288.0,19140992.0,19173696.0,19206400.0,19239104.0,19046272.0,19079040.0,19111808.0,19144576.0,19177344.0,19210112.0,19242880.0,19275648.0,19082368.0,19115200.0,19148032.0,19180864.0,19213696.0,19246528.0,19279360.0,19312192.0,19118464.0,19151360.0,19184256.0,19217152.0,19250048.0,19282944.0,19315840.0,19348736.0,19154560.0,19187520.0,19220480.0,19253440.0,19286400.0,19319360.0,19352320.0,19385280.0,19190656.0,19223680.0,19256704.0,19289728.0,19322752.0,19355776.0,19388800.0,19421824.0,19226752.0,19259840.0,19292928.0,19326016.0,19359104.0,19392192.0,19425280.0,19458368.0,19262848.0,19296000.0,19329152.0,19362304.0,19395456.0,19428608.0,19461760.0,19494912.0,19298944.0,19332160.0,19365376.0,19398592.0,19431808.0,19465024.0,19498240.0,19531456.0,19335040.0,19368320.0,19401600.0,19434880.0,19468160.0,19501440.0,19534720.0,19568000.0,19371136.0,19404480.0,19437824.0,19471168.0,19504512.0,19537856.0,19571200.0,19604544.0,19407232.0,19440640.0,19474048.0,19507456.0,19540864.0,19574272.0,19607680.0,19641088.0,19443328.0,19476800.0,19510272.0,19543744.0,19577216.0,19610688.0,19644160.0,19677632.0,19479424.0,19512960.0,19546496.0,19580032.0,19613568.0,19647104.0,19680640.0,19714176.0,19515520.0,19549120.0,19582720.0,19616320.0,19649920.0,19683520.0,19717120.0,19750720.0,19551616.0,19585280.0,19618944.0,19652608.0,19686272.0,19719936.0,19753600.0,19787264.0,19587712.0,19621440.0,19655168.0,19688896.0,19722624.0,19756352.0,19790080.0,19823808.0,19623808.0,19657600.0,19691392.0,19725184.0,19758976.0,19792768.0,19826560.0,19860352.0,19659904.0,19693760.0,19727616.0,19761472.0,19795328.0,19829184.0,19863040.0,19896896.0,18860416.0,18893312.0,18926208.0,18959104.0,18992000.0,19024896.0,19057792.0,19090688.0,18052736.0,18084672.0,18116608.0,18148544.0,18180480.0,18212416.0,18244352.0,18276288.0,17253248.0,17284224.0,17315200.0,17346176.0,17377152.0,17408128.0,17439104.0,17470080.0,16445568.0,16475584.0,16505600.0,16535616.0,16565632.0,16595648.0,16625664.0,16655680.0,16481664.0,16511744.0,16541824.0,16571904.0,16601984.0,16632064.0,16662144.0,16692224.0,16517760.0,16547904.0,16578048.0,16608192.0,16638336.0,16668480.0,16698624.0,16728768.0,16553856.0,16584064.0,16614272.0,16644480.0,16674688.0,16704896.0,16735104.0,16765312.0,16589952.0,16620224.0,16650496.0,16680768.0,16711040.0,16741312.0,16771584.0,16801856.0,16626048.0,16656384.0,16686720.0,16717056.0,16747392.0,16777728.0,16808064.0,16838400.0,16662144.0,16692544.0,16722944.0,16753344.0,16783744.0,16814144.0,16844544.0,16874944.0,16698240.0,16728704.0,16759168.0,16789632.0,16820096.0,16850560.0,16881024.0,16911488.0,16734336.0,16764864.0,16795392.0,16825920.0,16856448.0,16886976.0,16917504.0,16948032.0,16770432.0,16801024.0,16831616.0,16862208.0,16892800.0,16923392.0,16953984.0,16984576.0,16806528.0,16837184.0,16867840.0,16898496.0,16929152.0,16959808.0,16990464.0,17021120.0,16842624.0,16873344.0,16904064.0,16934784.0,16965504.0,16996224.0,17026944.0,17057664.0,16878720.0,16909504.0,16940288.0,16971072.0,17001856.0,17032640.0,17063424.0,17094208.0,16914816.0,16945664.0,16976512.0,17007360.0,17038208.0,17069056.0,17099904.0,17130752.0,16950912.0,16981824.0,17012736.0,17043648.0,17074560.0,17105472.0,17136384.0,17167296.0,16987008.0,17017984.0,17048960.0,17079936.0,17110912.0,17141888.0,17172864.0,17203840.0,17023104.0,17054144.0,17085184.0,17116224.0,17147264.0,17178304.0,17209344.0,17240384.0,17059200.0,17090304.0,17121408.0,17152512.0,17183616.0,17214720.0,17245824.0,17276928.0,17095296.0,17126464.0,17157632.0,17188800.0,17219968.0,17251136.0,17282304.0,17313472.0,17131392.0,17162624.0,17193856.0,17225088.0,17256320.0,17287552.0,17318784.0,17350016.0,17167488.0,17198784.0,17230080.0,17261376.0,17292672.0,17323968.0,17355264.0,17386560.0,17203584.0,17234944.0,17266304.0,17297664.0,17329024.0,17360384.0,17391744.0,17423104.0,17239680.0,17271104.0,17302528.0,17333952.0,17365376.0,17396800.0,17428224.0,17459648.0,17275776.0,17307264.0,17338752.0,17370240.0,17401728.0,17433216.0,17464704.0,17496192.0,17311872.0,17343424.0,17374976.0,17406528.0,17438080.0,17469632.0,17501184.0,17532736.0,17347968.0,17379584.0,17411200.0,17442816.0,17474432.0,17506048.0,17537664.0,17569280.0,17384064.0,17415744.0,17447424.0,17479104.0,17510784.0,17542464.0,17574144.0,17605824.0,17420160.0,17451904.0,17483648.0,17515392.0,17547136.0,17578880.0,17610624.0,17642368.0,17456256.0,17488064.0,17519872.0,17551680.0,17583488.0,17615296.0,17647104.0,17678912.0,17492352.0,17524224.0,17556096.0,17587968.0,17619840.0,17651712.0,17683584.0,17715456.0,17528448.0,17560384.0,17592320.0,17624256.0,17656192.0,17688128.0,17720064.0,17752000.0,17564544.0,17596544.0,17628544.0,17660544.0,17692544.0,17724544.0,17756544.0,17788544.0,17600640.0,17632704.0,17664768.0,17696832.0,17728896.0,17760960.0,17793024.0,17825088.0,17636736.0,17668864.0,17700992.0,17733120.0,17765248.0,17797376.0,17829504.0,17861632.0,17672832.0,17705024.0,17737216.0,17769408.0,17801600.0,17833792.0,17865984.0,17898176.0,17708928.0,17741184.0,17773440.0,17805696.0,17837952.0,17870208.0,17902464.0,17934720.0,17745024.0,17777344.0,17809664.0,17841984.0,17874304.0,17906624.0,17938944.0,17971264.0,17781120.0,17813504.0,17845888.0,17878272.0,17910656.0,17943040.0,17975424.0,18007808.0,17505920.0,17537344.0,17568768.0,17600192.0,17631616.0,17663040.0,17694464.0,17725888.0,17222528.0,17252992.0,17283456.0,17313920.0,17344384.0,17374848.0,17405312.0,17435776.0,17258624.0,17289152.0,17319680.0,17350208.0,17380736.0,17411264.0,17441792.0,17472320.0,17294720.0,17325312.0,17355904.0,17386496.0,17417088.0,17447680.0,17478272.0,17508864.0,17330816.0,17361472.0,17392128.0,17422784.0,17453440.0,17484096.0,17514752.0,17545408.0,17366912.0,17397632.0,17428352.0,17459072.0,17489792.0,17520512.0,17551232.0,17581952.0,17403008.0,17433792.0,17464576.0,17495360.0,17526144.0,17556928.0,17587712.0,17618496.0,17439104.0,17469952.0,17500800.0,17531648.0,17562496.0,17593344.0,17624192.0,17655040.0,17475200.0,17506112.0,17537024.0,17567936.0,17598848.0,17629760.0,17660672.0,17691584.0,17511296.0,17542272.0,17573248.0,17604224.0,17635200.0,17666176.0,17697152.0,17728128.0,17547392.0,17578432.0,17609472.0,17640512.0,17671552.0,17702592.0,17733632.0,17764672.0,17583488.0,17614592.0,17645696.0,17676800.0,17707904.0,17739008.0,17770112.0,17801216.0,17619584.0,17650752.0,17681920.0,17713088.0,17744256.0,17775424.0,17806592.0,17837760.0,17655680.0,17686912.0,17718144.0,17749376.0,17780608.0,17811840.0,17843072.0,17874304.0,17691776.0,17723072.0,17754368.0,17785664.0,17816960.0,17848256.0,17879552.0,17910848.0,17727872.0,17759232.0,17790592.0,17821952.0,17853312.0,17884672.0,17916032.0,17947392.0,17763968.0,17795392.0,17826816.0,17858240.0,17889664.0,17921088.0,17952512.0,17983936.0,17800064.0,17831552.0,17863040.0,17894528.0,17926016.0,17957504.0,17988992.0,18020480.0,17836160.0,17867712.0,17899264.0,17930816.0,17962368.0,17993920.0,18025472.0,18057024.0,17872256.0,17903872.0,17935488.0,17967104.0,17998720.0,18030336.0,18061952.0,18093568.0,17908352.0,17940032.0,17971712.0,18003392.0,18035072.0,18066752.0,18098432.0,18130112.0,17944448.0,17976192.0,18007936.0,18039680.0,18071424.0,18103168.0,18134912.0,18166656.0,17980544.0,18012352.0,18044160.0,18075968.0,18107776.0,18139584.0,18171392.0,18203200.0,18016640.0,18048512.0,18080384.0,18112256.0,18144128.0,18176000.0,18207872.0,18239744.0,18052736.0,18084672.0,18116608.0,18148544.0,18180480.0,18212416.0,18244352.0,18276288.0,18088832.0,18120832.0,18152832.0,18184832.0,18216832.0,18248832.0,18280832.0,18312832.0,18124928.0,18156992.0,18189056.0,18221120.0,18253184.0,18285248.0,18317312.0,18349376.0,18161024.0,18193152.0,18225280.0,18257408.0,18289536.0,18321664.0,18353792.0,18385920.0,18197120.0,18229312.0,18261504.0,18293696.0,18325888.0,18358080.0,18390272.0,18422464.0,18233216.0,18265472.0,18297728.0,18329984.0,18362240.0,18394496.0,18426752.0,18459008.0,18269312.0,18301632.0,18333952.0,18366272.0,18398592.0,18430912.0,18463232.0,18495552.0,18305408.0,18337792.0,18370176.0,18402560.0,18434944.0,18467328.0,18499712.0,18532096.0,18341504.0,18373952.0,18406400.0,18438848.0,18471296.0,18503744.0,18536192.0,18568640.0,18377600.0,18410112.0,18442624.0,18475136.0,18507648.0,18540160.0,18572672.0,18605184.0,18413696.0,18446272.0,18478848.0,18511424.0,18544000.0,18576576.0,18609152.0,18641728.0,18449792.0,18482432.0,18515072.0,18547712.0,18580352.0,18612992.0,18645632.0,18678272.0,18485888.0,18518592.0,18551296.0,18584000.0,18616704.0,18649408.0,18682112.0,18714816.0,18521984.0,18554752.0,18587520.0,18620288.0,18653056.0,18685824.0,18718592.0,18751360.0,18558080.0,18590912.0,18623744.0,18656576.0,18689408.0,18722240.0,18755072.0,18787904.0,17758592.0,17790464.0,17822336.0,17854208.0,17886080.0,17917952.0,17949824.0,17981696.0,16950912.0,16981824.0,17012736.0,17043648.0,17074560.0,17105472.0,17136384.0,17167296.0,16987008.0,17017984.0,17048960.0,17079936.0,17110912.0,17141888.0,17172864.0,17203840.0,17023104.0,17054144.0,17085184.0,17116224.0,17147264.0,17178304.0,17209344.0,17240384.0,17059200.0,17090304.0,17121408.0,17152512.0,17183616.0,17214720.0,17245824.0,17276928.0,17095296.0,17126464.0,17157632.0,17188800.0,17219968.0,17251136.0,17282304.0,17313472.0,17131392.0,17162624.0,17193856.0,17225088.0,17256320.0,17287552.0,17318784.0,17350016.0,17167488.0,17198784.0,17230080.0,17261376.0,17292672.0,17323968.0,17355264.0,17386560.0,17203584.0,17234944.0,17266304.0,17297664.0,17329024.0,17360384.0,17391744.0,17423104.0,17239680.0,17271104.0,17302528.0,17333952.0,17365376.0,17396800.0,17428224.0,17459648.0,17275776.0,17307264.0,17338752.0,17370240.0,17401728.0,17433216.0,17464704.0,17496192.0,17311872.0,17343424.0,17374976.0,17406528.0,17438080.0,17469632.0,17501184.0,17532736.0,17347968.0,17379584.0,17411200.0,17442816.0,17474432.0,17506048.0,17537664.0,17569280.0,17384064.0,17415744.0,17447424.0,17479104.0,17510784.0,17542464.0,17574144.0,17605824.0,17420160.0,17451904.0,17483648.0,17515392.0,17547136.0,17578880.0,17610624.0,17642368.0,17456256.0,17488064.0,17519872.0,17551680.0,17583488.0,17615296.0,17647104.0,17678912.0,17492352.0,17524224.0,17556096.0,17587968.0,17619840.0,17651712.0,17683584.0,17715456.0,17528448.0,17560384.0,17592320.0,17624256.0,17656192.0,17688128.0,17720064.0,17752000.0,17564544.0,17596544.0,17628544.0,17660544.0,17692544.0,17724544.0,17756544.0,17788544.0,17600640.0,17632704.0,17664768.0,17696832.0,17728896.0,17760960.0,17793024.0,17825088.0,17636736.0,17668864.0,17700992.0,17733120.0,17765248.0,17797376.0,17829504.0,17861632.0,17672832.0,17705024.0,17737216.0,17769408.0,17801600.0,17833792.0,17865984.0,17898176.0,17708928.0,17741184.0,17773440.0,17805696.0,17837952.0,17870208.0,17902464.0,17934720.0,17745024.0,17777344.0,17809664.0,17841984.0,17874304.0,17906624.0,17938944.0,17971264.0,17781120.0,17813504.0,17845888.0,17878272.0,17910656.0,17943040.0,17975424.0,18007808.0,17817216.0,17849664.0,17882112.0,17914560.0,17947008.0,17979456.0,18011904.0,18044352.0,17853312.0,17885824.0,17918336.0,17950848.0,17983360.0,18015872.0,18048384.0,18080896.0,17889408.0,17921984.0,17954560.0,17987136.0,18019712.0,18052288.0,18084864.0,18117440.0,17925504.0,17958144.0,17990784.0,18023424.0,18056064.0,18088704.0,18121344.0,18153984.0,17961600.0,17994304.0,18027008.0,18059712.0,18092416.0,18125120.0,18157824.0,18190528.0,17997696.0,18030464.0,18063232.0,18096000.0,18128768.0,18161536.0,18194304.0,18227072.0,18033792.0,18066624.0,18099456.0,18132288.0,18165120.0,18197952.0,18230784.0,18263616.0,18069888.0,18102784.0,18135680.0,18168576.0,18201472.0,18234368.0,18267264.0,18300160.0,18105984.0,18138944.0,18171904.0,18204864.0,18237824.0,18270784.0,18303744.0,18336704.0,18142080.0,18175104.0,18208128.0,18241152.0,18274176.0,18307200.0,18340224.0,18373248.0,18178176.0,18211264.0,18244352.0,18277440.0,18310528.0,18343616.0,18376704.0,18409792.0,18214272.0,18247424.0,18280576.0,18313728.0,18346880.0,18380032.0,18413184.0,18446336.0,18250368.0,18283584.0,18316800.0,18350016.0,18383232.0,18416448.0,18449664.0,18482880.0,18286464.0,18319744.0,18353024.0,18386304.0,18419584.0,18452864.0,18486144.0,18519424.0,18011264.0,18043584.0,18075904.0,18108224.0,18140544.0,18172864.0,18205184.0,18237504.0,17727872.0,17759232.0,17790592.0,17821952.0,17853312.0,17884672.0,17916032.0,17947392.0,17763968.0,17795392.0,17826816.0,17858240.0,17889664.0,17921088.0,17952512.0,17983936.0,17800064.0,17831552.0,17863040.0,17894528.0,17926016.0,17957504.0,17988992.0,18020480.0,17836160.0,17867712.0,17899264.0,17930816.0,17962368.0,17993920.0,18025472.0,18057024.0,17872256.0,17903872.0,17935488.0,17967104.0,17998720.0,18030336.0,18061952.0,18093568.0,17908352.0,17940032.0,17971712.0,18003392.0,18035072.0,18066752.0,18098432.0,18130112.0,17944448.0,17976192.0,18007936.0,18039680.0,18071424.0,18103168.0,18134912.0,18166656.0,17980544.0,18012352.0,18044160.0,18075968.0,18107776.0,18139584.0,18171392.0,18203200.0,18016640.0,18048512.0,18080384.0,18112256.0,18144128.0,18176000.0,18207872.0,18239744.0,18052736.0,18084672.0,18116608.0,18148544.0,18180480.0,18212416.0,18244352.0,18276288.0,18088832.0,18120832.0,18152832.0,18184832.0,18216832.0,18248832.0,18280832.0,18312832.0,18124928.0,18156992.0,18189056.0,18221120.0,18253184.0,18285248.0,18317312.0,18349376.0,18161024.0,18193152.0,18225280.0,18257408.0,18289536.0,18321664.0,18353792.0,18385920.0,18197120.0,18229312.0,18261504.0,18293696.0,18325888.0,18358080.0,18390272.0,18422464.0,18233216.0,18265472.0,18297728.0,18329984.0,18362240.0,18394496.0,18426752.0,18459008.0,18269312.0,18301632.0,18333952.0,18366272.0,18398592.0,18430912.0,18463232.0,18495552.0,18305408.0,18337792.0,18370176.0,18402560.0,18434944.0,18467328.0,18499712.0,18532096.0,18341504.0,18373952.0,18406400.0,18438848.0,18471296.0,18503744.0,18536192.0,18568640.0,18377600.0,18410112.0,18442624.0,18475136.0,18507648.0,18540160.0,18572672.0,18605184.0,18413696.0,18446272.0,18478848.0,18511424.0,18544000.0,18576576.0,18609152.0,18641728.0,18449792.0,18482432.0,18515072.0,18547712.0,18580352.0,18612992.0,18645632.0,18678272.0,18485888.0,18518592.0,18551296.0,18584000.0,18616704.0,18649408.0,18682112.0,18714816.0,18521984.0,18554752.0,18587520.0,18620288.0,18653056.0,18685824.0,18718592.0,18751360.0,18558080.0,18590912.0,18623744.0,18656576.0,18689408.0,18722240.0,18755072.0,18787904.0,18594176.0,18627072.0,18659968.0,18692864.0,18725760.0,18758656.0,18791552.0,18824448.0,18630272.0,18663232.0,18696192.0,18729152.0,18762112.0,18795072.0,18828032.0,18860992.0,18666368.0,18699392.0,18732416.0,18765440.0,18798464.0,18831488.0,18864512.0,18897536.0,18702464.0,18735552.0,18768640.0,18801728.0,18834816.0,18867904.0,18900992.0,18934080.0,18738560.0,18771712.0,18804864.0,18838016.0,18871168.0,18904320.0,18937472.0,18970624.0,18774656.0,18807872.0,18841088.0,18874304.0,18907520.0,18940736.0,18973952.0,19007168.0,18810752.0,18844032.0,18877312.0,18910592.0,18943872.0,18977152.0,19010432.0,19043712.0,18846848.0,18880192.0,18913536.0,18946880.0,18980224.0,19013568.0,19046912.0,19080256.0,18882944.0,18916352.0,18949760.0,18983168.0,19016576.0,19049984.0,19083392.0,19116800.0,18919040.0,18952512.0,18985984.0,19019456.0,19052928.0,19086400.0,19119872.0,19153344.0,18955136.0,18988672.0,19022208.0,19055744.0,19089280.0,19122816.0,19156352.0,19189888.0,18991232.0,19024832.0,19058432.0,19092032.0,19125632.0,19159232.0,19192832.0,19226432.0,19027328.0,19060992.0,19094656.0,19128320.0,19161984.0,19195648.0,19229312.0,19262976.0,19063424.0,19097152.0,19130880.0,19164608.0,19198336.0,19232064.0,19265792.0,19299520.0,18263936.0,18296704.0,18329472.0,18362240.0,18395008.0,18427776.0,18460544.0,18493312.0,17456256.0,17488064.0,17519872.0,17551680.0,17583488.0,17615296.0,17647104.0,17678912.0,8800896.0,8816816.0,8832736.0,8848656.0,8864576.0,8880496.0,8896416.0,8912336.0,9378816.0,9394608.0,9410400.0,9426192.0,9441984.0,9457776.0,9473568.0,9489360.0,18902400.0,18934016.0,18965632.0,18997248.0,19028864.0,19060480.0,19092096.0,19123712.0,18939520.0,18971200.0,19002880.0,19034560.0,19066240.0,19097920.0,19129600.0,19161280.0,18976640.0,19008384.0,19040128.0,19071872.0,19103616.0,19135360.0,19167104.0,19198848.0,19013760.0,19045568.0,19077376.0,19109184.0,19140992.0,19172800.0,19204608.0,19236416.0,19050880.0,19082752.0,19114624.0,19146496.0,19178368.0,19210240.0,19242112.0,19273984.0,19088000.0,19119936.0,19151872.0,19183808.0,19215744.0,19247680.0,19279616.0,19311552.0,19125120.0,19157120.0,19189120.0,19221120.0,19253120.0,19285120.0,19317120.0,19349120.0,19162240.0,19194304.0,19226368.0,19258432.0,19290496.0,19322560.0,19354624.0,19386688.0,19199360.0,19231488.0,19263616.0,19295744.0,19327872.0,19360000.0,19392128.0,19424256.0,19236480.0,19268672.0,19300864.0,19333056.0,19365248.0,19397440.0,19429632.0,19461824.0,19273600.0,19305856.0,19338112.0,19370368.0,19402624.0,19434880.0,19467136.0,19499392.0,19310720.0,19343040.0,19375360.0,19407680.0,19440000.0,19472320.0,19504640.0,19536960.0,19347840.0,19380224.0,19412608.0,19444992.0,19477376.0,19509760.0,19542144.0,19574528.0,19384960.0,19417408.0,19449856.0,19482304.0,19514752.0,19547200.0,19579648.0,19612096.0,19422080.0,19454592.0,19487104.0,19519616.0,19552128.0,19584640.0,19617152.0,19649664.0,19459200.0,19491776.0,19524352.0,19556928.0,19589504.0,19622080.0,19654656.0,19687232.0,19496320.0,19528960.0,19561600.0,19594240.0,19626880.0,19659520.0,19692160.0,19724800.0,19533440.0,19566144.0,19598848.0,19631552.0,19664256.0,19696960.0,19729664.0,19762368.0,19570560.0,19603328.0,19636096.0,19668864.0,19701632.0,19734400.0,19767168.0,19799936.0,19607680.0,19640512.0,19673344.0,19706176.0,19739008.0,19771840.0,19804672.0,19837504.0,19644800.0,19677696.0,19710592.0,19743488.0,19776384.0,19809280.0,19842176.0,19875072.0,19681920.0,19714880.0,19747840.0,19780800.0,19813760.0,19846720.0,19879680.0,19912640.0,19719040.0,19752064.0,19785088.0,19818112.0,19851136.0,19884160.0,19917184.0,19950208.0,19756160.0,19789248.0,19822336.0,19855424.0,19888512.0,19921600.0,19954688.0,19987776.0,19793280.0,19826432.0,19859584.0,19892736.0,19925888.0,19959040.0,19992192.0,20025344.0,19830400.0,19863616.0,19896832.0,19930048.0,19963264.0,19996480.0,20029696.0,20062912.0,19867520.0,19900800.0,19934080.0,19967360.0,20000640.0,20033920.0,20067200.0,20100480.0,19904640.0,19937984.0,19971328.0,20004672.0,20038016.0,20071360.0,20104704.0,20138048.0,19941760.0,19975168.0,20008576.0,20041984.0,20075392.0,20108800.0,20142208.0,20175616.0,19978880.0,20012352.0,20045824.0,20079296.0,20112768.0,20146240.0,20179712.0,20213184.0,20016000.0,20049536.0,20083072.0,20116608.0,20150144.0,20183680.0,20217216.0,20250752.0,20053120.0,20086720.0,20120320.0,20153920.0,20187520.0,20221120.0,20254720.0,20288320.0,20090240.0,20123904.0,20157568.0,20191232.0,20224896.0,20258560.0,20292224.0,20325888.0,20127360.0,20161088.0,20194816.0,20228544.0,20262272.0,20296000.0,20329728.0,20363456.0,20164480.0,20198272.0,20232064.0,20265856.0,20299648.0,20333440.0,20367232.0,20401024.0,20201600.0,20235456.0,20269312.0,20303168.0,20337024.0,20370880.0,20404736.0,20438592.0,19386752.0,19419648.0,19452544.0,19485440.0,19518336.0,19551232.0,19584128.0,19617024.0,18563712.0,18595648.0,18627584.0,18659520.0,18691456.0,18723392.0,18755328.0,18787264.0,17748864.0,17779840.0,17810816.0,17841792.0,17872768.0,17903744.0,17934720.0,17965696.0,16925824.0,16955840.0,16985856.0,17015872.0,17045888.0,17075904.0,17105920.0,17135936.0,16962944.0,16993024.0,17023104.0,17053184.0,17083264.0,17113344.0,17143424.0,17173504.0,17000064.0,17030208.0,17060352.0,17090496.0,17120640.0,17150784.0,17180928.0,17211072.0,17037184.0,17067392.0,17097600.0,17127808.0,17158016.0,17188224.0,17218432.0,17248640.0,17074304.0,17104576.0,17134848.0,17165120.0,17195392.0,17225664.0,17255936.0,17286208.0,17111424.0,17141760.0,17172096.0,17202432.0,17232768.0,17263104.0,17293440.0,17323776.0,17148544.0,17178944.0,17209344.0,17239744.0,17270144.0,17300544.0,17330944.0,17361344.0,17185664.0,17216128.0,17246592.0,17277056.0,17307520.0,17337984.0,17368448.0,17398912.0,17222784.0,17253312.0,17283840.0,17314368.0,17344896.0,17375424.0,17405952.0,17436480.0,17259904.0,17290496.0,17321088.0,17351680.0,17382272.0,17412864.0,17443456.0,17474048.0,17297024.0,17327680.0,17358336.0,17388992.0,17419648.0,17450304.0,17480960.0,17511616.0,17334144.0,17364864.0,17395584.0,17426304.0,17457024.0,17487744.0,17518464.0,17549184.0,17371264.0,17402048.0,17432832.0,17463616.0,17494400.0,17525184.0,17555968.0,17586752.0,17408384.0,17439232.0,17470080.0,17500928.0,17531776.0,17562624.0,17593472.0,17624320.0,17445504.0,17476416.0,17507328.0,17538240.0,17569152.0,17600064.0,17630976.0,17661888.0,17482624.0,17513600.0,17544576.0,17575552.0,17606528.0,17637504.0,17668480.0,17699456.0,17519744.0,17550784.0,17581824.0,17612864.0,17643904.0,17674944.0,17705984.0,17737024.0,17556864.0,17587968.0,17619072.0,17650176.0,17681280.0,17712384.0,17743488.0,17774592.0,17593984.0,17625152.0,17656320.0,17687488.0,17718656.0,17749824.0,17780992.0,17812160.0,17631104.0,17662336.0,17693568.0,17724800.0,17756032.0,17787264.0,17818496.0,17849728.0,17668224.0,17699520.0,17730816.0,17762112.0,17793408.0,17824704.0,17856000.0,17887296.0,17705344.0,17736704.0,17768064.0,17799424.0,17830784.0,17862144.0,17893504.0,17924864.0,17742464.0,17773888.0,17805312.0,17836736.0,17868160.0,17899584.0,17931008.0,17962432.0,17779584.0,17811072.0,17842560.0,17874048.0,17905536.0,17937024.0,17968512.0,18000000.0,17816704.0,17848256.0,17879808.0,17911360.0,17942912.0,17974464.0,18006016.0,18037568.0,17853824.0,17885440.0,17917056.0,17948672.0,17980288.0,18011904.0,18043520.0,18075136.0,17890944.0,17922624.0,17954304.0,17985984.0,18017664.0,18049344.0,18081024.0,18112704.0,17928064.0,17959808.0,17991552.0,18023296.0,18055040.0,18086784.0,18118528.0,18150272.0,17965184.0,17996992.0,18028800.0,18060608.0,18092416.0,18124224.0,18156032.0,18187840.0,18002304.0,18034176.0,18066048.0,18097920.0,18129792.0,18161664.0,18193536.0,18225408.0,18039424.0,18071360.0,18103296.0,18135232.0,18167168.0,18199104.0,18231040.0,18262976.0,18076544.0,18108544.0,18140544.0,18172544.0,18204544.0,18236544.0,18268544.0,18300544.0,18113664.0,18145728.0,18177792.0,18209856.0,18241920.0,18273984.0,18306048.0,18338112.0,18150784.0,18182912.0,18215040.0,18247168.0,18279296.0,18311424.0,18343552.0,18375680.0,18187904.0,18220096.0,18252288.0,18284480.0,18316672.0,18348864.0,18381056.0,18413248.0,18225024.0,18257280.0,18289536.0,18321792.0,18354048.0,18386304.0,18418560.0,18450816.0,18262144.0,18294464.0,18326784.0,18359104.0,18391424.0,18423744.0,18456064.0,18488384.0,18299264.0,18331648.0,18364032.0,18396416.0,18428800.0,18461184.0,18493568.0,18525952.0,18008704.0,18040128.0,18071552.0,18102976.0,18134400.0,18165824.0,18197248.0,18228672.0,17709952.0,17740416.0,17770880.0,17801344.0,17831808.0,17862272.0,17892736.0,17923200.0,17747072.0,17777600.0,17808128.0,17838656.0,17869184.0,17899712.0,17930240.0,17960768.0,17784192.0,17814784.0,17845376.0,17875968.0,17906560.0,17937152.0,17967744.0,17998336.0,17821312.0,17851968.0,17882624.0,17913280.0,17943936.0,17974592.0,18005248.0,18035904.0,17858432.0,17889152.0,17919872.0,17950592.0,17981312.0,18012032.0,18042752.0,18073472.0,17895552.0,17926336.0,17957120.0,17987904.0,18018688.0,18049472.0,18080256.0,18111040.0,17932672.0,17963520.0,17994368.0,18025216.0,18056064.0,18086912.0,18117760.0,18148608.0,17969792.0,18000704.0,18031616.0,18062528.0,18093440.0,18124352.0,18155264.0,18186176.0,18006912.0,18037888.0,18068864.0,18099840.0,18130816.0,18161792.0,18192768.0,18223744.0,18044032.0,18075072.0,18106112.0,18137152.0,18168192.0,18199232.0,18230272.0,18261312.0,18081152.0,18112256.0,18143360.0,18174464.0,18205568.0,18236672.0,18267776.0,18298880.0,18118272.0,18149440.0,18180608.0,18211776.0,18242944.0,18274112.0,18305280.0,18336448.0,18155392.0,18186624.0,18217856.0,18249088.0,18280320.0,18311552.0,18342784.0,18374016.0,18192512.0,18223808.0,18255104.0,18286400.0,18317696.0,18348992.0,18380288.0,18411584.0,18229632.0,18260992.0,18292352.0,18323712.0,18355072.0,18386432.0,18417792.0,18449152.0,18266752.0,18298176.0,18329600.0,18361024.0,18392448.0,18423872.0,18455296.0,18486720.0,18303872.0,18335360.0,18366848.0,18398336.0,18429824.0,18461312.0,18492800.0,18524288.0,18340992.0,18372544.0,18404096.0,18435648.0,18467200.0,18498752.0,18530304.0,18561856.0,18378112.0,18409728.0,18441344.0,18472960.0,18504576.0,18536192.0,18567808.0,18599424.0,18415232.0,18446912.0,18478592.0,18510272.0,18541952.0,18573632.0,18605312.0,18636992.0,18452352.0,18484096.0,18515840.0,18547584.0,18579328.0,18611072.0,18642816.0,18674560.0,18489472.0,18521280.0,18553088.0,18584896.0,18616704.0,18648512.0,18680320.0,18712128.0,18526592.0,18558464.0,18590336.0,18622208.0,18654080.0,18685952.0,18717824.0,18749696.0,18563712.0,18595648.0,18627584.0,18659520.0,18691456.0,18723392.0,18755328.0,18787264.0,18600832.0,18632832.0,18664832.0,18696832.0,18728832.0,18760832.0,18792832.0,18824832.0,18637952.0,18670016.0,18702080.0,18734144.0,18766208.0,18798272.0,18830336.0,18862400.0,18675072.0,18707200.0,18739328.0,18771456.0,18803584.0,18835712.0,18867840.0,18899968.0,18712192.0,18744384.0,18776576.0,18808768.0,18840960.0,18873152.0,18905344.0,18937536.0,18749312.0,18781568.0,18813824.0,18846080.0,18878336.0,18910592.0,18942848.0,18975104.0,18786432.0,18818752.0,18851072.0,18883392.0,18915712.0,18948032.0,18980352.0,19012672.0,18823552.0,18855936.0,18888320.0,18920704.0,18953088.0,18985472.0,19017856.0,19050240.0,18860672.0,18893120.0,18925568.0,18958016.0,18990464.0,19022912.0,19055360.0,19087808.0,18897792.0,18930304.0,18962816.0,18995328.0,19027840.0,19060352.0,19092864.0,19125376.0,18934912.0,18967488.0,19000064.0,19032640.0,19065216.0,19097792.0,19130368.0,19162944.0,18972032.0,19004672.0,19037312.0,19069952.0,19102592.0,19135232.0,19167872.0,19200512.0,19009152.0,19041856.0,19074560.0,19107264.0,19139968.0,19172672.0,19205376.0,19238080.0,19046272.0,19079040.0,19111808.0,19144576.0,19177344.0,19210112.0,19242880.0,19275648.0,19083392.0,19116224.0,19149056.0,19181888.0,19214720.0,19247552.0,19280384.0,19313216.0,18268544.0,18300416.0,18332288.0,18364160.0,18396032.0,18427904.0,18459776.0,18491648.0,17445504.0,17476416.0,17507328.0,17538240.0,17569152.0,17600064.0,17630976.0,17661888.0,17482624.0,17513600.0,17544576.0,17575552.0,17606528.0,17637504.0,17668480.0,17699456.0,17519744.0,17550784.0,17581824.0,17612864.0,17643904.0,17674944.0,17705984.0,17737024.0,17556864.0,17587968.0,17619072.0,17650176.0,17681280.0,17712384.0,17743488.0,17774592.0,17593984.0,17625152.0,17656320.0,17687488.0,17718656.0,17749824.0,17780992.0,17812160.0,17631104.0,17662336.0,17693568.0,17724800.0,17756032.0,17787264.0,17818496.0,17849728.0,17668224.0,17699520.0,17730816.0,17762112.0,17793408.0,17824704.0,17856000.0,17887296.0,17705344.0,17736704.0,17768064.0,17799424.0,17830784.0,17862144.0,17893504.0,17924864.0,17742464.0,17773888.0,17805312.0,17836736.0,17868160.0,17899584.0,17931008.0,17962432.0,17779584.0,17811072.0,17842560.0,17874048.0,17905536.0,17937024.0,17968512.0,18000000.0,17816704.0,17848256.0,17879808.0,17911360.0,17942912.0,17974464.0,18006016.0,18037568.0,17853824.0,17885440.0,17917056.0,17948672.0,17980288.0,18011904.0,18043520.0,18075136.0,17890944.0,17922624.0,17954304.0,17985984.0,18017664.0,18049344.0,18081024.0,18112704.0,17928064.0,17959808.0,17991552.0,18023296.0,18055040.0,18086784.0,18118528.0,18150272.0,17965184.0,17996992.0,18028800.0,18060608.0,18092416.0,18124224.0,18156032.0,18187840.0,18002304.0,18034176.0,18066048.0,18097920.0,18129792.0,18161664.0,18193536.0,18225408.0,18039424.0,18071360.0,18103296.0,18135232.0,18167168.0,18199104.0,18231040.0,18262976.0,18076544.0,18108544.0,18140544.0,18172544.0,18204544.0,18236544.0,18268544.0,18300544.0,18113664.0,18145728.0,18177792.0,18209856.0,18241920.0,18273984.0,18306048.0,18338112.0,18150784.0,18182912.0,18215040.0,18247168.0,18279296.0,18311424.0,18343552.0,18375680.0,18187904.0,18220096.0,18252288.0,18284480.0,18316672.0,18348864.0,18381056.0,18413248.0,18225024.0,18257280.0,18289536.0,18321792.0,18354048.0,18386304.0,18418560.0,18450816.0,18262144.0,18294464.0,18326784.0,18359104.0,18391424.0,18423744.0,18456064.0,18488384.0,18299264.0,18331648.0,18364032.0,18396416.0,18428800.0,18461184.0,18493568.0,18525952.0,18336384.0,18368832.0,18401280.0,18433728.0,18466176.0,18498624.0,18531072.0,18563520.0,18373504.0,18406016.0,18438528.0,18471040.0,18503552.0,18536064.0,18568576.0,18601088.0,18410624.0,18443200.0,18475776.0,18508352.0,18540928.0,18573504.0,18606080.0,18638656.0,18447744.0,18480384.0,18513024.0,18545664.0,18578304.0,18610944.0,18643584.0,18676224.0,18484864.0,18517568.0,18550272.0,18582976.0,18615680.0,18648384.0,18681088.0,18713792.0,18521984.0,18554752.0,18587520.0,18620288.0,18653056.0,18685824.0,18718592.0,18751360.0,18559104.0,18591936.0,18624768.0,18657600.0,18690432.0,18723264.0,18756096.0,18788928.0,18596224.0,18629120.0,18662016.0,18694912.0,18727808.0,18760704.0,18793600.0,18826496.0,18633344.0,18666304.0,18699264.0,18732224.0,18765184.0,18798144.0,18831104.0,18864064.0,18670464.0,18703488.0,18736512.0,18769536.0,18802560.0,18835584.0,18868608.0,18901632.0,18707584.0,18740672.0,18773760.0,18806848.0,18839936.0,18873024.0,18906112.0,18939200.0,18744704.0,18777856.0,18811008.0,18844160.0,18877312.0,18910464.0,18943616.0,18976768.0,18781824.0,18815040.0,18848256.0,18881472.0,18914688.0,18947904.0,18981120.0,19014336.0,18818944.0,18852224.0,18885504.0,18918784.0,18952064.0,18985344.0,19018624.0,19051904.0,18528384.0,18560704.0,18593024.0,18625344.0,18657664.0,18689984.0,18722304.0,18754624.0,18229632.0,18260992.0,18292352.0,18323712.0,18355072.0,18386432.0,18417792.0,18449152.0,18266752.0,18298176.0,18329600.0,18361024.0,18392448.0,18423872.0,18455296.0,18486720.0,18303872.0,18335360.0,18366848.0,18398336.0,18429824.0,18461312.0,18492800.0,18524288.0,18340992.0,18372544.0,18404096.0,18435648.0,18467200.0,18498752.0,18530304.0,18561856.0,18378112.0,18409728.0,18441344.0,18472960.0,18504576.0,18536192.0,18567808.0,18599424.0,18415232.0,18446912.0,18478592.0,18510272.0,18541952.0,18573632.0,18605312.0,18636992.0,18452352.0,18484096.0,18515840.0,18547584.0,18579328.0,18611072.0,18642816.0,18674560.0,18489472.0,18521280.0,18553088.0,18584896.0,18616704.0,18648512.0,18680320.0,18712128.0,18526592.0,18558464.0,18590336.0,18622208.0,18654080.0,18685952.0,18717824.0,18749696.0,18563712.0,18595648.0,18627584.0,18659520.0,18691456.0,18723392.0,18755328.0,18787264.0,18600832.0,18632832.0,18664832.0,18696832.0,18728832.0,18760832.0,18792832.0,18824832.0,18637952.0,18670016.0,18702080.0,18734144.0,18766208.0,18798272.0,18830336.0,18862400.0,18675072.0,18707200.0,18739328.0,18771456.0,18803584.0,18835712.0,18867840.0,18899968.0,18712192.0,18744384.0,18776576.0,18808768.0,18840960.0,18873152.0,18905344.0,18937536.0,18749312.0,18781568.0,18813824.0,18846080.0,18878336.0,18910592.0,18942848.0,18975104.0,18786432.0,18818752.0,18851072.0,18883392.0,18915712.0,18948032.0,18980352.0,19012672.0,18823552.0,18855936.0,18888320.0,18920704.0,18953088.0,18985472.0,19017856.0,19050240.0,18860672.0,18893120.0,18925568.0,18958016.0,18990464.0,19022912.0,19055360.0,19087808.0,18897792.0,18930304.0,18962816.0,18995328.0,19027840.0,19060352.0,19092864.0,19125376.0,18934912.0,18967488.0,19000064.0,19032640.0,19065216.0,19097792.0,19130368.0,19162944.0,18972032.0,19004672.0,19037312.0,19069952.0,19102592.0,19135232.0,19167872.0,19200512.0,19009152.0,19041856.0,19074560.0,19107264.0,19139968.0,19172672.0,19205376.0,19238080.0,19046272.0,19079040.0,19111808.0,19144576.0,19177344.0,19210112.0,19242880.0,19275648.0,19083392.0,19116224.0,19149056.0,19181888.0,19214720.0,19247552.0,19280384.0,19313216.0,19120512.0,19153408.0,19186304.0,19219200.0,19252096.0,19284992.0,19317888.0,19350784.0,19157632.0,19190592.0,19223552.0,19256512.0,19289472.0,19322432.0,19355392.0,19388352.0,19194752.0,19227776.0,19260800.0,19293824.0,19326848.0,19359872.0,19392896.0,19425920.0,19231872.0,19264960.0,19298048.0,19331136.0,19364224.0,19397312.0,19430400.0,19463488.0,19268992.0,19302144.0,19335296.0,19368448.0,19401600.0,19434752.0,19467904.0,19501056.0,19306112.0,19339328.0,19372544.0,19405760.0,19438976.0,19472192.0,19505408.0,19538624.0,19343232.0,19376512.0,19409792.0,19443072.0,19476352.0,19509632.0,19542912.0,19576192.0,19380352.0,19413696.0,19447040.0,19480384.0,19513728.0,19547072.0,19580416.0,19613760.0,19417472.0,19450880.0,19484288.0,19517696.0,19551104.0,19584512.0,19617920.0,19651328.0,19454592.0,19488064.0,19521536.0,19555008.0,19588480.0,19621952.0,19655424.0,19688896.0,19491712.0,19525248.0,19558784.0,19592320.0,19625856.0,19659392.0,19692928.0,19726464.0,19528832.0,19562432.0,19596032.0,19629632.0,19663232.0,19696832.0,19730432.0,19764032.0,19565952.0,19599616.0,19633280.0,19666944.0,19700608.0,19734272.0,19767936.0,19801600.0,19603072.0,19636800.0,19670528.0,19704256.0,19737984.0,19771712.0,19805440.0,19839168.0,18788224.0,18820992.0,18853760.0,18886528.0,18919296.0,18952064.0,18984832.0,19017600.0,17965184.0,17996992.0,18028800.0,18060608.0,18092416.0,18124224.0,18156032.0,18187840.0,9055616.0,9071536.0,9087456.0,9103376.0,9119296.0,9135216.0,9151136.0,9167056.0,9631488.0,9647280.0,9663072.0,9678864.0,9694656.0,9710448.0,9726240.0,9742032.0,19408256.0,19439872.0,19471488.0,19503104.0,19534720.0,19566336.0,19597952.0,19629568.0,19446400.0,19478080.0,19509760.0,19541440.0,19573120.0,19604800.0,19636480.0,19668160.0,19484544.0,19516288.0,19548032.0,19579776.0,19611520.0,19643264.0,19675008.0,19706752.0,19522688.0,19554496.0,19586304.0,19618112.0,19649920.0,19681728.0,19713536.0,19745344.0,19560832.0,19592704.0,19624576.0,19656448.0,19688320.0,19720192.0,19752064.0,19783936.0,19598976.0,19630912.0,19662848.0,19694784.0,19726720.0,19758656.0,19790592.0,19822528.0,19637120.0,19669120.0,19701120.0,19733120.0,19765120.0,19797120.0,19829120.0,19861120.0,19675264.0,19707328.0,19739392.0,19771456.0,19803520.0,19835584.0,19867648.0,19899712.0,19713408.0,19745536.0,19777664.0,19809792.0,19841920.0,19874048.0,19906176.0,19938304.0,19751552.0,19783744.0,19815936.0,19848128.0,19880320.0,19912512.0,19944704.0,19976896.0,19789696.0,19821952.0,19854208.0,19886464.0,19918720.0,19950976.0,19983232.0,20015488.0,19827840.0,19860160.0,19892480.0,19924800.0,19957120.0,19989440.0,20021760.0,20054080.0,19865984.0,19898368.0,19930752.0,19963136.0,19995520.0,20027904.0,20060288.0,20092672.0,19904128.0,19936576.0,19969024.0,20001472.0,20033920.0,20066368.0,20098816.0,20131264.0,19942272.0,19974784.0,20007296.0,20039808.0,20072320.0,20104832.0,20137344.0,20169856.0,19980416.0,20012992.0,20045568.0,20078144.0,20110720.0,20143296.0,20175872.0,20208448.0,20018560.0,20051200.0,20083840.0,20116480.0,20149120.0,20181760.0,20214400.0,20247040.0,20056704.0,20089408.0,20122112.0,20154816.0,20187520.0,20220224.0,20252928.0,20285632.0,20094848.0,20127616.0,20160384.0,20193152.0,20225920.0,20258688.0,20291456.0,20324224.0,20132992.0,20165824.0,20198656.0,20231488.0,20264320.0,20297152.0,20329984.0,20362816.0,20171136.0,20204032.0,20236928.0,20269824.0,20302720.0,20335616.0,20368512.0,20401408.0,20209280.0,20242240.0,20275200.0,20308160.0,20341120.0,20374080.0,20407040.0,20440000.0,20247424.0,20280448.0,20313472.0,20346496.0,20379520.0,20412544.0,20445568.0,20478592.0,20285568.0,20318656.0,20351744.0,20384832.0,20417920.0,20451008.0,20484096.0,20517184.0,20323712.0,20356864.0,20390016.0,20423168.0,20456320.0,20489472.0,20522624.0,20555776.0,20361856.0,20395072.0,20428288.0,20461504.0,20494720.0,20527936.0,20561152.0,20594368.0,20400000.0,20433280.0,20466560.0,20499840.0,20533120.0,20566400.0,20599680.0,20632960.0,20438144.0,20471488.0,20504832.0,20538176.0,20571520.0,20604864.0,20638208.0,20671552.0,20476288.0,20509696.0,20543104.0,20576512.0,20609920.0,20643328.0,20676736.0,20710144.0,20514432.0,20547904.0,20581376.0,20614848.0,20648320.0,20681792.0,20715264.0,20748736.0,20552576.0,20586112.0,20619648.0,20653184.0,20686720.0,20720256.0,20753792.0,20787328.0,20590720.0,20624320.0,20657920.0,20691520.0,20725120.0,20758720.0,20792320.0,20825920.0,20628864.0,20662528.0,20696192.0,20729856.0,20763520.0,20797184.0,20830848.0,20864512.0,20667008.0,20700736.0,20734464.0,20768192.0,20801920.0,20835648.0,20869376.0,20903104.0,20705152.0,20738944.0,20772736.0,20806528.0,20840320.0,20874112.0,20907904.0,20941696.0,20743296.0,20777152.0,20811008.0,20844864.0,20878720.0,20912576.0,20946432.0,20980288.0,19913088.0,19945984.0,19978880.0,20011776.0,20044672.0,20077568.0,20110464.0,20143360.0,19074688.0,19106624.0,19138560.0,19170496.0,19202432.0,19234368.0,19266304.0,19298240.0,18244480.0,18275456.0,18306432.0,18337408.0,18368384.0,18399360.0,18430336.0,18461312.0,17406080.0,17436096.0,17466112.0,17496128.0,17526144.0,17556160.0,17586176.0,17616192.0,17444224.0,17474304.0,17504384.0,17534464.0,17564544.0,17594624.0,17624704.0,17654784.0,17482368.0,17512512.0,17542656.0,17572800.0,17602944.0,17633088.0,17663232.0,17693376.0,17520512.0,17550720.0,17580928.0,17611136.0,17641344.0,17671552.0,17701760.0,17731968.0,17558656.0,17588928.0,17619200.0,17649472.0,17679744.0,17710016.0,17740288.0,17770560.0,17596800.0,17627136.0,17657472.0,17687808.0,17718144.0,17748480.0,17778816.0,17809152.0,17634944.0,17665344.0,17695744.0,17726144.0,17756544.0,17786944.0,17817344.0,17847744.0,17673088.0,17703552.0,17734016.0,17764480.0,17794944.0,17825408.0,17855872.0,17886336.0,17711232.0,17741760.0,17772288.0,17802816.0,17833344.0,17863872.0,17894400.0,17924928.0,17749376.0,17779968.0,17810560.0,17841152.0,17871744.0,17902336.0,17932928.0,17963520.0,17787520.0,17818176.0,17848832.0,17879488.0,17910144.0,17940800.0,17971456.0,18002112.0,17825664.0,17856384.0,17887104.0,17917824.0,17948544.0,17979264.0,18009984.0,18040704.0,17863808.0,17894592.0,17925376.0,17956160.0,17986944.0,18017728.0,18048512.0,18079296.0,17901952.0,17932800.0,17963648.0,17994496.0,18025344.0,18056192.0,18087040.0,18117888.0,17940096.0,17971008.0,18001920.0,18032832.0,18063744.0,18094656.0,18125568.0,18156480.0,17978240.0,18009216.0,18040192.0,18071168.0,18102144.0,18133120.0,18164096.0,18195072.0,18016384.0,18047424.0,18078464.0,18109504.0,18140544.0,18171584.0,18202624.0,18233664.0,18054528.0,18085632.0,18116736.0,18147840.0,18178944.0,18210048.0,18241152.0,18272256.0,18092672.0,18123840.0,18155008.0,18186176.0,18217344.0,18248512.0,18279680.0,18310848.0,18130816.0,18162048.0,18193280.0,18224512.0,18255744.0,18286976.0,18318208.0,18349440.0,18168960.0,18200256.0,18231552.0,18262848.0,18294144.0,18325440.0,18356736.0,18388032.0,18207104.0,18238464.0,18269824.0,18301184.0,18332544.0,18363904.0,18395264.0,18426624.0,18245248.0,18276672.0,18308096.0,18339520.0,18370944.0,18402368.0,18433792.0,18465216.0,18283392.0,18314880.0,18346368.0,18377856.0,18409344.0,18440832.0,18472320.0,18503808.0,18321536.0,18353088.0,18384640.0,18416192.0,18447744.0,18479296.0,18510848.0,18542400.0,18359680.0,18391296.0,18422912.0,18454528.0,18486144.0,18517760.0,18549376.0,18580992.0,18397824.0,18429504.0,18461184.0,18492864.0,18524544.0,18556224.0,18587904.0,18619584.0,18435968.0,18467712.0,18499456.0,18531200.0,18562944.0,18594688.0,18626432.0,18658176.0,18474112.0,18505920.0,18537728.0,18569536.0,18601344.0,18633152.0,18664960.0,18696768.0,18512256.0,18544128.0,18576000.0,18607872.0,18639744.0,18671616.0,18703488.0,18735360.0,18550400.0,18582336.0,18614272.0,18646208.0,18678144.0,18710080.0,18742016.0,18773952.0,18588544.0,18620544.0,18652544.0,18684544.0,18716544.0,18748544.0,18780544.0,18812544.0,18626688.0,18658752.0,18690816.0,18722880.0,18754944.0,18787008.0,18819072.0,18851136.0,18664832.0,18696960.0,18729088.0,18761216.0,18793344.0,18825472.0,18857600.0,18889728.0,18702976.0,18735168.0,18767360.0,18799552.0,18831744.0,18863936.0,18896128.0,18928320.0,18741120.0,18773376.0,18805632.0,18837888.0,18870144.0,18902400.0,18934656.0,18966912.0,18779264.0,18811584.0,18843904.0,18876224.0,18908544.0,18940864.0,18973184.0,19005504.0,18817408.0,18849792.0,18882176.0,18914560.0,18946944.0,18979328.0,19011712.0,19044096.0,18511488.0,18542912.0,18574336.0,18605760.0,18637184.0,18668608.0,18700032.0,18731456.0,18197376.0,18227840.0,18258304.0,18288768.0,18319232.0,18349696.0,18380160.0,18410624.0,18235520.0,18266048.0,18296576.0,18327104.0,18357632.0,18388160.0,18418688.0,18449216.0,18273664.0,18304256.0,18334848.0,18365440.0,18396032.0,18426624.0,18457216.0,18487808.0,18311808.0,18342464.0,18373120.0,18403776.0,18434432.0,18465088.0,18495744.0,18526400.0,18349952.0,18380672.0,18411392.0,18442112.0,18472832.0,18503552.0,18534272.0,18564992.0,18388096.0,18418880.0,18449664.0,18480448.0,18511232.0,18542016.0,18572800.0,18603584.0,18426240.0,18457088.0,18487936.0,18518784.0,18549632.0,18580480.0,18611328.0,18642176.0,18464384.0,18495296.0,18526208.0,18557120.0,18588032.0,18618944.0,18649856.0,18680768.0,18502528.0,18533504.0,18564480.0,18595456.0,18626432.0,18657408.0,18688384.0,18719360.0,18540672.0,18571712.0,18602752.0,18633792.0,18664832.0,18695872.0,18726912.0,18757952.0,18578816.0,18609920.0,18641024.0,18672128.0,18703232.0,18734336.0,18765440.0,18796544.0,18616960.0,18648128.0,18679296.0,18710464.0,18741632.0,18772800.0,18803968.0,18835136.0,18655104.0,18686336.0,18717568.0,18748800.0,18780032.0,18811264.0,18842496.0,18873728.0,18693248.0,18724544.0,18755840.0,18787136.0,18818432.0,18849728.0,18881024.0,18912320.0,18731392.0,18762752.0,18794112.0,18825472.0,18856832.0,18888192.0,18919552.0,18950912.0,18769536.0,18800960.0,18832384.0,18863808.0,18895232.0,18926656.0,18958080.0,18989504.0,18807680.0,18839168.0,18870656.0,18902144.0,18933632.0,18965120.0,18996608.0,19028096.0,18845824.0,18877376.0,18908928.0,18940480.0,18972032.0,19003584.0,19035136.0,19066688.0,18883968.0,18915584.0,18947200.0,18978816.0,19010432.0,19042048.0,19073664.0,19105280.0,18922112.0,18953792.0,18985472.0,19017152.0,19048832.0,19080512.0,19112192.0,19143872.0,18960256.0,18992000.0,19023744.0,19055488.0,19087232.0,19118976.0,19150720.0,19182464.0,18998400.0,19030208.0,19062016.0,19093824.0,19125632.0,19157440.0,19189248.0,19221056.0,19036544.0,19068416.0,19100288.0,19132160.0,19164032.0,19195904.0,19227776.0,19259648.0,19074688.0,19106624.0,19138560.0,19170496.0,19202432.0,19234368.0,19266304.0,19298240.0,19112832.0,19144832.0,19176832.0,19208832.0,19240832.0,19272832.0,19304832.0,19336832.0,19150976.0,19183040.0,19215104.0,19247168.0,19279232.0,19311296.0,19343360.0,19375424.0,19189120.0,19221248.0,19253376.0,19285504.0,19317632.0,19349760.0,19381888.0,19414016.0,19227264.0,19259456.0,19291648.0,19323840.0,19356032.0,19388224.0,19420416.0,19452608.0,19265408.0,19297664.0,19329920.0,19362176.0,19394432.0,19426688.0,19458944.0,19491200.0,19303552.0,19335872.0,19368192.0,19400512.0,19432832.0,19465152.0,19497472.0,19529792.0,19341696.0,19374080.0,19406464.0,19438848.0,19471232.0,19503616.0,19536000.0,19568384.0,19379840.0,19412288.0,19444736.0,19477184.0,19509632.0,19542080.0,19574528.0,19606976.0,19417984.0,19450496.0,19483008.0,19515520.0,19548032.0,19580544.0,19613056.0,19645568.0,19456128.0,19488704.0,19521280.0,19553856.0,19586432.0,19619008.0,19651584.0,19684160.0,19494272.0,19526912.0,19559552.0,19592192.0,19624832.0,19657472.0,19690112.0,19722752.0,19532416.0,19565120.0,19597824.0,19630528.0,19663232.0,19695936.0,19728640.0,19761344.0,19570560.0,19603328.0,19636096.0,19668864.0,19701632.0,19734400.0,19767168.0,19799936.0,19608704.0,19641536.0,19674368.0,19707200.0,19740032.0,19772864.0,19805696.0,19838528.0,18778496.0,18810368.0,18842240.0,18874112.0,18905984.0,18937856.0,18969728.0,19001600.0,17940096.0,17971008.0,18001920.0,18032832.0,18063744.0,18094656.0,18125568.0,18156480.0,17978240.0,18009216.0,18040192.0,18071168.0,18102144.0,18133120.0,18164096.0,18195072.0,18016384.0,18047424.0,18078464.0,18109504.0,18140544.0,18171584.0,18202624.0,18233664.0,18054528.0,18085632.0,18116736.0,18147840.0,18178944.0,18210048.0,18241152.0,18272256.0,18092672.0,18123840.0,18155008.0,18186176.0,18217344.0,18248512.0,18279680.0,18310848.0,18130816.0,18162048.0,18193280.0,18224512.0,18255744.0,18286976.0,18318208.0,18349440.0,18168960.0,18200256.0,18231552.0,18262848.0,18294144.0,18325440.0,18356736.0,18388032.0,18207104.0,18238464.0,18269824.0,18301184.0,18332544.0,18363904.0,18395264.0,18426624.0,18245248.0,18276672.0,18308096.0,18339520.0,18370944.0,18402368.0,18433792.0,18465216.0,18283392.0,18314880.0,18346368.0,18377856.0,18409344.0,18440832.0,18472320.0,18503808.0,18321536.0,18353088.0,18384640.0,18416192.0,18447744.0,18479296.0,18510848.0,18542400.0,18359680.0,18391296.0,18422912.0,18454528.0,18486144.0,18517760.0,18549376.0,18580992.0,18397824.0,18429504.0,18461184.0,18492864.0,18524544.0,18556224.0,18587904.0,18619584.0,18435968.0,18467712.0,18499456.0,18531200.0,18562944.0,18594688.0,18626432.0,18658176.0,18474112.0,18505920.0,18537728.0,18569536.0,18601344.0,18633152.0,18664960.0,18696768.0,18512256.0,18544128.0,18576000.0,18607872.0,18639744.0,18671616.0,18703488.0,18735360.0,18550400.0,18582336.0,18614272.0,18646208.0,18678144.0,18710080.0,18742016.0,18773952.0,18588544.0,18620544.0,18652544.0,18684544.0,18716544.0,18748544.0,18780544.0,18812544.0,18626688.0,18658752.0,18690816.0,18722880.0,18754944.0,18787008.0,18819072.0,18851136.0,18664832.0,18696960.0,18729088.0,18761216.0,18793344.0,18825472.0,18857600.0,18889728.0,18702976.0,18735168.0,18767360.0,18799552.0,18831744.0,18863936.0,18896128.0,18928320.0,18741120.0,18773376.0,18805632.0,18837888.0,18870144.0,18902400.0,18934656.0,18966912.0,18779264.0,18811584.0,18843904.0,18876224.0,18908544.0,18940864.0,18973184.0,19005504.0,18817408.0,18849792.0,18882176.0,18914560.0,18946944.0,18979328.0,19011712.0,19044096.0,18855552.0,18888000.0,18920448.0,18952896.0,18985344.0,19017792.0,19050240.0,19082688.0,18893696.0,18926208.0,18958720.0,18991232.0,19023744.0,19056256.0,19088768.0,19121280.0,18931840.0,18964416.0,18996992.0,19029568.0,19062144.0,19094720.0,19127296.0,19159872.0,18969984.0,19002624.0,19035264.0,19067904.0,19100544.0,19133184.0,19165824.0,19198464.0,19008128.0,19040832.0,19073536.0,19106240.0,19138944.0,19171648.0,19204352.0,19237056.0,19046272.0,19079040.0,19111808.0,19144576.0,19177344.0,19210112.0,19242880.0,19275648.0,19084416.0,19117248.0,19150080.0,19182912.0,19215744.0,19248576.0,19281408.0,19314240.0,19122560.0,19155456.0,19188352.0,19221248.0,19254144.0,19287040.0,19319936.0,19352832.0,19160704.0,19193664.0,19226624.0,19259584.0,19292544.0,19325504.0,19358464.0,19391424.0,19198848.0,19231872.0,19264896.0,19297920.0,19330944.0,19363968.0,19396992.0,19430016.0,19236992.0,19270080.0,19303168.0,19336256.0,19369344.0,19402432.0,19435520.0,19468608.0,19275136.0,19308288.0,19341440.0,19374592.0,19407744.0,19440896.0,19474048.0,19507200.0,19313280.0,19346496.0,19379712.0,19412928.0,19446144.0,19479360.0,19512576.0,19545792.0,19351424.0,19384704.0,19417984.0,19451264.0,19484544.0,19517824.0,19551104.0,19584384.0,19045504.0,19077824.0,19110144.0,19142464.0,19174784.0,19207104.0,19239424.0,19271744.0,18731392.0,18762752.0,18794112.0,18825472.0,18856832.0,18888192.0,18919552.0,18950912.0,18769536.0,18800960.0,18832384.0,18863808.0,18895232.0,18926656.0,18958080.0,18989504.0,18807680.0,18839168.0,18870656.0,18902144.0,18933632.0,18965120.0,18996608.0,19028096.0,18845824.0,18877376.0,18908928.0,18940480.0,18972032.0,19003584.0,19035136.0,19066688.0,18883968.0,18915584.0,18947200.0,18978816.0,19010432.0,19042048.0,19073664.0,19105280.0,18922112.0,18953792.0,18985472.0,19017152.0,19048832.0,19080512.0,19112192.0,19143872.0,18960256.0,18992000.0,19023744.0,19055488.0,19087232.0,19118976.0,19150720.0,19182464.0,18998400.0,19030208.0,19062016.0,19093824.0,19125632.0,19157440.0,19189248.0,19221056.0,19036544.0,19068416.0,19100288.0,19132160.0,19164032.0,19195904.0,19227776.0,19259648.0,19074688.0,19106624.0,19138560.0,19170496.0,19202432.0,19234368.0,19266304.0,19298240.0,19112832.0,19144832.0,19176832.0,19208832.0,19240832.0,19272832.0,19304832.0,19336832.0,19150976.0,19183040.0,19215104.0,19247168.0,19279232.0,19311296.0,19343360.0,19375424.0,19189120.0,19221248.0,19253376.0,19285504.0,19317632.0,19349760.0,19381888.0,19414016.0,19227264.0,19259456.0,19291648.0,19323840.0,19356032.0,19388224.0,19420416.0,19452608.0,19265408.0,19297664.0,19329920.0,19362176.0,19394432.0,19426688.0,19458944.0,19491200.0,19303552.0,19335872.0,19368192.0,19400512.0,19432832.0,19465152.0,19497472.0,19529792.0,19341696.0,19374080.0,19406464.0,19438848.0,19471232.0,19503616.0,19536000.0,19568384.0,19379840.0,19412288.0,19444736.0,19477184.0,19509632.0,19542080.0,19574528.0,19606976.0,19417984.0,19450496.0,19483008.0,19515520.0,19548032.0,19580544.0,19613056.0,19645568.0,19456128.0,19488704.0,19521280.0,19553856.0,19586432.0,19619008.0,19651584.0,19684160.0,19494272.0,19526912.0,19559552.0,19592192.0,19624832.0,19657472.0,19690112.0,19722752.0,19532416.0,19565120.0,19597824.0,19630528.0,19663232.0,19695936.0,19728640.0,19761344.0,19570560.0,19603328.0,19636096.0,19668864.0,19701632.0,19734400.0,19767168.0,19799936.0,19608704.0,19641536.0,19674368.0,19707200.0,19740032.0,19772864.0,19805696.0,19838528.0,19646848.0,19679744.0,19712640.0,19745536.0,19778432.0,19811328.0,19844224.0,19877120.0,19684992.0,19717952.0,19750912.0,19783872.0,19816832.0,19849792.0,19882752.0,19915712.0,19723136.0,19756160.0,19789184.0,19822208.0,19855232.0,19888256.0,19921280.0,19954304.0,19761280.0,19794368.0,19827456.0,19860544.0,19893632.0,19926720.0,19959808.0,19992896.0,19799424.0,19832576.0,19865728.0,19898880.0,19932032.0,19965184.0,19998336.0,20031488.0,19837568.0,19870784.0,19904000.0,19937216.0,19970432.0,20003648.0,20036864.0,20070080.0,19875712.0,19908992.0,19942272.0,19975552.0,20008832.0,20042112.0,20075392.0,20108672.0,19913856.0,19947200.0,19980544.0,20013888.0,20047232.0,20080576.0,20113920.0,20147264.0,19952000.0,19985408.0,20018816.0,20052224.0,20085632.0,20119040.0,20152448.0,20185856.0,19990144.0,20023616.0,20057088.0,20090560.0,20124032.0,20157504.0,20190976.0,20224448.0,20028288.0,20061824.0,20095360.0,20128896.0,20162432.0,20195968.0,20229504.0,20263040.0,20066432.0,20100032.0,20133632.0,20167232.0,20200832.0,20234432.0,20268032.0,20301632.0,20104576.0,20138240.0,20171904.0,20205568.0,20239232.0,20272896.0,20306560.0,20340224.0,20142720.0,20176448.0,20210176.0,20243904.0,20277632.0,20311360.0,20345088.0,20378816.0,19312512.0,19345280.0,19378048.0,19410816.0,19443584.0,19476352.0,19509120.0,19541888.0,18474112.0,18505920.0,18537728.0,18569536.0,18601344.0,18633152.0,18664960.0,18696768.0,9310336.0,9326256.0,9342176.0,9358096.0,9374016.0,9389936.0,9405856.0,9421776.0,9884160.0,9899952.0,9915744.0,9931536.0,9947328.0,9963120.0,9978912.0,9994704.0,19914112.0,19945728.0,19977344.0,20008960.0,20040576.0,20072192.0,20103808.0,20135424.0,19953280.0,19984960.0,20016640.0,20048320.0,20080000.0,20111680.0,20143360.0,20175040.0,19992448.0,20024192.0,20055936.0,20087680.0,20119424.0,20151168.0,20182912.0,20214656.0,20031616.0,20063424.0,20095232.0,20127040.0,20158848.0,20190656.0,20222464.0,20254272.0,20070784.0,20102656.0,20134528.0,20166400.0,20198272.0,20230144.0,20262016.0,20293888.0,20109952.0,20141888.0,20173824.0,20205760.0,20237696.0,20269632.0,20301568.0,20333504.0,20149120.0,20181120.0,20213120.0,20245120.0,20277120.0,20309120.0,20341120.0,20373120.0,20188288.0,20220352.0,20252416.0,20284480.0,20316544.0,20348608.0,20380672.0,20412736.0,20227456.0,20259584.0,20291712.0,20323840.0,20355968.0,20388096.0,20420224.0,20452352.0,20266624.0,20298816.0,20331008.0,20363200.0,20395392.0,20427584.0,20459776.0,20491968.0,20305792.0,20338048.0,20370304.0,20402560.0,20434816.0,20467072.0,20499328.0,20531584.0,20344960.0,20377280.0,20409600.0,20441920.0,20474240.0,20506560.0,20538880.0,20571200.0,20384128.0,20416512.0,20448896.0,20481280.0,20513664.0,20546048.0,20578432.0,20610816.0,20423296.0,20455744.0,20488192.0,20520640.0,20553088.0,20585536.0,20617984.0,20650432.0,20462464.0,20494976.0,20527488.0,20560000.0,20592512.0,20625024.0,20657536.0,20690048.0,20501632.0,20534208.0,20566784.0,20599360.0,20631936.0,20664512.0,20697088.0,20729664.0,20540800.0,20573440.0,20606080.0,20638720.0,20671360.0,20704000.0,20736640.0,20769280.0,20579968.0,20612672.0,20645376.0,20678080.0,20710784.0,20743488.0,20776192.0,20808896.0,20619136.0,20651904.0,20684672.0,20717440.0,20750208.0,20782976.0,20815744.0,20848512.0,20658304.0,20691136.0,20723968.0,20756800.0,20789632.0,20822464.0,20855296.0,20888128.0,20697472.0,20730368.0,20763264.0,20796160.0,20829056.0,20861952.0,20894848.0,20927744.0,20736640.0,20769600.0,20802560.0,20835520.0,20868480.0,20901440.0,20934400.0,20967360.0,20775808.0,20808832.0,20841856.0,20874880.0,20907904.0,20940928.0,20973952.0,21006976.0,20814976.0,20848064.0,20881152.0,20914240.0,20947328.0,20980416.0,21013504.0,21046592.0,20854144.0,20887296.0,20920448.0,20953600.0,20986752.0,21019904.0,21053056.0,21086208.0,20893312.0,20926528.0,20959744.0,20992960.0,21026176.0,21059392.0,21092608.0,21125824.0,20932480.0,20965760.0,20999040.0,21032320.0,21065600.0,21098880.0,21132160.0,21165440.0,20971648.0,21004992.0,21038336.0,21071680.0,21105024.0,21138368.0,21171712.0,21205056.0,21010816.0,21044224.0,21077632.0,21111040.0,21144448.0,21177856.0,21211264.0,21244672.0,21049984.0,21083456.0,21116928.0,21150400.0,21183872.0,21217344.0,21250816.0,21284288.0,21089152.0,21122688.0,21156224.0,21189760.0,21223296.0,21256832.0,21290368.0,21323904.0,21128320.0,21161920.0,21195520.0,21229120.0,21262720.0,21296320.0,21329920.0,21363520.0,21167488.0,21201152.0,21234816.0,21268480.0,21302144.0,21335808.0,21369472.0,21403136.0,21206656.0,21240384.0,21274112.0,21307840.0,21341568.0,21375296.0,21409024.0,21442752.0,21245824.0,21279616.0,21313408.0,21347200.0,21380992.0,21414784.0,21448576.0,21482368.0,21284992.0,21318848.0,21352704.0,21386560.0,21420416.0,21454272.0,21488128.0,21521984.0,20439424.0,20472320.0,20505216.0,20538112.0,20571008.0,20603904.0,20636800.0,20669696.0,19585664.0,19617600.0,19649536.0,19681472.0,19713408.0,19745344.0,19777280.0,19809216.0,18740096.0,18771072.0,18802048.0,18833024.0,18864000.0,18894976.0,18925952.0,18956928.0,17886336.0,17916352.0,17946368.0,17976384.0,18006400.0,18036416.0,18066432.0,18096448.0,17925504.0,17955584.0,17985664.0,18015744.0,18045824.0,18075904.0,18105984.0,18136064.0,17964672.0,17994816.0,18024960.0,18055104.0,18085248.0,18115392.0,18145536.0,18175680.0,18003840.0,18034048.0,18064256.0,18094464.0,18124672.0,18154880.0,18185088.0,18215296.0,18043008.0,18073280.0,18103552.0,18133824.0,18164096.0,18194368.0,18224640.0,18254912.0,18082176.0,18112512.0,18142848.0,18173184.0,18203520.0,18233856.0,18264192.0,18294528.0,18121344.0,18151744.0,18182144.0,18212544.0,18242944.0,18273344.0,18303744.0,18334144.0,18160512.0,18190976.0,18221440.0,18251904.0,18282368.0,18312832.0,18343296.0,18373760.0,18199680.0,18230208.0,18260736.0,18291264.0,18321792.0,18352320.0,18382848.0,18413376.0,18238848.0,18269440.0,18300032.0,18330624.0,18361216.0,18391808.0,18422400.0,18452992.0,18278016.0,18308672.0,18339328.0,18369984.0,18400640.0,18431296.0,18461952.0,18492608.0,18317184.0,18347904.0,18378624.0,18409344.0,18440064.0,18470784.0,18501504.0,18532224.0,18356352.0,18387136.0,18417920.0,18448704.0,18479488.0,18510272.0,18541056.0,18571840.0,18395520.0,18426368.0,18457216.0,18488064.0,18518912.0,18549760.0,18580608.0,18611456.0,18434688.0,18465600.0,18496512.0,18527424.0,18558336.0,18589248.0,18620160.0,18651072.0,18473856.0,18504832.0,18535808.0,18566784.0,18597760.0,18628736.0,18659712.0,18690688.0,18513024.0,18544064.0,18575104.0,18606144.0,18637184.0,18668224.0,18699264.0,18730304.0,18552192.0,18583296.0,18614400.0,18645504.0,18676608.0,18707712.0,18738816.0,18769920.0,18591360.0,18622528.0,18653696.0,18684864.0,18716032.0,18747200.0,18778368.0,18809536.0,18630528.0,18661760.0,18692992.0,18724224.0,18755456.0,18786688.0,18817920.0,18849152.0,18669696.0,18700992.0,18732288.0,18763584.0,18794880.0,18826176.0,18857472.0,18888768.0,18708864.0,18740224.0,18771584.0,18802944.0,18834304.0,18865664.0,18897024.0,18928384.0,18748032.0,18779456.0,18810880.0,18842304.0,18873728.0,18905152.0,18936576.0,18968000.0,18787200.0,18818688.0,18850176.0,18881664.0,18913152.0,18944640.0,18976128.0,19007616.0,18826368.0,18857920.0,18889472.0,18921024.0,18952576.0,18984128.0,19015680.0,19047232.0,18865536.0,18897152.0,18928768.0,18960384.0,18992000.0,19023616.0,19055232.0,19086848.0,18904704.0,18936384.0,18968064.0,18999744.0,19031424.0,19063104.0,19094784.0,19126464.0,18943872.0,18975616.0,19007360.0,19039104.0,19070848.0,19102592.0,19134336.0,19166080.0,18983040.0,19014848.0,19046656.0,19078464.0,19110272.0,19142080.0,19173888.0,19205696.0,19022208.0,19054080.0,19085952.0,19117824.0,19149696.0,19181568.0,19213440.0,19245312.0,19061376.0,19093312.0,19125248.0,19157184.0,19189120.0,19221056.0,19252992.0,19284928.0,19100544.0,19132544.0,19164544.0,19196544.0,19228544.0,19260544.0,19292544.0,19324544.0,19139712.0,19171776.0,19203840.0,19235904.0,19267968.0,19300032.0,19332096.0,19364160.0,19178880.0,19211008.0,19243136.0,19275264.0,19307392.0,19339520.0,19371648.0,19403776.0,19218048.0,19250240.0,19282432.0,19314624.0,19346816.0,19379008.0,19411200.0,19443392.0,19257216.0,19289472.0,19321728.0,19353984.0,19386240.0,19418496.0,19450752.0,19483008.0,19296384.0,19328704.0,19361024.0,19393344.0,19425664.0,19457984.0,19490304.0,19522624.0,19335552.0,19367936.0,19400320.0,19432704.0,19465088.0,19497472.0,19529856.0,19562240.0,19014272.0,19045696.0,19077120.0,19108544.0,19139968.0,19171392.0,19202816.0,19234240.0,18684800.0,18715264.0,18745728.0,18776192.0,18806656.0,18837120.0,18867584.0,18898048.0,18723968.0,18754496.0,18785024.0,18815552.0,18846080.0,18876608.0,18907136.0,18937664.0,18763136.0,18793728.0,18824320.0,18854912.0,18885504.0,18916096.0,18946688.0,18977280.0,18802304.0,18832960.0,18863616.0,18894272.0,18924928.0,18955584.0,18986240.0,19016896.0,18841472.0,18872192.0,18902912.0,18933632.0,18964352.0,18995072.0,19025792.0,19056512.0,18880640.0,18911424.0,18942208.0,18972992.0,19003776.0,19034560.0,19065344.0,19096128.0,18919808.0,18950656.0,18981504.0,19012352.0,19043200.0,19074048.0,19104896.0,19135744.0,18958976.0,18989888.0,19020800.0,19051712.0,19082624.0,19113536.0,19144448.0,19175360.0,18998144.0,19029120.0,19060096.0,19091072.0,19122048.0,19153024.0,19184000.0,19214976.0,19037312.0,19068352.0,19099392.0,19130432.0,19161472.0,19192512.0,19223552.0,19254592.0,19076480.0,19107584.0,19138688.0,19169792.0,19200896.0,19232000.0,19263104.0,19294208.0,19115648.0,19146816.0,19177984.0,19209152.0,19240320.0,19271488.0,19302656.0,19333824.0,19154816.0,19186048.0,19217280.0,19248512.0,19279744.0,19310976.0,19342208.0,19373440.0,19193984.0,19225280.0,19256576.0,19287872.0,19319168.0,19350464.0,19381760.0,19413056.0,19233152.0,19264512.0,19295872.0,19327232.0,19358592.0,19389952.0,19421312.0,19452672.0,19272320.0,19303744.0,19335168.0,19366592.0,19398016.0,19429440.0,19460864.0,19492288.0,19311488.0,19342976.0,19374464.0,19405952.0,19437440.0,19468928.0,19500416.0,19531904.0,19350656.0,19382208.0,19413760.0,19445312.0,19476864.0,19508416.0,19539968.0,19571520.0,19389824.0,19421440.0,19453056.0,19484672.0,19516288.0,19547904.0,19579520.0,19611136.0,19428992.0,19460672.0,19492352.0,19524032.0,19555712.0,19587392.0,19619072.0,19650752.0,19468160.0,19499904.0,19531648.0,19563392.0,19595136.0,19626880.0,19658624.0,19690368.0,19507328.0,19539136.0,19570944.0,19602752.0,19634560.0,19666368.0,19698176.0,19729984.0,19546496.0,19578368.0,19610240.0,19642112.0,19673984.0,19705856.0,19737728.0,19769600.0,19585664.0,19617600.0,19649536.0,19681472.0,19713408.0,19745344.0,19777280.0,19809216.0,19624832.0,19656832.0,19688832.0,19720832.0,19752832.0,19784832.0,19816832.0,19848832.0,19664000.0,19696064.0,19728128.0,19760192.0,19792256.0,19824320.0,19856384.0,19888448.0,19703168.0,19735296.0,19767424.0,19799552.0,19831680.0,19863808.0,19895936.0,19928064.0,19742336.0,19774528.0,19806720.0,19838912.0,19871104.0,19903296.0,19935488.0,19967680.0,19781504.0,19813760.0,19846016.0,19878272.0,19910528.0,19942784.0,19975040.0,20007296.0,19820672.0,19852992.0,19885312.0,19917632.0,19949952.0,19982272.0,20014592.0,20046912.0,19859840.0,19892224.0,19924608.0,19956992.0,19989376.0,20021760.0,20054144.0,20086528.0,19899008.0,19931456.0,19963904.0,19996352.0,20028800.0,20061248.0,20093696.0,20126144.0,19938176.0,19970688.0,20003200.0,20035712.0,20068224.0,20100736.0,20133248.0,20165760.0,19977344.0,20009920.0,20042496.0,20075072.0,20107648.0,20140224.0,20172800.0,20205376.0,20016512.0,20049152.0,20081792.0,20114432.0,20147072.0,20179712.0,20212352.0,20244992.0,20055680.0,20088384.0,20121088.0,20153792.0,20186496.0,20219200.0,20251904.0,20284608.0,20094848.0,20127616.0,20160384.0,20193152.0,20225920.0,20258688.0,20291456.0,20324224.0,20134016.0,20166848.0,20199680.0,20232512.0,20265344.0,20298176.0,20331008.0,20363840.0,19288448.0,19320320.0,19352192.0,19384064.0,19415936.0,19447808.0,19479680.0,19511552.0,18434688.0,18465600.0,18496512.0,18527424.0,18558336.0,18589248.0,18620160.0,18651072.0,18473856.0,18504832.0,18535808.0,18566784.0,18597760.0,18628736.0,18659712.0,18690688.0,18513024.0,18544064.0,18575104.0,18606144.0,18637184.0,18668224.0,18699264.0,18730304.0,18552192.0,18583296.0,18614400.0,18645504.0,18676608.0,18707712.0,18738816.0,18769920.0,18591360.0,18622528.0,18653696.0,18684864.0,18716032.0,18747200.0,18778368.0,18809536.0,18630528.0,18661760.0,18692992.0,18724224.0,18755456.0,18786688.0,18817920.0,18849152.0,18669696.0,18700992.0,18732288.0,18763584.0,18794880.0,18826176.0,18857472.0,18888768.0,18708864.0,18740224.0,18771584.0,18802944.0,18834304.0,18865664.0,18897024.0,18928384.0,18748032.0,18779456.0,18810880.0,18842304.0,18873728.0,18905152.0,18936576.0,18968000.0,18787200.0,18818688.0,18850176.0,18881664.0,18913152.0,18944640.0,18976128.0,19007616.0,18826368.0,18857920.0,18889472.0,18921024.0,18952576.0,18984128.0,19015680.0,19047232.0,18865536.0,18897152.0,18928768.0,18960384.0,18992000.0,19023616.0,19055232.0,19086848.0,18904704.0,18936384.0,18968064.0,18999744.0,19031424.0,19063104.0,19094784.0,19126464.0,18943872.0,18975616.0,19007360.0,19039104.0,19070848.0,19102592.0,19134336.0,19166080.0,18983040.0,19014848.0,19046656.0,19078464.0,19110272.0,19142080.0,19173888.0,19205696.0,19022208.0,19054080.0,19085952.0,19117824.0,19149696.0,19181568.0,19213440.0,19245312.0,19061376.0,19093312.0,19125248.0,19157184.0,19189120.0,19221056.0,19252992.0,19284928.0,19100544.0,19132544.0,19164544.0,19196544.0,19228544.0,19260544.0,19292544.0,19324544.0,19139712.0,19171776.0,19203840.0,19235904.0,19267968.0,19300032.0,19332096.0,19364160.0,19178880.0,19211008.0,19243136.0,19275264.0,19307392.0,19339520.0,19371648.0,19403776.0,19218048.0,19250240.0,19282432.0,19314624.0,19346816.0,19379008.0,19411200.0,19443392.0,19257216.0,19289472.0,19321728.0,19353984.0,19386240.0,19418496.0,19450752.0,19483008.0,19296384.0,19328704.0,19361024.0,19393344.0,19425664.0,19457984.0,19490304.0,19522624.0,19335552.0,19367936.0,19400320.0,19432704.0,19465088.0,19497472.0,19529856.0,19562240.0,19374720.0,19407168.0,19439616.0,19472064.0,19504512.0,19536960.0,19569408.0,19601856.0,19413888.0,19446400.0,19478912.0,19511424.0,19543936.0,19576448.0,19608960.0,19641472.0,19453056.0,19485632.0,19518208.0,19550784.0,19583360.0,19615936.0,19648512.0,19681088.0,19492224.0,19524864.0,19557504.0,19590144.0,19622784.0,19655424.0,19688064.0,19720704.0,19531392.0,19564096.0,19596800.0,19629504.0,19662208.0,19694912.0,19727616.0,19760320.0,19570560.0,19603328.0,19636096.0,19668864.0,19701632.0,19734400.0,19767168.0,19799936.0,19609728.0,19642560.0,19675392.0,19708224.0,19741056.0,19773888.0,19806720.0,19839552.0,19648896.0,19681792.0,19714688.0,19747584.0,19780480.0,19813376.0,19846272.0,19879168.0,19688064.0,19721024.0,19753984.0,19786944.0,19819904.0,19852864.0,19885824.0,19918784.0,19727232.0,19760256.0,19793280.0,19826304.0,19859328.0,19892352.0,19925376.0,19958400.0,19766400.0,19799488.0,19832576.0,19865664.0,19898752.0,19931840.0,19964928.0,19998016.0,19805568.0,19838720.0,19871872.0,19905024.0,19938176.0,19971328.0,20004480.0,20037632.0,19844736.0,19877952.0,19911168.0,19944384.0,19977600.0,20010816.0,20044032.0,20077248.0,19883904.0,19917184.0,19950464.0,19983744.0,20017024.0,20050304.0,20083584.0,20116864.0,19562624.0,19594944.0,19627264.0,19659584.0,19691904.0,19724224.0,19756544.0,19788864.0,19233152.0,19264512.0,19295872.0,19327232.0,19358592.0,19389952.0,19421312.0,19452672.0,19272320.0,19303744.0,19335168.0,19366592.0,19398016.0,19429440.0,19460864.0,19492288.0,19311488.0,19342976.0,19374464.0,19405952.0,19437440.0,19468928.0,19500416.0,19531904.0,19350656.0,19382208.0,19413760.0,19445312.0,19476864.0,19508416.0,19539968.0,19571520.0,19389824.0,19421440.0,19453056.0,19484672.0,19516288.0,19547904.0,19579520.0,19611136.0,19428992.0,19460672.0,19492352.0,19524032.0,19555712.0,19587392.0,19619072.0,19650752.0,19468160.0,19499904.0,19531648.0,19563392.0,19595136.0,19626880.0,19658624.0,19690368.0,19507328.0,19539136.0,19570944.0,19602752.0,19634560.0,19666368.0,19698176.0,19729984.0,19546496.0,19578368.0,19610240.0,19642112.0,19673984.0,19705856.0,19737728.0,19769600.0,19585664.0,19617600.0,19649536.0,19681472.0,19713408.0,19745344.0,19777280.0,19809216.0,19624832.0,19656832.0,19688832.0,19720832.0,19752832.0,19784832.0,19816832.0,19848832.0,19664000.0,19696064.0,19728128.0,19760192.0,19792256.0,19824320.0,19856384.0,19888448.0,19703168.0,19735296.0,19767424.0,19799552.0,19831680.0,19863808.0,19895936.0,19928064.0,19742336.0,19774528.0,19806720.0,19838912.0,19871104.0,19903296.0,19935488.0,19967680.0,19781504.0,19813760.0,19846016.0,19878272.0,19910528.0,19942784.0,19975040.0,20007296.0,19820672.0,19852992.0,19885312.0,19917632.0,19949952.0,19982272.0,20014592.0,20046912.0,19859840.0,19892224.0,19924608.0,19956992.0,19989376.0,20021760.0,20054144.0,20086528.0,19899008.0,19931456.0,19963904.0,19996352.0,20028800.0,20061248.0,20093696.0,20126144.0,19938176.0,19970688.0,20003200.0,20035712.0,20068224.0,20100736.0,20133248.0,20165760.0,19977344.0,20009920.0,20042496.0,20075072.0,20107648.0,20140224.0,20172800.0,20205376.0,20016512.0,20049152.0,20081792.0,20114432.0,20147072.0,20179712.0,20212352.0,20244992.0,20055680.0,20088384.0,20121088.0,20153792.0,20186496.0,20219200.0,20251904.0,20284608.0,20094848.0,20127616.0,20160384.0,20193152.0,20225920.0,20258688.0,20291456.0,20324224.0,20134016.0,20166848.0,20199680.0,20232512.0,20265344.0,20298176.0,20331008.0,20363840.0,20173184.0,20206080.0,20238976.0,20271872.0,20304768.0,20337664.0,20370560.0,20403456.0,20212352.0,20245312.0,20278272.0,20311232.0,20344192.0,20377152.0,20410112.0,20443072.0,20251520.0,20284544.0,20317568.0,20350592.0,20383616.0,20416640.0,20449664.0,20482688.0,20290688.0,20323776.0,20356864.0,20389952.0,20423040.0,20456128.0,20489216.0,20522304.0,20329856.0,20363008.0,20396160.0,20429312.0,20462464.0,20495616.0,20528768.0,20561920.0,20369024.0,20402240.0,20435456.0,20468672.0,20501888.0,20535104.0,20568320.0,20601536.0,20408192.0,20441472.0,20474752.0,20508032.0,20541312.0,20574592.0,20607872.0,20641152.0,20447360.0,20480704.0,20514048.0,20547392.0,20580736.0,20614080.0,20647424.0,20680768.0,20486528.0,20519936.0,20553344.0,20586752.0,20620160.0,20653568.0,20686976.0,20720384.0,20525696.0,20559168.0,20592640.0,20626112.0,20659584.0,20693056.0,20726528.0,20760000.0,20564864.0,20598400.0,20631936.0,20665472.0,20699008.0,20732544.0,20766080.0,20799616.0,20604032.0,20637632.0,20671232.0,20704832.0,20738432.0,20772032.0,20805632.0,20839232.0,20643200.0,20676864.0,20710528.0,20744192.0,20777856.0,20811520.0,20845184.0,20878848.0,20682368.0,20716096.0,20749824.0,20783552.0,20817280.0,20851008.0,20884736.0,20918464.0,19836800.0,19869568.0,19902336.0,19935104.0,19967872.0,20000640.0,20033408.0,20066176.0,18983040.0,19014848.0,19046656.0,19078464.0,19110272.0,19142080.0,19173888.0,19205696.0,9565056.0,9580976.0,9596896.0,9612816.0,9628736.0,9644656.0,9660576.0,9676496.0,10136832.0,10152624.0,10168416.0,10184208.0,10200000.0,10215792.0,10231584.0,10247376.0,20419968.0,20451584.0,20483200.0,20514816.0,20546432.0,20578048.0,20609664.0,20641280.0,20460160.0,20491840.0,20523520.0,20555200.0,20586880.0,20618560.0,20650240.0,20681920.0,20500352.0,20532096.0,20563840.0,20595584.0,20627328.0,20659072.0,20690816.0,20722560.0,20540544.0,20572352.0,20604160.0,20635968.0,20667776.0,20699584.0,20731392.0,20763200.0,20580736.0,20612608.0,20644480.0,20676352.0,20708224.0,20740096.0,20771968.0,20803840.0,20620928.0,20652864.0,20684800.0,20716736.0,20748672.0,20780608.0,20812544.0,20844480.0,20661120.0,20693120.0,20725120.0,20757120.0,20789120.0,20821120.0,20853120.0,20885120.0,20701312.0,20733376.0,20765440.0,20797504.0,20829568.0,20861632.0,20893696.0,20925760.0,20741504.0,20773632.0,20805760.0,20837888.0,20870016.0,20902144.0,20934272.0,20966400.0,20781696.0,20813888.0,20846080.0,20878272.0,20910464.0,20942656.0,20974848.0,21007040.0,20821888.0,20854144.0,20886400.0,20918656.0,20950912.0,20983168.0,21015424.0,21047680.0,20862080.0,20894400.0,20926720.0,20959040.0,20991360.0,21023680.0,21056000.0,21088320.0,20902272.0,20934656.0,20967040.0,20999424.0,21031808.0,21064192.0,21096576.0,21128960.0,20942464.0,20974912.0,21007360.0,21039808.0,21072256.0,21104704.0,21137152.0,21169600.0,20982656.0,21015168.0,21047680.0,21080192.0,21112704.0,21145216.0,21177728.0,21210240.0,21022848.0,21055424.0,21088000.0,21120576.0,21153152.0,21185728.0,21218304.0,21250880.0,21063040.0,21095680.0,21128320.0,21160960.0,21193600.0,21226240.0,21258880.0,21291520.0,21103232.0,21135936.0,21168640.0,21201344.0,21234048.0,21266752.0,21299456.0,21332160.0,21143424.0,21176192.0,21208960.0,21241728.0,21274496.0,21307264.0,21340032.0,21372800.0,21183616.0,21216448.0,21249280.0,21282112.0,21314944.0,21347776.0,21380608.0,21413440.0,21223808.0,21256704.0,21289600.0,21322496.0,21355392.0,21388288.0,21421184.0,21454080.0,21264000.0,21296960.0,21329920.0,21362880.0,21395840.0,21428800.0,21461760.0,21494720.0,21304192.0,21337216.0,21370240.0,21403264.0,21436288.0,21469312.0,21502336.0,21535360.0,21344384.0,21377472.0,21410560.0,21443648.0,21476736.0,21509824.0,21542912.0,21576000.0,21384576.0,21417728.0,21450880.0,21484032.0,21517184.0,21550336.0,21583488.0,21616640.0,21424768.0,21457984.0,21491200.0,21524416.0,21557632.0,21590848.0,21624064.0,21657280.0,21464960.0,21498240.0,21531520.0,21564800.0,21598080.0,21631360.0,21664640.0,21697920.0,21505152.0,21538496.0,21571840.0,21605184.0,21638528.0,21671872.0,21705216.0,21738560.0,21545344.0,21578752.0,21612160.0,21645568.0,21678976.0,21712384.0,21745792.0,21779200.0,21585536.0,21619008.0,21652480.0,21685952.0,21719424.0,21752896.0,21786368.0,21819840.0,21625728.0,21659264.0,21692800.0,21726336.0,21759872.0,21793408.0,21826944.0,21860480.0,21665920.0,21699520.0,21733120.0,21766720.0,21800320.0,21833920.0,21867520.0,21901120.0,21706112.0,21739776.0,21773440.0,21807104.0,21840768.0,21874432.0,21908096.0,21941760.0,21746304.0,21780032.0,21813760.0,21847488.0,21881216.0,21914944.0,21948672.0,21982400.0,21786496.0,21820288.0,21854080.0,21887872.0,21921664.0,21955456.0,21989248.0,22023040.0,21826688.0,21860544.0,21894400.0,21928256.0,21962112.0,21995968.0,22029824.0,22063680.0,20965760.0,20998656.0,21031552.0,21064448.0,21097344.0,21130240.0,21163136.0,21196032.0,20096640.0,20128576.0,20160512.0,20192448.0,20224384.0,20256320.0,20288256.0,20320192.0,19235712.0,19266688.0,19297664.0,19328640.0,19359616.0,19390592.0,19421568.0,19452544.0,18366592.0,18396608.0,18426624.0,18456640.0,18486656.0,18516672.0,18546688.0,18576704.0,18406784.0,18436864.0,18466944.0,18497024.0,18527104.0,18557184.0,18587264.0,18617344.0,18446976.0,18477120.0,18507264.0,18537408.0,18567552.0,18597696.0,18627840.0,18657984.0,18487168.0,18517376.0,18547584.0,18577792.0,18608000.0,18638208.0,18668416.0,18698624.0,18527360.0,18557632.0,18587904.0,18618176.0,18648448.0,18678720.0,18708992.0,18739264.0,18567552.0,18597888.0,18628224.0,18658560.0,18688896.0,18719232.0,18749568.0,18779904.0,18607744.0,18638144.0,18668544.0,18698944.0,18729344.0,18759744.0,18790144.0,18820544.0,18647936.0,18678400.0,18708864.0,18739328.0,18769792.0,18800256.0,18830720.0,18861184.0,18688128.0,18718656.0,18749184.0,18779712.0,18810240.0,18840768.0,18871296.0,18901824.0,18728320.0,18758912.0,18789504.0,18820096.0,18850688.0,18881280.0,18911872.0,18942464.0,18768512.0,18799168.0,18829824.0,18860480.0,18891136.0,18921792.0,18952448.0,18983104.0,18808704.0,18839424.0,18870144.0,18900864.0,18931584.0,18962304.0,18993024.0,19023744.0,18848896.0,18879680.0,18910464.0,18941248.0,18972032.0,19002816.0,19033600.0,19064384.0,18889088.0,18919936.0,18950784.0,18981632.0,19012480.0,19043328.0,19074176.0,19105024.0,18929280.0,18960192.0,18991104.0,19022016.0,19052928.0,19083840.0,19114752.0,19145664.0,18969472.0,19000448.0,19031424.0,19062400.0,19093376.0,19124352.0,19155328.0,19186304.0,19009664.0,19040704.0,19071744.0,19102784.0,19133824.0,19164864.0,19195904.0,19226944.0,19049856.0,19080960.0,19112064.0,19143168.0,19174272.0,19205376.0,19236480.0,19267584.0,19090048.0,19121216.0,19152384.0,19183552.0,19214720.0,19245888.0,19277056.0,19308224.0,19130240.0,19161472.0,19192704.0,19223936.0,19255168.0,19286400.0,19317632.0,19348864.0,19170432.0,19201728.0,19233024.0,19264320.0,19295616.0,19326912.0,19358208.0,19389504.0,19210624.0,19241984.0,19273344.0,19304704.0,19336064.0,19367424.0,19398784.0,19430144.0,19250816.0,19282240.0,19313664.0,19345088.0,19376512.0,19407936.0,19439360.0,19470784.0,19291008.0,19322496.0,19353984.0,19385472.0,19416960.0,19448448.0,19479936.0,19511424.0,19331200.0,19362752.0,19394304.0,19425856.0,19457408.0,19488960.0,19520512.0,19552064.0,19371392.0,19403008.0,19434624.0,19466240.0,19497856.0,19529472.0,19561088.0,19592704.0,19411584.0,19443264.0,19474944.0,19506624.0,19538304.0,19569984.0,19601664.0,19633344.0,19451776.0,19483520.0,19515264.0,19547008.0,19578752.0,19610496.0,19642240.0,19673984.0,19491968.0,19523776.0,19555584.0,19587392.0,19619200.0,19651008.0,19682816.0,19714624.0,19532160.0,19564032.0,19595904.0,19627776.0,19659648.0,19691520.0,19723392.0,19755264.0,19572352.0,19604288.0,19636224.0,19668160.0,19700096.0,19732032.0,19763968.0,19795904.0,19612544.0,19644544.0,19676544.0,19708544.0,19740544.0,19772544.0,19804544.0,19836544.0,19652736.0,19684800.0,19716864.0,19748928.0,19780992.0,19813056.0,19845120.0,19877184.0,19692928.0,19725056.0,19757184.0,19789312.0,19821440.0,19853568.0,19885696.0,19917824.0,19733120.0,19765312.0,19797504.0,19829696.0,19861888.0,19894080.0,19926272.0,19958464.0,19773312.0,19805568.0,19837824.0,19870080.0,19902336.0,19934592.0,19966848.0,19999104.0,19813504.0,19845824.0,19878144.0,19910464.0,19942784.0,19975104.0,20007424.0,20039744.0,19853696.0,19886080.0,19918464.0,19950848.0,19983232.0,20015616.0,20048000.0,20080384.0,19517056.0,19548480.0,19579904.0,19611328.0,19642752.0,19674176.0,19705600.0,19737024.0,19172224.0,19202688.0,19233152.0,19263616.0,19294080.0,19324544.0,19355008.0,19385472.0,19212416.0,19242944.0,19273472.0,19304000.0,19334528.0,19365056.0,19395584.0,19426112.0,19252608.0,19283200.0,19313792.0,19344384.0,19374976.0,19405568.0,19436160.0,19466752.0,19292800.0,19323456.0,19354112.0,19384768.0,19415424.0,19446080.0,19476736.0,19507392.0,19332992.0,19363712.0,19394432.0,19425152.0,19455872.0,19486592.0,19517312.0,19548032.0,19373184.0,19403968.0,19434752.0,19465536.0,19496320.0,19527104.0,19557888.0,19588672.0,19413376.0,19444224.0,19475072.0,19505920.0,19536768.0,19567616.0,19598464.0,19629312.0,19453568.0,19484480.0,19515392.0,19546304.0,19577216.0,19608128.0,19639040.0,19669952.0,19493760.0,19524736.0,19555712.0,19586688.0,19617664.0,19648640.0,19679616.0,19710592.0,19533952.0,19564992.0,19596032.0,19627072.0,19658112.0,19689152.0,19720192.0,19751232.0,19574144.0,19605248.0,19636352.0,19667456.0,19698560.0,19729664.0,19760768.0,19791872.0,19614336.0,19645504.0,19676672.0,19707840.0,19739008.0,19770176.0,19801344.0,19832512.0,19654528.0,19685760.0,19716992.0,19748224.0,19779456.0,19810688.0,19841920.0,19873152.0,19694720.0,19726016.0,19757312.0,19788608.0,19819904.0,19851200.0,19882496.0,19913792.0,19734912.0,19766272.0,19797632.0,19828992.0,19860352.0,19891712.0,19923072.0,19954432.0,19775104.0,19806528.0,19837952.0,19869376.0,19900800.0,19932224.0,19963648.0,19995072.0,19815296.0,19846784.0,19878272.0,19909760.0,19941248.0,19972736.0,20004224.0,20035712.0,19855488.0,19887040.0,19918592.0,19950144.0,19981696.0,20013248.0,20044800.0,20076352.0,19895680.0,19927296.0,19958912.0,19990528.0,20022144.0,20053760.0,20085376.0,20116992.0,19935872.0,19967552.0,19999232.0,20030912.0,20062592.0,20094272.0,20125952.0,20157632.0,19976064.0,20007808.0,20039552.0,20071296.0,20103040.0,20134784.0,20166528.0,20198272.0,20016256.0,20048064.0,20079872.0,20111680.0,20143488.0,20175296.0,20207104.0,20238912.0,20056448.0,20088320.0,20120192.0,20152064.0,20183936.0,20215808.0,20247680.0,20279552.0,20096640.0,20128576.0,20160512.0,20192448.0,20224384.0,20256320.0,20288256.0,20320192.0,20136832.0,20168832.0,20200832.0,20232832.0,20264832.0,20296832.0,20328832.0,20360832.0,20177024.0,20209088.0,20241152.0,20273216.0,20305280.0,20337344.0,20369408.0,20401472.0,20217216.0,20249344.0,20281472.0,20313600.0,20345728.0,20377856.0,20409984.0,20442112.0,20257408.0,20289600.0,20321792.0,20353984.0,20386176.0,20418368.0,20450560.0,20482752.0,20297600.0,20329856.0,20362112.0,20394368.0,20426624.0,20458880.0,20491136.0,20523392.0,20337792.0,20370112.0,20402432.0,20434752.0,20467072.0,20499392.0,20531712.0,20564032.0,20377984.0,20410368.0,20442752.0,20475136.0,20507520.0,20539904.0,20572288.0,20604672.0,20418176.0,20450624.0,20483072.0,20515520.0,20547968.0,20580416.0,20612864.0,20645312.0,20458368.0,20490880.0,20523392.0,20555904.0,20588416.0,20620928.0,20653440.0,20685952.0,20498560.0,20531136.0,20563712.0,20596288.0,20628864.0,20661440.0,20694016.0,20726592.0,20538752.0,20571392.0,20604032.0,20636672.0,20669312.0,20701952.0,20734592.0,20767232.0,20578944.0,20611648.0,20644352.0,20677056.0,20709760.0,20742464.0,20775168.0,20807872.0,20619136.0,20651904.0,20684672.0,20717440.0,20750208.0,20782976.0,20815744.0,20848512.0,20659328.0,20692160.0,20724992.0,20757824.0,20790656.0,20823488.0,20856320.0,20889152.0,19798400.0,19830272.0,19862144.0,19894016.0,19925888.0,19957760.0,19989632.0,20021504.0,18929280.0,18960192.0,18991104.0,19022016.0,19052928.0,19083840.0,19114752.0,19145664.0,18969472.0,19000448.0,19031424.0,19062400.0,19093376.0,19124352.0,19155328.0,19186304.0,19009664.0,19040704.0,19071744.0,19102784.0,19133824.0,19164864.0,19195904.0,19226944.0,19049856.0,19080960.0,19112064.0,19143168.0,19174272.0,19205376.0,19236480.0,19267584.0,19090048.0,19121216.0,19152384.0,19183552.0,19214720.0,19245888.0,19277056.0,19308224.0,19130240.0,19161472.0,19192704.0,19223936.0,19255168.0,19286400.0,19317632.0,19348864.0,19170432.0,19201728.0,19233024.0,19264320.0,19295616.0,19326912.0,19358208.0,19389504.0,19210624.0,19241984.0,19273344.0,19304704.0,19336064.0,19367424.0,19398784.0,19430144.0,19250816.0,19282240.0,19313664.0,19345088.0,19376512.0,19407936.0,19439360.0,19470784.0,19291008.0,19322496.0,19353984.0,19385472.0,19416960.0,19448448.0,19479936.0,19511424.0,19331200.0,19362752.0,19394304.0,19425856.0,19457408.0,19488960.0,19520512.0,19552064.0,19371392.0,19403008.0,19434624.0,19466240.0,19497856.0,19529472.0,19561088.0,19592704.0,19411584.0,19443264.0,19474944.0,19506624.0,19538304.0,19569984.0,19601664.0,19633344.0,19451776.0,19483520.0,19515264.0,19547008.0,19578752.0,19610496.0,19642240.0,19673984.0,19491968.0,19523776.0,19555584.0,19587392.0,19619200.0,19651008.0,19682816.0,19714624.0,19532160.0,19564032.0,19595904.0,19627776.0,19659648.0,19691520.0,19723392.0,19755264.0,19572352.0,19604288.0,19636224.0,19668160.0,19700096.0,19732032.0,19763968.0,19795904.0,19612544.0,19644544.0,19676544.0,19708544.0,19740544.0,19772544.0,19804544.0,19836544.0,19652736.0,19684800.0,19716864.0,19748928.0,19780992.0,19813056.0,19845120.0,19877184.0,19692928.0,19725056.0,19757184.0,19789312.0,19821440.0,19853568.0,19885696.0,19917824.0,19733120.0,19765312.0,19797504.0,19829696.0,19861888.0,19894080.0,19926272.0,19958464.0,19773312.0,19805568.0,19837824.0,19870080.0,19902336.0,19934592.0,19966848.0,19999104.0,19813504.0,19845824.0,19878144.0,19910464.0,19942784.0,19975104.0,20007424.0,20039744.0,19853696.0,19886080.0,19918464.0,19950848.0,19983232.0,20015616.0,20048000.0,20080384.0,19893888.0,19926336.0,19958784.0,19991232.0,20023680.0,20056128.0,20088576.0,20121024.0,19934080.0,19966592.0,19999104.0,20031616.0,20064128.0,20096640.0,20129152.0,20161664.0,19974272.0,20006848.0,20039424.0,20072000.0,20104576.0,20137152.0,20169728.0,20202304.0,20014464.0,20047104.0,20079744.0,20112384.0,20145024.0,20177664.0,20210304.0,20242944.0,20054656.0,20087360.0,20120064.0,20152768.0,20185472.0,20218176.0,20250880.0,20283584.0,20094848.0,20127616.0,20160384.0,20193152.0,20225920.0,20258688.0,20291456.0,20324224.0,20135040.0,20167872.0,20200704.0,20233536.0,20266368.0,20299200.0,20332032.0,20364864.0,20175232.0,20208128.0,20241024.0,20273920.0,20306816.0,20339712.0,20372608.0,20405504.0,20215424.0,20248384.0,20281344.0,20314304.0,20347264.0,20380224.0,20413184.0,20446144.0,20255616.0,20288640.0,20321664.0,20354688.0,20387712.0,20420736.0,20453760.0,20486784.0,20295808.0,20328896.0,20361984.0,20395072.0,20428160.0,20461248.0,20494336.0,20527424.0,20336000.0,20369152.0,20402304.0,20435456.0,20468608.0,20501760.0,20534912.0,20568064.0,20376192.0,20409408.0,20442624.0,20475840.0,20509056.0,20542272.0,20575488.0,20608704.0,20416384.0,20449664.0,20482944.0,20516224.0,20549504.0,20582784.0,20616064.0,20649344.0,20079744.0,20112064.0,20144384.0,20176704.0,20209024.0,20241344.0,20273664.0,20305984.0,19734912.0,19766272.0,19797632.0,19828992.0,19860352.0,19891712.0,19923072.0,19954432.0,19775104.0,19806528.0,19837952.0,19869376.0,19900800.0,19932224.0,19963648.0,19995072.0,19815296.0,19846784.0,19878272.0,19909760.0,19941248.0,19972736.0,20004224.0,20035712.0,19855488.0,19887040.0,19918592.0,19950144.0,19981696.0,20013248.0,20044800.0,20076352.0,19895680.0,19927296.0,19958912.0,19990528.0,20022144.0,20053760.0,20085376.0,20116992.0,19935872.0,19967552.0,19999232.0,20030912.0,20062592.0,20094272.0,20125952.0,20157632.0,19976064.0,20007808.0,20039552.0,20071296.0,20103040.0,20134784.0,20166528.0,20198272.0,20016256.0,20048064.0,20079872.0,20111680.0,20143488.0,20175296.0,20207104.0,20238912.0,20056448.0,20088320.0,20120192.0,20152064.0,20183936.0,20215808.0,20247680.0,20279552.0,20096640.0,20128576.0,20160512.0,20192448.0,20224384.0,20256320.0,20288256.0,20320192.0,20136832.0,20168832.0,20200832.0,20232832.0,20264832.0,20296832.0,20328832.0,20360832.0,20177024.0,20209088.0,20241152.0,20273216.0,20305280.0,20337344.0,20369408.0,20401472.0,20217216.0,20249344.0,20281472.0,20313600.0,20345728.0,20377856.0,20409984.0,20442112.0,20257408.0,20289600.0,20321792.0,20353984.0,20386176.0,20418368.0,20450560.0,20482752.0,20297600.0,20329856.0,20362112.0,20394368.0,20426624.0,20458880.0,20491136.0,20523392.0,20337792.0,20370112.0,20402432.0,20434752.0,20467072.0,20499392.0,20531712.0,20564032.0,20377984.0,20410368.0,20442752.0,20475136.0,20507520.0,20539904.0,20572288.0,20604672.0,20418176.0,20450624.0,20483072.0,20515520.0,20547968.0,20580416.0,20612864.0,20645312.0,20458368.0,20490880.0,20523392.0,20555904.0,20588416.0,20620928.0,20653440.0,20685952.0,20498560.0,20531136.0,20563712.0,20596288.0,20628864.0,20661440.0,20694016.0,20726592.0,20538752.0,20571392.0,20604032.0,20636672.0,20669312.0,20701952.0,20734592.0,20767232.0,20578944.0,20611648.0,20644352.0,20677056.0,20709760.0,20742464.0,20775168.0,20807872.0,20619136.0,20651904.0,20684672.0,20717440.0,20750208.0,20782976.0,20815744.0,20848512.0,20659328.0,20692160.0,20724992.0,20757824.0,20790656.0,20823488.0,20856320.0,20889152.0,20699520.0,20732416.0,20765312.0,20798208.0,20831104.0,20864000.0,20896896.0,20929792.0,20739712.0,20772672.0,20805632.0,20838592.0,20871552.0,20904512.0,20937472.0,20970432.0,20779904.0,20812928.0,20845952.0,20878976.0,20912000.0,20945024.0,20978048.0,21011072.0,20820096.0,20853184.0,20886272.0,20919360.0,20952448.0,20985536.0,21018624.0,21051712.0,20860288.0,20893440.0,20926592.0,20959744.0,20992896.0,21026048.0,21059200.0,21092352.0,20900480.0,20933696.0,20966912.0,21000128.0,21033344.0,21066560.0,21099776.0,21132992.0,20940672.0,20973952.0,21007232.0,21040512.0,21073792.0,21107072.0,21140352.0,21173632.0,20980864.0,21014208.0,21047552.0,21080896.0,21114240.0,21147584.0,21180928.0,21214272.0,21021056.0,21054464.0,21087872.0,21121280.0,21154688.0,21188096.0,21221504.0,21254912.0,21061248.0,21094720.0,21128192.0,21161664.0,21195136.0,21228608.0,21262080.0,21295552.0,21101440.0,21134976.0,21168512.0,21202048.0,21235584.0,21269120.0,21302656.0,21336192.0,21141632.0,21175232.0,21208832.0,21242432.0,21276032.0,21309632.0,21343232.0,21376832.0,21181824.0,21215488.0,21249152.0,21282816.0,21316480.0,21350144.0,21383808.0,21417472.0,21222016.0,21255744.0,21289472.0,21323200.0,21356928.0,21390656.0,21424384.0,21458112.0,20361088.0,20393856.0,20426624.0,20459392.0,20492160.0,20524928.0,20557696.0,20590464.0,19491968.0,19523776.0,19555584.0,19587392.0,19619200.0,19651008.0,19682816.0,19714624.0,9819776.0,9835696.0,9851616.0,9867536.0,9883456.0,9899376.0,9915296.0,9931216.0,10389504.0,10405296.0,10421088.0,10436880.0,10452672.0,10468464.0,10484256.0,10500048.0,20925824.0,20957440.0,20989056.0,21020672.0,21052288.0,21083904.0,21115520.0,21147136.0,20967040.0,20998720.0,21030400.0,21062080.0,21093760.0,21125440.0,21157120.0,21188800.0,21008256.0,21040000.0,21071744.0,21103488.0,21135232.0,21166976.0,21198720.0,21230464.0,21049472.0,21081280.0,21113088.0,21144896.0,21176704.0,21208512.0,21240320.0,21272128.0,21090688.0,21122560.0,21154432.0,21186304.0,21218176.0,21250048.0,21281920.0,21313792.0,21131904.0,21163840.0,21195776.0,21227712.0,21259648.0,21291584.0,21323520.0,21355456.0,21173120.0,21205120.0,21237120.0,21269120.0,21301120.0,21333120.0,21365120.0,21397120.0,21214336.0,21246400.0,21278464.0,21310528.0,21342592.0,21374656.0,21406720.0,21438784.0,21255552.0,21287680.0,21319808.0,21351936.0,21384064.0,21416192.0,21448320.0,21480448.0,21296768.0,21328960.0,21361152.0,21393344.0,21425536.0,21457728.0,21489920.0,21522112.0,21337984.0,21370240.0,21402496.0,21434752.0,21467008.0,21499264.0,21531520.0,21563776.0,21379200.0,21411520.0,21443840.0,21476160.0,21508480.0,21540800.0,21573120.0,21605440.0,21420416.0,21452800.0,21485184.0,21517568.0,21549952.0,21582336.0,21614720.0,21647104.0,21461632.0,21494080.0,21526528.0,21558976.0,21591424.0,21623872.0,21656320.0,21688768.0,21502848.0,21535360.0,21567872.0,21600384.0,21632896.0,21665408.0,21697920.0,21730432.0,21544064.0,21576640.0,21609216.0,21641792.0,21674368.0,21706944.0,21739520.0,21772096.0,21585280.0,21617920.0,21650560.0,21683200.0,21715840.0,21748480.0,21781120.0,21813760.0,21626496.0,21659200.0,21691904.0,21724608.0,21757312.0,21790016.0,21822720.0,21855424.0,21667712.0,21700480.0,21733248.0,21766016.0,21798784.0,21831552.0,21864320.0,21897088.0,21708928.0,21741760.0,21774592.0,21807424.0,21840256.0,21873088.0,21905920.0,21938752.0,21750144.0,21783040.0,21815936.0,21848832.0,21881728.0,21914624.0,21947520.0,21980416.0,21791360.0,21824320.0,21857280.0,21890240.0,21923200.0,21956160.0,21989120.0,22022080.0,21832576.0,21865600.0,21898624.0,21931648.0,21964672.0,21997696.0,22030720.0,22063744.0,21873792.0,21906880.0,21939968.0,21973056.0,22006144.0,22039232.0,22072320.0,22105408.0,21915008.0,21948160.0,21981312.0,22014464.0,22047616.0,22080768.0,22113920.0,22147072.0,21956224.0,21989440.0,22022656.0,22055872.0,22089088.0,22122304.0,22155520.0,22188736.0,21997440.0,22030720.0,22064000.0,22097280.0,22130560.0,22163840.0,22197120.0,22230400.0,22038656.0,22072000.0,22105344.0,22138688.0,22172032.0,22205376.0,22238720.0,22272064.0,22079872.0,22113280.0,22146688.0,22180096.0,22213504.0,22246912.0,22280320.0,22313728.0,22121088.0,22154560.0,22188032.0,22221504.0,22254976.0,22288448.0,22321920.0,22355392.0,22162304.0,22195840.0,22229376.0,22262912.0,22296448.0,22329984.0,22363520.0,22397056.0,22203520.0,22237120.0,22270720.0,22304320.0,22337920.0,22371520.0,22405120.0,22438720.0,22244736.0,22278400.0,22312064.0,22345728.0,22379392.0,22413056.0,22446720.0,22480384.0,22285952.0,22319680.0,22353408.0,22387136.0,22420864.0,22454592.0,22488320.0,22522048.0,22327168.0,22360960.0,22394752.0,22428544.0,22462336.0,22496128.0,22529920.0,22563712.0,22368384.0,22402240.0,22436096.0,22469952.0,22503808.0,22537664.0,22571520.0,22605376.0,21492096.0,21524992.0,21557888.0,21590784.0,21623680.0,21656576.0,21689472.0,21722368.0,20607616.0,20639552.0,20671488.0,20703424.0,20735360.0,20767296.0,20799232.0,20831168.0,19731328.0,19762304.0,19793280.0,19824256.0,19855232.0,19886208.0,19917184.0,19948160.0,18846848.0,18876864.0,18906880.0,18936896.0,18966912.0,18996928.0,19026944.0,19056960.0,18888064.0,18918144.0,18948224.0,18978304.0,19008384.0,19038464.0,19068544.0,19098624.0,18929280.0,18959424.0,18989568.0,19019712.0,19049856.0,19080000.0,19110144.0,19140288.0,18970496.0,19000704.0,19030912.0,19061120.0,19091328.0,19121536.0,19151744.0,19181952.0,19011712.0,19041984.0,19072256.0,19102528.0,19132800.0,19163072.0,19193344.0,19223616.0,19052928.0,19083264.0,19113600.0,19143936.0,19174272.0,19204608.0,19234944.0,19265280.0,19094144.0,19124544.0,19154944.0,19185344.0,19215744.0,19246144.0,19276544.0,19306944.0,19135360.0,19165824.0,19196288.0,19226752.0,19257216.0,19287680.0,19318144.0,19348608.0,19176576.0,19207104.0,19237632.0,19268160.0,19298688.0,19329216.0,19359744.0,19390272.0,19217792.0,19248384.0,19278976.0,19309568.0,19340160.0,19370752.0,19401344.0,19431936.0,19259008.0,19289664.0,19320320.0,19350976.0,19381632.0,19412288.0,19442944.0,19473600.0,19300224.0,19330944.0,19361664.0,19392384.0,19423104.0,19453824.0,19484544.0,19515264.0,19341440.0,19372224.0,19403008.0,19433792.0,19464576.0,19495360.0,19526144.0,19556928.0,19382656.0,19413504.0,19444352.0,19475200.0,19506048.0,19536896.0,19567744.0,19598592.0,19423872.0,19454784.0,19485696.0,19516608.0,19547520.0,19578432.0,19609344.0,19640256.0,19465088.0,19496064.0,19527040.0,19558016.0,19588992.0,19619968.0,19650944.0,19681920.0,19506304.0,19537344.0,19568384.0,19599424.0,19630464.0,19661504.0,19692544.0,19723584.0,19547520.0,19578624.0,19609728.0,19640832.0,19671936.0,19703040.0,19734144.0,19765248.0,19588736.0,19619904.0,19651072.0,19682240.0,19713408.0,19744576.0,19775744.0,19806912.0,19629952.0,19661184.0,19692416.0,19723648.0,19754880.0,19786112.0,19817344.0,19848576.0,19671168.0,19702464.0,19733760.0,19765056.0,19796352.0,19827648.0,19858944.0,19890240.0,19712384.0,19743744.0,19775104.0,19806464.0,19837824.0,19869184.0,19900544.0,19931904.0,19753600.0,19785024.0,19816448.0,19847872.0,19879296.0,19910720.0,19942144.0,19973568.0,19794816.0,19826304.0,19857792.0,19889280.0,19920768.0,19952256.0,19983744.0,20015232.0,19836032.0,19867584.0,19899136.0,19930688.0,19962240.0,19993792.0,20025344.0,20056896.0,19877248.0,19908864.0,19940480.0,19972096.0,20003712.0,20035328.0,20066944.0,20098560.0,19918464.0,19950144.0,19981824.0,20013504.0,20045184.0,20076864.0,20108544.0,20140224.0,19959680.0,19991424.0,20023168.0,20054912.0,20086656.0,20118400.0,20150144.0,20181888.0,20000896.0,20032704.0,20064512.0,20096320.0,20128128.0,20159936.0,20191744.0,20223552.0,20042112.0,20073984.0,20105856.0,20137728.0,20169600.0,20201472.0,20233344.0,20265216.0,20083328.0,20115264.0,20147200.0,20179136.0,20211072.0,20243008.0,20274944.0,20306880.0,20124544.0,20156544.0,20188544.0,20220544.0,20252544.0,20284544.0,20316544.0,20348544.0,20165760.0,20197824.0,20229888.0,20261952.0,20294016.0,20326080.0,20358144.0,20390208.0,20206976.0,20239104.0,20271232.0,20303360.0,20335488.0,20367616.0,20399744.0,20431872.0,20248192.0,20280384.0,20312576.0,20344768.0,20376960.0,20409152.0,20441344.0,20473536.0,20289408.0,20321664.0,20353920.0,20386176.0,20418432.0,20450688.0,20482944.0,20515200.0,20330624.0,20362944.0,20395264.0,20427584.0,20459904.0,20492224.0,20524544.0,20556864.0,20371840.0,20404224.0,20436608.0,20468992.0,20501376.0,20533760.0,20566144.0,20598528.0,20019840.0,20051264.0,20082688.0,20114112.0,20145536.0,20176960.0,20208384.0,20239808.0,19659648.0,19690112.0,19720576.0,19751040.0,19781504.0,19811968.0,19842432.0,19872896.0,19700864.0,19731392.0,19761920.0,19792448.0,19822976.0,19853504.0,19884032.0,19914560.0,19742080.0,19772672.0,19803264.0,19833856.0,19864448.0,19895040.0,19925632.0,19956224.0,19783296.0,19813952.0,19844608.0,19875264.0,19905920.0,19936576.0,19967232.0,19997888.0,19824512.0,19855232.0,19885952.0,19916672.0,19947392.0,19978112.0,20008832.0,20039552.0,19865728.0,19896512.0,19927296.0,19958080.0,19988864.0,20019648.0,20050432.0,20081216.0,19906944.0,19937792.0,19968640.0,19999488.0,20030336.0,20061184.0,20092032.0,20122880.0,19948160.0,19979072.0,20009984.0,20040896.0,20071808.0,20102720.0,20133632.0,20164544.0,19989376.0,20020352.0,20051328.0,20082304.0,20113280.0,20144256.0,20175232.0,20206208.0,20030592.0,20061632.0,20092672.0,20123712.0,20154752.0,20185792.0,20216832.0,20247872.0,20071808.0,20102912.0,20134016.0,20165120.0,20196224.0,20227328.0,20258432.0,20289536.0,20113024.0,20144192.0,20175360.0,20206528.0,20237696.0,20268864.0,20300032.0,20331200.0,20154240.0,20185472.0,20216704.0,20247936.0,20279168.0,20310400.0,20341632.0,20372864.0,20195456.0,20226752.0,20258048.0,20289344.0,20320640.0,20351936.0,20383232.0,20414528.0,20236672.0,20268032.0,20299392.0,20330752.0,20362112.0,20393472.0,20424832.0,20456192.0,20277888.0,20309312.0,20340736.0,20372160.0,20403584.0,20435008.0,20466432.0,20497856.0,20319104.0,20350592.0,20382080.0,20413568.0,20445056.0,20476544.0,20508032.0,20539520.0,20360320.0,20391872.0,20423424.0,20454976.0,20486528.0,20518080.0,20549632.0,20581184.0,20401536.0,20433152.0,20464768.0,20496384.0,20528000.0,20559616.0,20591232.0,20622848.0,20442752.0,20474432.0,20506112.0,20537792.0,20569472.0,20601152.0,20632832.0,20664512.0,20483968.0,20515712.0,20547456.0,20579200.0,20610944.0,20642688.0,20674432.0,20706176.0,20525184.0,20556992.0,20588800.0,20620608.0,20652416.0,20684224.0,20716032.0,20747840.0,20566400.0,20598272.0,20630144.0,20662016.0,20693888.0,20725760.0,20757632.0,20789504.0,20607616.0,20639552.0,20671488.0,20703424.0,20735360.0,20767296.0,20799232.0,20831168.0,20648832.0,20680832.0,20712832.0,20744832.0,20776832.0,20808832.0,20840832.0,20872832.0,20690048.0,20722112.0,20754176.0,20786240.0,20818304.0,20850368.0,20882432.0,20914496.0,20731264.0,20763392.0,20795520.0,20827648.0,20859776.0,20891904.0,20924032.0,20956160.0,20772480.0,20804672.0,20836864.0,20869056.0,20901248.0,20933440.0,20965632.0,20997824.0,20813696.0,20845952.0,20878208.0,20910464.0,20942720.0,20974976.0,21007232.0,21039488.0,20854912.0,20887232.0,20919552.0,20951872.0,20984192.0,21016512.0,21048832.0,21081152.0,20896128.0,20928512.0,20960896.0,20993280.0,21025664.0,21058048.0,21090432.0,21122816.0,20937344.0,20969792.0,21002240.0,21034688.0,21067136.0,21099584.0,21132032.0,21164480.0,20978560.0,21011072.0,21043584.0,21076096.0,21108608.0,21141120.0,21173632.0,21206144.0,21019776.0,21052352.0,21084928.0,21117504.0,21150080.0,21182656.0,21215232.0,21247808.0,21060992.0,21093632.0,21126272.0,21158912.0,21191552.0,21224192.0,21256832.0,21289472.0,21102208.0,21134912.0,21167616.0,21200320.0,21233024.0,21265728.0,21298432.0,21331136.0,21143424.0,21176192.0,21208960.0,21241728.0,21274496.0,21307264.0,21340032.0,21372800.0,21184640.0,21217472.0,21250304.0,21283136.0,21315968.0,21348800.0,21381632.0,21414464.0,20308352.0,20340224.0,20372096.0,20403968.0,20435840.0,20467712.0,20499584.0,20531456.0,19423872.0,19454784.0,19485696.0,19516608.0,19547520.0,19578432.0,19609344.0,19640256.0,19465088.0,19496064.0,19527040.0,19558016.0,19588992.0,19619968.0,19650944.0,19681920.0,19506304.0,19537344.0,19568384.0,19599424.0,19630464.0,19661504.0,19692544.0,19723584.0,19547520.0,19578624.0,19609728.0,19640832.0,19671936.0,19703040.0,19734144.0,19765248.0,19588736.0,19619904.0,19651072.0,19682240.0,19713408.0,19744576.0,19775744.0,19806912.0,19629952.0,19661184.0,19692416.0,19723648.0,19754880.0,19786112.0,19817344.0,19848576.0,19671168.0,19702464.0,19733760.0,19765056.0,19796352.0,19827648.0,19858944.0,19890240.0,19712384.0,19743744.0,19775104.0,19806464.0,19837824.0,19869184.0,19900544.0,19931904.0,19753600.0,19785024.0,19816448.0,19847872.0,19879296.0,19910720.0,19942144.0,19973568.0,19794816.0,19826304.0,19857792.0,19889280.0,19920768.0,19952256.0,19983744.0,20015232.0,19836032.0,19867584.0,19899136.0,19930688.0,19962240.0,19993792.0,20025344.0,20056896.0,19877248.0,19908864.0,19940480.0,19972096.0,20003712.0,20035328.0,20066944.0,20098560.0,19918464.0,19950144.0,19981824.0,20013504.0,20045184.0,20076864.0,20108544.0,20140224.0,19959680.0,19991424.0,20023168.0,20054912.0,20086656.0,20118400.0,20150144.0,20181888.0,20000896.0,20032704.0,20064512.0,20096320.0,20128128.0,20159936.0,20191744.0,20223552.0,20042112.0,20073984.0,20105856.0,20137728.0,20169600.0,20201472.0,20233344.0,20265216.0,20083328.0,20115264.0,20147200.0,20179136.0,20211072.0,20243008.0,20274944.0,20306880.0,20124544.0,20156544.0,20188544.0,20220544.0,20252544.0,20284544.0,20316544.0,20348544.0,20165760.0,20197824.0,20229888.0,20261952.0,20294016.0,20326080.0,20358144.0,20390208.0,20206976.0,20239104.0,20271232.0,20303360.0,20335488.0,20367616.0,20399744.0,20431872.0,20248192.0,20280384.0,20312576.0,20344768.0,20376960.0,20409152.0,20441344.0,20473536.0,20289408.0,20321664.0,20353920.0,20386176.0,20418432.0,20450688.0,20482944.0,20515200.0,20330624.0,20362944.0,20395264.0,20427584.0,20459904.0,20492224.0,20524544.0,20556864.0,20371840.0,20404224.0,20436608.0,20468992.0,20501376.0,20533760.0,20566144.0,20598528.0,20413056.0,20445504.0,20477952.0,20510400.0,20542848.0,20575296.0,20607744.0,20640192.0,20454272.0,20486784.0,20519296.0,20551808.0,20584320.0,20616832.0,20649344.0,20681856.0,20495488.0,20528064.0,20560640.0,20593216.0,20625792.0,20658368.0,20690944.0,20723520.0,20536704.0,20569344.0,20601984.0,20634624.0,20667264.0,20699904.0,20732544.0,20765184.0,20577920.0,20610624.0,20643328.0,20676032.0,20708736.0,20741440.0,20774144.0,20806848.0,20619136.0,20651904.0,20684672.0,20717440.0,20750208.0,20782976.0,20815744.0,20848512.0,20660352.0,20693184.0,20726016.0,20758848.0,20791680.0,20824512.0,20857344.0,20890176.0,20701568.0,20734464.0,20767360.0,20800256.0,20833152.0,20866048.0,20898944.0,20931840.0,20742784.0,20775744.0,20808704.0,20841664.0,20874624.0,20907584.0,20940544.0,20973504.0,20784000.0,20817024.0,20850048.0,20883072.0,20916096.0,20949120.0,20982144.0,21015168.0,20825216.0,20858304.0,20891392.0,20924480.0,20957568.0,20990656.0,21023744.0,21056832.0,20866432.0,20899584.0,20932736.0,20965888.0,20999040.0,21032192.0,21065344.0,21098496.0,20907648.0,20940864.0,20974080.0,21007296.0,21040512.0,21073728.0,21106944.0,21140160.0,20948864.0,20982144.0,21015424.0,21048704.0,21081984.0,21115264.0,21148544.0,21181824.0,20596864.0,20629184.0,20661504.0,20693824.0,20726144.0,20758464.0,20790784.0,20823104.0,20236672.0,20268032.0,20299392.0,20330752.0,20362112.0,20393472.0,20424832.0,20456192.0,20277888.0,20309312.0,20340736.0,20372160.0,20403584.0,20435008.0,20466432.0,20497856.0,20319104.0,20350592.0,20382080.0,20413568.0,20445056.0,20476544.0,20508032.0,20539520.0,20360320.0,20391872.0,20423424.0,20454976.0,20486528.0,20518080.0,20549632.0,20581184.0,20401536.0,20433152.0,20464768.0,20496384.0,20528000.0,20559616.0,20591232.0,20622848.0,20442752.0,20474432.0,20506112.0,20537792.0,20569472.0,20601152.0,20632832.0,20664512.0,20483968.0,20515712.0,20547456.0,20579200.0,20610944.0,20642688.0,20674432.0,20706176.0,20525184.0,20556992.0,20588800.0,20620608.0,20652416.0,20684224.0,20716032.0,20747840.0,20566400.0,20598272.0,20630144.0,20662016.0,20693888.0,20725760.0,20757632.0,20789504.0,20607616.0,20639552.0,20671488.0,20703424.0,20735360.0,20767296.0,20799232.0,20831168.0,20648832.0,20680832.0,20712832.0,20744832.0,20776832.0,20808832.0,20840832.0,20872832.0,20690048.0,20722112.0,20754176.0,20786240.0,20818304.0,20850368.0,20882432.0,20914496.0,20731264.0,20763392.0,20795520.0,20827648.0,20859776.0,20891904.0,20924032.0,20956160.0,20772480.0,20804672.0,20836864.0,20869056.0,20901248.0,20933440.0,20965632.0,20997824.0,20813696.0,20845952.0,20878208.0,20910464.0,20942720.0,20974976.0,21007232.0,21039488.0,20854912.0,20887232.0,20919552.0,20951872.0,20984192.0,21016512.0,21048832.0,21081152.0,20896128.0,20928512.0,20960896.0,20993280.0,21025664.0,21058048.0,21090432.0,21122816.0,20937344.0,20969792.0,21002240.0,21034688.0,21067136.0,21099584.0,21132032.0,21164480.0,20978560.0,21011072.0,21043584.0,21076096.0,21108608.0,21141120.0,21173632.0,21206144.0,21019776.0,21052352.0,21084928.0,21117504.0,21150080.0,21182656.0,21215232.0,21247808.0,21060992.0,21093632.0,21126272.0,21158912.0,21191552.0,21224192.0,21256832.0,21289472.0,21102208.0,21134912.0,21167616.0,21200320.0,21233024.0,21265728.0,21298432.0,21331136.0,21143424.0,21176192.0,21208960.0,21241728.0,21274496.0,21307264.0,21340032.0,21372800.0,21184640.0,21217472.0,21250304.0,21283136.0,21315968.0,21348800.0,21381632.0,21414464.0,21225856.0,21258752.0,21291648.0,21324544.0,21357440.0,21390336.0,21423232.0,21456128.0,21267072.0,21300032.0,21332992.0,21365952.0,21398912.0,21431872.0,21464832.0,21497792.0,21308288.0,21341312.0,21374336.0,21407360.0,21440384.0,21473408.0,21506432.0,21539456.0,21349504.0,21382592.0,21415680.0,21448768.0,21481856.0,21514944.0,21548032.0,21581120.0,21390720.0,21423872.0,21457024.0,21490176.0,21523328.0,21556480.0,21589632.0,21622784.0,21431936.0,21465152.0,21498368.0,21531584.0,21564800.0,21598016.0,21631232.0,21664448.0,21473152.0,21506432.0,21539712.0,21572992.0,21606272.0,21639552.0,21672832.0,21706112.0,21514368.0,21547712.0,21581056.0,21614400.0,21647744.0,21681088.0,21714432.0,21747776.0,21555584.0,21588992.0,21622400.0,21655808.0,21689216.0,21722624.0,21756032.0,21789440.0,21596800.0,21630272.0,21663744.0,21697216.0,21730688.0,21764160.0,21797632.0,21831104.0,21638016.0,21671552.0,21705088.0,21738624.0,21772160.0,21805696.0,21839232.0,21872768.0,21679232.0,21712832.0,21746432.0,21780032.0,21813632.0,21847232.0,21880832.0,21914432.0,21720448.0,21754112.0,21787776.0,21821440.0,21855104.0,21888768.0,21922432.0,21956096.0,21761664.0,21795392.0,21829120.0,21862848.0,21896576.0,21930304.0,21964032.0,21997760.0,20885376.0,20918144.0,20950912.0,20983680.0,21016448.0,21049216.0,21081984.0,21114752.0,20000896.0,20032704.0,20064512.0,20096320.0,20128128.0,20159936.0,20191744.0,20223552.0,10074496.0,10090416.0,10106336.0,10122256.0,10138176.0,10154096.0,10170016.0,10185936.0,10642176.0,10657968.0,10673760.0,10689552.0,10705344.0,10721136.0,10736928.0,10752720.0,21431680.0,21463296.0,21494912.0,21526528.0,21558144.0,21589760.0,21621376.0,21652992.0,21473920.0,21505600.0,21537280.0,21568960.0,21600640.0,21632320.0,21664000.0,21695680.0,21516160.0,21547904.0,21579648.0,21611392.0,21643136.0,21674880.0,21706624.0,21738368.0,21558400.0,21590208.0,21622016.0,21653824.0,21685632.0,21717440.0,21749248.0,21781056.0,21600640.0,21632512.0,21664384.0,21696256.0,21728128.0,21760000.0,21791872.0,21823744.0,21642880.0,21674816.0,21706752.0,21738688.0,21770624.0,21802560.0,21834496.0,21866432.0,21685120.0,21717120.0,21749120.0,21781120.0,21813120.0,21845120.0,21877120.0,21909120.0,21727360.0,21759424.0,21791488.0,21823552.0,21855616.0,21887680.0,21919744.0,21951808.0,21769600.0,21801728.0,21833856.0,21865984.0,21898112.0,21930240.0,21962368.0,21994496.0,21811840.0,21844032.0,21876224.0,21908416.0,21940608.0,21972800.0,22004992.0,22037184.0,21854080.0,21886336.0,21918592.0,21950848.0,21983104.0,22015360.0,22047616.0,22079872.0,21896320.0,21928640.0,21960960.0,21993280.0,22025600.0,22057920.0,22090240.0,22122560.0,21938560.0,21970944.0,22003328.0,22035712.0,22068096.0,22100480.0,22132864.0,22165248.0,21980800.0,22013248.0,22045696.0,22078144.0,22110592.0,22143040.0,22175488.0,22207936.0,22023040.0,22055552.0,22088064.0,22120576.0,22153088.0,22185600.0,22218112.0,22250624.0,22065280.0,22097856.0,22130432.0,22163008.0,22195584.0,22228160.0,22260736.0,22293312.0,22107520.0,22140160.0,22172800.0,22205440.0,22238080.0,22270720.0,22303360.0,22336000.0,22149760.0,22182464.0,22215168.0,22247872.0,22280576.0,22313280.0,22345984.0,22378688.0,22192000.0,22224768.0,22257536.0,22290304.0,22323072.0,22355840.0,22388608.0,22421376.0,22234240.0,22267072.0,22299904.0,22332736.0,22365568.0,22398400.0,22431232.0,22464064.0,22276480.0,22309376.0,22342272.0,22375168.0,22408064.0,22440960.0,22473856.0,22506752.0,22318720.0,22351680.0,22384640.0,22417600.0,22450560.0,22483520.0,22516480.0,22549440.0,22360960.0,22393984.0,22427008.0,22460032.0,22493056.0,22526080.0,22559104.0,22592128.0,22403200.0,22436288.0,22469376.0,22502464.0,22535552.0,22568640.0,22601728.0,22634816.0,22445440.0,22478592.0,22511744.0,22544896.0,22578048.0,22611200.0,22644352.0,22677504.0,22487680.0,22520896.0,22554112.0,22587328.0,22620544.0,22653760.0,22686976.0,22720192.0,22529920.0,22563200.0,22596480.0,22629760.0,22663040.0,22696320.0,22729600.0,22762880.0,22572160.0,22605504.0,22638848.0,22672192.0,22705536.0,22738880.0,22772224.0,22805568.0,22614400.0,22647808.0,22681216.0,22714624.0,22748032.0,22781440.0,22814848.0,22848256.0,22656640.0,22690112.0,22723584.0,22757056.0,22790528.0,22824000.0,22857472.0,22890944.0,22698880.0,22732416.0,22765952.0,22799488.0,22833024.0,22866560.0,22900096.0,22933632.0,22741120.0,22774720.0,22808320.0,22841920.0,22875520.0,22909120.0,22942720.0,22976320.0,22783360.0,22817024.0,22850688.0,22884352.0,22918016.0,22951680.0,22985344.0,23019008.0,22825600.0,22859328.0,22893056.0,22926784.0,22960512.0,22994240.0,23027968.0,23061696.0,22867840.0,22901632.0,22935424.0,22969216.0,23003008.0,23036800.0,23070592.0,23104384.0,22910080.0,22943936.0,22977792.0,23011648.0,23045504.0,23079360.0,23113216.0,23147072.0,22018432.0,22051328.0,22084224.0,22117120.0,22150016.0,22182912.0,22215808.0,22248704.0,21118592.0,21150528.0,21182464.0,21214400.0,21246336.0,21278272.0,21310208.0,21342144.0,20226944.0,20257920.0,20288896.0,20319872.0,20350848.0,20381824.0,20412800.0,20443776.0,19327104.0,19357120.0,19387136.0,19417152.0,19447168.0,19477184.0,19507200.0,19537216.0,19369344.0,19399424.0,19429504.0,19459584.0,19489664.0,19519744.0,19549824.0,19579904.0,19411584.0,19441728.0,19471872.0,19502016.0,19532160.0,19562304.0,19592448.0,19622592.0,19453824.0,19484032.0,19514240.0,19544448.0,19574656.0,19604864.0,19635072.0,19665280.0,19496064.0,19526336.0,19556608.0,19586880.0,19617152.0,19647424.0,19677696.0,19707968.0,19538304.0,19568640.0,19598976.0,19629312.0,19659648.0,19689984.0,19720320.0,19750656.0,19580544.0,19610944.0,19641344.0,19671744.0,19702144.0,19732544.0,19762944.0,19793344.0,19622784.0,19653248.0,19683712.0,19714176.0,19744640.0,19775104.0,19805568.0,19836032.0,19665024.0,19695552.0,19726080.0,19756608.0,19787136.0,19817664.0,19848192.0,19878720.0,19707264.0,19737856.0,19768448.0,19799040.0,19829632.0,19860224.0,19890816.0,19921408.0,19749504.0,19780160.0,19810816.0,19841472.0,19872128.0,19902784.0,19933440.0,19964096.0,19791744.0,19822464.0,19853184.0,19883904.0,19914624.0,19945344.0,19976064.0,20006784.0,19833984.0,19864768.0,19895552.0,19926336.0,19957120.0,19987904.0,20018688.0,20049472.0,19876224.0,19907072.0,19937920.0,19968768.0,19999616.0,20030464.0,20061312.0,20092160.0,19918464.0,19949376.0,19980288.0,20011200.0,20042112.0,20073024.0,20103936.0,20134848.0,19960704.0,19991680.0,20022656.0,20053632.0,20084608.0,20115584.0,20146560.0,20177536.0,20002944.0,20033984.0,20065024.0,20096064.0,20127104.0,20158144.0,20189184.0,20220224.0,20045184.0,20076288.0,20107392.0,20138496.0,20169600.0,20200704.0,20231808.0,20262912.0,20087424.0,20118592.0,20149760.0,20180928.0,20212096.0,20243264.0,20274432.0,20305600.0,20129664.0,20160896.0,20192128.0,20223360.0,20254592.0,20285824.0,20317056.0,20348288.0,20171904.0,20203200.0,20234496.0,20265792.0,20297088.0,20328384.0,20359680.0,20390976.0,20214144.0,20245504.0,20276864.0,20308224.0,20339584.0,20370944.0,20402304.0,20433664.0,20256384.0,20287808.0,20319232.0,20350656.0,20382080.0,20413504.0,20444928.0,20476352.0,20298624.0,20330112.0,20361600.0,20393088.0,20424576.0,20456064.0,20487552.0,20519040.0,20340864.0,20372416.0,20403968.0,20435520.0,20467072.0,20498624.0,20530176.0,20561728.0,20383104.0,20414720.0,20446336.0,20477952.0,20509568.0,20541184.0,20572800.0,20604416.0,20425344.0,20457024.0,20488704.0,20520384.0,20552064.0,20583744.0,20615424.0,20647104.0,20467584.0,20499328.0,20531072.0,20562816.0,20594560.0,20626304.0,20658048.0,20689792.0,20509824.0,20541632.0,20573440.0,20605248.0,20637056.0,20668864.0,20700672.0,20732480.0,20552064.0,20583936.0,20615808.0,20647680.0,20679552.0,20711424.0,20743296.0,20775168.0,20594304.0,20626240.0,20658176.0,20690112.0,20722048.0,20753984.0,20785920.0,20817856.0,20636544.0,20668544.0,20700544.0,20732544.0,20764544.0,20796544.0,20828544.0,20860544.0,20678784.0,20710848.0,20742912.0,20774976.0,20807040.0,20839104.0,20871168.0,20903232.0,20721024.0,20753152.0,20785280.0,20817408.0,20849536.0,20881664.0,20913792.0,20945920.0,20763264.0,20795456.0,20827648.0,20859840.0,20892032.0,20924224.0,20956416.0,20988608.0,20805504.0,20837760.0,20870016.0,20902272.0,20934528.0,20966784.0,20999040.0,21031296.0,20847744.0,20880064.0,20912384.0,20944704.0,20977024.0,21009344.0,21041664.0,21073984.0,20889984.0,20922368.0,20954752.0,20987136.0,21019520.0,21051904.0,21084288.0,21116672.0,20522624.0,20554048.0,20585472.0,20616896.0,20648320.0,20679744.0,20711168.0,20742592.0,20147072.0,20177536.0,20208000.0,20238464.0,20268928.0,20299392.0,20329856.0,20360320.0,20189312.0,20219840.0,20250368.0,20280896.0,20311424.0,20341952.0,20372480.0,20403008.0,20231552.0,20262144.0,20292736.0,20323328.0,20353920.0,20384512.0,20415104.0,20445696.0,20273792.0,20304448.0,20335104.0,20365760.0,20396416.0,20427072.0,20457728.0,20488384.0,20316032.0,20346752.0,20377472.0,20408192.0,20438912.0,20469632.0,20500352.0,20531072.0,20358272.0,20389056.0,20419840.0,20450624.0,20481408.0,20512192.0,20542976.0,20573760.0,20400512.0,20431360.0,20462208.0,20493056.0,20523904.0,20554752.0,20585600.0,20616448.0,20442752.0,20473664.0,20504576.0,20535488.0,20566400.0,20597312.0,20628224.0,20659136.0,20484992.0,20515968.0,20546944.0,20577920.0,20608896.0,20639872.0,20670848.0,20701824.0,20527232.0,20558272.0,20589312.0,20620352.0,20651392.0,20682432.0,20713472.0,20744512.0,20569472.0,20600576.0,20631680.0,20662784.0,20693888.0,20724992.0,20756096.0,20787200.0,20611712.0,20642880.0,20674048.0,20705216.0,20736384.0,20767552.0,20798720.0,20829888.0,20653952.0,20685184.0,20716416.0,20747648.0,20778880.0,20810112.0,20841344.0,20872576.0,20696192.0,20727488.0,20758784.0,20790080.0,20821376.0,20852672.0,20883968.0,20915264.0,20738432.0,20769792.0,20801152.0,20832512.0,20863872.0,20895232.0,20926592.0,20957952.0,20780672.0,20812096.0,20843520.0,20874944.0,20906368.0,20937792.0,20969216.0,21000640.0,20822912.0,20854400.0,20885888.0,20917376.0,20948864.0,20980352.0,21011840.0,21043328.0,20865152.0,20896704.0,20928256.0,20959808.0,20991360.0,21022912.0,21054464.0,21086016.0,20907392.0,20939008.0,20970624.0,21002240.0,21033856.0,21065472.0,21097088.0,21128704.0,20949632.0,20981312.0,21012992.0,21044672.0,21076352.0,21108032.0,21139712.0,21171392.0,20991872.0,21023616.0,21055360.0,21087104.0,21118848.0,21150592.0,21182336.0,21214080.0,21034112.0,21065920.0,21097728.0,21129536.0,21161344.0,21193152.0,21224960.0,21256768.0,21076352.0,21108224.0,21140096.0,21171968.0,21203840.0,21235712.0,21267584.0,21299456.0,21118592.0,21150528.0,21182464.0,21214400.0,21246336.0,21278272.0,21310208.0,21342144.0,21160832.0,21192832.0,21224832.0,21256832.0,21288832.0,21320832.0,21352832.0,21384832.0,21203072.0,21235136.0,21267200.0,21299264.0,21331328.0,21363392.0,21395456.0,21427520.0,21245312.0,21277440.0,21309568.0,21341696.0,21373824.0,21405952.0,21438080.0,21470208.0,21287552.0,21319744.0,21351936.0,21384128.0,21416320.0,21448512.0,21480704.0,21512896.0,21329792.0,21362048.0,21394304.0,21426560.0,21458816.0,21491072.0,21523328.0,21555584.0,21372032.0,21404352.0,21436672.0,21468992.0,21501312.0,21533632.0,21565952.0,21598272.0,21414272.0,21446656.0,21479040.0,21511424.0,21543808.0,21576192.0,21608576.0,21640960.0,21456512.0,21488960.0,21521408.0,21553856.0,21586304.0,21618752.0,21651200.0,21683648.0,21498752.0,21531264.0,21563776.0,21596288.0,21628800.0,21661312.0,21693824.0,21726336.0,21540992.0,21573568.0,21606144.0,21638720.0,21671296.0,21703872.0,21736448.0,21769024.0,21583232.0,21615872.0,21648512.0,21681152.0,21713792.0,21746432.0,21779072.0,21811712.0,21625472.0,21658176.0,21690880.0,21723584.0,21756288.0,21788992.0,21821696.0,21854400.0,21667712.0,21700480.0,21733248.0,21766016.0,21798784.0,21831552.0,21864320.0,21897088.0,21709952.0,21742784.0,21775616.0,21808448.0,21841280.0,21874112.0,21906944.0,21939776.0,20818304.0,20850176.0,20882048.0,20913920.0,20945792.0,20977664.0,21009536.0,21041408.0,19918464.0,19949376.0,19980288.0,20011200.0,20042112.0,20073024.0,20103936.0,20134848.0,19960704.0,19991680.0,20022656.0,20053632.0,20084608.0,20115584.0,20146560.0,20177536.0,20002944.0,20033984.0,20065024.0,20096064.0,20127104.0,20158144.0,20189184.0,20220224.0,20045184.0,20076288.0,20107392.0,20138496.0,20169600.0,20200704.0,20231808.0,20262912.0,20087424.0,20118592.0,20149760.0,20180928.0,20212096.0,20243264.0,20274432.0,20305600.0,20129664.0,20160896.0,20192128.0,20223360.0,20254592.0,20285824.0,20317056.0,20348288.0,20171904.0,20203200.0,20234496.0,20265792.0,20297088.0,20328384.0,20359680.0,20390976.0,20214144.0,20245504.0,20276864.0,20308224.0,20339584.0,20370944.0,20402304.0,20433664.0,20256384.0,20287808.0,20319232.0,20350656.0,20382080.0,20413504.0,20444928.0,20476352.0,20298624.0,20330112.0,20361600.0,20393088.0,20424576.0,20456064.0,20487552.0,20519040.0,20340864.0,20372416.0,20403968.0,20435520.0,20467072.0,20498624.0,20530176.0,20561728.0,20383104.0,20414720.0,20446336.0,20477952.0,20509568.0,20541184.0,20572800.0,20604416.0,20425344.0,20457024.0,20488704.0,20520384.0,20552064.0,20583744.0,20615424.0,20647104.0,20467584.0,20499328.0,20531072.0,20562816.0,20594560.0,20626304.0,20658048.0,20689792.0,20509824.0,20541632.0,20573440.0,20605248.0,20637056.0,20668864.0,20700672.0,20732480.0,20552064.0,20583936.0,20615808.0,20647680.0,20679552.0,20711424.0,20743296.0,20775168.0,20594304.0,20626240.0,20658176.0,20690112.0,20722048.0,20753984.0,20785920.0,20817856.0,20636544.0,20668544.0,20700544.0,20732544.0,20764544.0,20796544.0,20828544.0,20860544.0,20678784.0,20710848.0,20742912.0,20774976.0,20807040.0,20839104.0,20871168.0,20903232.0,20721024.0,20753152.0,20785280.0,20817408.0,20849536.0,20881664.0,20913792.0,20945920.0,20763264.0,20795456.0,20827648.0,20859840.0,20892032.0,20924224.0,20956416.0,20988608.0,20805504.0,20837760.0,20870016.0,20902272.0,20934528.0,20966784.0,20999040.0,21031296.0,20847744.0,20880064.0,20912384.0,20944704.0,20977024.0,21009344.0,21041664.0,21073984.0,20889984.0,20922368.0,20954752.0,20987136.0,21019520.0,21051904.0,21084288.0,21116672.0,20932224.0,20964672.0,20997120.0,21029568.0,21062016.0,21094464.0,21126912.0,21159360.0,20974464.0,21006976.0,21039488.0,21072000.0,21104512.0,21137024.0,21169536.0,21202048.0,21016704.0,21049280.0,21081856.0,21114432.0,21147008.0,21179584.0,21212160.0,21244736.0,21058944.0,21091584.0,21124224.0,21156864.0,21189504.0,21222144.0,21254784.0,21287424.0,21101184.0,21133888.0,21166592.0,21199296.0,21232000.0,21264704.0,21297408.0,21330112.0,21143424.0,21176192.0,21208960.0,21241728.0,21274496.0,21307264.0,21340032.0,21372800.0,21185664.0,21218496.0,21251328.0,21284160.0,21316992.0,21349824.0,21382656.0,21415488.0,21227904.0,21260800.0,21293696.0,21326592.0,21359488.0,21392384.0,21425280.0,21458176.0,21270144.0,21303104.0,21336064.0,21369024.0,21401984.0,21434944.0,21467904.0,21500864.0,21312384.0,21345408.0,21378432.0,21411456.0,21444480.0,21477504.0,21510528.0,21543552.0,21354624.0,21387712.0,21420800.0,21453888.0,21486976.0,21520064.0,21553152.0,21586240.0,21396864.0,21430016.0,21463168.0,21496320.0,21529472.0,21562624.0,21595776.0,21628928.0,21439104.0,21472320.0,21505536.0,21538752.0,21571968.0,21605184.0,21638400.0,21671616.0,21481344.0,21514624.0,21547904.0,21581184.0,21614464.0,21647744.0,21681024.0,21714304.0,21113984.0,21146304.0,21178624.0,21210944.0,21243264.0,21275584.0,21307904.0,21340224.0,20738432.0,20769792.0,20801152.0,20832512.0,20863872.0,20895232.0,20926592.0,20957952.0,20780672.0,20812096.0,20843520.0,20874944.0,20906368.0,20937792.0,20969216.0,21000640.0,20822912.0,20854400.0,20885888.0,20917376.0,20948864.0,20980352.0,21011840.0,21043328.0,20865152.0,20896704.0,20928256.0,20959808.0,20991360.0,21022912.0,21054464.0,21086016.0,20907392.0,20939008.0,20970624.0,21002240.0,21033856.0,21065472.0,21097088.0,21128704.0,20949632.0,20981312.0,21012992.0,21044672.0,21076352.0,21108032.0,21139712.0,21171392.0,20991872.0,21023616.0,21055360.0,21087104.0,21118848.0,21150592.0,21182336.0,21214080.0,21034112.0,21065920.0,21097728.0,21129536.0,21161344.0,21193152.0,21224960.0,21256768.0,21076352.0,21108224.0,21140096.0,21171968.0,21203840.0,21235712.0,21267584.0,21299456.0,21118592.0,21150528.0,21182464.0,21214400.0,21246336.0,21278272.0,21310208.0,21342144.0,21160832.0,21192832.0,21224832.0,21256832.0,21288832.0,21320832.0,21352832.0,21384832.0,21203072.0,21235136.0,21267200.0,21299264.0,21331328.0,21363392.0,21395456.0,21427520.0,21245312.0,21277440.0,21309568.0,21341696.0,21373824.0,21405952.0,21438080.0,21470208.0,21287552.0,21319744.0,21351936.0,21384128.0,21416320.0,21448512.0,21480704.0,21512896.0,21329792.0,21362048.0,21394304.0,21426560.0,21458816.0,21491072.0,21523328.0,21555584.0,21372032.0,21404352.0,21436672.0,21468992.0,21501312.0,21533632.0,21565952.0,21598272.0,21414272.0,21446656.0,21479040.0,21511424.0,21543808.0,21576192.0,21608576.0,21640960.0,21456512.0,21488960.0,21521408.0,21553856.0,21586304.0,21618752.0,21651200.0,21683648.0,21498752.0,21531264.0,21563776.0,21596288.0,21628800.0,21661312.0,21693824.0,21726336.0,21540992.0,21573568.0,21606144.0,21638720.0,21671296.0,21703872.0,21736448.0,21769024.0,21583232.0,21615872.0,21648512.0,21681152.0,21713792.0,21746432.0,21779072.0,21811712.0,21625472.0,21658176.0,21690880.0,21723584.0,21756288.0,21788992.0,21821696.0,21854400.0,21667712.0,21700480.0,21733248.0,21766016.0,21798784.0,21831552.0,21864320.0,21897088.0,21709952.0,21742784.0,21775616.0,21808448.0,21841280.0,21874112.0,21906944.0,21939776.0,21752192.0,21785088.0,21817984.0,21850880.0,21883776.0,21916672.0,21949568.0,21982464.0,21794432.0,21827392.0,21860352.0,21893312.0,21926272.0,21959232.0,21992192.0,22025152.0,21836672.0,21869696.0,21902720.0,21935744.0,21968768.0,22001792.0,22034816.0,22067840.0,21878912.0,21912000.0,21945088.0,21978176.0,22011264.0,22044352.0,22077440.0,22110528.0,21921152.0,21954304.0,21987456.0,22020608.0,22053760.0,22086912.0,22120064.0,22153216.0,21963392.0,21996608.0,22029824.0,22063040.0,22096256.0,22129472.0,22162688.0,22195904.0,22005632.0,22038912.0,22072192.0,22105472.0,22138752.0,22172032.0,22205312.0,22238592.0,22047872.0,22081216.0,22114560.0,22147904.0,22181248.0,22214592.0,22247936.0,22281280.0,22090112.0,22123520.0,22156928.0,22190336.0,22223744.0,22257152.0,22290560.0,22323968.0,22132352.0,22165824.0,22199296.0,22232768.0,22266240.0,22299712.0,22333184.0,22366656.0,22174592.0,22208128.0,22241664.0,22275200.0,22308736.0,22342272.0,22375808.0,22409344.0,22216832.0,22250432.0,22284032.0,22317632.0,22351232.0,22384832.0,22418432.0,22452032.0,22259072.0,22292736.0,22326400.0,22360064.0,22393728.0,22427392.0,22461056.0,22494720.0,22301312.0,22335040.0,22368768.0,22402496.0,22436224.0,22469952.0,22503680.0,22537408.0,21409664.0,21442432.0,21475200.0,21507968.0,21540736.0,21573504.0,21606272.0,21639040.0,20509824.0,20541632.0,20573440.0,20605248.0,20637056.0,20668864.0,20700672.0,20732480.0,10329216.0,10345136.0,10361056.0,10376976.0,10392896.0,10408816.0,10424736.0,10440656.0,10894848.0,10910640.0,10926432.0,10942224.0,10958016.0,10973808.0,10989600.0,11005392.0,21937536.0,21969152.0,22000768.0,22032384.0,22064000.0,22095616.0,22127232.0,22158848.0,21980800.0,22012480.0,22044160.0,22075840.0,22107520.0,22139200.0,22170880.0,22202560.0,22024064.0,22055808.0,22087552.0,22119296.0,22151040.0,22182784.0,22214528.0,22246272.0,22067328.0,22099136.0,22130944.0,22162752.0,22194560.0,22226368.0,22258176.0,22289984.0,22110592.0,22142464.0,22174336.0,22206208.0,22238080.0,22269952.0,22301824.0,22333696.0,22153856.0,22185792.0,22217728.0,22249664.0,22281600.0,22313536.0,22345472.0,22377408.0,22197120.0,22229120.0,22261120.0,22293120.0,22325120.0,22357120.0,22389120.0,22421120.0,22240384.0,22272448.0,22304512.0,22336576.0,22368640.0,22400704.0,22432768.0,22464832.0,22283648.0,22315776.0,22347904.0,22380032.0,22412160.0,22444288.0,22476416.0,22508544.0,22326912.0,22359104.0,22391296.0,22423488.0,22455680.0,22487872.0,22520064.0,22552256.0,22370176.0,22402432.0,22434688.0,22466944.0,22499200.0,22531456.0,22563712.0,22595968.0,22413440.0,22445760.0,22478080.0,22510400.0,22542720.0,22575040.0,22607360.0,22639680.0,22456704.0,22489088.0,22521472.0,22553856.0,22586240.0,22618624.0,22651008.0,22683392.0,22499968.0,22532416.0,22564864.0,22597312.0,22629760.0,22662208.0,22694656.0,22727104.0,22543232.0,22575744.0,22608256.0,22640768.0,22673280.0,22705792.0,22738304.0,22770816.0,22586496.0,22619072.0,22651648.0,22684224.0,22716800.0,22749376.0,22781952.0,22814528.0,22629760.0,22662400.0,22695040.0,22727680.0,22760320.0,22792960.0,22825600.0,22858240.0,22673024.0,22705728.0,22738432.0,22771136.0,22803840.0,22836544.0,22869248.0,22901952.0,22716288.0,22749056.0,22781824.0,22814592.0,22847360.0,22880128.0,22912896.0,22945664.0,22759552.0,22792384.0,22825216.0,22858048.0,22890880.0,22923712.0,22956544.0,22989376.0,22802816.0,22835712.0,22868608.0,22901504.0,22934400.0,22967296.0,23000192.0,23033088.0,22846080.0,22879040.0,22912000.0,22944960.0,22977920.0,23010880.0,23043840.0,23076800.0,22889344.0,22922368.0,22955392.0,22988416.0,23021440.0,23054464.0,23087488.0,23120512.0,22932608.0,22965696.0,22998784.0,23031872.0,23064960.0,23098048.0,23131136.0,23164224.0,22975872.0,23009024.0,23042176.0,23075328.0,23108480.0,23141632.0,23174784.0,23207936.0,23019136.0,23052352.0,23085568.0,23118784.0,23152000.0,23185216.0,23218432.0,23251648.0,23062400.0,23095680.0,23128960.0,23162240.0,23195520.0,23228800.0,23262080.0,23295360.0,23105664.0,23139008.0,23172352.0,23205696.0,23239040.0,23272384.0,23305728.0,23339072.0,23148928.0,23182336.0,23215744.0,23249152.0,23282560.0,23315968.0,23349376.0,23382784.0,23192192.0,23225664.0,23259136.0,23292608.0,23326080.0,23359552.0,23393024.0,23426496.0,23235456.0,23268992.0,23302528.0,23336064.0,23369600.0,23403136.0,23436672.0,23470208.0,23278720.0,23312320.0,23345920.0,23379520.0,23413120.0,23446720.0,23480320.0,23513920.0,23321984.0,23355648.0,23389312.0,23422976.0,23456640.0,23490304.0,23523968.0,23557632.0,23365248.0,23398976.0,23432704.0,23466432.0,23500160.0,23533888.0,23567616.0,23601344.0,23408512.0,23442304.0,23476096.0,23509888.0,23543680.0,23577472.0,23611264.0,23645056.0,23451776.0,23485632.0,23519488.0,23553344.0,23587200.0,23621056.0,23654912.0,23688768.0,22544768.0,22577664.0,22610560.0,22643456.0,22676352.0,22709248.0,22742144.0,22775040.0,21629568.0,21661504.0,21693440.0,21725376.0,21757312.0,21789248.0,21821184.0,21853120.0,20722560.0,20753536.0,20784512.0,20815488.0,20846464.0,20877440.0,20908416.0,20939392.0,19807360.0,19837376.0,19867392.0,19897408.0,19927424.0,19957440.0,19987456.0,20017472.0,19850624.0,19880704.0,19910784.0,19940864.0,19970944.0,20001024.0,20031104.0,20061184.0,19893888.0,19924032.0,19954176.0,19984320.0,20014464.0,20044608.0,20074752.0,20104896.0,19937152.0,19967360.0,19997568.0,20027776.0,20057984.0,20088192.0,20118400.0,20148608.0,19980416.0,20010688.0,20040960.0,20071232.0,20101504.0,20131776.0,20162048.0,20192320.0,20023680.0,20054016.0,20084352.0,20114688.0,20145024.0,20175360.0,20205696.0,20236032.0,20066944.0,20097344.0,20127744.0,20158144.0,20188544.0,20218944.0,20249344.0,20279744.0,20110208.0,20140672.0,20171136.0,20201600.0,20232064.0,20262528.0,20292992.0,20323456.0,20153472.0,20184000.0,20214528.0,20245056.0,20275584.0,20306112.0,20336640.0,20367168.0,20196736.0,20227328.0,20257920.0,20288512.0,20319104.0,20349696.0,20380288.0,20410880.0,20240000.0,20270656.0,20301312.0,20331968.0,20362624.0,20393280.0,20423936.0,20454592.0,20283264.0,20313984.0,20344704.0,20375424.0,20406144.0,20436864.0,20467584.0,20498304.0,20326528.0,20357312.0,20388096.0,20418880.0,20449664.0,20480448.0,20511232.0,20542016.0,20369792.0,20400640.0,20431488.0,20462336.0,20493184.0,20524032.0,20554880.0,20585728.0,20413056.0,20443968.0,20474880.0,20505792.0,20536704.0,20567616.0,20598528.0,20629440.0,20456320.0,20487296.0,20518272.0,20549248.0,20580224.0,20611200.0,20642176.0,20673152.0,20499584.0,20530624.0,20561664.0,20592704.0,20623744.0,20654784.0,20685824.0,20716864.0,20542848.0,20573952.0,20605056.0,20636160.0,20667264.0,20698368.0,20729472.0,20760576.0,20586112.0,20617280.0,20648448.0,20679616.0,20710784.0,20741952.0,20773120.0,20804288.0,20629376.0,20660608.0,20691840.0,20723072.0,20754304.0,20785536.0,20816768.0,20848000.0,20672640.0,20703936.0,20735232.0,20766528.0,20797824.0,20829120.0,20860416.0,20891712.0,20715904.0,20747264.0,20778624.0,20809984.0,20841344.0,20872704.0,20904064.0,20935424.0,20759168.0,20790592.0,20822016.0,20853440.0,20884864.0,20916288.0,20947712.0,20979136.0,20802432.0,20833920.0,20865408.0,20896896.0,20928384.0,20959872.0,20991360.0,21022848.0,20845696.0,20877248.0,20908800.0,20940352.0,20971904.0,21003456.0,21035008.0,21066560.0,20888960.0,20920576.0,20952192.0,20983808.0,21015424.0,21047040.0,21078656.0,21110272.0,20932224.0,20963904.0,20995584.0,21027264.0,21058944.0,21090624.0,21122304.0,21153984.0,20975488.0,21007232.0,21038976.0,21070720.0,21102464.0,21134208.0,21165952.0,21197696.0,21018752.0,21050560.0,21082368.0,21114176.0,21145984.0,21177792.0,21209600.0,21241408.0,21062016.0,21093888.0,21125760.0,21157632.0,21189504.0,21221376.0,21253248.0,21285120.0,21105280.0,21137216.0,21169152.0,21201088.0,21233024.0,21264960.0,21296896.0,21328832.0,21148544.0,21180544.0,21212544.0,21244544.0,21276544.0,21308544.0,21340544.0,21372544.0,21191808.0,21223872.0,21255936.0,21288000.0,21320064.0,21352128.0,21384192.0,21416256.0,21235072.0,21267200.0,21299328.0,21331456.0,21363584.0,21395712.0,21427840.0,21459968.0,21278336.0,21310528.0,21342720.0,21374912.0,21407104.0,21439296.0,21471488.0,21503680.0,21321600.0,21353856.0,21386112.0,21418368.0,21450624.0,21482880.0,21515136.0,21547392.0,21364864.0,21397184.0,21429504.0,21461824.0,21494144.0,21526464.0,21558784.0,21591104.0,21408128.0,21440512.0,21472896.0,21505280.0,21537664.0,21570048.0,21602432.0,21634816.0,21025408.0,21056832.0,21088256.0,21119680.0,21151104.0,21182528.0,21213952.0,21245376.0,20634496.0,20664960.0,20695424.0,20725888.0,20756352.0,20786816.0,20817280.0,20847744.0,20677760.0,20708288.0,20738816.0,20769344.0,20799872.0,20830400.0,20860928.0,20891456.0,20721024.0,20751616.0,20782208.0,20812800.0,20843392.0,20873984.0,20904576.0,20935168.0,20764288.0,20794944.0,20825600.0,20856256.0,20886912.0,20917568.0,20948224.0,20978880.0,20807552.0,20838272.0,20868992.0,20899712.0,20930432.0,20961152.0,20991872.0,21022592.0,20850816.0,20881600.0,20912384.0,20943168.0,20973952.0,21004736.0,21035520.0,21066304.0,20894080.0,20924928.0,20955776.0,20986624.0,21017472.0,21048320.0,21079168.0,21110016.0,20937344.0,20968256.0,20999168.0,21030080.0,21060992.0,21091904.0,21122816.0,21153728.0,20980608.0,21011584.0,21042560.0,21073536.0,21104512.0,21135488.0,21166464.0,21197440.0,21023872.0,21054912.0,21085952.0,21116992.0,21148032.0,21179072.0,21210112.0,21241152.0,21067136.0,21098240.0,21129344.0,21160448.0,21191552.0,21222656.0,21253760.0,21284864.0,21110400.0,21141568.0,21172736.0,21203904.0,21235072.0,21266240.0,21297408.0,21328576.0,21153664.0,21184896.0,21216128.0,21247360.0,21278592.0,21309824.0,21341056.0,21372288.0,21196928.0,21228224.0,21259520.0,21290816.0,21322112.0,21353408.0,21384704.0,21416000.0,21240192.0,21271552.0,21302912.0,21334272.0,21365632.0,21396992.0,21428352.0,21459712.0,21283456.0,21314880.0,21346304.0,21377728.0,21409152.0,21440576.0,21472000.0,21503424.0,21326720.0,21358208.0,21389696.0,21421184.0,21452672.0,21484160.0,21515648.0,21547136.0,21369984.0,21401536.0,21433088.0,21464640.0,21496192.0,21527744.0,21559296.0,21590848.0,21413248.0,21444864.0,21476480.0,21508096.0,21539712.0,21571328.0,21602944.0,21634560.0,21456512.0,21488192.0,21519872.0,21551552.0,21583232.0,21614912.0,21646592.0,21678272.0,21499776.0,21531520.0,21563264.0,21595008.0,21626752.0,21658496.0,21690240.0,21721984.0,21543040.0,21574848.0,21606656.0,21638464.0,21670272.0,21702080.0,21733888.0,21765696.0,21586304.0,21618176.0,21650048.0,21681920.0,21713792.0,21745664.0,21777536.0,21809408.0,21629568.0,21661504.0,21693440.0,21725376.0,21757312.0,21789248.0,21821184.0,21853120.0,21672832.0,21704832.0,21736832.0,21768832.0,21800832.0,21832832.0,21864832.0,21896832.0,21716096.0,21748160.0,21780224.0,21812288.0,21844352.0,21876416.0,21908480.0,21940544.0,21759360.0,21791488.0,21823616.0,21855744.0,21887872.0,21920000.0,21952128.0,21984256.0,21802624.0,21834816.0,21867008.0,21899200.0,21931392.0,21963584.0,21995776.0,22027968.0,21845888.0,21878144.0,21910400.0,21942656.0,21974912.0,22007168.0,22039424.0,22071680.0,21889152.0,21921472.0,21953792.0,21986112.0,22018432.0,22050752.0,22083072.0,22115392.0,21932416.0,21964800.0,21997184.0,22029568.0,22061952.0,22094336.0,22126720.0,22159104.0,21975680.0,22008128.0,22040576.0,22073024.0,22105472.0,22137920.0,22170368.0,22202816.0,22018944.0,22051456.0,22083968.0,22116480.0,22148992.0,22181504.0,22214016.0,22246528.0,22062208.0,22094784.0,22127360.0,22159936.0,22192512.0,22225088.0,22257664.0,22290240.0,22105472.0,22138112.0,22170752.0,22203392.0,22236032.0,22268672.0,22301312.0,22333952.0,22148736.0,22181440.0,22214144.0,22246848.0,22279552.0,22312256.0,22344960.0,22377664.0,22192000.0,22224768.0,22257536.0,22290304.0,22323072.0,22355840.0,22388608.0,22421376.0,22235264.0,22268096.0,22300928.0,22333760.0,22366592.0,22399424.0,22432256.0,22465088.0,21328256.0,21360128.0,21392000.0,21423872.0,21455744.0,21487616.0,21519488.0,21551360.0,20413056.0,20443968.0,20474880.0,20505792.0,20536704.0,20567616.0,20598528.0,20629440.0,20456320.0,20487296.0,20518272.0,20549248.0,20580224.0,20611200.0,20642176.0,20673152.0,20499584.0,20530624.0,20561664.0,20592704.0,20623744.0,20654784.0,20685824.0,20716864.0,20542848.0,20573952.0,20605056.0,20636160.0,20667264.0,20698368.0,20729472.0,20760576.0,20586112.0,20617280.0,20648448.0,20679616.0,20710784.0,20741952.0,20773120.0,20804288.0,20629376.0,20660608.0,20691840.0,20723072.0,20754304.0,20785536.0,20816768.0,20848000.0,20672640.0,20703936.0,20735232.0,20766528.0,20797824.0,20829120.0,20860416.0,20891712.0,20715904.0,20747264.0,20778624.0,20809984.0,20841344.0,20872704.0,20904064.0,20935424.0,20759168.0,20790592.0,20822016.0,20853440.0,20884864.0,20916288.0,20947712.0,20979136.0,20802432.0,20833920.0,20865408.0,20896896.0,20928384.0,20959872.0,20991360.0,21022848.0,20845696.0,20877248.0,20908800.0,20940352.0,20971904.0,21003456.0,21035008.0,21066560.0,20888960.0,20920576.0,20952192.0,20983808.0,21015424.0,21047040.0,21078656.0,21110272.0,20932224.0,20963904.0,20995584.0,21027264.0,21058944.0,21090624.0,21122304.0,21153984.0,20975488.0,21007232.0,21038976.0,21070720.0,21102464.0,21134208.0,21165952.0,21197696.0,21018752.0,21050560.0,21082368.0,21114176.0,21145984.0,21177792.0,21209600.0,21241408.0,21062016.0,21093888.0,21125760.0,21157632.0,21189504.0,21221376.0,21253248.0,21285120.0,21105280.0,21137216.0,21169152.0,21201088.0,21233024.0,21264960.0,21296896.0,21328832.0,21148544.0,21180544.0,21212544.0,21244544.0,21276544.0,21308544.0,21340544.0,21372544.0,21191808.0,21223872.0,21255936.0,21288000.0,21320064.0,21352128.0,21384192.0,21416256.0,21235072.0,21267200.0,21299328.0,21331456.0,21363584.0,21395712.0,21427840.0,21459968.0,21278336.0,21310528.0,21342720.0,21374912.0,21407104.0,21439296.0,21471488.0,21503680.0,21321600.0,21353856.0,21386112.0,21418368.0,21450624.0,21482880.0,21515136.0,21547392.0,21364864.0,21397184.0,21429504.0,21461824.0,21494144.0,21526464.0,21558784.0,21591104.0,21408128.0,21440512.0,21472896.0,21505280.0,21537664.0,21570048.0,21602432.0,21634816.0,21451392.0,21483840.0,21516288.0,21548736.0,21581184.0,21613632.0,21646080.0,21678528.0,21494656.0,21527168.0,21559680.0,21592192.0,21624704.0,21657216.0,21689728.0,21722240.0,21537920.0,21570496.0,21603072.0,21635648.0,21668224.0,21700800.0,21733376.0,21765952.0,21581184.0,21613824.0,21646464.0,21679104.0,21711744.0,21744384.0,21777024.0,21809664.0,21624448.0,21657152.0,21689856.0,21722560.0,21755264.0,21787968.0,21820672.0,21853376.0,21667712.0,21700480.0,21733248.0,21766016.0,21798784.0,21831552.0,21864320.0,21897088.0,21710976.0,21743808.0,21776640.0,21809472.0,21842304.0,21875136.0,21907968.0,21940800.0,21754240.0,21787136.0,21820032.0,21852928.0,21885824.0,21918720.0,21951616.0,21984512.0,21797504.0,21830464.0,21863424.0,21896384.0,21929344.0,21962304.0,21995264.0,22028224.0,21840768.0,21873792.0,21906816.0,21939840.0,21972864.0,22005888.0,22038912.0,22071936.0,21884032.0,21917120.0,21950208.0,21983296.0,22016384.0,22049472.0,22082560.0,22115648.0,21927296.0,21960448.0,21993600.0,22026752.0,22059904.0,22093056.0,22126208.0,22159360.0,21970560.0,22003776.0,22036992.0,22070208.0,22103424.0,22136640.0,22169856.0,22203072.0,22013824.0,22047104.0,22080384.0,22113664.0,22146944.0,22180224.0,22213504.0,22246784.0,21631104.0,21663424.0,21695744.0,21728064.0,21760384.0,21792704.0,21825024.0,21857344.0,21240192.0,21271552.0,21302912.0,21334272.0,21365632.0,21396992.0,21428352.0,21459712.0,21283456.0,21314880.0,21346304.0,21377728.0,21409152.0,21440576.0,21472000.0,21503424.0,21326720.0,21358208.0,21389696.0,21421184.0,21452672.0,21484160.0,21515648.0,21547136.0,21369984.0,21401536.0,21433088.0,21464640.0,21496192.0,21527744.0,21559296.0,21590848.0,21413248.0,21444864.0,21476480.0,21508096.0,21539712.0,21571328.0,21602944.0,21634560.0,21456512.0,21488192.0,21519872.0,21551552.0,21583232.0,21614912.0,21646592.0,21678272.0,21499776.0,21531520.0,21563264.0,21595008.0,21626752.0,21658496.0,21690240.0,21721984.0,21543040.0,21574848.0,21606656.0,21638464.0,21670272.0,21702080.0,21733888.0,21765696.0,21586304.0,21618176.0,21650048.0,21681920.0,21713792.0,21745664.0,21777536.0,21809408.0,21629568.0,21661504.0,21693440.0,21725376.0,21757312.0,21789248.0,21821184.0,21853120.0,21672832.0,21704832.0,21736832.0,21768832.0,21800832.0,21832832.0,21864832.0,21896832.0,21716096.0,21748160.0,21780224.0,21812288.0,21844352.0,21876416.0,21908480.0,21940544.0,21759360.0,21791488.0,21823616.0,21855744.0,21887872.0,21920000.0,21952128.0,21984256.0,21802624.0,21834816.0,21867008.0,21899200.0,21931392.0,21963584.0,21995776.0,22027968.0,21845888.0,21878144.0,21910400.0,21942656.0,21974912.0,22007168.0,22039424.0,22071680.0,21889152.0,21921472.0,21953792.0,21986112.0,22018432.0,22050752.0,22083072.0,22115392.0,21932416.0,21964800.0,21997184.0,22029568.0,22061952.0,22094336.0,22126720.0,22159104.0,21975680.0,22008128.0,22040576.0,22073024.0,22105472.0,22137920.0,22170368.0,22202816.0,22018944.0,22051456.0,22083968.0,22116480.0,22148992.0,22181504.0,22214016.0,22246528.0,22062208.0,22094784.0,22127360.0,22159936.0,22192512.0,22225088.0,22257664.0,22290240.0,22105472.0,22138112.0,22170752.0,22203392.0,22236032.0,22268672.0,22301312.0,22333952.0,22148736.0,22181440.0,22214144.0,22246848.0,22279552.0,22312256.0,22344960.0,22377664.0,22192000.0,22224768.0,22257536.0,22290304.0,22323072.0,22355840.0,22388608.0,22421376.0,22235264.0,22268096.0,22300928.0,22333760.0,22366592.0,22399424.0,22432256.0,22465088.0,22278528.0,22311424.0,22344320.0,22377216.0,22410112.0,22443008.0,22475904.0,22508800.0,22321792.0,22354752.0,22387712.0,22420672.0,22453632.0,22486592.0,22519552.0,22552512.0,22365056.0,22398080.0,22431104.0,22464128.0,22497152.0,22530176.0,22563200.0,22596224.0,22408320.0,22441408.0,22474496.0,22507584.0,22540672.0,22573760.0,22606848.0,22639936.0,22451584.0,22484736.0,22517888.0,22551040.0,22584192.0,22617344.0,22650496.0,22683648.0,22494848.0,22528064.0,22561280.0,22594496.0,22627712.0,22660928.0,22694144.0,22727360.0,22538112.0,22571392.0,22604672.0,22637952.0,22671232.0,22704512.0,22737792.0,22771072.0,22581376.0,22614720.0,22648064.0,22681408.0,22714752.0,22748096.0,22781440.0,22814784.0,22624640.0,22658048.0,22691456.0,22724864.0,22758272.0,22791680.0,22825088.0,22858496.0,22667904.0,22701376.0,22734848.0,22768320.0,22801792.0,22835264.0,22868736.0,22902208.0,22711168.0,22744704.0,22778240.0,22811776.0,22845312.0,22878848.0,22912384.0,22945920.0,22754432.0,22788032.0,22821632.0,22855232.0,22888832.0,22922432.0,22956032.0,22989632.0,22797696.0,22831360.0,22865024.0,22898688.0,22932352.0,22966016.0,22999680.0,23033344.0,22840960.0,22874688.0,22908416.0,22942144.0,22975872.0,23009600.0,23043328.0,23077056.0,21933952.0,21966720.0,21999488.0,22032256.0,22065024.0,22097792.0,22130560.0,22163328.0,21018752.0,21050560.0,21082368.0,21114176.0,21145984.0,21177792.0,21209600.0,21241408.0,10583936.0,10599856.0,10615776.0,10631696.0,10647616.0,10663536.0,10679456.0,10695376.0,11147520.0,11163312.0,11179104.0,11194896.0,11210688.0,11226480.0,11242272.0,11258064.0,22443392.0,22475008.0,22506624.0,22538240.0,22569856.0,22601472.0,22633088.0,22664704.0,22487680.0,22519360.0,22551040.0,22582720.0,22614400.0,22646080.0,22677760.0,22709440.0,22531968.0,22563712.0,22595456.0,22627200.0,22658944.0,22690688.0,22722432.0,22754176.0,22576256.0,22608064.0,22639872.0,22671680.0,22703488.0,22735296.0,22767104.0,22798912.0,22620544.0,22652416.0,22684288.0,22716160.0,22748032.0,22779904.0,22811776.0,22843648.0,22664832.0,22696768.0,22728704.0,22760640.0,22792576.0,22824512.0,22856448.0,22888384.0,22709120.0,22741120.0,22773120.0,22805120.0,22837120.0,22869120.0,22901120.0,22933120.0,22753408.0,22785472.0,22817536.0,22849600.0,22881664.0,22913728.0,22945792.0,22977856.0,22797696.0,22829824.0,22861952.0,22894080.0,22926208.0,22958336.0,22990464.0,23022592.0,22841984.0,22874176.0,22906368.0,22938560.0,22970752.0,23002944.0,23035136.0,23067328.0,22886272.0,22918528.0,22950784.0,22983040.0,23015296.0,23047552.0,23079808.0,23112064.0,22930560.0,22962880.0,22995200.0,23027520.0,23059840.0,23092160.0,23124480.0,23156800.0,22974848.0,23007232.0,23039616.0,23072000.0,23104384.0,23136768.0,23169152.0,23201536.0,23019136.0,23051584.0,23084032.0,23116480.0,23148928.0,23181376.0,23213824.0,23246272.0,23063424.0,23095936.0,23128448.0,23160960.0,23193472.0,23225984.0,23258496.0,23291008.0,23107712.0,23140288.0,23172864.0,23205440.0,23238016.0,23270592.0,23303168.0,23335744.0,23152000.0,23184640.0,23217280.0,23249920.0,23282560.0,23315200.0,23347840.0,23380480.0,23196288.0,23228992.0,23261696.0,23294400.0,23327104.0,23359808.0,23392512.0,23425216.0,23240576.0,23273344.0,23306112.0,23338880.0,23371648.0,23404416.0,23437184.0,23469952.0,23284864.0,23317696.0,23350528.0,23383360.0,23416192.0,23449024.0,23481856.0,23514688.0,23329152.0,23362048.0,23394944.0,23427840.0,23460736.0,23493632.0,23526528.0,23559424.0,23373440.0,23406400.0,23439360.0,23472320.0,23505280.0,23538240.0,23571200.0,23604160.0,23417728.0,23450752.0,23483776.0,23516800.0,23549824.0,23582848.0,23615872.0,23648896.0,23462016.0,23495104.0,23528192.0,23561280.0,23594368.0,23627456.0,23660544.0,23693632.0,23506304.0,23539456.0,23572608.0,23605760.0,23638912.0,23672064.0,23705216.0,23738368.0,23550592.0,23583808.0,23617024.0,23650240.0,23683456.0,23716672.0,23749888.0,23783104.0,23594880.0,23628160.0,23661440.0,23694720.0,23728000.0,23761280.0,23794560.0,23827840.0,23639168.0,23672512.0,23705856.0,23739200.0,23772544.0,23805888.0,23839232.0,23872576.0,23683456.0,23716864.0,23750272.0,23783680.0,23817088.0,23850496.0,23883904.0,23917312.0,23727744.0,23761216.0,23794688.0,23828160.0,23861632.0,23895104.0,23928576.0,23962048.0,23772032.0,23805568.0,23839104.0,23872640.0,23906176.0,23939712.0,23973248.0,24006784.0,23816320.0,23849920.0,23883520.0,23917120.0,23950720.0,23984320.0,24017920.0,24051520.0,23860608.0,23894272.0,23927936.0,23961600.0,23995264.0,24028928.0,24062592.0,24096256.0,23904896.0,23938624.0,23972352.0,24006080.0,24039808.0,24073536.0,24107264.0,24140992.0,23949184.0,23982976.0,24016768.0,24050560.0,24084352.0,24118144.0,24151936.0,24185728.0,23993472.0,24027328.0,24061184.0,24095040.0,24128896.0,24162752.0,24196608.0,24230464.0,23071104.0,23104000.0,23136896.0,23169792.0,23202688.0,23235584.0,23268480.0,23301376.0,22140544.0,22172480.0,22204416.0,22236352.0,22268288.0,22300224.0,22332160.0,22364096.0,21218176.0,21249152.0,21280128.0,21311104.0,21342080.0,21373056.0,21404032.0,21435008.0,20287616.0,20317632.0,20347648.0,20377664.0,20407680.0,20437696.0,20467712.0,20497728.0,20331904.0,20361984.0,20392064.0,20422144.0,20452224.0,20482304.0,20512384.0,20542464.0,20376192.0,20406336.0,20436480.0,20466624.0,20496768.0,20526912.0,20557056.0,20587200.0,20420480.0,20450688.0,20480896.0,20511104.0,20541312.0,20571520.0,20601728.0,20631936.0,20464768.0,20495040.0,20525312.0,20555584.0,20585856.0,20616128.0,20646400.0,20676672.0,20509056.0,20539392.0,20569728.0,20600064.0,20630400.0,20660736.0,20691072.0,20721408.0,20553344.0,20583744.0,20614144.0,20644544.0,20674944.0,20705344.0,20735744.0,20766144.0,20597632.0,20628096.0,20658560.0,20689024.0,20719488.0,20749952.0,20780416.0,20810880.0,20641920.0,20672448.0,20702976.0,20733504.0,20764032.0,20794560.0,20825088.0,20855616.0,20686208.0,20716800.0,20747392.0,20777984.0,20808576.0,20839168.0,20869760.0,20900352.0,20730496.0,20761152.0,20791808.0,20822464.0,20853120.0,20883776.0,20914432.0,20945088.0,20774784.0,20805504.0,20836224.0,20866944.0,20897664.0,20928384.0,20959104.0,20989824.0,20819072.0,20849856.0,20880640.0,20911424.0,20942208.0,20972992.0,21003776.0,21034560.0,20863360.0,20894208.0,20925056.0,20955904.0,20986752.0,21017600.0,21048448.0,21079296.0,20907648.0,20938560.0,20969472.0,21000384.0,21031296.0,21062208.0,21093120.0,21124032.0,20951936.0,20982912.0,21013888.0,21044864.0,21075840.0,21106816.0,21137792.0,21168768.0,20996224.0,21027264.0,21058304.0,21089344.0,21120384.0,21151424.0,21182464.0,21213504.0,21040512.0,21071616.0,21102720.0,21133824.0,21164928.0,21196032.0,21227136.0,21258240.0,21084800.0,21115968.0,21147136.0,21178304.0,21209472.0,21240640.0,21271808.0,21302976.0,21129088.0,21160320.0,21191552.0,21222784.0,21254016.0,21285248.0,21316480.0,21347712.0,21173376.0,21204672.0,21235968.0,21267264.0,21298560.0,21329856.0,21361152.0,21392448.0,21217664.0,21249024.0,21280384.0,21311744.0,21343104.0,21374464.0,21405824.0,21437184.0,21261952.0,21293376.0,21324800.0,21356224.0,21387648.0,21419072.0,21450496.0,21481920.0,21306240.0,21337728.0,21369216.0,21400704.0,21432192.0,21463680.0,21495168.0,21526656.0,21350528.0,21382080.0,21413632.0,21445184.0,21476736.0,21508288.0,21539840.0,21571392.0,21394816.0,21426432.0,21458048.0,21489664.0,21521280.0,21552896.0,21584512.0,21616128.0,21439104.0,21470784.0,21502464.0,21534144.0,21565824.0,21597504.0,21629184.0,21660864.0,21483392.0,21515136.0,21546880.0,21578624.0,21610368.0,21642112.0,21673856.0,21705600.0,21527680.0,21559488.0,21591296.0,21623104.0,21654912.0,21686720.0,21718528.0,21750336.0,21571968.0,21603840.0,21635712.0,21667584.0,21699456.0,21731328.0,21763200.0,21795072.0,21616256.0,21648192.0,21680128.0,21712064.0,21744000.0,21775936.0,21807872.0,21839808.0,21660544.0,21692544.0,21724544.0,21756544.0,21788544.0,21820544.0,21852544.0,21884544.0,21704832.0,21736896.0,21768960.0,21801024.0,21833088.0,21865152.0,21897216.0,21929280.0,21749120.0,21781248.0,21813376.0,21845504.0,21877632.0,21909760.0,21941888.0,21974016.0,21793408.0,21825600.0,21857792.0,21889984.0,21922176.0,21954368.0,21986560.0,22018752.0,21837696.0,21869952.0,21902208.0,21934464.0,21966720.0,21998976.0,22031232.0,22063488.0,21881984.0,21914304.0,21946624.0,21978944.0,22011264.0,22043584.0,22075904.0,22108224.0,21926272.0,21958656.0,21991040.0,22023424.0,22055808.0,22088192.0,22120576.0,22152960.0,21528192.0,21559616.0,21591040.0,21622464.0,21653888.0,21685312.0,21716736.0,21748160.0,21121920.0,21152384.0,21182848.0,21213312.0,21243776.0,21274240.0,21304704.0,21335168.0,21166208.0,21196736.0,21227264.0,21257792.0,21288320.0,21318848.0,21349376.0,21379904.0,21210496.0,21241088.0,21271680.0,21302272.0,21332864.0,21363456.0,21394048.0,21424640.0,21254784.0,21285440.0,21316096.0,21346752.0,21377408.0,21408064.0,21438720.0,21469376.0,21299072.0,21329792.0,21360512.0,21391232.0,21421952.0,21452672.0,21483392.0,21514112.0,21343360.0,21374144.0,21404928.0,21435712.0,21466496.0,21497280.0,21528064.0,21558848.0,21387648.0,21418496.0,21449344.0,21480192.0,21511040.0,21541888.0,21572736.0,21603584.0,21431936.0,21462848.0,21493760.0,21524672.0,21555584.0,21586496.0,21617408.0,21648320.0,21476224.0,21507200.0,21538176.0,21569152.0,21600128.0,21631104.0,21662080.0,21693056.0,21520512.0,21551552.0,21582592.0,21613632.0,21644672.0,21675712.0,21706752.0,21737792.0,21564800.0,21595904.0,21627008.0,21658112.0,21689216.0,21720320.0,21751424.0,21782528.0,21609088.0,21640256.0,21671424.0,21702592.0,21733760.0,21764928.0,21796096.0,21827264.0,21653376.0,21684608.0,21715840.0,21747072.0,21778304.0,21809536.0,21840768.0,21872000.0,21697664.0,21728960.0,21760256.0,21791552.0,21822848.0,21854144.0,21885440.0,21916736.0,21741952.0,21773312.0,21804672.0,21836032.0,21867392.0,21898752.0,21930112.0,21961472.0,21786240.0,21817664.0,21849088.0,21880512.0,21911936.0,21943360.0,21974784.0,22006208.0,21830528.0,21862016.0,21893504.0,21924992.0,21956480.0,21987968.0,22019456.0,22050944.0,21874816.0,21906368.0,21937920.0,21969472.0,22001024.0,22032576.0,22064128.0,22095680.0,21919104.0,21950720.0,21982336.0,22013952.0,22045568.0,22077184.0,22108800.0,22140416.0,21963392.0,21995072.0,22026752.0,22058432.0,22090112.0,22121792.0,22153472.0,22185152.0,22007680.0,22039424.0,22071168.0,22102912.0,22134656.0,22166400.0,22198144.0,22229888.0,22051968.0,22083776.0,22115584.0,22147392.0,22179200.0,22211008.0,22242816.0,22274624.0,22096256.0,22128128.0,22160000.0,22191872.0,22223744.0,22255616.0,22287488.0,22319360.0,22140544.0,22172480.0,22204416.0,22236352.0,22268288.0,22300224.0,22332160.0,22364096.0,22184832.0,22216832.0,22248832.0,22280832.0,22312832.0,22344832.0,22376832.0,22408832.0,22229120.0,22261184.0,22293248.0,22325312.0,22357376.0,22389440.0,22421504.0,22453568.0,22273408.0,22305536.0,22337664.0,22369792.0,22401920.0,22434048.0,22466176.0,22498304.0,22317696.0,22349888.0,22382080.0,22414272.0,22446464.0,22478656.0,22510848.0,22543040.0,22361984.0,22394240.0,22426496.0,22458752.0,22491008.0,22523264.0,22555520.0,22587776.0,22406272.0,22438592.0,22470912.0,22503232.0,22535552.0,22567872.0,22600192.0,22632512.0,22450560.0,22482944.0,22515328.0,22547712.0,22580096.0,22612480.0,22644864.0,22677248.0,22494848.0,22527296.0,22559744.0,22592192.0,22624640.0,22657088.0,22689536.0,22721984.0,22539136.0,22571648.0,22604160.0,22636672.0,22669184.0,22701696.0,22734208.0,22766720.0,22583424.0,22616000.0,22648576.0,22681152.0,22713728.0,22746304.0,22778880.0,22811456.0,22627712.0,22660352.0,22692992.0,22725632.0,22758272.0,22790912.0,22823552.0,22856192.0,22672000.0,22704704.0,22737408.0,22770112.0,22802816.0,22835520.0,22868224.0,22900928.0,22716288.0,22749056.0,22781824.0,22814592.0,22847360.0,22880128.0,22912896.0,22945664.0,22760576.0,22793408.0,22826240.0,22859072.0,22891904.0,22924736.0,22957568.0,22990400.0,21838208.0,21870080.0,21901952.0,21933824.0,21965696.0,21997568.0,22029440.0,22061312.0,20907648.0,20938560.0,20969472.0,21000384.0,21031296.0,21062208.0,21093120.0,21124032.0,20951936.0,20982912.0,21013888.0,21044864.0,21075840.0,21106816.0,21137792.0,21168768.0,20996224.0,21027264.0,21058304.0,21089344.0,21120384.0,21151424.0,21182464.0,21213504.0,21040512.0,21071616.0,21102720.0,21133824.0,21164928.0,21196032.0,21227136.0,21258240.0,21084800.0,21115968.0,21147136.0,21178304.0,21209472.0,21240640.0,21271808.0,21302976.0,21129088.0,21160320.0,21191552.0,21222784.0,21254016.0,21285248.0,21316480.0,21347712.0,21173376.0,21204672.0,21235968.0,21267264.0,21298560.0,21329856.0,21361152.0,21392448.0,21217664.0,21249024.0,21280384.0,21311744.0,21343104.0,21374464.0,21405824.0,21437184.0,21261952.0,21293376.0,21324800.0,21356224.0,21387648.0,21419072.0,21450496.0,21481920.0,21306240.0,21337728.0,21369216.0,21400704.0,21432192.0,21463680.0,21495168.0,21526656.0,21350528.0,21382080.0,21413632.0,21445184.0,21476736.0,21508288.0,21539840.0,21571392.0,21394816.0,21426432.0,21458048.0,21489664.0,21521280.0,21552896.0,21584512.0,21616128.0,21439104.0,21470784.0,21502464.0,21534144.0,21565824.0,21597504.0,21629184.0,21660864.0,21483392.0,21515136.0,21546880.0,21578624.0,21610368.0,21642112.0,21673856.0,21705600.0,21527680.0,21559488.0,21591296.0,21623104.0,21654912.0,21686720.0,21718528.0,21750336.0,21571968.0,21603840.0,21635712.0,21667584.0,21699456.0,21731328.0,21763200.0,21795072.0,21616256.0,21648192.0,21680128.0,21712064.0,21744000.0,21775936.0,21807872.0,21839808.0,21660544.0,21692544.0,21724544.0,21756544.0,21788544.0,21820544.0,21852544.0,21884544.0,21704832.0,21736896.0,21768960.0,21801024.0,21833088.0,21865152.0,21897216.0,21929280.0,21749120.0,21781248.0,21813376.0,21845504.0,21877632.0,21909760.0,21941888.0,21974016.0,21793408.0,21825600.0,21857792.0,21889984.0,21922176.0,21954368.0,21986560.0,22018752.0,21837696.0,21869952.0,21902208.0,21934464.0,21966720.0,21998976.0,22031232.0,22063488.0,21881984.0,21914304.0,21946624.0,21978944.0,22011264.0,22043584.0,22075904.0,22108224.0,21926272.0,21958656.0,21991040.0,22023424.0,22055808.0,22088192.0,22120576.0,22152960.0,21970560.0,22003008.0,22035456.0,22067904.0,22100352.0,22132800.0,22165248.0,22197696.0,22014848.0,22047360.0,22079872.0,22112384.0,22144896.0,22177408.0,22209920.0,22242432.0,22059136.0,22091712.0,22124288.0,22156864.0,22189440.0,22222016.0,22254592.0,22287168.0,22103424.0,22136064.0,22168704.0,22201344.0,22233984.0,22266624.0,22299264.0,22331904.0,22147712.0,22180416.0,22213120.0,22245824.0,22278528.0,22311232.0,22343936.0,22376640.0,22192000.0,22224768.0,22257536.0,22290304.0,22323072.0,22355840.0,22388608.0,22421376.0,22236288.0,22269120.0,22301952.0,22334784.0,22367616.0,22400448.0,22433280.0,22466112.0,22280576.0,22313472.0,22346368.0,22379264.0,22412160.0,22445056.0,22477952.0,22510848.0,22324864.0,22357824.0,22390784.0,22423744.0,22456704.0,22489664.0,22522624.0,22555584.0,22369152.0,22402176.0,22435200.0,22468224.0,22501248.0,22534272.0,22567296.0,22600320.0,22413440.0,22446528.0,22479616.0,22512704.0,22545792.0,22578880.0,22611968.0,22645056.0,22457728.0,22490880.0,22524032.0,22557184.0,22590336.0,22623488.0,22656640.0,22689792.0,22502016.0,22535232.0,22568448.0,22601664.0,22634880.0,22668096.0,22701312.0,22734528.0,22546304.0,22579584.0,22612864.0,22646144.0,22679424.0,22712704.0,22745984.0,22779264.0,22148224.0,22180544.0,22212864.0,22245184.0,22277504.0,22309824.0,22342144.0,22374464.0,21741952.0,21773312.0,21804672.0,21836032.0,21867392.0,21898752.0,21930112.0,21961472.0,21786240.0,21817664.0,21849088.0,21880512.0,21911936.0,21943360.0,21974784.0,22006208.0,21830528.0,21862016.0,21893504.0,21924992.0,21956480.0,21987968.0,22019456.0,22050944.0,21874816.0,21906368.0,21937920.0,21969472.0,22001024.0,22032576.0,22064128.0,22095680.0,21919104.0,21950720.0,21982336.0,22013952.0,22045568.0,22077184.0,22108800.0,22140416.0,21963392.0,21995072.0,22026752.0,22058432.0,22090112.0,22121792.0,22153472.0,22185152.0,22007680.0,22039424.0,22071168.0,22102912.0,22134656.0,22166400.0,22198144.0,22229888.0,22051968.0,22083776.0,22115584.0,22147392.0,22179200.0,22211008.0,22242816.0,22274624.0,22096256.0,22128128.0,22160000.0,22191872.0,22223744.0,22255616.0,22287488.0,22319360.0,22140544.0,22172480.0,22204416.0,22236352.0,22268288.0,22300224.0,22332160.0,22364096.0,22184832.0,22216832.0,22248832.0,22280832.0,22312832.0,22344832.0,22376832.0,22408832.0,22229120.0,22261184.0,22293248.0,22325312.0,22357376.0,22389440.0,22421504.0,22453568.0,22273408.0,22305536.0,22337664.0,22369792.0,22401920.0,22434048.0,22466176.0,22498304.0,22317696.0,22349888.0,22382080.0,22414272.0,22446464.0,22478656.0,22510848.0,22543040.0,22361984.0,22394240.0,22426496.0,22458752.0,22491008.0,22523264.0,22555520.0,22587776.0,22406272.0,22438592.0,22470912.0,22503232.0,22535552.0,22567872.0,22600192.0,22632512.0,22450560.0,22482944.0,22515328.0,22547712.0,22580096.0,22612480.0,22644864.0,22677248.0,22494848.0,22527296.0,22559744.0,22592192.0,22624640.0,22657088.0,22689536.0,22721984.0,22539136.0,22571648.0,22604160.0,22636672.0,22669184.0,22701696.0,22734208.0,22766720.0,22583424.0,22616000.0,22648576.0,22681152.0,22713728.0,22746304.0,22778880.0,22811456.0,22627712.0,22660352.0,22692992.0,22725632.0,22758272.0,22790912.0,22823552.0,22856192.0,22672000.0,22704704.0,22737408.0,22770112.0,22802816.0,22835520.0,22868224.0,22900928.0,22716288.0,22749056.0,22781824.0,22814592.0,22847360.0,22880128.0,22912896.0,22945664.0,22760576.0,22793408.0,22826240.0,22859072.0,22891904.0,22924736.0,22957568.0,22990400.0,22804864.0,22837760.0,22870656.0,22903552.0,22936448.0,22969344.0,23002240.0,23035136.0,22849152.0,22882112.0,22915072.0,22948032.0,22980992.0,23013952.0,23046912.0,23079872.0,22893440.0,22926464.0,22959488.0,22992512.0,23025536.0,23058560.0,23091584.0,23124608.0,22937728.0,22970816.0,23003904.0,23036992.0,23070080.0,23103168.0,23136256.0,23169344.0,22982016.0,23015168.0,23048320.0,23081472.0,23114624.0,23147776.0,23180928.0,23214080.0,23026304.0,23059520.0,23092736.0,23125952.0,23159168.0,23192384.0,23225600.0,23258816.0,23070592.0,23103872.0,23137152.0,23170432.0,23203712.0,23236992.0,23270272.0,23303552.0,23114880.0,23148224.0,23181568.0,23214912.0,23248256.0,23281600.0,23314944.0,23348288.0,23159168.0,23192576.0,23225984.0,23259392.0,23292800.0,23326208.0,23359616.0,23393024.0,23203456.0,23236928.0,23270400.0,23303872.0,23337344.0,23370816.0,23404288.0,23437760.0,23247744.0,23281280.0,23314816.0,23348352.0,23381888.0,23415424.0,23448960.0,23482496.0,23292032.0,23325632.0,23359232.0,23392832.0,23426432.0,23460032.0,23493632.0,23527232.0,23336320.0,23369984.0,23403648.0,23437312.0,23470976.0,23504640.0,23538304.0,23571968.0,23380608.0,23414336.0,23448064.0,23481792.0,23515520.0,23549248.0,23582976.0,23616704.0,22458240.0,22491008.0,22523776.0,22556544.0,22589312.0,22622080.0,22654848.0,22687616.0,21527680.0,21559488.0,21591296.0,21623104.0,21654912.0,21686720.0,21718528.0,21750336.0,10838656.0,10854576.0,10870496.0,10886416.0,10902336.0,10918256.0,10934176.0,10950096.0,11400192.0,11415984.0,11431776.0,11447568.0,11463360.0,11479152.0,11494944.0,11510736.0,22949248.0,22980864.0,23012480.0,23044096.0,23075712.0,23107328.0,23138944.0,23170560.0,22994560.0,23026240.0,23057920.0,23089600.0,23121280.0,23152960.0,23184640.0,23216320.0,23039872.0,23071616.0,23103360.0,23135104.0,23166848.0,23198592.0,23230336.0,23262080.0,23085184.0,23116992.0,23148800.0,23180608.0,23212416.0,23244224.0,23276032.0,23307840.0,23130496.0,23162368.0,23194240.0,23226112.0,23257984.0,23289856.0,23321728.0,23353600.0,23175808.0,23207744.0,23239680.0,23271616.0,23303552.0,23335488.0,23367424.0,23399360.0,23221120.0,23253120.0,23285120.0,23317120.0,23349120.0,23381120.0,23413120.0,23445120.0,23266432.0,23298496.0,23330560.0,23362624.0,23394688.0,23426752.0,23458816.0,23490880.0,23311744.0,23343872.0,23376000.0,23408128.0,23440256.0,23472384.0,23504512.0,23536640.0,23357056.0,23389248.0,23421440.0,23453632.0,23485824.0,23518016.0,23550208.0,23582400.0,23402368.0,23434624.0,23466880.0,23499136.0,23531392.0,23563648.0,23595904.0,23628160.0,23447680.0,23480000.0,23512320.0,23544640.0,23576960.0,23609280.0,23641600.0,23673920.0,23492992.0,23525376.0,23557760.0,23590144.0,23622528.0,23654912.0,23687296.0,23719680.0,23538304.0,23570752.0,23603200.0,23635648.0,23668096.0,23700544.0,23732992.0,23765440.0,23583616.0,23616128.0,23648640.0,23681152.0,23713664.0,23746176.0,23778688.0,23811200.0,23628928.0,23661504.0,23694080.0,23726656.0,23759232.0,23791808.0,23824384.0,23856960.0,23674240.0,23706880.0,23739520.0,23772160.0,23804800.0,23837440.0,23870080.0,23902720.0,23719552.0,23752256.0,23784960.0,23817664.0,23850368.0,23883072.0,23915776.0,23948480.0,23764864.0,23797632.0,23830400.0,23863168.0,23895936.0,23928704.0,23961472.0,23994240.0,23810176.0,23843008.0,23875840.0,23908672.0,23941504.0,23974336.0,24007168.0,24040000.0,23855488.0,23888384.0,23921280.0,23954176.0,23987072.0,24019968.0,24052864.0,24085760.0,23900800.0,23933760.0,23966720.0,23999680.0,24032640.0,24065600.0,24098560.0,24131520.0,23946112.0,23979136.0,24012160.0,24045184.0,24078208.0,24111232.0,24144256.0,24177280.0,23991424.0,24024512.0,24057600.0,24090688.0,24123776.0,24156864.0,24189952.0,24223040.0,24036736.0,24069888.0,24103040.0,24136192.0,24169344.0,24202496.0,24235648.0,24268800.0,24082048.0,24115264.0,24148480.0,24181696.0,24214912.0,24248128.0,24281344.0,24314560.0,24127360.0,24160640.0,24193920.0,24227200.0,24260480.0,24293760.0,24327040.0,24360320.0,24172672.0,24206016.0,24239360.0,24272704.0,24306048.0,24339392.0,24372736.0,24406080.0,24217984.0,24251392.0,24284800.0,24318208.0,24351616.0,24385024.0,24418432.0,24451840.0,24263296.0,24296768.0,24330240.0,24363712.0,24397184.0,24430656.0,24464128.0,24497600.0,24308608.0,24342144.0,24375680.0,24409216.0,24442752.0,24476288.0,24509824.0,24543360.0,24353920.0,24387520.0,24421120.0,24454720.0,24488320.0,24521920.0,24555520.0,24589120.0,24399232.0,24432896.0,24466560.0,24500224.0,24533888.0,24567552.0,24601216.0,24634880.0,24444544.0,24478272.0,24512000.0,24545728.0,24579456.0,24613184.0,24646912.0,24680640.0,24489856.0,24523648.0,24557440.0,24591232.0,24625024.0,24658816.0,24692608.0,24726400.0,24535168.0,24569024.0,24602880.0,24636736.0,24670592.0,24704448.0,24738304.0,24772160.0,23597440.0,23630336.0,23663232.0,23696128.0,23729024.0,23761920.0,23794816.0,23827712.0,22651520.0,22683456.0,22715392.0,22747328.0,22779264.0,22811200.0,22843136.0,22875072.0,21713792.0,21744768.0,21775744.0,21806720.0,21837696.0,21868672.0,21899648.0,21930624.0,20767872.0,20797888.0,20827904.0,20857920.0,20887936.0,20917952.0,20947968.0,20977984.0,20813184.0,20843264.0,20873344.0,20903424.0,20933504.0,20963584.0,20993664.0,21023744.0,20858496.0,20888640.0,20918784.0,20948928.0,20979072.0,21009216.0,21039360.0,21069504.0,20903808.0,20934016.0,20964224.0,20994432.0,21024640.0,21054848.0,21085056.0,21115264.0,20949120.0,20979392.0,21009664.0,21039936.0,21070208.0,21100480.0,21130752.0,21161024.0,20994432.0,21024768.0,21055104.0,21085440.0,21115776.0,21146112.0,21176448.0,21206784.0,21039744.0,21070144.0,21100544.0,21130944.0,21161344.0,21191744.0,21222144.0,21252544.0,21085056.0,21115520.0,21145984.0,21176448.0,21206912.0,21237376.0,21267840.0,21298304.0,21130368.0,21160896.0,21191424.0,21221952.0,21252480.0,21283008.0,21313536.0,21344064.0,21175680.0,21206272.0,21236864.0,21267456.0,21298048.0,21328640.0,21359232.0,21389824.0,21220992.0,21251648.0,21282304.0,21312960.0,21343616.0,21374272.0,21404928.0,21435584.0,21266304.0,21297024.0,21327744.0,21358464.0,21389184.0,21419904.0,21450624.0,21481344.0,21311616.0,21342400.0,21373184.0,21403968.0,21434752.0,21465536.0,21496320.0,21527104.0,21356928.0,21387776.0,21418624.0,21449472.0,21480320.0,21511168.0,21542016.0,21572864.0,21402240.0,21433152.0,21464064.0,21494976.0,21525888.0,21556800.0,21587712.0,21618624.0,21447552.0,21478528.0,21509504.0,21540480.0,21571456.0,21602432.0,21633408.0,21664384.0,21492864.0,21523904.0,21554944.0,21585984.0,21617024.0,21648064.0,21679104.0,21710144.0,21538176.0,21569280.0,21600384.0,21631488.0,21662592.0,21693696.0,21724800.0,21755904.0,21583488.0,21614656.0,21645824.0,21676992.0,21708160.0,21739328.0,21770496.0,21801664.0,21628800.0,21660032.0,21691264.0,21722496.0,21753728.0,21784960.0,21816192.0,21847424.0,21674112.0,21705408.0,21736704.0,21768000.0,21799296.0,21830592.0,21861888.0,21893184.0,21719424.0,21750784.0,21782144.0,21813504.0,21844864.0,21876224.0,21907584.0,21938944.0,21764736.0,21796160.0,21827584.0,21859008.0,21890432.0,21921856.0,21953280.0,21984704.0,21810048.0,21841536.0,21873024.0,21904512.0,21936000.0,21967488.0,21998976.0,22030464.0,21855360.0,21886912.0,21918464.0,21950016.0,21981568.0,22013120.0,22044672.0,22076224.0,21900672.0,21932288.0,21963904.0,21995520.0,22027136.0,22058752.0,22090368.0,22121984.0,21945984.0,21977664.0,22009344.0,22041024.0,22072704.0,22104384.0,22136064.0,22167744.0,21991296.0,22023040.0,22054784.0,22086528.0,22118272.0,22150016.0,22181760.0,22213504.0,22036608.0,22068416.0,22100224.0,22132032.0,22163840.0,22195648.0,22227456.0,22259264.0,22081920.0,22113792.0,22145664.0,22177536.0,22209408.0,22241280.0,22273152.0,22305024.0,22127232.0,22159168.0,22191104.0,22223040.0,22254976.0,22286912.0,22318848.0,22350784.0,22172544.0,22204544.0,22236544.0,22268544.0,22300544.0,22332544.0,22364544.0,22396544.0,22217856.0,22249920.0,22281984.0,22314048.0,22346112.0,22378176.0,22410240.0,22442304.0,22263168.0,22295296.0,22327424.0,22359552.0,22391680.0,22423808.0,22455936.0,22488064.0,22308480.0,22340672.0,22372864.0,22405056.0,22437248.0,22469440.0,22501632.0,22533824.0,22353792.0,22386048.0,22418304.0,22450560.0,22482816.0,22515072.0,22547328.0,22579584.0,22399104.0,22431424.0,22463744.0,22496064.0,22528384.0,22560704.0,22593024.0,22625344.0,22444416.0,22476800.0,22509184.0,22541568.0,22573952.0,22606336.0,22638720.0,22671104.0,22030976.0,22062400.0,22093824.0,22125248.0,22156672.0,22188096.0,22219520.0,22250944.0,21609344.0,21639808.0,21670272.0,21700736.0,21731200.0,21761664.0,21792128.0,21822592.0,21654656.0,21685184.0,21715712.0,21746240.0,21776768.0,21807296.0,21837824.0,21868352.0,21699968.0,21730560.0,21761152.0,21791744.0,21822336.0,21852928.0,21883520.0,21914112.0,21745280.0,21775936.0,21806592.0,21837248.0,21867904.0,21898560.0,21929216.0,21959872.0,21790592.0,21821312.0,21852032.0,21882752.0,21913472.0,21944192.0,21974912.0,22005632.0,21835904.0,21866688.0,21897472.0,21928256.0,21959040.0,21989824.0,22020608.0,22051392.0,21881216.0,21912064.0,21942912.0,21973760.0,22004608.0,22035456.0,22066304.0,22097152.0,21926528.0,21957440.0,21988352.0,22019264.0,22050176.0,22081088.0,22112000.0,22142912.0,21971840.0,22002816.0,22033792.0,22064768.0,22095744.0,22126720.0,22157696.0,22188672.0,22017152.0,22048192.0,22079232.0,22110272.0,22141312.0,22172352.0,22203392.0,22234432.0,22062464.0,22093568.0,22124672.0,22155776.0,22186880.0,22217984.0,22249088.0,22280192.0,22107776.0,22138944.0,22170112.0,22201280.0,22232448.0,22263616.0,22294784.0,22325952.0,22153088.0,22184320.0,22215552.0,22246784.0,22278016.0,22309248.0,22340480.0,22371712.0,22198400.0,22229696.0,22260992.0,22292288.0,22323584.0,22354880.0,22386176.0,22417472.0,22243712.0,22275072.0,22306432.0,22337792.0,22369152.0,22400512.0,22431872.0,22463232.0,22289024.0,22320448.0,22351872.0,22383296.0,22414720.0,22446144.0,22477568.0,22508992.0,22334336.0,22365824.0,22397312.0,22428800.0,22460288.0,22491776.0,22523264.0,22554752.0,22379648.0,22411200.0,22442752.0,22474304.0,22505856.0,22537408.0,22568960.0,22600512.0,22424960.0,22456576.0,22488192.0,22519808.0,22551424.0,22583040.0,22614656.0,22646272.0,22470272.0,22501952.0,22533632.0,22565312.0,22596992.0,22628672.0,22660352.0,22692032.0,22515584.0,22547328.0,22579072.0,22610816.0,22642560.0,22674304.0,22706048.0,22737792.0,22560896.0,22592704.0,22624512.0,22656320.0,22688128.0,22719936.0,22751744.0,22783552.0,22606208.0,22638080.0,22669952.0,22701824.0,22733696.0,22765568.0,22797440.0,22829312.0,22651520.0,22683456.0,22715392.0,22747328.0,22779264.0,22811200.0,22843136.0,22875072.0,22696832.0,22728832.0,22760832.0,22792832.0,22824832.0,22856832.0,22888832.0,22920832.0,22742144.0,22774208.0,22806272.0,22838336.0,22870400.0,22902464.0,22934528.0,22966592.0,22787456.0,22819584.0,22851712.0,22883840.0,22915968.0,22948096.0,22980224.0,23012352.0,22832768.0,22864960.0,22897152.0,22929344.0,22961536.0,22993728.0,23025920.0,23058112.0,22878080.0,22910336.0,22942592.0,22974848.0,23007104.0,23039360.0,23071616.0,23103872.0,22923392.0,22955712.0,22988032.0,23020352.0,23052672.0,23084992.0,23117312.0,23149632.0,22968704.0,23001088.0,23033472.0,23065856.0,23098240.0,23130624.0,23163008.0,23195392.0,23014016.0,23046464.0,23078912.0,23111360.0,23143808.0,23176256.0,23208704.0,23241152.0,23059328.0,23091840.0,23124352.0,23156864.0,23189376.0,23221888.0,23254400.0,23286912.0,23104640.0,23137216.0,23169792.0,23202368.0,23234944.0,23267520.0,23300096.0,23332672.0,23149952.0,23182592.0,23215232.0,23247872.0,23280512.0,23313152.0,23345792.0,23378432.0,23195264.0,23227968.0,23260672.0,23293376.0,23326080.0,23358784.0,23391488.0,23424192.0,23240576.0,23273344.0,23306112.0,23338880.0,23371648.0,23404416.0,23437184.0,23469952.0,23285888.0,23318720.0,23351552.0,23384384.0,23417216.0,23450048.0,23482880.0,23515712.0,22348160.0,22380032.0,22411904.0,22443776.0,22475648.0,22507520.0,22539392.0,22571264.0,21402240.0,21433152.0,21464064.0,21494976.0,21525888.0,21556800.0,21587712.0,21618624.0,21447552.0,21478528.0,21509504.0,21540480.0,21571456.0,21602432.0,21633408.0,21664384.0,21492864.0,21523904.0,21554944.0,21585984.0,21617024.0,21648064.0,21679104.0,21710144.0,21538176.0,21569280.0,21600384.0,21631488.0,21662592.0,21693696.0,21724800.0,21755904.0,21583488.0,21614656.0,21645824.0,21676992.0,21708160.0,21739328.0,21770496.0,21801664.0,21628800.0,21660032.0,21691264.0,21722496.0,21753728.0,21784960.0,21816192.0,21847424.0,21674112.0,21705408.0,21736704.0,21768000.0,21799296.0,21830592.0,21861888.0,21893184.0,21719424.0,21750784.0,21782144.0,21813504.0,21844864.0,21876224.0,21907584.0,21938944.0,21764736.0,21796160.0,21827584.0,21859008.0,21890432.0,21921856.0,21953280.0,21984704.0,21810048.0,21841536.0,21873024.0,21904512.0,21936000.0,21967488.0,21998976.0,22030464.0,21855360.0,21886912.0,21918464.0,21950016.0,21981568.0,22013120.0,22044672.0,22076224.0,21900672.0,21932288.0,21963904.0,21995520.0,22027136.0,22058752.0,22090368.0,22121984.0,21945984.0,21977664.0,22009344.0,22041024.0,22072704.0,22104384.0,22136064.0,22167744.0,21991296.0,22023040.0,22054784.0,22086528.0,22118272.0,22150016.0,22181760.0,22213504.0,22036608.0,22068416.0,22100224.0,22132032.0,22163840.0,22195648.0,22227456.0,22259264.0,22081920.0,22113792.0,22145664.0,22177536.0,22209408.0,22241280.0,22273152.0,22305024.0,22127232.0,22159168.0,22191104.0,22223040.0,22254976.0,22286912.0,22318848.0,22350784.0,22172544.0,22204544.0,22236544.0,22268544.0,22300544.0,22332544.0,22364544.0,22396544.0,22217856.0,22249920.0,22281984.0,22314048.0,22346112.0,22378176.0,22410240.0,22442304.0,22263168.0,22295296.0,22327424.0,22359552.0,22391680.0,22423808.0,22455936.0,22488064.0,22308480.0,22340672.0,22372864.0,22405056.0,22437248.0,22469440.0,22501632.0,22533824.0,22353792.0,22386048.0,22418304.0,22450560.0,22482816.0,22515072.0,22547328.0,22579584.0,22399104.0,22431424.0,22463744.0,22496064.0,22528384.0,22560704.0,22593024.0,22625344.0,22444416.0,22476800.0,22509184.0,22541568.0,22573952.0,22606336.0,22638720.0,22671104.0,22489728.0,22522176.0,22554624.0,22587072.0,22619520.0,22651968.0,22684416.0,22716864.0,22535040.0,22567552.0,22600064.0,22632576.0,22665088.0,22697600.0,22730112.0,22762624.0,22580352.0,22612928.0,22645504.0,22678080.0,22710656.0,22743232.0,22775808.0,22808384.0,22625664.0,22658304.0,22690944.0,22723584.0,22756224.0,22788864.0,22821504.0,22854144.0,22670976.0,22703680.0,22736384.0,22769088.0,22801792.0,22834496.0,22867200.0,22899904.0,22716288.0,22749056.0,22781824.0,22814592.0,22847360.0,22880128.0,22912896.0,22945664.0,22761600.0,22794432.0,22827264.0,22860096.0,22892928.0,22925760.0,22958592.0,22991424.0,22806912.0,22839808.0,22872704.0,22905600.0,22938496.0,22971392.0,23004288.0,23037184.0,22852224.0,22885184.0,22918144.0,22951104.0,22984064.0,23017024.0,23049984.0,23082944.0,22897536.0,22930560.0,22963584.0,22996608.0,23029632.0,23062656.0,23095680.0,23128704.0,22942848.0,22975936.0,23009024.0,23042112.0,23075200.0,23108288.0,23141376.0,23174464.0,22988160.0,23021312.0,23054464.0,23087616.0,23120768.0,23153920.0,23187072.0,23220224.0,23033472.0,23066688.0,23099904.0,23133120.0,23166336.0,23199552.0,23232768.0,23265984.0,23078784.0,23112064.0,23145344.0,23178624.0,23211904.0,23245184.0,23278464.0,23311744.0,22665344.0,22697664.0,22729984.0,22762304.0,22794624.0,22826944.0,22859264.0,22891584.0,22243712.0,22275072.0,22306432.0,22337792.0,22369152.0,22400512.0,22431872.0,22463232.0,22289024.0,22320448.0,22351872.0,22383296.0,22414720.0,22446144.0,22477568.0,22508992.0,22334336.0,22365824.0,22397312.0,22428800.0,22460288.0,22491776.0,22523264.0,22554752.0,22379648.0,22411200.0,22442752.0,22474304.0,22505856.0,22537408.0,22568960.0,22600512.0,22424960.0,22456576.0,22488192.0,22519808.0,22551424.0,22583040.0,22614656.0,22646272.0,22470272.0,22501952.0,22533632.0,22565312.0,22596992.0,22628672.0,22660352.0,22692032.0,22515584.0,22547328.0,22579072.0,22610816.0,22642560.0,22674304.0,22706048.0,22737792.0,22560896.0,22592704.0,22624512.0,22656320.0,22688128.0,22719936.0,22751744.0,22783552.0,22606208.0,22638080.0,22669952.0,22701824.0,22733696.0,22765568.0,22797440.0,22829312.0,22651520.0,22683456.0,22715392.0,22747328.0,22779264.0,22811200.0,22843136.0,22875072.0,22696832.0,22728832.0,22760832.0,22792832.0,22824832.0,22856832.0,22888832.0,22920832.0,22742144.0,22774208.0,22806272.0,22838336.0,22870400.0,22902464.0,22934528.0,22966592.0,22787456.0,22819584.0,22851712.0,22883840.0,22915968.0,22948096.0,22980224.0,23012352.0,22832768.0,22864960.0,22897152.0,22929344.0,22961536.0,22993728.0,23025920.0,23058112.0,22878080.0,22910336.0,22942592.0,22974848.0,23007104.0,23039360.0,23071616.0,23103872.0,22923392.0,22955712.0,22988032.0,23020352.0,23052672.0,23084992.0,23117312.0,23149632.0,22968704.0,23001088.0,23033472.0,23065856.0,23098240.0,23130624.0,23163008.0,23195392.0,23014016.0,23046464.0,23078912.0,23111360.0,23143808.0,23176256.0,23208704.0,23241152.0,23059328.0,23091840.0,23124352.0,23156864.0,23189376.0,23221888.0,23254400.0,23286912.0,23104640.0,23137216.0,23169792.0,23202368.0,23234944.0,23267520.0,23300096.0,23332672.0,23149952.0,23182592.0,23215232.0,23247872.0,23280512.0,23313152.0,23345792.0,23378432.0,23195264.0,23227968.0,23260672.0,23293376.0,23326080.0,23358784.0,23391488.0,23424192.0,23240576.0,23273344.0,23306112.0,23338880.0,23371648.0,23404416.0,23437184.0,23469952.0,23285888.0,23318720.0,23351552.0,23384384.0,23417216.0,23450048.0,23482880.0,23515712.0,23331200.0,23364096.0,23396992.0,23429888.0,23462784.0,23495680.0,23528576.0,23561472.0,23376512.0,23409472.0,23442432.0,23475392.0,23508352.0,23541312.0,23574272.0,23607232.0,23421824.0,23454848.0,23487872.0,23520896.0,23553920.0,23586944.0,23619968.0,23652992.0,23467136.0,23500224.0,23533312.0,23566400.0,23599488.0,23632576.0,23665664.0,23698752.0,23512448.0,23545600.0,23578752.0,23611904.0,23645056.0,23678208.0,23711360.0,23744512.0,23557760.0,23590976.0,23624192.0,23657408.0,23690624.0,23723840.0,23757056.0,23790272.0,23603072.0,23636352.0,23669632.0,23702912.0,23736192.0,23769472.0,23802752.0,23836032.0,23648384.0,23681728.0,23715072.0,23748416.0,23781760.0,23815104.0,23848448.0,23881792.0,23693696.0,23727104.0,23760512.0,23793920.0,23827328.0,23860736.0,23894144.0,23927552.0,23739008.0,23772480.0,23805952.0,23839424.0,23872896.0,23906368.0,23939840.0,23973312.0,23784320.0,23817856.0,23851392.0,23884928.0,23918464.0,23952000.0,23985536.0,24019072.0,23829632.0,23863232.0,23896832.0,23930432.0,23964032.0,23997632.0,24031232.0,24064832.0,23874944.0,23908608.0,23942272.0,23975936.0,24009600.0,24043264.0,24076928.0,24110592.0,23920256.0,23953984.0,23987712.0,24021440.0,24055168.0,24088896.0,24122624.0,24156352.0,22982528.0,23015296.0,23048064.0,23080832.0,23113600.0,23146368.0,23179136.0,23211904.0,22036608.0,22068416.0,22100224.0,22132032.0,22163840.0,22195648.0,22227456.0,22259264.0,11093376.0,11109296.0,11125216.0,11141136.0,11157056.0,11172976.0,11188896.0,11204816.0,11652864.0,11668656.0,11684448.0,11700240.0,11716032.0,11731824.0,11747616.0,11763408.0,23455104.0,23486720.0,23518336.0,23549952.0,23581568.0,23613184.0,23644800.0,23676416.0,23501440.0,23533120.0,23564800.0,23596480.0,23628160.0,23659840.0,23691520.0,23723200.0,23547776.0,23579520.0,23611264.0,23643008.0,23674752.0,23706496.0,23738240.0,23769984.0,23594112.0,23625920.0,23657728.0,23689536.0,23721344.0,23753152.0,23784960.0,23816768.0,23640448.0,23672320.0,23704192.0,23736064.0,23767936.0,23799808.0,23831680.0,23863552.0,23686784.0,23718720.0,23750656.0,23782592.0,23814528.0,23846464.0,23878400.0,23910336.0,23733120.0,23765120.0,23797120.0,23829120.0,23861120.0,23893120.0,23925120.0,23957120.0,23779456.0,23811520.0,23843584.0,23875648.0,23907712.0,23939776.0,23971840.0,24003904.0,23825792.0,23857920.0,23890048.0,23922176.0,23954304.0,23986432.0,24018560.0,24050688.0,23872128.0,23904320.0,23936512.0,23968704.0,24000896.0,24033088.0,24065280.0,24097472.0,23918464.0,23950720.0,23982976.0,24015232.0,24047488.0,24079744.0,24112000.0,24144256.0,23964800.0,23997120.0,24029440.0,24061760.0,24094080.0,24126400.0,24158720.0,24191040.0,24011136.0,24043520.0,24075904.0,24108288.0,24140672.0,24173056.0,24205440.0,24237824.0,24057472.0,24089920.0,24122368.0,24154816.0,24187264.0,24219712.0,24252160.0,24284608.0,24103808.0,24136320.0,24168832.0,24201344.0,24233856.0,24266368.0,24298880.0,24331392.0,24150144.0,24182720.0,24215296.0,24247872.0,24280448.0,24313024.0,24345600.0,24378176.0,24196480.0,24229120.0,24261760.0,24294400.0,24327040.0,24359680.0,24392320.0,24424960.0,24242816.0,24275520.0,24308224.0,24340928.0,24373632.0,24406336.0,24439040.0,24471744.0,24289152.0,24321920.0,24354688.0,24387456.0,24420224.0,24452992.0,24485760.0,24518528.0,24335488.0,24368320.0,24401152.0,24433984.0,24466816.0,24499648.0,24532480.0,24565312.0,24381824.0,24414720.0,24447616.0,24480512.0,24513408.0,24546304.0,24579200.0,24612096.0,24428160.0,24461120.0,24494080.0,24527040.0,24560000.0,24592960.0,24625920.0,24658880.0,24474496.0,24507520.0,24540544.0,24573568.0,24606592.0,24639616.0,24672640.0,24705664.0,24520832.0,24553920.0,24587008.0,24620096.0,24653184.0,24686272.0,24719360.0,24752448.0,24567168.0,24600320.0,24633472.0,24666624.0,24699776.0,24732928.0,24766080.0,24799232.0,24613504.0,24646720.0,24679936.0,24713152.0,24746368.0,24779584.0,24812800.0,24846016.0,24659840.0,24693120.0,24726400.0,24759680.0,24792960.0,24826240.0,24859520.0,24892800.0,24706176.0,24739520.0,24772864.0,24806208.0,24839552.0,24872896.0,24906240.0,24939584.0,24752512.0,24785920.0,24819328.0,24852736.0,24886144.0,24919552.0,24952960.0,24986368.0,24798848.0,24832320.0,24865792.0,24899264.0,24932736.0,24966208.0,24999680.0,25033152.0,24845184.0,24878720.0,24912256.0,24945792.0,24979328.0,25012864.0,25046400.0,25079936.0,24891520.0,24925120.0,24958720.0,24992320.0,25025920.0,25059520.0,25093120.0,25126720.0,24937856.0,24971520.0,25005184.0,25038848.0,25072512.0,25106176.0,25139840.0,25173504.0,24984192.0,25017920.0,25051648.0,25085376.0,25119104.0,25152832.0,25186560.0,25220288.0,25030528.0,25064320.0,25098112.0,25131904.0,25165696.0,25199488.0,25233280.0,25267072.0,25076864.0,25110720.0,25144576.0,25178432.0,25212288.0,25246144.0,25280000.0,25313856.0,24123776.0,24156672.0,24189568.0,24222464.0,24255360.0,24288256.0,24321152.0,24354048.0,23162496.0,23194432.0,23226368.0,23258304.0,23290240.0,23322176.0,23354112.0,23386048.0,22209408.0,22240384.0,22271360.0,22302336.0,22333312.0,22364288.0,22395264.0,22426240.0,21248128.0,21278144.0,21308160.0,21338176.0,21368192.0,21398208.0,21428224.0,21458240.0,21294464.0,21324544.0,21354624.0,21384704.0,21414784.0,21444864.0,21474944.0,21505024.0,21340800.0,21370944.0,21401088.0,21431232.0,21461376.0,21491520.0,21521664.0,21551808.0,21387136.0,21417344.0,21447552.0,21477760.0,21507968.0,21538176.0,21568384.0,21598592.0,21433472.0,21463744.0,21494016.0,21524288.0,21554560.0,21584832.0,21615104.0,21645376.0,21479808.0,21510144.0,21540480.0,21570816.0,21601152.0,21631488.0,21661824.0,21692160.0,21526144.0,21556544.0,21586944.0,21617344.0,21647744.0,21678144.0,21708544.0,21738944.0,21572480.0,21602944.0,21633408.0,21663872.0,21694336.0,21724800.0,21755264.0,21785728.0,21618816.0,21649344.0,21679872.0,21710400.0,21740928.0,21771456.0,21801984.0,21832512.0,21665152.0,21695744.0,21726336.0,21756928.0,21787520.0,21818112.0,21848704.0,21879296.0,21711488.0,21742144.0,21772800.0,21803456.0,21834112.0,21864768.0,21895424.0,21926080.0,21757824.0,21788544.0,21819264.0,21849984.0,21880704.0,21911424.0,21942144.0,21972864.0,21804160.0,21834944.0,21865728.0,21896512.0,21927296.0,21958080.0,21988864.0,22019648.0,21850496.0,21881344.0,21912192.0,21943040.0,21973888.0,22004736.0,22035584.0,22066432.0,21896832.0,21927744.0,21958656.0,21989568.0,22020480.0,22051392.0,22082304.0,22113216.0,21943168.0,21974144.0,22005120.0,22036096.0,22067072.0,22098048.0,22129024.0,22160000.0,21989504.0,22020544.0,22051584.0,22082624.0,22113664.0,22144704.0,22175744.0,22206784.0,22035840.0,22066944.0,22098048.0,22129152.0,22160256.0,22191360.0,22222464.0,22253568.0,22082176.0,22113344.0,22144512.0,22175680.0,22206848.0,22238016.0,22269184.0,22300352.0,22128512.0,22159744.0,22190976.0,22222208.0,22253440.0,22284672.0,22315904.0,22347136.0,22174848.0,22206144.0,22237440.0,22268736.0,22300032.0,22331328.0,22362624.0,22393920.0,22221184.0,22252544.0,22283904.0,22315264.0,22346624.0,22377984.0,22409344.0,22440704.0,22267520.0,22298944.0,22330368.0,22361792.0,22393216.0,22424640.0,22456064.0,22487488.0,22313856.0,22345344.0,22376832.0,22408320.0,22439808.0,22471296.0,22502784.0,22534272.0,22360192.0,22391744.0,22423296.0,22454848.0,22486400.0,22517952.0,22549504.0,22581056.0,22406528.0,22438144.0,22469760.0,22501376.0,22532992.0,22564608.0,22596224.0,22627840.0,22452864.0,22484544.0,22516224.0,22547904.0,22579584.0,22611264.0,22642944.0,22674624.0,22499200.0,22530944.0,22562688.0,22594432.0,22626176.0,22657920.0,22689664.0,22721408.0,22545536.0,22577344.0,22609152.0,22640960.0,22672768.0,22704576.0,22736384.0,22768192.0,22591872.0,22623744.0,22655616.0,22687488.0,22719360.0,22751232.0,22783104.0,22814976.0,22638208.0,22670144.0,22702080.0,22734016.0,22765952.0,22797888.0,22829824.0,22861760.0,22684544.0,22716544.0,22748544.0,22780544.0,22812544.0,22844544.0,22876544.0,22908544.0,22730880.0,22762944.0,22795008.0,22827072.0,22859136.0,22891200.0,22923264.0,22955328.0,22777216.0,22809344.0,22841472.0,22873600.0,22905728.0,22937856.0,22969984.0,23002112.0,22823552.0,22855744.0,22887936.0,22920128.0,22952320.0,22984512.0,23016704.0,23048896.0,22869888.0,22902144.0,22934400.0,22966656.0,22998912.0,23031168.0,23063424.0,23095680.0,22916224.0,22948544.0,22980864.0,23013184.0,23045504.0,23077824.0,23110144.0,23142464.0,22962560.0,22994944.0,23027328.0,23059712.0,23092096.0,23124480.0,23156864.0,23189248.0,22533760.0,22565184.0,22596608.0,22628032.0,22659456.0,22690880.0,22722304.0,22753728.0,22096768.0,22127232.0,22157696.0,22188160.0,22218624.0,22249088.0,22279552.0,22310016.0,22143104.0,22173632.0,22204160.0,22234688.0,22265216.0,22295744.0,22326272.0,22356800.0,22189440.0,22220032.0,22250624.0,22281216.0,22311808.0,22342400.0,22372992.0,22403584.0,22235776.0,22266432.0,22297088.0,22327744.0,22358400.0,22389056.0,22419712.0,22450368.0,22282112.0,22312832.0,22343552.0,22374272.0,22404992.0,22435712.0,22466432.0,22497152.0,22328448.0,22359232.0,22390016.0,22420800.0,22451584.0,22482368.0,22513152.0,22543936.0,22374784.0,22405632.0,22436480.0,22467328.0,22498176.0,22529024.0,22559872.0,22590720.0,22421120.0,22452032.0,22482944.0,22513856.0,22544768.0,22575680.0,22606592.0,22637504.0,22467456.0,22498432.0,22529408.0,22560384.0,22591360.0,22622336.0,22653312.0,22684288.0,22513792.0,22544832.0,22575872.0,22606912.0,22637952.0,22668992.0,22700032.0,22731072.0,22560128.0,22591232.0,22622336.0,22653440.0,22684544.0,22715648.0,22746752.0,22777856.0,22606464.0,22637632.0,22668800.0,22699968.0,22731136.0,22762304.0,22793472.0,22824640.0,22652800.0,22684032.0,22715264.0,22746496.0,22777728.0,22808960.0,22840192.0,22871424.0,22699136.0,22730432.0,22761728.0,22793024.0,22824320.0,22855616.0,22886912.0,22918208.0,22745472.0,22776832.0,22808192.0,22839552.0,22870912.0,22902272.0,22933632.0,22964992.0,22791808.0,22823232.0,22854656.0,22886080.0,22917504.0,22948928.0,22980352.0,23011776.0,22838144.0,22869632.0,22901120.0,22932608.0,22964096.0,22995584.0,23027072.0,23058560.0,22884480.0,22916032.0,22947584.0,22979136.0,23010688.0,23042240.0,23073792.0,23105344.0,22930816.0,22962432.0,22994048.0,23025664.0,23057280.0,23088896.0,23120512.0,23152128.0,22977152.0,23008832.0,23040512.0,23072192.0,23103872.0,23135552.0,23167232.0,23198912.0,23023488.0,23055232.0,23086976.0,23118720.0,23150464.0,23182208.0,23213952.0,23245696.0,23069824.0,23101632.0,23133440.0,23165248.0,23197056.0,23228864.0,23260672.0,23292480.0,23116160.0,23148032.0,23179904.0,23211776.0,23243648.0,23275520.0,23307392.0,23339264.0,23162496.0,23194432.0,23226368.0,23258304.0,23290240.0,23322176.0,23354112.0,23386048.0,23208832.0,23240832.0,23272832.0,23304832.0,23336832.0,23368832.0,23400832.0,23432832.0,23255168.0,23287232.0,23319296.0,23351360.0,23383424.0,23415488.0,23447552.0,23479616.0,23301504.0,23333632.0,23365760.0,23397888.0,23430016.0,23462144.0,23494272.0,23526400.0,23347840.0,23380032.0,23412224.0,23444416.0,23476608.0,23508800.0,23540992.0,23573184.0,23394176.0,23426432.0,23458688.0,23490944.0,23523200.0,23555456.0,23587712.0,23619968.0,23440512.0,23472832.0,23505152.0,23537472.0,23569792.0,23602112.0,23634432.0,23666752.0,23486848.0,23519232.0,23551616.0,23584000.0,23616384.0,23648768.0,23681152.0,23713536.0,23533184.0,23565632.0,23598080.0,23630528.0,23662976.0,23695424.0,23727872.0,23760320.0,23579520.0,23612032.0,23644544.0,23677056.0,23709568.0,23742080.0,23774592.0,23807104.0,23625856.0,23658432.0,23691008.0,23723584.0,23756160.0,23788736.0,23821312.0,23853888.0,23672192.0,23704832.0,23737472.0,23770112.0,23802752.0,23835392.0,23868032.0,23900672.0,23718528.0,23751232.0,23783936.0,23816640.0,23849344.0,23882048.0,23914752.0,23947456.0,23764864.0,23797632.0,23830400.0,23863168.0,23895936.0,23928704.0,23961472.0,23994240.0,23811200.0,23844032.0,23876864.0,23909696.0,23942528.0,23975360.0,24008192.0,24041024.0,22858112.0,22889984.0,22921856.0,22953728.0,22985600.0,23017472.0,23049344.0,23081216.0,21896832.0,21927744.0,21958656.0,21989568.0,22020480.0,22051392.0,22082304.0,22113216.0,21943168.0,21974144.0,22005120.0,22036096.0,22067072.0,22098048.0,22129024.0,22160000.0,21989504.0,22020544.0,22051584.0,22082624.0,22113664.0,22144704.0,22175744.0,22206784.0,22035840.0,22066944.0,22098048.0,22129152.0,22160256.0,22191360.0,22222464.0,22253568.0,22082176.0,22113344.0,22144512.0,22175680.0,22206848.0,22238016.0,22269184.0,22300352.0,22128512.0,22159744.0,22190976.0,22222208.0,22253440.0,22284672.0,22315904.0,22347136.0,22174848.0,22206144.0,22237440.0,22268736.0,22300032.0,22331328.0,22362624.0,22393920.0,22221184.0,22252544.0,22283904.0,22315264.0,22346624.0,22377984.0,22409344.0,22440704.0,22267520.0,22298944.0,22330368.0,22361792.0,22393216.0,22424640.0,22456064.0,22487488.0,22313856.0,22345344.0,22376832.0,22408320.0,22439808.0,22471296.0,22502784.0,22534272.0,22360192.0,22391744.0,22423296.0,22454848.0,22486400.0,22517952.0,22549504.0,22581056.0,22406528.0,22438144.0,22469760.0,22501376.0,22532992.0,22564608.0,22596224.0,22627840.0,22452864.0,22484544.0,22516224.0,22547904.0,22579584.0,22611264.0,22642944.0,22674624.0,22499200.0,22530944.0,22562688.0,22594432.0,22626176.0,22657920.0,22689664.0,22721408.0,22545536.0,22577344.0,22609152.0,22640960.0,22672768.0,22704576.0,22736384.0,22768192.0,22591872.0,22623744.0,22655616.0,22687488.0,22719360.0,22751232.0,22783104.0,22814976.0,22638208.0,22670144.0,22702080.0,22734016.0,22765952.0,22797888.0,22829824.0,22861760.0,22684544.0,22716544.0,22748544.0,22780544.0,22812544.0,22844544.0,22876544.0,22908544.0,22730880.0,22762944.0,22795008.0,22827072.0,22859136.0,22891200.0,22923264.0,22955328.0,22777216.0,22809344.0,22841472.0,22873600.0,22905728.0,22937856.0,22969984.0,23002112.0,22823552.0,22855744.0,22887936.0,22920128.0,22952320.0,22984512.0,23016704.0,23048896.0,22869888.0,22902144.0,22934400.0,22966656.0,22998912.0,23031168.0,23063424.0,23095680.0,22916224.0,22948544.0,22980864.0,23013184.0,23045504.0,23077824.0,23110144.0,23142464.0,22962560.0,22994944.0,23027328.0,23059712.0,23092096.0,23124480.0,23156864.0,23189248.0,23008896.0,23041344.0,23073792.0,23106240.0,23138688.0,23171136.0,23203584.0,23236032.0,23055232.0,23087744.0,23120256.0,23152768.0,23185280.0,23217792.0,23250304.0,23282816.0,23101568.0,23134144.0,23166720.0,23199296.0,23231872.0,23264448.0,23297024.0,23329600.0,23147904.0,23180544.0,23213184.0,23245824.0,23278464.0,23311104.0,23343744.0,23376384.0,23194240.0,23226944.0,23259648.0,23292352.0,23325056.0,23357760.0,23390464.0,23423168.0,23240576.0,23273344.0,23306112.0,23338880.0,23371648.0,23404416.0,23437184.0,23469952.0,23286912.0,23319744.0,23352576.0,23385408.0,23418240.0,23451072.0,23483904.0,23516736.0,23333248.0,23366144.0,23399040.0,23431936.0,23464832.0,23497728.0,23530624.0,23563520.0,23379584.0,23412544.0,23445504.0,23478464.0,23511424.0,23544384.0,23577344.0,23610304.0,23425920.0,23458944.0,23491968.0,23524992.0,23558016.0,23591040.0,23624064.0,23657088.0,23472256.0,23505344.0,23538432.0,23571520.0,23604608.0,23637696.0,23670784.0,23703872.0,23518592.0,23551744.0,23584896.0,23618048.0,23651200.0,23684352.0,23717504.0,23750656.0,23564928.0,23598144.0,23631360.0,23664576.0,23697792.0,23731008.0,23764224.0,23797440.0,23611264.0,23644544.0,23677824.0,23711104.0,23744384.0,23777664.0,23810944.0,23844224.0,23182464.0,23214784.0,23247104.0,23279424.0,23311744.0,23344064.0,23376384.0,23408704.0,22745472.0,22776832.0,22808192.0,22839552.0,22870912.0,22902272.0,22933632.0,22964992.0,22791808.0,22823232.0,22854656.0,22886080.0,22917504.0,22948928.0,22980352.0,23011776.0,22838144.0,22869632.0,22901120.0,22932608.0,22964096.0,22995584.0,23027072.0,23058560.0,22884480.0,22916032.0,22947584.0,22979136.0,23010688.0,23042240.0,23073792.0,23105344.0,22930816.0,22962432.0,22994048.0,23025664.0,23057280.0,23088896.0,23120512.0,23152128.0,22977152.0,23008832.0,23040512.0,23072192.0,23103872.0,23135552.0,23167232.0,23198912.0,23023488.0,23055232.0,23086976.0,23118720.0,23150464.0,23182208.0,23213952.0,23245696.0,23069824.0,23101632.0,23133440.0,23165248.0,23197056.0,23228864.0,23260672.0,23292480.0,23116160.0,23148032.0,23179904.0,23211776.0,23243648.0,23275520.0,23307392.0,23339264.0,23162496.0,23194432.0,23226368.0,23258304.0,23290240.0,23322176.0,23354112.0,23386048.0,23208832.0,23240832.0,23272832.0,23304832.0,23336832.0,23368832.0,23400832.0,23432832.0,23255168.0,23287232.0,23319296.0,23351360.0,23383424.0,23415488.0,23447552.0,23479616.0,23301504.0,23333632.0,23365760.0,23397888.0,23430016.0,23462144.0,23494272.0,23526400.0,23347840.0,23380032.0,23412224.0,23444416.0,23476608.0,23508800.0,23540992.0,23573184.0,23394176.0,23426432.0,23458688.0,23490944.0,23523200.0,23555456.0,23587712.0,23619968.0,23440512.0,23472832.0,23505152.0,23537472.0,23569792.0,23602112.0,23634432.0,23666752.0,23486848.0,23519232.0,23551616.0,23584000.0,23616384.0,23648768.0,23681152.0,23713536.0,23533184.0,23565632.0,23598080.0,23630528.0,23662976.0,23695424.0,23727872.0,23760320.0,23579520.0,23612032.0,23644544.0,23677056.0,23709568.0,23742080.0,23774592.0,23807104.0,23625856.0,23658432.0,23691008.0,23723584.0,23756160.0,23788736.0,23821312.0,23853888.0,23672192.0,23704832.0,23737472.0,23770112.0,23802752.0,23835392.0,23868032.0,23900672.0,23718528.0,23751232.0,23783936.0,23816640.0,23849344.0,23882048.0,23914752.0,23947456.0,23764864.0,23797632.0,23830400.0,23863168.0,23895936.0,23928704.0,23961472.0,23994240.0,23811200.0,23844032.0,23876864.0,23909696.0,23942528.0,23975360.0,24008192.0,24041024.0,23857536.0,23890432.0,23923328.0,23956224.0,23989120.0,24022016.0,24054912.0,24087808.0,23903872.0,23936832.0,23969792.0,24002752.0,24035712.0,24068672.0,24101632.0,24134592.0,23950208.0,23983232.0,24016256.0,24049280.0,24082304.0,24115328.0,24148352.0,24181376.0,23996544.0,24029632.0,24062720.0,24095808.0,24128896.0,24161984.0,24195072.0,24228160.0,24042880.0,24076032.0,24109184.0,24142336.0,24175488.0,24208640.0,24241792.0,24274944.0,24089216.0,24122432.0,24155648.0,24188864.0,24222080.0,24255296.0,24288512.0,24321728.0,24135552.0,24168832.0,24202112.0,24235392.0,24268672.0,24301952.0,24335232.0,24368512.0,24181888.0,24215232.0,24248576.0,24281920.0,24315264.0,24348608.0,24381952.0,24415296.0,24228224.0,24261632.0,24295040.0,24328448.0,24361856.0,24395264.0,24428672.0,24462080.0,24274560.0,24308032.0,24341504.0,24374976.0,24408448.0,24441920.0,24475392.0,24508864.0,24320896.0,24354432.0,24387968.0,24421504.0,24455040.0,24488576.0,24522112.0,24555648.0,24367232.0,24400832.0,24434432.0,24468032.0,24501632.0,24535232.0,24568832.0,24602432.0,24413568.0,24447232.0,24480896.0,24514560.0,24548224.0,24581888.0,24615552.0,24649216.0,24459904.0,24493632.0,24527360.0,24561088.0,24594816.0,24628544.0,24662272.0,24696000.0,23506816.0,23539584.0,23572352.0,23605120.0,23637888.0,23670656.0,23703424.0,23736192.0,22545536.0,22577344.0,22609152.0,22640960.0,22672768.0,22704576.0,22736384.0,22768192.0,11348096.0,11364016.0,11379936.0,11395856.0,11411776.0,11427696.0,11443616.0,11459536.0,11905536.0,11921328.0,11937120.0,11952912.0,11968704.0,11984496.0,12000288.0,12016080.0,23960960.0,23992576.0,24024192.0,24055808.0,24087424.0,24119040.0,24150656.0,24182272.0,24008320.0,24040000.0,24071680.0,24103360.0,24135040.0,24166720.0,24198400.0,24230080.0,24055680.0,24087424.0,24119168.0,24150912.0,24182656.0,24214400.0,24246144.0,24277888.0,24103040.0,24134848.0,24166656.0,24198464.0,24230272.0,24262080.0,24293888.0,24325696.0,24150400.0,24182272.0,24214144.0,24246016.0,24277888.0,24309760.0,24341632.0,24373504.0,24197760.0,24229696.0,24261632.0,24293568.0,24325504.0,24357440.0,24389376.0,24421312.0,24245120.0,24277120.0,24309120.0,24341120.0,24373120.0,24405120.0,24437120.0,24469120.0,24292480.0,24324544.0,24356608.0,24388672.0,24420736.0,24452800.0,24484864.0,24516928.0,24339840.0,24371968.0,24404096.0,24436224.0,24468352.0,24500480.0,24532608.0,24564736.0,24387200.0,24419392.0,24451584.0,24483776.0,24515968.0,24548160.0,24580352.0,24612544.0,24434560.0,24466816.0,24499072.0,24531328.0,24563584.0,24595840.0,24628096.0,24660352.0,24481920.0,24514240.0,24546560.0,24578880.0,24611200.0,24643520.0,24675840.0,24708160.0,24529280.0,24561664.0,24594048.0,24626432.0,24658816.0,24691200.0,24723584.0,24755968.0,24576640.0,24609088.0,24641536.0,24673984.0,24706432.0,24738880.0,24771328.0,24803776.0,24624000.0,24656512.0,24689024.0,24721536.0,24754048.0,24786560.0,24819072.0,24851584.0,24671360.0,24703936.0,24736512.0,24769088.0,24801664.0,24834240.0,24866816.0,24899392.0,24718720.0,24751360.0,24784000.0,24816640.0,24849280.0,24881920.0,24914560.0,24947200.0,24766080.0,24798784.0,24831488.0,24864192.0,24896896.0,24929600.0,24962304.0,24995008.0,24813440.0,24846208.0,24878976.0,24911744.0,24944512.0,24977280.0,25010048.0,25042816.0,24860800.0,24893632.0,24926464.0,24959296.0,24992128.0,25024960.0,25057792.0,25090624.0,24908160.0,24941056.0,24973952.0,25006848.0,25039744.0,25072640.0,25105536.0,25138432.0,24955520.0,24988480.0,25021440.0,25054400.0,25087360.0,25120320.0,25153280.0,25186240.0,25002880.0,25035904.0,25068928.0,25101952.0,25134976.0,25168000.0,25201024.0,25234048.0,25050240.0,25083328.0,25116416.0,25149504.0,25182592.0,25215680.0,25248768.0,25281856.0,25097600.0,25130752.0,25163904.0,25197056.0,25230208.0,25263360.0,25296512.0,25329664.0,25144960.0,25178176.0,25211392.0,25244608.0,25277824.0,25311040.0,25344256.0,25377472.0,25192320.0,25225600.0,25258880.0,25292160.0,25325440.0,25358720.0,25392000.0,25425280.0,25239680.0,25273024.0,25306368.0,25339712.0,25373056.0,25406400.0,25439744.0,25473088.0,25287040.0,25320448.0,25353856.0,25387264.0,25420672.0,25454080.0,25487488.0,25520896.0,25334400.0,25367872.0,25401344.0,25434816.0,25468288.0,25501760.0,25535232.0,25568704.0,25381760.0,25415296.0,25448832.0,25482368.0,25515904.0,25549440.0,25582976.0,25616512.0,25429120.0,25462720.0,25496320.0,25529920.0,25563520.0,25597120.0,25630720.0,25664320.0,25476480.0,25510144.0,25543808.0,25577472.0,25611136.0,25644800.0,25678464.0,25712128.0,25523840.0,25557568.0,25591296.0,25625024.0,25658752.0,25692480.0,25726208.0,25759936.0,25571200.0,25604992.0,25638784.0,25672576.0,25706368.0,25740160.0,25773952.0,25807744.0,25618560.0,25652416.0,25686272.0,25720128.0,25753984.0,25787840.0,25821696.0,25855552.0,24650112.0,24683008.0,24715904.0,24748800.0,24781696.0,24814592.0,24847488.0,24880384.0,23673472.0,23705408.0,23737344.0,23769280.0,23801216.0,23833152.0,23865088.0,23897024.0,22705024.0,22736000.0,22766976.0,22797952.0,22828928.0,22859904.0,22890880.0,22921856.0,21728384.0,21758400.0,21788416.0,21818432.0,21848448.0,21878464.0,21908480.0,21938496.0,21775744.0,21805824.0,21835904.0,21865984.0,21896064.0,21926144.0,21956224.0,21986304.0,21823104.0,21853248.0,21883392.0,21913536.0,21943680.0,21973824.0,22003968.0,22034112.0,21870464.0,21900672.0,21930880.0,21961088.0,21991296.0,22021504.0,22051712.0,22081920.0,21917824.0,21948096.0,21978368.0,22008640.0,22038912.0,22069184.0,22099456.0,22129728.0,21965184.0,21995520.0,22025856.0,22056192.0,22086528.0,22116864.0,22147200.0,22177536.0,22012544.0,22042944.0,22073344.0,22103744.0,22134144.0,22164544.0,22194944.0,22225344.0,22059904.0,22090368.0,22120832.0,22151296.0,22181760.0,22212224.0,22242688.0,22273152.0,22107264.0,22137792.0,22168320.0,22198848.0,22229376.0,22259904.0,22290432.0,22320960.0,22154624.0,22185216.0,22215808.0,22246400.0,22276992.0,22307584.0,22338176.0,22368768.0,22201984.0,22232640.0,22263296.0,22293952.0,22324608.0,22355264.0,22385920.0,22416576.0,22249344.0,22280064.0,22310784.0,22341504.0,22372224.0,22402944.0,22433664.0,22464384.0,22296704.0,22327488.0,22358272.0,22389056.0,22419840.0,22450624.0,22481408.0,22512192.0,22344064.0,22374912.0,22405760.0,22436608.0,22467456.0,22498304.0,22529152.0,22560000.0,22391424.0,22422336.0,22453248.0,22484160.0,22515072.0,22545984.0,22576896.0,22607808.0,22438784.0,22469760.0,22500736.0,22531712.0,22562688.0,22593664.0,22624640.0,22655616.0,22486144.0,22517184.0,22548224.0,22579264.0,22610304.0,22641344.0,22672384.0,22703424.0,22533504.0,22564608.0,22595712.0,22626816.0,22657920.0,22689024.0,22720128.0,22751232.0,22580864.0,22612032.0,22643200.0,22674368.0,22705536.0,22736704.0,22767872.0,22799040.0,22628224.0,22659456.0,22690688.0,22721920.0,22753152.0,22784384.0,22815616.0,22846848.0,22675584.0,22706880.0,22738176.0,22769472.0,22800768.0,22832064.0,22863360.0,22894656.0,22722944.0,22754304.0,22785664.0,22817024.0,22848384.0,22879744.0,22911104.0,22942464.0,22770304.0,22801728.0,22833152.0,22864576.0,22896000.0,22927424.0,22958848.0,22990272.0,22817664.0,22849152.0,22880640.0,22912128.0,22943616.0,22975104.0,23006592.0,23038080.0,22865024.0,22896576.0,22928128.0,22959680.0,22991232.0,23022784.0,23054336.0,23085888.0,22912384.0,22944000.0,22975616.0,23007232.0,23038848.0,23070464.0,23102080.0,23133696.0,22959744.0,22991424.0,23023104.0,23054784.0,23086464.0,23118144.0,23149824.0,23181504.0,23007104.0,23038848.0,23070592.0,23102336.0,23134080.0,23165824.0,23197568.0,23229312.0,23054464.0,23086272.0,23118080.0,23149888.0,23181696.0,23213504.0,23245312.0,23277120.0,23101824.0,23133696.0,23165568.0,23197440.0,23229312.0,23261184.0,23293056.0,23324928.0,23149184.0,23181120.0,23213056.0,23244992.0,23276928.0,23308864.0,23340800.0,23372736.0,23196544.0,23228544.0,23260544.0,23292544.0,23324544.0,23356544.0,23388544.0,23420544.0,23243904.0,23275968.0,23308032.0,23340096.0,23372160.0,23404224.0,23436288.0,23468352.0,23291264.0,23323392.0,23355520.0,23387648.0,23419776.0,23451904.0,23484032.0,23516160.0,23338624.0,23370816.0,23403008.0,23435200.0,23467392.0,23499584.0,23531776.0,23563968.0,23385984.0,23418240.0,23450496.0,23482752.0,23515008.0,23547264.0,23579520.0,23611776.0,23433344.0,23465664.0,23497984.0,23530304.0,23562624.0,23594944.0,23627264.0,23659584.0,23480704.0,23513088.0,23545472.0,23577856.0,23610240.0,23642624.0,23675008.0,23707392.0,23036544.0,23067968.0,23099392.0,23130816.0,23162240.0,23193664.0,23225088.0,23256512.0,22584192.0,22614656.0,22645120.0,22675584.0,22706048.0,22736512.0,22766976.0,22797440.0,22631552.0,22662080.0,22692608.0,22723136.0,22753664.0,22784192.0,22814720.0,22845248.0,22678912.0,22709504.0,22740096.0,22770688.0,22801280.0,22831872.0,22862464.0,22893056.0,22726272.0,22756928.0,22787584.0,22818240.0,22848896.0,22879552.0,22910208.0,22940864.0,22773632.0,22804352.0,22835072.0,22865792.0,22896512.0,22927232.0,22957952.0,22988672.0,22820992.0,22851776.0,22882560.0,22913344.0,22944128.0,22974912.0,23005696.0,23036480.0,22868352.0,22899200.0,22930048.0,22960896.0,22991744.0,23022592.0,23053440.0,23084288.0,22915712.0,22946624.0,22977536.0,23008448.0,23039360.0,23070272.0,23101184.0,23132096.0,22963072.0,22994048.0,23025024.0,23056000.0,23086976.0,23117952.0,23148928.0,23179904.0,23010432.0,23041472.0,23072512.0,23103552.0,23134592.0,23165632.0,23196672.0,23227712.0,23057792.0,23088896.0,23120000.0,23151104.0,23182208.0,23213312.0,23244416.0,23275520.0,23105152.0,23136320.0,23167488.0,23198656.0,23229824.0,23260992.0,23292160.0,23323328.0,23152512.0,23183744.0,23214976.0,23246208.0,23277440.0,23308672.0,23339904.0,23371136.0,23199872.0,23231168.0,23262464.0,23293760.0,23325056.0,23356352.0,23387648.0,23418944.0,23247232.0,23278592.0,23309952.0,23341312.0,23372672.0,23404032.0,23435392.0,23466752.0,23294592.0,23326016.0,23357440.0,23388864.0,23420288.0,23451712.0,23483136.0,23514560.0,23341952.0,23373440.0,23404928.0,23436416.0,23467904.0,23499392.0,23530880.0,23562368.0,23389312.0,23420864.0,23452416.0,23483968.0,23515520.0,23547072.0,23578624.0,23610176.0,23436672.0,23468288.0,23499904.0,23531520.0,23563136.0,23594752.0,23626368.0,23657984.0,23484032.0,23515712.0,23547392.0,23579072.0,23610752.0,23642432.0,23674112.0,23705792.0,23531392.0,23563136.0,23594880.0,23626624.0,23658368.0,23690112.0,23721856.0,23753600.0,23578752.0,23610560.0,23642368.0,23674176.0,23705984.0,23737792.0,23769600.0,23801408.0,23626112.0,23657984.0,23689856.0,23721728.0,23753600.0,23785472.0,23817344.0,23849216.0,23673472.0,23705408.0,23737344.0,23769280.0,23801216.0,23833152.0,23865088.0,23897024.0,23720832.0,23752832.0,23784832.0,23816832.0,23848832.0,23880832.0,23912832.0,23944832.0,23768192.0,23800256.0,23832320.0,23864384.0,23896448.0,23928512.0,23960576.0,23992640.0,23815552.0,23847680.0,23879808.0,23911936.0,23944064.0,23976192.0,24008320.0,24040448.0,23862912.0,23895104.0,23927296.0,23959488.0,23991680.0,24023872.0,24056064.0,24088256.0,23910272.0,23942528.0,23974784.0,24007040.0,24039296.0,24071552.0,24103808.0,24136064.0,23957632.0,23989952.0,24022272.0,24054592.0,24086912.0,24119232.0,24151552.0,24183872.0,24004992.0,24037376.0,24069760.0,24102144.0,24134528.0,24166912.0,24199296.0,24231680.0,24052352.0,24084800.0,24117248.0,24149696.0,24182144.0,24214592.0,24247040.0,24279488.0,24099712.0,24132224.0,24164736.0,24197248.0,24229760.0,24262272.0,24294784.0,24327296.0,24147072.0,24179648.0,24212224.0,24244800.0,24277376.0,24309952.0,24342528.0,24375104.0,24194432.0,24227072.0,24259712.0,24292352.0,24324992.0,24357632.0,24390272.0,24422912.0,24241792.0,24274496.0,24307200.0,24339904.0,24372608.0,24405312.0,24438016.0,24470720.0,24289152.0,24321920.0,24354688.0,24387456.0,24420224.0,24452992.0,24485760.0,24518528.0,24336512.0,24369344.0,24402176.0,24435008.0,24467840.0,24500672.0,24533504.0,24566336.0,23368064.0,23399936.0,23431808.0,23463680.0,23495552.0,23527424.0,23559296.0,23591168.0,22391424.0,22422336.0,22453248.0,22484160.0,22515072.0,22545984.0,22576896.0,22607808.0,22438784.0,22469760.0,22500736.0,22531712.0,22562688.0,22593664.0,22624640.0,22655616.0,22486144.0,22517184.0,22548224.0,22579264.0,22610304.0,22641344.0,22672384.0,22703424.0,22533504.0,22564608.0,22595712.0,22626816.0,22657920.0,22689024.0,22720128.0,22751232.0,22580864.0,22612032.0,22643200.0,22674368.0,22705536.0,22736704.0,22767872.0,22799040.0,22628224.0,22659456.0,22690688.0,22721920.0,22753152.0,22784384.0,22815616.0,22846848.0,22675584.0,22706880.0,22738176.0,22769472.0,22800768.0,22832064.0,22863360.0,22894656.0,22722944.0,22754304.0,22785664.0,22817024.0,22848384.0,22879744.0,22911104.0,22942464.0,22770304.0,22801728.0,22833152.0,22864576.0,22896000.0,22927424.0,22958848.0,22990272.0,22817664.0,22849152.0,22880640.0,22912128.0,22943616.0,22975104.0,23006592.0,23038080.0,22865024.0,22896576.0,22928128.0,22959680.0,22991232.0,23022784.0,23054336.0,23085888.0,22912384.0,22944000.0,22975616.0,23007232.0,23038848.0,23070464.0,23102080.0,23133696.0,22959744.0,22991424.0,23023104.0,23054784.0,23086464.0,23118144.0,23149824.0,23181504.0,23007104.0,23038848.0,23070592.0,23102336.0,23134080.0,23165824.0,23197568.0,23229312.0,23054464.0,23086272.0,23118080.0,23149888.0,23181696.0,23213504.0,23245312.0,23277120.0,23101824.0,23133696.0,23165568.0,23197440.0,23229312.0,23261184.0,23293056.0,23324928.0,23149184.0,23181120.0,23213056.0,23244992.0,23276928.0,23308864.0,23340800.0,23372736.0,23196544.0,23228544.0,23260544.0,23292544.0,23324544.0,23356544.0,23388544.0,23420544.0,23243904.0,23275968.0,23308032.0,23340096.0,23372160.0,23404224.0,23436288.0,23468352.0,23291264.0,23323392.0,23355520.0,23387648.0,23419776.0,23451904.0,23484032.0,23516160.0,23338624.0,23370816.0,23403008.0,23435200.0,23467392.0,23499584.0,23531776.0,23563968.0,23385984.0,23418240.0,23450496.0,23482752.0,23515008.0,23547264.0,23579520.0,23611776.0,23433344.0,23465664.0,23497984.0,23530304.0,23562624.0,23594944.0,23627264.0,23659584.0,23480704.0,23513088.0,23545472.0,23577856.0,23610240.0,23642624.0,23675008.0,23707392.0,23528064.0,23560512.0,23592960.0,23625408.0,23657856.0,23690304.0,23722752.0,23755200.0,23575424.0,23607936.0,23640448.0,23672960.0,23705472.0,23737984.0,23770496.0,23803008.0,23622784.0,23655360.0,23687936.0,23720512.0,23753088.0,23785664.0,23818240.0,23850816.0,23670144.0,23702784.0,23735424.0,23768064.0,23800704.0,23833344.0,23865984.0,23898624.0,23717504.0,23750208.0,23782912.0,23815616.0,23848320.0,23881024.0,23913728.0,23946432.0,23764864.0,23797632.0,23830400.0,23863168.0,23895936.0,23928704.0,23961472.0,23994240.0,23812224.0,23845056.0,23877888.0,23910720.0,23943552.0,23976384.0,24009216.0,24042048.0,23859584.0,23892480.0,23925376.0,23958272.0,23991168.0,24024064.0,24056960.0,24089856.0,23906944.0,23939904.0,23972864.0,24005824.0,24038784.0,24071744.0,24104704.0,24137664.0,23954304.0,23987328.0,24020352.0,24053376.0,24086400.0,24119424.0,24152448.0,24185472.0,24001664.0,24034752.0,24067840.0,24100928.0,24134016.0,24167104.0,24200192.0,24233280.0,24049024.0,24082176.0,24115328.0,24148480.0,24181632.0,24214784.0,24247936.0,24281088.0,24096384.0,24129600.0,24162816.0,24196032.0,24229248.0,24262464.0,24295680.0,24328896.0,24143744.0,24177024.0,24210304.0,24243584.0,24276864.0,24310144.0,24343424.0,24376704.0,23699584.0,23731904.0,23764224.0,23796544.0,23828864.0,23861184.0,23893504.0,23925824.0,23247232.0,23278592.0,23309952.0,23341312.0,23372672.0,23404032.0,23435392.0,23466752.0,23294592.0,23326016.0,23357440.0,23388864.0,23420288.0,23451712.0,23483136.0,23514560.0,23341952.0,23373440.0,23404928.0,23436416.0,23467904.0,23499392.0,23530880.0,23562368.0,23389312.0,23420864.0,23452416.0,23483968.0,23515520.0,23547072.0,23578624.0,23610176.0,23436672.0,23468288.0,23499904.0,23531520.0,23563136.0,23594752.0,23626368.0,23657984.0,23484032.0,23515712.0,23547392.0,23579072.0,23610752.0,23642432.0,23674112.0,23705792.0,23531392.0,23563136.0,23594880.0,23626624.0,23658368.0,23690112.0,23721856.0,23753600.0,23578752.0,23610560.0,23642368.0,23674176.0,23705984.0,23737792.0,23769600.0,23801408.0,23626112.0,23657984.0,23689856.0,23721728.0,23753600.0,23785472.0,23817344.0,23849216.0,23673472.0,23705408.0,23737344.0,23769280.0,23801216.0,23833152.0,23865088.0,23897024.0,23720832.0,23752832.0,23784832.0,23816832.0,23848832.0,23880832.0,23912832.0,23944832.0,23768192.0,23800256.0,23832320.0,23864384.0,23896448.0,23928512.0,23960576.0,23992640.0,23815552.0,23847680.0,23879808.0,23911936.0,23944064.0,23976192.0,24008320.0,24040448.0,23862912.0,23895104.0,23927296.0,23959488.0,23991680.0,24023872.0,24056064.0,24088256.0,23910272.0,23942528.0,23974784.0,24007040.0,24039296.0,24071552.0,24103808.0,24136064.0,23957632.0,23989952.0,24022272.0,24054592.0,24086912.0,24119232.0,24151552.0,24183872.0,24004992.0,24037376.0,24069760.0,24102144.0,24134528.0,24166912.0,24199296.0,24231680.0,24052352.0,24084800.0,24117248.0,24149696.0,24182144.0,24214592.0,24247040.0,24279488.0,24099712.0,24132224.0,24164736.0,24197248.0,24229760.0,24262272.0,24294784.0,24327296.0,24147072.0,24179648.0,24212224.0,24244800.0,24277376.0,24309952.0,24342528.0,24375104.0,24194432.0,24227072.0,24259712.0,24292352.0,24324992.0,24357632.0,24390272.0,24422912.0,24241792.0,24274496.0,24307200.0,24339904.0,24372608.0,24405312.0,24438016.0,24470720.0,24289152.0,24321920.0,24354688.0,24387456.0,24420224.0,24452992.0,24485760.0,24518528.0,24336512.0,24369344.0,24402176.0,24435008.0,24467840.0,24500672.0,24533504.0,24566336.0,24383872.0,24416768.0,24449664.0,24482560.0,24515456.0,24548352.0,24581248.0,24614144.0,24431232.0,24464192.0,24497152.0,24530112.0,24563072.0,24596032.0,24628992.0,24661952.0,24478592.0,24511616.0,24544640.0,24577664.0,24610688.0,24643712.0,24676736.0,24709760.0,24525952.0,24559040.0,24592128.0,24625216.0,24658304.0,24691392.0,24724480.0,24757568.0,24573312.0,24606464.0,24639616.0,24672768.0,24705920.0,24739072.0,24772224.0,24805376.0,24620672.0,24653888.0,24687104.0,24720320.0,24753536.0,24786752.0,24819968.0,24853184.0,24668032.0,24701312.0,24734592.0,24767872.0,24801152.0,24834432.0,24867712.0,24900992.0,24715392.0,24748736.0,24782080.0,24815424.0,24848768.0,24882112.0,24915456.0,24948800.0,24762752.0,24796160.0,24829568.0,24862976.0,24896384.0,24929792.0,24963200.0,24996608.0,24810112.0,24843584.0,24877056.0,24910528.0,24944000.0,24977472.0,25010944.0,25044416.0,24857472.0,24891008.0,24924544.0,24958080.0,24991616.0,25025152.0,25058688.0,25092224.0,24904832.0,24938432.0,24972032.0,25005632.0,25039232.0,25072832.0,25106432.0,25140032.0,24952192.0,24985856.0,25019520.0,25053184.0,25086848.0,25120512.0,25154176.0,25187840.0,24999552.0,25033280.0,25067008.0,25100736.0,25134464.0,25168192.0,25201920.0,25235648.0,24031104.0,24063872.0,24096640.0,24129408.0,24162176.0,24194944.0,24227712.0,24260480.0,23054464.0,23086272.0,23118080.0,23149888.0,23181696.0,23213504.0,23245312.0,23277120.0,11602816.0,11618736.0,11634656.0,11650576.0,11666496.0,11682416.0,11698336.0,11714256.0,12158208.0,12174000.0,12189792.0,12205584.0,12221376.0,12237168.0,12252960.0,12268752.0,24466816.0,24498432.0,24530048.0,24561664.0,24593280.0,24624896.0,24656512.0,24688128.0,24515200.0,24546880.0,24578560.0,24610240.0,24641920.0,24673600.0,24705280.0,24736960.0,24563584.0,24595328.0,24627072.0,24658816.0,24690560.0,24722304.0,24754048.0,24785792.0,24611968.0,24643776.0,24675584.0,24707392.0,24739200.0,24771008.0,24802816.0,24834624.0,24660352.0,24692224.0,24724096.0,24755968.0,24787840.0,24819712.0,24851584.0,24883456.0,24708736.0,24740672.0,24772608.0,24804544.0,24836480.0,24868416.0,24900352.0,24932288.0,24757120.0,24789120.0,24821120.0,24853120.0,24885120.0,24917120.0,24949120.0,24981120.0,24805504.0,24837568.0,24869632.0,24901696.0,24933760.0,24965824.0,24997888.0,25029952.0,24853888.0,24886016.0,24918144.0,24950272.0,24982400.0,25014528.0,25046656.0,25078784.0,24902272.0,24934464.0,24966656.0,24998848.0,25031040.0,25063232.0,25095424.0,25127616.0,24950656.0,24982912.0,25015168.0,25047424.0,25079680.0,25111936.0,25144192.0,25176448.0,24999040.0,25031360.0,25063680.0,25096000.0,25128320.0,25160640.0,25192960.0,25225280.0,25047424.0,25079808.0,25112192.0,25144576.0,25176960.0,25209344.0,25241728.0,25274112.0,25095808.0,25128256.0,25160704.0,25193152.0,25225600.0,25258048.0,25290496.0,25322944.0,25144192.0,25176704.0,25209216.0,25241728.0,25274240.0,25306752.0,25339264.0,25371776.0,25192576.0,25225152.0,25257728.0,25290304.0,25322880.0,25355456.0,25388032.0,25420608.0,25240960.0,25273600.0,25306240.0,25338880.0,25371520.0,25404160.0,25436800.0,25469440.0,25289344.0,25322048.0,25354752.0,25387456.0,25420160.0,25452864.0,25485568.0,25518272.0,25337728.0,25370496.0,25403264.0,25436032.0,25468800.0,25501568.0,25534336.0,25567104.0,25386112.0,25418944.0,25451776.0,25484608.0,25517440.0,25550272.0,25583104.0,25615936.0,25434496.0,25467392.0,25500288.0,25533184.0,25566080.0,25598976.0,25631872.0,25664768.0,25482880.0,25515840.0,25548800.0,25581760.0,25614720.0,25647680.0,25680640.0,25713600.0,25531264.0,25564288.0,25597312.0,25630336.0,25663360.0,25696384.0,25729408.0,25762432.0,25579648.0,25612736.0,25645824.0,25678912.0,25712000.0,25745088.0,25778176.0,25811264.0,25628032.0,25661184.0,25694336.0,25727488.0,25760640.0,25793792.0,25826944.0,25860096.0,25676416.0,25709632.0,25742848.0,25776064.0,25809280.0,25842496.0,25875712.0,25908928.0,25724800.0,25758080.0,25791360.0,25824640.0,25857920.0,25891200.0,25924480.0,25957760.0,25773184.0,25806528.0,25839872.0,25873216.0,25906560.0,25939904.0,25973248.0,26006592.0,25821568.0,25854976.0,25888384.0,25921792.0,25955200.0,25988608.0,26022016.0,26055424.0,25869952.0,25903424.0,25936896.0,25970368.0,26003840.0,26037312.0,26070784.0,26104256.0,25918336.0,25951872.0,25985408.0,26018944.0,26052480.0,26086016.0,26119552.0,26153088.0,25966720.0,26000320.0,26033920.0,26067520.0,26101120.0,26134720.0,26168320.0,26201920.0,26015104.0,26048768.0,26082432.0,26116096.0,26149760.0,26183424.0,26217088.0,26250752.0,26063488.0,26097216.0,26130944.0,26164672.0,26198400.0,26232128.0,26265856.0,26299584.0,26111872.0,26145664.0,26179456.0,26213248.0,26247040.0,26280832.0,26314624.0,26348416.0,26160256.0,26194112.0,26227968.0,26261824.0,26295680.0,26329536.0,26363392.0,26397248.0,25176448.0,25209344.0,25242240.0,25275136.0,25308032.0,25340928.0,25373824.0,25406720.0,24184448.0,24216384.0,24248320.0,24280256.0,24312192.0,24344128.0,24376064.0,24408000.0,23200640.0,23231616.0,23262592.0,23293568.0,23324544.0,23355520.0,23386496.0,23417472.0,22208640.0,22238656.0,22268672.0,22298688.0,22328704.0,22358720.0,22388736.0,22418752.0,22257024.0,22287104.0,22317184.0,22347264.0,22377344.0,22407424.0,22437504.0,22467584.0,22305408.0,22335552.0,22365696.0,22395840.0,22425984.0,22456128.0,22486272.0,22516416.0,22353792.0,22384000.0,22414208.0,22444416.0,22474624.0,22504832.0,22535040.0,22565248.0,22402176.0,22432448.0,22462720.0,22492992.0,22523264.0,22553536.0,22583808.0,22614080.0,22450560.0,22480896.0,22511232.0,22541568.0,22571904.0,22602240.0,22632576.0,22662912.0,22498944.0,22529344.0,22559744.0,22590144.0,22620544.0,22650944.0,22681344.0,22711744.0,22547328.0,22577792.0,22608256.0,22638720.0,22669184.0,22699648.0,22730112.0,22760576.0,22595712.0,22626240.0,22656768.0,22687296.0,22717824.0,22748352.0,22778880.0,22809408.0,22644096.0,22674688.0,22705280.0,22735872.0,22766464.0,22797056.0,22827648.0,22858240.0,22692480.0,22723136.0,22753792.0,22784448.0,22815104.0,22845760.0,22876416.0,22907072.0,22740864.0,22771584.0,22802304.0,22833024.0,22863744.0,22894464.0,22925184.0,22955904.0,22789248.0,22820032.0,22850816.0,22881600.0,22912384.0,22943168.0,22973952.0,23004736.0,22837632.0,22868480.0,22899328.0,22930176.0,22961024.0,22991872.0,23022720.0,23053568.0,22886016.0,22916928.0,22947840.0,22978752.0,23009664.0,23040576.0,23071488.0,23102400.0,22934400.0,22965376.0,22996352.0,23027328.0,23058304.0,23089280.0,23120256.0,23151232.0,22982784.0,23013824.0,23044864.0,23075904.0,23106944.0,23137984.0,23169024.0,23200064.0,23031168.0,23062272.0,23093376.0,23124480.0,23155584.0,23186688.0,23217792.0,23248896.0,23079552.0,23110720.0,23141888.0,23173056.0,23204224.0,23235392.0,23266560.0,23297728.0,23127936.0,23159168.0,23190400.0,23221632.0,23252864.0,23284096.0,23315328.0,23346560.0,23176320.0,23207616.0,23238912.0,23270208.0,23301504.0,23332800.0,23364096.0,23395392.0,23224704.0,23256064.0,23287424.0,23318784.0,23350144.0,23381504.0,23412864.0,23444224.0,23273088.0,23304512.0,23335936.0,23367360.0,23398784.0,23430208.0,23461632.0,23493056.0,23321472.0,23352960.0,23384448.0,23415936.0,23447424.0,23478912.0,23510400.0,23541888.0,23369856.0,23401408.0,23432960.0,23464512.0,23496064.0,23527616.0,23559168.0,23590720.0,23418240.0,23449856.0,23481472.0,23513088.0,23544704.0,23576320.0,23607936.0,23639552.0,23466624.0,23498304.0,23529984.0,23561664.0,23593344.0,23625024.0,23656704.0,23688384.0,23515008.0,23546752.0,23578496.0,23610240.0,23641984.0,23673728.0,23705472.0,23737216.0,23563392.0,23595200.0,23627008.0,23658816.0,23690624.0,23722432.0,23754240.0,23786048.0,23611776.0,23643648.0,23675520.0,23707392.0,23739264.0,23771136.0,23803008.0,23834880.0,23660160.0,23692096.0,23724032.0,23755968.0,23787904.0,23819840.0,23851776.0,23883712.0,23708544.0,23740544.0,23772544.0,23804544.0,23836544.0,23868544.0,23900544.0,23932544.0,23756928.0,23788992.0,23821056.0,23853120.0,23885184.0,23917248.0,23949312.0,23981376.0,23805312.0,23837440.0,23869568.0,23901696.0,23933824.0,23965952.0,23998080.0,24030208.0,23853696.0,23885888.0,23918080.0,23950272.0,23982464.0,24014656.0,24046848.0,24079040.0,23902080.0,23934336.0,23966592.0,23998848.0,24031104.0,24063360.0,24095616.0,24127872.0,23950464.0,23982784.0,24015104.0,24047424.0,24079744.0,24112064.0,24144384.0,24176704.0,23998848.0,24031232.0,24063616.0,24096000.0,24128384.0,24160768.0,24193152.0,24225536.0,23539328.0,23570752.0,23602176.0,23633600.0,23665024.0,23696448.0,23727872.0,23759296.0,23071616.0,23102080.0,23132544.0,23163008.0,23193472.0,23223936.0,23254400.0,23284864.0,23120000.0,23150528.0,23181056.0,23211584.0,23242112.0,23272640.0,23303168.0,23333696.0,23168384.0,23198976.0,23229568.0,23260160.0,23290752.0,23321344.0,23351936.0,23382528.0,23216768.0,23247424.0,23278080.0,23308736.0,23339392.0,23370048.0,23400704.0,23431360.0,23265152.0,23295872.0,23326592.0,23357312.0,23388032.0,23418752.0,23449472.0,23480192.0,23313536.0,23344320.0,23375104.0,23405888.0,23436672.0,23467456.0,23498240.0,23529024.0,23361920.0,23392768.0,23423616.0,23454464.0,23485312.0,23516160.0,23547008.0,23577856.0,23410304.0,23441216.0,23472128.0,23503040.0,23533952.0,23564864.0,23595776.0,23626688.0,23458688.0,23489664.0,23520640.0,23551616.0,23582592.0,23613568.0,23644544.0,23675520.0,23507072.0,23538112.0,23569152.0,23600192.0,23631232.0,23662272.0,23693312.0,23724352.0,23555456.0,23586560.0,23617664.0,23648768.0,23679872.0,23710976.0,23742080.0,23773184.0,23603840.0,23635008.0,23666176.0,23697344.0,23728512.0,23759680.0,23790848.0,23822016.0,23652224.0,23683456.0,23714688.0,23745920.0,23777152.0,23808384.0,23839616.0,23870848.0,23700608.0,23731904.0,23763200.0,23794496.0,23825792.0,23857088.0,23888384.0,23919680.0,23748992.0,23780352.0,23811712.0,23843072.0,23874432.0,23905792.0,23937152.0,23968512.0,23797376.0,23828800.0,23860224.0,23891648.0,23923072.0,23954496.0,23985920.0,24017344.0,23845760.0,23877248.0,23908736.0,23940224.0,23971712.0,24003200.0,24034688.0,24066176.0,23894144.0,23925696.0,23957248.0,23988800.0,24020352.0,24051904.0,24083456.0,24115008.0,23942528.0,23974144.0,24005760.0,24037376.0,24068992.0,24100608.0,24132224.0,24163840.0,23990912.0,24022592.0,24054272.0,24085952.0,24117632.0,24149312.0,24180992.0,24212672.0,24039296.0,24071040.0,24102784.0,24134528.0,24166272.0,24198016.0,24229760.0,24261504.0,24087680.0,24119488.0,24151296.0,24183104.0,24214912.0,24246720.0,24278528.0,24310336.0,24136064.0,24167936.0,24199808.0,24231680.0,24263552.0,24295424.0,24327296.0,24359168.0,24184448.0,24216384.0,24248320.0,24280256.0,24312192.0,24344128.0,24376064.0,24408000.0,24232832.0,24264832.0,24296832.0,24328832.0,24360832.0,24392832.0,24424832.0,24456832.0,24281216.0,24313280.0,24345344.0,24377408.0,24409472.0,24441536.0,24473600.0,24505664.0,24329600.0,24361728.0,24393856.0,24425984.0,24458112.0,24490240.0,24522368.0,24554496.0,24377984.0,24410176.0,24442368.0,24474560.0,24506752.0,24538944.0,24571136.0,24603328.0,24426368.0,24458624.0,24490880.0,24523136.0,24555392.0,24587648.0,24619904.0,24652160.0,24474752.0,24507072.0,24539392.0,24571712.0,24604032.0,24636352.0,24668672.0,24700992.0,24523136.0,24555520.0,24587904.0,24620288.0,24652672.0,24685056.0,24717440.0,24749824.0,24571520.0,24603968.0,24636416.0,24668864.0,24701312.0,24733760.0,24766208.0,24798656.0,24619904.0,24652416.0,24684928.0,24717440.0,24749952.0,24782464.0,24814976.0,24847488.0,24668288.0,24700864.0,24733440.0,24766016.0,24798592.0,24831168.0,24863744.0,24896320.0,24716672.0,24749312.0,24781952.0,24814592.0,24847232.0,24879872.0,24912512.0,24945152.0,24765056.0,24797760.0,24830464.0,24863168.0,24895872.0,24928576.0,24961280.0,24993984.0,24813440.0,24846208.0,24878976.0,24911744.0,24944512.0,24977280.0,25010048.0,25042816.0,24861824.0,24894656.0,24927488.0,24960320.0,24993152.0,25025984.0,25058816.0,25091648.0,23878016.0,23909888.0,23941760.0,23973632.0,24005504.0,24037376.0,24069248.0,24101120.0,22886016.0,22916928.0,22947840.0,22978752.0,23009664.0,23040576.0,23071488.0,23102400.0,22934400.0,22965376.0,22996352.0,23027328.0,23058304.0,23089280.0,23120256.0,23151232.0,22982784.0,23013824.0,23044864.0,23075904.0,23106944.0,23137984.0,23169024.0,23200064.0,23031168.0,23062272.0,23093376.0,23124480.0,23155584.0,23186688.0,23217792.0,23248896.0,23079552.0,23110720.0,23141888.0,23173056.0,23204224.0,23235392.0,23266560.0,23297728.0,23127936.0,23159168.0,23190400.0,23221632.0,23252864.0,23284096.0,23315328.0,23346560.0,23176320.0,23207616.0,23238912.0,23270208.0,23301504.0,23332800.0,23364096.0,23395392.0,23224704.0,23256064.0,23287424.0,23318784.0,23350144.0,23381504.0,23412864.0,23444224.0,23273088.0,23304512.0,23335936.0,23367360.0,23398784.0,23430208.0,23461632.0,23493056.0,23321472.0,23352960.0,23384448.0,23415936.0,23447424.0,23478912.0,23510400.0,23541888.0,23369856.0,23401408.0,23432960.0,23464512.0,23496064.0,23527616.0,23559168.0,23590720.0,23418240.0,23449856.0,23481472.0,23513088.0,23544704.0,23576320.0,23607936.0,23639552.0,23466624.0,23498304.0,23529984.0,23561664.0,23593344.0,23625024.0,23656704.0,23688384.0,23515008.0,23546752.0,23578496.0,23610240.0,23641984.0,23673728.0,23705472.0,23737216.0,23563392.0,23595200.0,23627008.0,23658816.0,23690624.0,23722432.0,23754240.0,23786048.0,23611776.0,23643648.0,23675520.0,23707392.0,23739264.0,23771136.0,23803008.0,23834880.0,23660160.0,23692096.0,23724032.0,23755968.0,23787904.0,23819840.0,23851776.0,23883712.0,23708544.0,23740544.0,23772544.0,23804544.0,23836544.0,23868544.0,23900544.0,23932544.0,23756928.0,23788992.0,23821056.0,23853120.0,23885184.0,23917248.0,23949312.0,23981376.0,23805312.0,23837440.0,23869568.0,23901696.0,23933824.0,23965952.0,23998080.0,24030208.0,23853696.0,23885888.0,23918080.0,23950272.0,23982464.0,24014656.0,24046848.0,24079040.0,23902080.0,23934336.0,23966592.0,23998848.0,24031104.0,24063360.0,24095616.0,24127872.0,23950464.0,23982784.0,24015104.0,24047424.0,24079744.0,24112064.0,24144384.0,24176704.0,23998848.0,24031232.0,24063616.0,24096000.0,24128384.0,24160768.0,24193152.0,24225536.0,24047232.0,24079680.0,24112128.0,24144576.0,24177024.0,24209472.0,24241920.0,24274368.0,24095616.0,24128128.0,24160640.0,24193152.0,24225664.0,24258176.0,24290688.0,24323200.0,24144000.0,24176576.0,24209152.0,24241728.0,24274304.0,24306880.0,24339456.0,24372032.0,24192384.0,24225024.0,24257664.0,24290304.0,24322944.0,24355584.0,24388224.0,24420864.0,24240768.0,24273472.0,24306176.0,24338880.0,24371584.0,24404288.0,24436992.0,24469696.0,24289152.0,24321920.0,24354688.0,24387456.0,24420224.0,24452992.0,24485760.0,24518528.0,24337536.0,24370368.0,24403200.0,24436032.0,24468864.0,24501696.0,24534528.0,24567360.0,24385920.0,24418816.0,24451712.0,24484608.0,24517504.0,24550400.0,24583296.0,24616192.0,24434304.0,24467264.0,24500224.0,24533184.0,24566144.0,24599104.0,24632064.0,24665024.0,24482688.0,24515712.0,24548736.0,24581760.0,24614784.0,24647808.0,24680832.0,24713856.0,24531072.0,24564160.0,24597248.0,24630336.0,24663424.0,24696512.0,24729600.0,24762688.0,24579456.0,24612608.0,24645760.0,24678912.0,24712064.0,24745216.0,24778368.0,24811520.0,24627840.0,24661056.0,24694272.0,24727488.0,24760704.0,24793920.0,24827136.0,24860352.0,24676224.0,24709504.0,24742784.0,24776064.0,24809344.0,24842624.0,24875904.0,24909184.0,24216704.0,24249024.0,24281344.0,24313664.0,24345984.0,24378304.0,24410624.0,24442944.0,23748992.0,23780352.0,23811712.0,23843072.0,23874432.0,23905792.0,23937152.0,23968512.0,23797376.0,23828800.0,23860224.0,23891648.0,23923072.0,23954496.0,23985920.0,24017344.0,23845760.0,23877248.0,23908736.0,23940224.0,23971712.0,24003200.0,24034688.0,24066176.0,23894144.0,23925696.0,23957248.0,23988800.0,24020352.0,24051904.0,24083456.0,24115008.0,23942528.0,23974144.0,24005760.0,24037376.0,24068992.0,24100608.0,24132224.0,24163840.0,23990912.0,24022592.0,24054272.0,24085952.0,24117632.0,24149312.0,24180992.0,24212672.0,24039296.0,24071040.0,24102784.0,24134528.0,24166272.0,24198016.0,24229760.0,24261504.0,24087680.0,24119488.0,24151296.0,24183104.0,24214912.0,24246720.0,24278528.0,24310336.0,24136064.0,24167936.0,24199808.0,24231680.0,24263552.0,24295424.0,24327296.0,24359168.0,24184448.0,24216384.0,24248320.0,24280256.0,24312192.0,24344128.0,24376064.0,24408000.0,24232832.0,24264832.0,24296832.0,24328832.0,24360832.0,24392832.0,24424832.0,24456832.0,24281216.0,24313280.0,24345344.0,24377408.0,24409472.0,24441536.0,24473600.0,24505664.0,24329600.0,24361728.0,24393856.0,24425984.0,24458112.0,24490240.0,24522368.0,24554496.0,24377984.0,24410176.0,24442368.0,24474560.0,24506752.0,24538944.0,24571136.0,24603328.0,24426368.0,24458624.0,24490880.0,24523136.0,24555392.0,24587648.0,24619904.0,24652160.0,24474752.0,24507072.0,24539392.0,24571712.0,24604032.0,24636352.0,24668672.0,24700992.0,24523136.0,24555520.0,24587904.0,24620288.0,24652672.0,24685056.0,24717440.0,24749824.0,24571520.0,24603968.0,24636416.0,24668864.0,24701312.0,24733760.0,24766208.0,24798656.0,24619904.0,24652416.0,24684928.0,24717440.0,24749952.0,24782464.0,24814976.0,24847488.0,24668288.0,24700864.0,24733440.0,24766016.0,24798592.0,24831168.0,24863744.0,24896320.0,24716672.0,24749312.0,24781952.0,24814592.0,24847232.0,24879872.0,24912512.0,24945152.0,24765056.0,24797760.0,24830464.0,24863168.0,24895872.0,24928576.0,24961280.0,24993984.0,24813440.0,24846208.0,24878976.0,24911744.0,24944512.0,24977280.0,25010048.0,25042816.0,24861824.0,24894656.0,24927488.0,24960320.0,24993152.0,25025984.0,25058816.0,25091648.0,24910208.0,24943104.0,24976000.0,25008896.0,25041792.0,25074688.0,25107584.0,25140480.0,24958592.0,24991552.0,25024512.0,25057472.0,25090432.0,25123392.0,25156352.0,25189312.0,25006976.0,25040000.0,25073024.0,25106048.0,25139072.0,25172096.0,25205120.0,25238144.0,25055360.0,25088448.0,25121536.0,25154624.0,25187712.0,25220800.0,25253888.0,25286976.0,25103744.0,25136896.0,25170048.0,25203200.0,25236352.0,25269504.0,25302656.0,25335808.0,25152128.0,25185344.0,25218560.0,25251776.0,25284992.0,25318208.0,25351424.0,25384640.0,25200512.0,25233792.0,25267072.0,25300352.0,25333632.0,25366912.0,25400192.0,25433472.0,25248896.0,25282240.0,25315584.0,25348928.0,25382272.0,25415616.0,25448960.0,25482304.0,25297280.0,25330688.0,25364096.0,25397504.0,25430912.0,25464320.0,25497728.0,25531136.0,25345664.0,25379136.0,25412608.0,25446080.0,25479552.0,25513024.0,25546496.0,25579968.0,25394048.0,25427584.0,25461120.0,25494656.0,25528192.0,25561728.0,25595264.0,25628800.0,25442432.0,25476032.0,25509632.0,25543232.0,25576832.0,25610432.0,25644032.0,25677632.0,25490816.0,25524480.0,25558144.0,25591808.0,25625472.0,25659136.0,25692800.0,25726464.0,25539200.0,25572928.0,25606656.0,25640384.0,25674112.0,25707840.0,25741568.0,25775296.0,24555392.0,24588160.0,24620928.0,24653696.0,24686464.0,24719232.0,24752000.0,24784768.0,23563392.0,23595200.0,23627008.0,23658816.0,23690624.0,23722432.0,23754240.0,23786048.0,11857536.0,11873456.0,11889376.0,11905296.0,11921216.0,11937136.0,11953056.0,11968976.0};

    printf("\nPerforming test:\n");

    bool passed = true;
    for(int i = 0; i < n_conv_transpose_1d_test_0; i++) {
        if(
            conv1d_transpose_data_0[i] != expected_conv1d_0[i]) {
            std::cout << "index: " << i << std::endl;
            std::cout << "expected: " << expected_conv1d_0[i] << std::endl;
            std::cout << "actual: " << conv1d_transpose_data_0[i] << std::endl;
            passed = false;
            break;
        }
    }

    printf("ggml_conv_1d_transpose (%d): %s\n", (int) ggml_nelements(conv1d_transpose_res_0), passed && (ggml_nelements(conv1d_transpose_res_0) == n_conv_transpose_1d_test_0) ? "\033[32mPASSED\033[0m" : "\033[31mFAILED\033[0m");

    passed = true;
    for(int i = 0; i < n_conv_transpose_1d_test_1; i++) {
        if(
            conv1d_transpose_data_1[i] != expected_conv1d_1[i]) {
            std::cout << "index: " << i << std::endl;
            std::cout << "expected: " << expected_conv1d_1[i] << std::endl;
            std::cout << "actual: " << conv1d_transpose_data_1[i] << std::endl;
            passed = false;
        }
    }

    printf("ggml_conv_1d_transpose (%d): %s\n", (int) ggml_nelements(conv1d_transpose_res_1), passed && (ggml_nelements(conv1d_transpose_res_1) == n_conv_transpose_1d_test_1) ? "\033[32mPASSED\033[0m" : "\033[31mFAILED\033[0m");

    passed = true;
    for(int i = 0; i < n_conv_transpose_1d_test_2; i++) {
        if(
            conv1d_transpose_data_2[i] != expected_conv1d_2[i]) {
            std::cout << "index: " << i << std::endl;
            std::cout << "expected: " << expected_conv1d_2[i] << std::endl;
            std::cout << "actual: " << conv1d_transpose_data_2[i] << std::endl;
            passed = false;
        }
    }

    printf("ggml_conv_1d_transpose (%d): %s\n", (int) ggml_nelements(conv1d_transpose_res_2), passed && (ggml_nelements(conv1d_transpose_res_2) == n_conv_transpose_1d_test_2) ? "\033[32mPASSED\033[0m" : "\033[31mFAILED\033[0m");


    passed = true;
    for(int i = 0; i < n_conv_transpose_1d_test_3; i++) {
        if(
            conv1d_transpose_data_3[i] != expected_conv1d_3[i]) {
            std::cout << "index: " << i << std::endl;
            std::cout << "expected: " << expected_conv1d_3[i] << std::endl;
            std::cout << "actual: " << conv1d_transpose_data_3[i] << std::endl;
            passed = false;
        }
    }

    printf("ggml_conv_1d_transpose (%d): %s\n", (int) ggml_nelements(conv1d_transpose_res_3), passed && (ggml_nelements(conv1d_transpose_res_3) == n_conv_transpose_1d_test_3) ? "\033[32mPASSED\033[0m" : "\033[31mFAILED\033[0m");

    passed = true;
    for(int i = 0; i < n_conv_transpose_1d_test_4; i++) {
        if(
            conv1d_transpose_data_4[i] != expected_conv1d_4[i]) {
            std::cout << "index: " << i << std::endl;
            std::cout << "expected: " << expected_conv1d_4[i] << std::endl;
            std::cout << "actual: " << conv1d_transpose_data_4[i] << std::endl;
            passed = false;
        }
    }

    printf("ggml_conv_1d_transpose (%d): %s\n", (int) ggml_nelements(conv1d_transpose_res_4), passed && (ggml_nelements(conv1d_transpose_res_4) == n_conv_transpose_1d_test_4) ? "\033[32mPASSED\033[0m" : "\033[31mFAILED\033[0m");

    passed = true;
    for(int i = 0; i < n_conv_transpose_1d_test_5; i++) {
        if(
            conv1d_transpose_data_5[i] != expected_conv1d_5[i]) {
            std::cout << "index: " << i << std::endl;
            std::cout << "expected: " << expected_conv1d_5[i] << std::endl;
            std::cout << "actual: " << conv1d_transpose_data_5[i] << std::endl;
            passed = false;
        }
    }

    printf("ggml_conv_1d_transpose (%d): %s\n", (int) ggml_nelements(conv1d_transpose_res_5), passed && (ggml_nelements(conv1d_transpose_res_5) == n_conv_transpose_1d_test_5) ? "\033[32mPASSED\033[0m" : "\033[31mFAILED\033[0m");

    passed = true;
    for(int i = 0; i < n_conv_transpose_1d_test_6; i++) {
        if(
            conv1d_transpose_data_6[i] != expected_conv1d_6[i]) {
            std::cout << "index: " << i << std::endl;
            std::cout << "expected: " << expected_conv1d_6[i] << std::endl;
            std::cout << "actual: " << conv1d_transpose_data_6[i] << std::endl;
            passed = false;
        }
    }


    printf("ggml_conv_1d_transpose (%d): %s\n", (int) ggml_nelements(conv1d_transpose_res_6), passed && (ggml_nelements(conv1d_transpose_res_6) == n_conv_transpose_1d_test_6) ? "\033[32mPASSED\033[0m" : "\033[31mFAILED\033[0m");


    passed = true;
    for(int i = 0; i < n_conv_transpose_1d_test_7; i++) {
        if(
            fabs(conv1d_transpose_data_7[i] - expected_conv1d_7[i])/fabs(expected_conv1d_7[i]) > .000001) {
            std::cout << "index: " << i << std::endl;
            std::cout << "expected: " << expected_conv1d_7[i] << std::endl;
            std::cout << "actual: " << conv1d_transpose_data_7[i] << std::endl;
            passed = false;
        }
    }


    printf("ggml_conv_1d_transpose (%d): %s\n", (int) ggml_nelements(conv1d_transpose_res_7), passed && (ggml_nelements(conv1d_transpose_res_7) == n_conv_transpose_1d_test_7) ? "\033[32mPASSED\033[0m" : "\033[31mFAILED\033[0m");


    ggml_free(model.ctx);

    ggml_backend_buffer_free(model.buffer);
    ggml_backend_free(model.backend);
    ggml_gallocr_free(allocr);

    if (passed) {
        return 0;
    }
    return 1;
}
