# Copyright 2016 NTT DATA
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from oslo_config import cfg

wsgi_group = cfg.OptGroup(
    'wsgi',
    title='WSGI Options')

api_paste_config = cfg.StrOpt(
    'api_paste_config',
    default="api-paste.ini",
    help='File name for the paste.deploy config for masakari-api')

# TODO(abhishekk): It is not possible to rename this to 'log_format'
# yet, as doing so would cause a conflict if '[DEFAULT] log_format'
# were used. When 'deprecated_group' is removed after Ocata, this
# should be changed.
wsgi_log_format = cfg.StrOpt(
    'wsgi_log_format',
    default='%(client_ip)s "%(request_line)s" status: %(status_code)s'
            ' len: %(body_length)s time: %(wall_seconds).7f',
    help='A python format string that is used as the template to '
         'generate log lines. The following values can be formatted '
         'into it: client_ip, date_time, request_line, status_code, '
         'body_length, wall_seconds.')

secure_proxy_ssl_header = cfg.StrOpt(
    'secure_proxy_ssl_header',
    deprecated_for_removal=True,
    deprecated_reason='The functionality of this parameter is duplicate of '
                      'the http_proxy_to_wsgi middleware of oslo.middleware '
                      'and will be completely replaced.',
    help='The HTTP header used to determine the scheme for the '
         'original request, even if it was removed by an SSL '
         'terminating proxy. Typical value is '
         '"HTTP_X_FORWARDED_PROTO".')

ssl_ca_file = cfg.StrOpt(
    'ssl_ca_file',
    help='CA certificate file to use to verify connecting clients')

ssl_cert_file = cfg.StrOpt(
    'ssl_cert_file',
    help='SSL certificate of API server')

ssl_key_file = cfg.StrOpt(
    'ssl_key_file',
    help='SSL private key of API server')

tcp_keepidle = cfg.IntOpt(
    'tcp_keepidle',
    default=600,
    help='Sets the value of TCP_KEEPIDLE in seconds for each '
         'server socket. Not supported on OS X.')

default_pool_size = cfg.IntOpt(
    'default_pool_size',
    default=1000,
    help='Size of the pool of greenthreads used by wsgi')

max_header_line = cfg.IntOpt(
    'max_header_line',
    default=16384,
    help='Maximum line size of message headers to be accepted. '
         'max_header_line may need to be increased when using '
         'large tokens (typically those generated by the '
         'Keystone v3 API with big service catalogs).')

keep_alive = cfg.BoolOpt(
    'keep_alive',
    default=True,
    help='If False, closes the client socket connection explicitly.')

client_socket_timeout = cfg.IntOpt(
    'client_socket_timeout',
    default=900,
    help="Timeout for client connections' socket operations. "
         "If an incoming connection is idle for this number of "
         "seconds it will be closed. A value of '0' means "
         "wait forever.")

ALL_OPTS = [api_paste_config,
            wsgi_log_format,
            secure_proxy_ssl_header,
            ssl_ca_file,
            ssl_cert_file,
            ssl_key_file,
            tcp_keepidle,
            default_pool_size,
            max_header_line,
            keep_alive,
            client_socket_timeout
            ]


def register_opts(conf):
    conf.register_group(wsgi_group)
    conf.register_opts(ALL_OPTS, group=wsgi_group)


def list_opts():
    return {wsgi_group: ALL_OPTS}
